// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "filter/EngineNodeDatabaseFilter.h"

#include "utils/Database.h"
#include "utils/ElementUtils.h"
#include "utils/Logger.h"

EngineNodeDatabaseFilter::~EngineNodeDatabaseFilter()
{}

void EngineNodeDatabaseFilter::run(Database& db)
{
    // First pass, obtain the hkeNode base class definition and use it for comparison instead of string comparison.
    const clang::CXXRecordDecl* hkeNodeDef = db.findRecordDefinition("hkeNode");

    if(hkeNodeDef == NULL)
    {
        // hkeNode wasn't found in any of the included files, this project can't include
        // new engine nodes because they should inherit from it.
        return;
    }

    // For every file in the database.
    for(Database::FileDeclMapType::const_iterator fit = db.inputFilesDecls.begin();
        fit != db.inputFilesDecls.end();
        ++fit)
    {
        const Database::DeclListType& declList = fit->second.decls;

        // For every named declaration.
        for(Database::DeclListType::const_iterator dit = declList.begin();
            dit != declList.end();
            ++dit)
        {
            if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(*dit))
            {
                const std::string& nodeName = recDecl->getQualifiedNameAsString();

                if (stringStartsWith(nodeName, "hkReflect::Typedef::"))
                {
                    continue;
                }

                // Check if it is an hkeNode (only explore the chain of main parents).
                bool isNode = definitionIsA(hkeNodeDef, recDecl);

                if(!isNode)
                {
                    continue;
                }

                hkLog.info("Found hkeNode: %s", nodeName.c_str());

                // 1: Look for the hke::EngineNode attribute, should always be used on hkeNodes as HK_DECLARE_CLASS_EngineNode should be used.
                {
                    bool markedAsEngineNode = db.getAttributeValue(recDecl, "hke::EngineNode", false);
                    if(!markedAsEngineNode)
                    {
                        hkLog.error(recDecl, "'%s' Add 'EngineNode' to the HK_DECLARE_CLASS() macro", nodeName.c_str());
                        continue;
                    }
                }

                // 2: Examine all functions, the rules for functions are as follows.
                //
                // Static functions are not treated specially.
                // Non-static functions are treated differently depending on whether they are public or not:
                // a) Non-public non-static functions are considered internal implementation functions, they are not
                //    reflected unless they are explicitly marked for reflection.
                // b) Public non-static functions are considered part of the interface available for users and as such they
                //    are automatically reflected unless they are explicitly marked to prevent reflection.
                //    If one of these reflected functions is successfully matched against one of the available message types,
                //    we also mark the function as a handler for the specific message type. If the function is explicitly marked
                //    as a message handler this automatic handler deduction is skipped. Regardless of how the function was
                //    discovered to be a message handler, we check to make sure that the function is virtual. If the function is
                //    not virtual, it is a potential source of trouble since message dispatching will always be polymorphic and
                //    dispatch to the exact type, therefore calling a virtual function or calling sendMessage() would be different,
                //    which we want to avoid. Errors will be thrown if the above criteria fails.
                {
                    // Check if methods are reflected/unreflected at class level
                    auto detailsAttr = db.getAttributeValue< Optional<Attribute> >(recDecl, "hk::ReflectDetails");
                    Optional<bool> explicitReflectAllMethods = detailsAttr ?
                        Optional<bool>((*detailsAttr)["methods"]->as<bool>()) :
                        Optional<bool>();
                    for(clang::CXXRecordDecl::method_iterator it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
                    {
                        clang::CXXMethodDecl* method = *it;
                        if(method->isStatic())
                        {
                            // Skip static methods.
                            continue;
                        }

                        clang::AccessSpecifier accessSpecifier = method->getAccess();

                        bool isCtor = false;
                        bool isDtor = false;
                        if( const clang::CXXConstructorDecl* ctor = clang::dyn_cast<clang::CXXConstructorDecl>(method) )
                        {
                            isCtor = true;
                        }
                        else if( const clang::CXXDestructorDecl* dtor = clang::dyn_cast<clang::CXXDestructorDecl>(method) )
                        {
                            isDtor = true;
                        }

                        bool reflectedMethod = false;

                        // Consider the method explicitly reflected if the method
                        // itself or the class are explicitly marked
                        auto explicitReflectCurrent = db.getAttributeValue< Optional<bool> >(method, "hk::Reflect");
                        if (!explicitReflectCurrent)
                            explicitReflectCurrent = explicitReflectAllMethods;

                        // If it is non-public, we don't do anything as normally methods are not reflected.
                        // It will be reflected only if there is additional generic markup to force the reflection.
                        if(accessSpecifier != clang::AS_public)
                        {
                            // Non-public non-static method. Reflect only if explicitly reflected.
                            bool explicitlyReflected = explicitReflectCurrent && *explicitReflectCurrent;
                            if(!method->isImplicit() && explicitlyReflected)
                            {
                                reflectedMethod = true;
                            }
                        }
                        else
                        {
                            // It is a public non-static method.
                            // Reflect by default unless explicitly non-reflected.
                            bool explicitlyNonReflected = explicitReflectCurrent && *explicitReflectCurrent == false;
                            if(!method->isImplicit() && !explicitlyNonReflected)
                            {
                                if( isCtor && isAbstract(recDecl) )
                                {
                                    hkLog.info("hkeNode %s public constructor not reflected as class is abstract.", nodeName.c_str());
                                }
                                else
                                {
                                    // If the method is not implicit and it's not explicitly non-reflected, we get in here.
                                    // Check that the return value and all the parameters are reflected.
                                    reflectedMethod = db.isReflected(method->getResultType());
                                    if(reflectedMethod == true)
                                    {
                                        for(clang::CXXMethodDecl::param_const_iterator p = method->param_begin(); p != method->param_end(); ++p)
                                        {
                                            if(!db.isReflected((*p)->getType()))
                                            {
                                                reflectedMethod = false;
                                                break;
                                            }
                                        }
                                    }

                                    if(reflectedMethod)
                                    {
                                        // Add additional markup to make the reflection actually happen (add the attribute if not already there).
                                        if(!explicitReflectCurrent)
                                        {
                                            db.addAttribute(method, "hk::Reflect");
                                        }
                                    }
                                    else
                                    {
                                        hkLog.warning(method, "Public method %s() in hkeNode class %s is not marked with hk::Reflect("
                                            "false), but it can't be reflected because the return type and/or parameter types are not reflected.",
                                            method->getNameAsString().c_str(), nodeName.c_str());
                                    }
                                }
                            }
                        }

                        if( reflectedMethod &&
                            !isCtor && !isDtor &&
                            accessSpecifier == clang::AS_public )
                        {
                            // Try to match the reflected method with an existing message type.
                            bool markedAsHandler = false;
                            if(db.getAttribute(method, "hk::HandledMessage"))
                            {
                                markedAsHandler = true;
                            }
                            else
                            {
                                if(method->getResultType()->isVoidType() && method->getNumParams() == 1)
                                {
                                    const clang::PointerType* pointerType = clang::dyn_cast<clang::PointerType>(method->getParamDecl(0)->getType().getTypePtr());
                                    const clang::ReferenceType* refType = clang::dyn_cast<clang::ReferenceType>(method->getParamDecl(0)->getType().getTypePtr());

                                    if (const clang::Type* pointedType = pointerType ? pointerType->getPointeeType().getTypePtr() :
                                        refType ? refType->getPointeeType().getTypePtr() : NULL)
                                    {
                                        if (const clang::CXXRecordDecl* recordType = pointedType->getAsCXXRecordDecl())
                                        {
                                            const clang::CXXRecordDecl* recordTypeDefinition = recordType->getDefinition();
                                            if (recordTypeDefinition && db.getAttribute(recordTypeDefinition, "hk::MessageType"))
                                            {
                                                markedAsHandler = true;
                                                if(!method->getName().equals_lower(recordType->getName()))
                                                {
                                                    hkLog.debug(method, "Reflected method %s() in hkeNode class %s handles messages of type %s, "
                                                        "but the function name differs from the message type name.",
                                                        method->getNameAsString().c_str(), nodeName.c_str(), recordTypeDefinition->getNameAsString().c_str());
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            if(!markedAsHandler)
                            {
                                // If it isn't explicitly marked as a message handler, we try to match it with an existing message
                                // type name and make it a handler for that message type.
                                std::string methodName = method->getNameAsString();
                                methodName[0] = static_cast<char>(::toupper(methodName[0]));
                                std::string messageName = "hke::";
                                messageName.append(methodName);
                                const clang::CXXRecordDecl* messageType = db.getMessageTypeDecl(messageName);
                                if(messageType)
                                {
                                    // Found a corresponding message.
                                    // No need to check the parameters, this will be done later on in the actual reflection generation.
                                    // We simply add the hk::HandledMessage attribute.
                                    std::string attributeText = "hk::HandledMessage(";
                                    attributeText.append(messageName);
                                    attributeText.append(")");
                                    db.addAttribute(method, attributeText);

                                    hkLog.info(method, "Reflected method %s() in hkeNode class %s was automatically "
                                        "set as handler for messages of type %s.", method->getNameAsString().c_str(), nodeName.c_str(),
                                        messageName.c_str());
                                    // Now it really is a handler.
                                    markedAsHandler = true;
                                }
                            }

                            if( markedAsHandler && !method->isVirtual())
                            {
                                // This function will be treated as a message handler and it is possible to call it normally and
                                // through the reflection-based message passing interface. Since the message dispatching is always
                                // polymorphic, here we make sure that the function is declared virtual (because otherwise
                                // the behavior would be different when called directly and through the message).
                                hkLog.error(method, "Reflected method %s() in hkeNode class %s is a message handler, "
                                    "but it is non-virtual. This is disallowed as different behavior will result when calling the method "
                                    "directly and through message dispatch.", method->getNameAsString().c_str(), nodeName.c_str());
                            }
                        }
                    }
                }

                // 3: Examine all fields, fields in the RUNTIME_DATA special sections will be treated specially.
                // All those fields will be automatically hk::Reflect(false), they will not be reflected and won't
                // be serialized.
                {
                    bool isRuntimeData = false;
                    for( clang::DeclContext::decl_iterator it = recDecl->decls_begin(); it != recDecl->decls_end(); ++it)
                    {
                        if(const clang::AccessSpecDecl* a = clang::dyn_cast<clang::AccessSpecDecl>(*it))
                        {
                            isRuntimeData = false;
                            if(db.getAttributeValue<bool>(a, "hke::RuntimeDataSection", false) == true)
                            {
                                isRuntimeData = true;
                            }
                        }
                        else if(clang::FieldDecl* f = clang::dyn_cast<clang::FieldDecl>(*it))
                        {
                            // Mark it as reflect false to make sure that is not reflected (unless explicitly reflected)
                            if(isRuntimeData && db.getAttribute(f, "hk::Reflect") == NULL)
                            {
                                db.addAttribute(f, "hk::Reflect(false)");
                            }
                        }
                    }
                }
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
