// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "filter/DeclareClassFilter.h"
#include "utils/Database.h"
#include "utils/ElementUtils.h"

// We need an "operator new" at least at the first concrete base class, so
// don't issue an error if there is such a class, since we would have issued an
// error for that.
static bool hasOperatorNewOrDeclaredConcreteBase( const clang::CXXRecordDecl* recDecl )
{
    for ( clang::CXXRecordDecl::base_class_const_iterator it = recDecl->bases_begin();
            it != recDecl->bases_end(); ++it )
    {
        if ( const clang::CXXRecordDecl* parent = it->getType()->getAsCXXRecordDecl() )
        {
            if ( !isAbstract(parent) && getHkDeclareClassDecl( parent ) )
            {
                // We can assume that either the parent is valid or we've issued an error for it.
                return true;
            }
            if ( hasOperatorNewOrDeclaredConcreteBase( parent ) )
            {
                return true;
            }
        }
    }
    for(clang::CXXRecordDecl::method_iterator it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
    {
        const clang::CXXMethodDecl* method = *it;
        if ( method->getOverloadedOperator() == clang::OO_New )
        {
            return true;
        }
    }
    return false;
};


class InterfaceChecker
{
public:
    InterfaceChecker(Database& db) : m_db(db) {}

        // A class should have a virtual getExactType method
        //  IF it has no parents AND is reflected AND has a vtable
    void check(const clang::CXXRecordDecl* rec)
    {
        if(rec->getNumBases() == 0)
        {
            if(m_db.isReflected(rec))
            {
                if(m_db.hasOwnVtable(rec))
                {
                    if(hasVirtualExactType(rec) == false)
                    {
                        const std::string& cxxName = m_db.getScopedMangledName(rec);
                        hkLog.warning(rec, "Found a reflected class (%s) which has virtual methods, but does not have virtual getExactType() method."
                            "This means dynamic casting will not work. Replace 'Reflect' with 'ReflectVirtualBase' in HK_DECLARE_CLASS to fix this.", cxxName.c_str());
                    }
                }
            }
        }
    }

private:
    void operator=(const InterfaceChecker&);

    bool hasVirtualExactType(const clang::CXXRecordDecl* rec)
    {
        for(auto cur = rec->method_begin(), end = rec->method_end(); cur != end; ++cur)
        {
            if(auto id = cur->getIdentifier())
            {
                if(id->getName() == "getExactType")
                {
                    if(cur->isVirtual())
                    {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    Database& m_db;
};


void DeclareClassFilter::run(Database& db)
{
    InterfaceChecker ichecker(db);
    // For every file in the database.
    for(Database::FileDeclMapType::const_iterator fit = db.inputFilesDecls.begin();
        fit != db.inputFilesDecls.end();
        ++fit)
    {
        const Database::DeclListType& declList = fit->second.decls;

        // For every named declaration.
        for(Database::DeclListType::const_iterator dit = declList.begin();
            dit != declList.end();
            ++dit)
        {
            if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(*dit))
            {
                // Consider classes which have a HK_DECLARE_CLASS or HK_DECLARE_REFLECTION macro.
                if ( const clang::TypedefDecl* havokDecl = getHkDeclareClassDecl( recDecl ) )
                {
                    // Check the given class name (void is used by the deprecated HK_DECLARE_REFLECTION macro to suppress this check).
                    if ( !havokDecl->getUnderlyingType()->isVoidType() && havokDecl->getUnderlyingType()->getAsCXXRecordDecl() != recDecl )
                    {
                        const std::string& cxxName = db.getScopedMangledName(recDecl);
                        hkLog.error( havokDecl, "The first argument to HK_DECLARE_CLASS in %s must be the name of the containing class.", cxxName.c_str() );
                    }
                    // Check that the macro is in the public section.
                    if ( havokDecl->getAccess() != clang::AccessSpecifier::AS_public )
                    {
                        hkLog.error( havokDecl, "A HK_DECLARE_CLASS macro must have public access." );
                    }
                    // Check that non-abstract classes have a new operator.
                    if ( !isAbstract(recDecl) )
                    {
                        if ( !hasOperatorNewOrDeclaredConcreteBase( recDecl ) )
                        {
                            const std::string& cxxName = db.getScopedMangledName(recDecl);
                            hkLog.error( havokDecl, "Class %s has a HK_DECLARE_CLASS declaration but no operator new. Built-in options include HK_DECLARE_CLASS( ..., New / PlacementNew / PrivateNew, ... ). PrivateNew suppresses the error.", cxxName.c_str() );
                        }
                    }
                }

                else if ( db.isReflected( recDecl ) )
                {
                    const std::string& cxxName = db.getScopedMangledName(recDecl);
                    hkLog.error( recDecl, "Class %s is reflected but has no HK_DECLARE_CLASS declaration.", cxxName.c_str() );
                }

                ichecker.check(recDecl);
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
