// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 MAC OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include "pch.h"
#include "filter/AddSetterFilter.h"
#include "utils/Database.h"
#include "utils/StlUtils.h"
#include "utils/ElementUtils.h"
#include <cctype>

AddSetterFilter::~AddSetterFilter()
{
}

namespace
{
    std::string getMethodName(const Database& db, const clang::RecordDecl* recDecl, clang::FieldDecl* field,
        const clang::AnnotateAttr* attr, const char* prefix, bool attrIsExplicit)
    {
        std::string lookFor;
        if (attrIsExplicit)
        {
            Attribute attrObj(db, attr);
            lookFor = attrObj.get("value");
        }
        if (lookFor.empty())
        {
            // Look for a method called "setX".
            const std::string& fieldPrefix = db.getAttributeValue<std::string>(recDecl, "hk::FieldPrefix");
            lookFor = prefix + fieldNameWithoutPrefix(fieldPrefix, field->getName());
        }
        return lookFor;
    }

    bool parameterMatchesField(clang::QualType parameterType, clang::QualType fieldType)
    {
        return parameterType.getCanonicalType().getUnqualifiedType() == fieldType.getCanonicalType().getUnqualifiedType();
    }

    bool passesCommonChecks(const clang::CXXMethodDecl* method, unsigned numParams)
    {
        if (method->isStatic())
        {
            hkLog.info(method, "Skipping %s because it is static", method->getNameAsString().c_str());
            return false;
        }

        clang::QualType resultType = method->getResultType();
        if (!(resultType->isVoidType() || resultType->isBooleanType()))
        {
            // Difficult to handle arbitrary return types safely, so setters are limited to void and bool return types.
            hkLog.info(method, "Skipping %s because it does not return void or bool", method->getNameAsString().c_str());
            return false;
        }

        if (method->getNumParams() != numParams)
        {
            hkLog.info(method, "Skipping %s because it does not have %d params", method->getNameAsString().c_str(), numParams);
            return false;
        }
        return true;
    }

    void tryAddTo(Database& db, const clang::CXXRecordDecl* recDecl, clang::FieldDecl* field,
        const clang::AnnotateAttr* attr, bool attrIsExplicit)
    {
        const std::string& lookFor = getMethodName(db, recDecl, field, attr, "addTo", attrIsExplicit);
        const std::string& cxxName = db.getScopedMangledName(recDecl);

        bool found = false;
        int candidates = 0;

        hkLog.info(field, "Searching addTo for %s::%s (%s)",
            cxxName.c_str(), field->getNameAsString().c_str(), lookFor.c_str());
        for (auto it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
        {
            const clang::CXXMethodDecl* method = *it;
            const std::string& methodName = method->getNameAsString();
            // Check the name
            if (llvm::StringRef(methodName).equals_lower(lookFor))
            {
                ++candidates;

                if (!passesCommonChecks(method, 2))
                {
                    continue;
                }

                if (!method->getParamDecl(0)->getType()->isIntegerType())
                {
                    hkLog.info(method, "Skipping %s because its first param is not an integer", methodName.c_str());
                    return;
                }

                found = true;
                clang::QualType fieldType = field->getType();

                clang::QualType parameterType = method->getParamDecl(1)->getType();

                const std::string& fieldTypeName = db.getScopedMangledName(fieldType);
                const std::string& setValueType = db.getScopedMangledName(parameterType);
                const std::string& setMethReturnType = db.getScopedMangledName(method->getResultType());
                const std::string& cxxNameNoTypename = db.getScopedMangledNameNoPrefix(db.astContext.getTypeDeclType(recDecl));
                //const std::string& cxxName = db.getScopedMangledName(recDecl);

                const std::string& attrString = formatString(
                    "hk::SetArrayAddOp(&hkReflect::Detail::erasedArrayAddCall< %s >::call< "
                    "HK_REFLECT_RESOLVE(%s), HK_REFLECT_RESOLVE(%s), "
                    "typename hkReflect::Detail::GetArraySubtype< HK_REFLECT_RESOLVE(%s) >::Type, "
                    "&HK_REFLECT_RESOLVE2(%s)::%s >)",
                    setMethReturnType.c_str(),
                    cxxName.c_str(),
                    setValueType.c_str(),
                    fieldTypeName.c_str(),
                    cxxNameNoTypename.c_str(),
                    methodName.c_str());

                if (attrIsExplicit)
                {
                    // Replace the contents of the explicit attribute.
                    db.replaceAttribute(field, attr, attrString);
                }
                else
                {
                    // Add the new attribute to the decl.
                    db.addAttribute(field, attrString);
                }

                hkLog.info(method, "SetArrayAddOp %s::%s.", cxxNameNoTypename.c_str(), methodName.c_str());
            }
        }

        if (attrIsExplicit && !found)
        {
            hkLog.error(db.astContext.getSourceManager(), attr->getLocation(),
                "No valid addTo callback named '%s' found for %s::%s (%d candidates)",
                lookFor.c_str(), cxxName.c_str(), field->getNameAsString().c_str(), candidates);
        }
    }

    void tryRemoveFrom(Database& db, const clang::CXXRecordDecl* recDecl, clang::FieldDecl* field,
        const clang::AnnotateAttr* attr, bool attrIsExplicit)
    {
        const std::string& lookFor = getMethodName(db, recDecl, field, attr, "removeFrom", attrIsExplicit);
        const std::string& cxxName = db.getScopedMangledName(recDecl);

        bool found = false;
        int candidates = 0;

        hkLog.info(field, "Searching removeFrom for %s::%s (%s)",
            cxxName.c_str(), field->getNameAsString().c_str(), lookFor.c_str());
        for (auto it = recDecl->method_begin(); it != recDecl->method_end(); ++it)
        {
            const clang::CXXMethodDecl* method = *it;
            const std::string& methodName = method->getNameAsString();
            // Check the name
            if (llvm::StringRef(methodName).equals_lower(lookFor))
            {
                ++candidates;

                if (!passesCommonChecks(method, 1))
                {
                    continue;
                }

                if (!method->getParamDecl(0)->getType()->isIntegerType())
                {
                    hkLog.info(method, "Skipping %s because its first param is not an integer", methodName.c_str());
                    return;
                }

                found = true;
                clang::QualType fieldType = field->getType();

                const std::string& setMethReturnType = db.getScopedMangledName(method->getResultType());
                const std::string& cxxNameNoTypename = db.getScopedMangledNameNoPrefix(db.astContext.getTypeDeclType(recDecl));
                //const std::string& cxxName = db.getScopedMangledName(recDecl);

                const std::string& attrString = formatString(
                    "hk::SetArrayRemoveOp(&hkReflect::Detail::erasedArrayRemoveCall< %s >::call<"
                    "HK_REFLECT_RESOLVE(%s), &HK_REFLECT_RESOLVE2(%s)::%s >)",
                    setMethReturnType.c_str(),
                    cxxName.c_str(),
                    cxxNameNoTypename.c_str(),
                    methodName.c_str());

                if (attrIsExplicit)
                {
                    // Replace the contents of the explicit attribute.
                    db.replaceAttribute(field, attr, attrString);
                }
                else
                {
                    // Add the new attribute to the decl.
                    db.addAttribute(field, attrString);
                }

                hkLog.info(method, "SetArrayRemoveOp %s::%s.", cxxNameNoTypename.c_str(), methodName.c_str());
            }
        }

        if (attrIsExplicit && !found)
        {
            hkLog.error(db.astContext.getSourceManager(), attr->getLocation(),
                "No valid removeFrom callback named '%s' found for %s::%s (%d candidates)",
                lookFor.c_str(), cxxName.c_str(), field->getNameAsString().c_str(), candidates);
        }
    }
}


void AddSetterFilter::run(Database& db)
{
    // For every file in the database.
    for(auto fit = db.inputFilesDecls.begin();
        fit != db.inputFilesDecls.end();
        ++fit)
    {
        const Database::DeclListType& declList = fit->second.decls;

        bool addExtraInclude = false;

        // For every named declaration.
        for(Database::DeclListType::const_iterator dit = declList.begin();
            dit != declList.end();
            ++dit)
        {
            if(const clang::CXXRecordDecl* recDecl = clang::dyn_cast<clang::CXXRecordDecl>(*dit))
            {
                if ( db.isReflected( recDecl ) )
                {
                    const std::string& cxxName = db.getScopedMangledName(recDecl); cxxName;

                    // Associate setters with their corresponding fields.
                    for (auto it = recDecl->field_begin(); it != recDecl->field_end(); ++it)
                    {
                        if (auto addToAttr = db.getAttribute(*it, "hk::SetArrayAddOp"))
                        {
                            tryAddTo(db, recDecl, *it, addToAttr, true);
                            addExtraInclude = true;
                        }
                        else if (auto removeFromAttr = db.getAttribute(*it, "hk::SetArrayRemoveOp"))
                        {
                            tryRemoveFrom(db, recDecl, *it, removeFromAttr, true);
                            addExtraInclude = true;
                        }
                        else if (auto diffAttr = db.getAttribute(*it, "hk::DiffArraySemantics"))
                        {
                            tryAddTo(db, recDecl, *it, diffAttr, false);
                            tryRemoveFrom(db, recDecl, *it, diffAttr, false);
                            addExtraInclude = true;
                        }
                    }
                }
            }
        }
        if (addExtraInclude)
        {
            fit->second.extraIncludes.insert("Common/Base/Reflect/Util/hkFieldSetterDetail.h");
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
