// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 X64 MAC
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Resource/hkResource.h>
#include <Common/Base/Serialize/Format/Xml/hkXmlWriteFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Base/Fwd/hkcstdio.h>
#include <Common/Base/System/Log/hkLog.h>

#include "FileConvert.h"

// todo.rt this should rather load DLLs dynamically
#include <Common/Base/KeyCode.h>

#if defined(HK_FEATURE_PRODUCT_ANIMATION) && defined(HK_FEATURE_PRODUCT_PHYSICS_2012) && defined(HK_FEATURE_PRODUCT_PHYSICS)
#include <Animation/PhysicsMigrationUtils/hkaPhysicsMigrationUtils.h>
#endif

#if defined(HK_FEATURE_PRODUCT_BEHAVIOR) && defined(HK_FEATURE_PRODUCT_PHYSICS_2012) && defined(HK_FEATURE_PRODUCT_PHYSICS)
#include <Behavior/PhysicsMigrationUtils/hkbPhysicsMigrationUtils.h>
#endif

#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012) && defined(HK_FEATURE_PRODUCT_PHYSICS)
#include <Physics2012/PhysicsMigrationUtils/hkpPhysicsMigrationUtils.h>
#endif

#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012) && defined(HK_FEATURE_PRODUCT_PHYSICS)
static bool s_convertPhysics2012Objects(
    const hkRootLevelContainer& contentsAsContainer,
    hkRootLevelContainer& convertedContainer )
{
    // Temp vars
    bool contentsConverted = false;
    convertedContainer = contentsAsContainer;
    hkRootLevelContainer tempContainer;
    hkPointerMap<const class hkpShape*, class hknpShape*> shapesMap;

#ifdef HK_FEATURE_PRODUCT_ANIMATION
    // Animation
    {
        contentsConverted |= hkaPhysicsMigrationUtils::convertRootLevelContainer(
            convertedContainer,
            tempContainer,
            HK_NULL,
            &shapesMap,
            false );
        convertedContainer = tempContainer;
        tempContainer.m_namedVariants.clear();
    }
#endif

#ifdef HK_FEATURE_PRODUCT_BEHAVIOR
    // Behavior
    {
        contentsConverted |= hkbPhysicsMigrationUtils::convertRootLevelContainer(
            convertedContainer,
            tempContainer,
            &shapesMap,
            false );
        convertedContainer = tempContainer;
        tempContainer.m_namedVariants.clear();
    }
#endif

    // Other Physics items
    {
        contentsConverted |= hkpPhysicsMigrationUtils::convertRootLevelContainer(
            convertedContainer,
            tempContainer,
            HK_NULL,
            &shapesMap,
            true );
        convertedContainer = tempContainer;
        tempContainer.m_namedVariants.clear();
    }

    return contentsConverted;
}

#else

static bool s_convertPhysics2012Objects(
    const hkRootLevelContainer& contentsAsContainer,
    hkRootLevelContainer& convertedContainer )
{
    convertedContainer = contentsAsContainer;
    return false;
}

#endif

#ifdef HK_DEBUG

void s_verificationCallback( const hkReflect::Var& var, hkSerialize::BundleBuilder&, void* args )
{

#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012) && defined(HK_FEATURE_PRODUCT_PHYSICS)
    // Physics versioning check
    HK_ASSERT(
        0x22440538,
        !reinterpret_cast< FileConvertOptions* >( args )->m_convertPhysics ||
        !hkpPhysicsMigrationUtils::isDeprecatedClass( var.getType()->getName() ), "Physics2012 class discovered in serialized output." );
#endif

}

#endif

static int FileConvert(int argc, const char** argv)
{
    FileConvertOptions options;
    hkArray<const char*> arguments;

    int parseReturnVal = options.parse(argc, argv, arguments);
    if( parseReturnVal != FILECONVERT_ERR_NONE )
    {
        return parseReturnVal;
    }

    if (options.m_quiet)
    {
        hkLog::setBaseSystemLogLevel(hkLog::Level::Disabled);
    }

    // Choose the appropriate reader
    {
        hkIstream is( arguments[0] );
        hkStreamReader* sreader = is.getStreamReader();

        if ((sreader == HK_NULL) || (!sreader->isOk()))
        {
            if (!options.m_quiet) printf("File '%s' was not found.\n", arguments[0]);
            return FILECONVERT_ERR_FILE_NOT_FOUND;
        }

        hkSerialize::Load loader;
        if (options.m_compendium)
        {
            loader.loadCompendium(options.m_compendium);
        }

        hkRefPtr<hkResource> resource = loader.withVersioning(true).toOwnedResource(sreader);
        if( resource == HK_NULL )
        {
            if (!options.m_quiet) printf("Could not load the file '%s'.\n", arguments[0]);
            return FILECONVERT_ERR_UNSUPPORTED_FORMAT;
        }

        hkReflect::Var contents = resource->getContents();
        if( !contents )
        {
            if (!options.m_quiet) printf("Could not get file contents.\n");
            return FILECONVERT_ERR_GET_CONTENTS_FAILED;
        }

        if(options.m_verbose >= 1)
        {
            printf("Contents class is %s.\n", contents.getType()->getName());
        }

        if(options.m_convertPhysics)
        {
            hkRootLevelContainer* contentsAsContainer = resource->getContents<hkRootLevelContainer>();

            if (contentsAsContainer != HK_NULL)
            {
                // Create a container to hold all converted items and items which were passed-along
                hkRootLevelContainer convertedContainer;

                // Bail if we haven't converted and the user has requested we skip such assets
                if ( !s_convertPhysics2012Objects( *contentsAsContainer, convertedContainer ) && options.m_skipUnchanged )
                {
                    printf("Nothing to convert.\n");
                    return FILECONVERT_ERR_CONVERSION_SKIPPED;
                }
                else
                {
                    *contentsAsContainer = convertedContainer;
                }
            }
            else
            {
                printf("File contents were not of an expected type.\n");
                return FILECONVERT_ERR_GET_CONTENTS_FAILED;
            }
        }

        if (arguments.getSize() == 2)
        {
            hkRefPtr<hkSerialize::WriteFormat> wformat;
            if (options.m_xml)
            {
                wformat.setAndDontIncrementRefCount(new hkSerialize::XmlWriteFormat);
            }
            else if (options.m_platform)
            {
                printf("Using platform: %s\n", options.m_platform);
                wformat.setAndDontIncrementRefCount(new hkSerialize::TagfileWriteFormat(options.m_platform));
            }
            else
            {
                wformat.setAndDontIncrementRefCount(new hkSerialize::TagfileWriteFormat);
            }

#ifdef HK_DEBUG
            if (hkSerialize::Save().withFormatState(wformat).withCallbacks(&options, s_verificationCallback, HK_NULL).contentsVar(contents, arguments[1]).isSuccess())
#else
            if (hkSerialize::Save().withFormatState(wformat).contentsVar(contents, arguments[1]).isSuccess())
#endif
            {
                if (options.m_verbose >= 1)
                {
                    printf("Saved %s.\n", arguments[1]);
                }
            }
            else
            {
                printf("Failed to save %s.\n", arguments[1]);
                return FILECONVERT_ERR_SAVE_FAILED;
            }
        }
        else if (arguments.getSize() == 1)
        {
            if (options.m_verbose >= 1)
            {
                printf("Load-only %s.\n", arguments[0]);
            }
        }
    }
    return FILECONVERT_ERR_NONE;
}

#if defined(HK_PLATFORM_WIN32)
#   define HK_MAIN_CALL _cdecl
#else
#   define HK_MAIN_CALL
#endif

static void HK_CALL errorReport(const char* msg, void* p = HK_NULL)
{
    printf("%s", msg);
}

int HK_MAIN_CALL main(int argc, const char** argv)
{
    FileConvertInit();
    int returnVal = FileConvert(argc, argv);
    FileConvertQuit();

    return returnVal;
}

#ifdef HK_DYNAMIC_DLL
    #include <Common/CommonAll/hkKeycodeDllInit.cxx>    
    #include <Engine/Base/hkeExport.h>                  
    #include <Engine/Base/World/hkePluginManager.h>     
    #include <Common/NewBase/Project/hkeProject.h>      
#else
    #include <Common/Base/keycode.cxx>
    #define HK_CLASSES_FILE <Common/Base/ClassLists/hkKeyCodeClassLists.cxx>
    #define HK_FEATURE_REFLECTION_CLOTH_SETUP
    #define HK_FEATURE_REFLECTION_CLOTH_SETUP_ANIMATION
    #define HK_FEATURE_REFLECTION_COMMON_VISUALIZE
    #undef HK_FEATURE_PRODUCT_NEW_TOOLS
    #define HK_EXCLUDE_FEATURE_MemoryTracker
    #define HK_EXCLUDE_FEATURE_hkFormatYamlFile
    #include <Common/Base/Config/hkProductFeatures.cxx>
#endif

#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>

void HK_CALL FileConvertInit()
{
#if defined(HK_COMPILER_HAS_INTRINSICS_IA32) && HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED
    // Flush all denormal/subnormal numbers (2^-1074 to 2^-1022) to zero.
    // Typically operations on denormals are very slow, up to 100 times slower than normal numbers.
    _MM_SET_FLUSH_ZERO_MODE(_MM_FLUSH_ZERO_ON);
#endif

    extern HK_EXPORT_COMMON hkBool hkBaseSystemInitVerbose;
    hkBaseSystemInitVerbose = false;

    hkMemoryRouter* memory = hkMemoryInitUtil::initDefault(hkMallocAllocator::m_defaultMallocAllocator, hkMemorySystem::FrameInfo(0));
    hkBaseSystem::init(memory, errorReport);
    hkError::getInstance().setEnabled(0x38d9223e, false); // disable stack performance warning

#ifdef HK_DYNAMIC_DLL
    commonInitWhenUsingDlls();
    hkePluginManager::getInstance().loadAll();
    //hkeProject::getInstance().init();
#endif
}

void HK_CALL FileConvertQuit()
{
    hkBaseSystem::quit();
    hkMemoryInitUtil::quit();
}

FileConvertOptions::FileConvertOptions()
{
    hkString::memSet(this, 0, sizeof(*this));
}

#include <Common/Base/System/Io/OptionParser/hkOptionParser.h>

int FileConvertOptions::parse(int argc, const char** argv, /*OUT*/hkArray<const char*>& arguments)
{
    hkOptionParser parser("File Converter",
        "Tool to load an asset, and optionally save it out in the same or a different format.\n"
        "\n"
        "Sample usage:\n"
        "Verify that a file loads cleanly. Exit status is zero on success:\n"
        "    FileConvert LoadOnly.hkt\n"
        "Use a 2nd argument for conversion\n"
        "    FileConvert Input.hkt Output.hkt\n"
        "To convert asset to a native console format use --platform-rules, for example for PlayStation(R)4:\n"
        "   FileConvert -r=64g MyAsset.xml MyAsset.hkt\n"
        "To convert a binary tagfile (hkt) to xml format:\n"
        "   FileConvert -x MyAsset.hkt MyAsset.xml");

    hkOptionParser::Option options[] =
    {
        // Core options
        hkOptionParser::Option("x", "xml", "Xml output", &m_xml),
        hkOptionParser::Option("v", "verbose", "Verbose logging", &m_verbose),
        hkOptionParser::Option("q", "quiet", "Do not log anything", &m_quiet),
        hkOptionParser::Option("r", "platform-rules", "Generate output optimised for a target platform (default is cross-platform). 'target' mnemonics are 'a':arm32, 'b':big endian, 'm':msvc, 'g':generic. Specify 'd' at the end for hkReal==double. Examples: \n - 32a:Arm32,PlayStation(R)Vita\n - 32b:WiiU\n - 32m:Win32\n - 32g:x86(except 32m)\n - 64m:Win64,xBoxOne\n - 64g:x64(except 64m),Arm64,PlayStation(R)4", &m_platform),
        hkOptionParser::Option("", "compendium", "Specifies the type compendium to use", &m_compendium),

        // Migration options
        hkOptionParser::Option("p", "physics", "Convert Physics 2012 assets to Physics assets", &m_convertPhysics),
        hkOptionParser::Option("k", "skip", "Skip files which don't need conversion (only used when '-p' is specified)", &m_skipUnchanged),

        // Deprecated options
        hkOptionParser::Option("g", "tagfile", "(Deprecated) Tagfile output (For compatibility, this option is always active)", &m_tagfile, true),
        hkOptionParser::Option("s", "strip", "(Deprecated) Strip metadata (Not supported)", &m_strip)
    };

    parser.setOptions(options, HK_COUNT_OF(options));

    const char* argumentArray[2];
    parser.setArguments("file", "The input filename, optionally followed by the output filename", hkOptionParser::ARGUMENTS_ONE_OR_MORE, argumentArray, 2);

    if(parser.parse(argc, argv) != hkOptionParser::PARSE_SUCCESS)
    {
        return FILECONVERT_ERR_PARSE_FAIL;
    }
    else // verify args
    {
        // Warn if the user specifies deprecated arguments
        if (!m_tagfile)
        {
            HK_WARN(0, "Deprecated flag --tagfile detected. Tagfiles are always outputted for compatibility.");
        }

        if (m_strip)
        {
            HK_WARN(0, "Deprecated flag --strip detected. This flag is no longer supported.");
        }

        // Convert arguments to hkArray
        for (int i = 0; i < 2; i++)
        {
            if (argumentArray[i] == HK_NULL)
            {
                break;
            }
            else
            {
                arguments.pushBack(argumentArray[i]);
            }
        }

        if (arguments.getSize() < 1 || arguments.getSize() > 2)
        {
            parser.usage("One or Two arguments required: the input filename and (optionally) the output filename are required");
            return FILECONVERT_ERR_PARSE_ARGS;
        }
    }
    return FILECONVERT_ERR_NONE;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
