// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/Vehicle/hkpVehicle.h>

#include <Physics2012/Vehicle/Camera/hkp1dAngularFollowCam.h>
#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>

hkp1dAngularFollowCam::hkp1dAngularFollowCam()
{
}


hkp1dAngularFollowCamCinfo::hkp1dAngularFollowCamCinfo()
{
    m_yawCorrection.setZero();
    m_yawSignCorrection = hkSimdReal_1;
    m_rigidBodyForwardDir.setConstant<HK_QUADREAL_0100>();
    m_upDirWS.setConstant<HK_QUADREAL_0010>();
}


hkp1dAngularFollowCamCinfo::CameraSet::CameraSet()
{
    m_positionUS.set( hkReal(0), hkReal(12), hkReal(5.5f));
    m_lookAtUS.setZero();
    m_fov = hkSimdReal_1;
    m_velocity.setZero();
    m_speedInfluenceOnCameraDirection.setFromFloat(hkReal(0.01f));
    m_angularRelaxation = hkSimdReal_4;
}


void hkp1dAngularFollowCam::reinitialize( const hkp1dAngularFollowCamCinfo &bp )
{
    m_yawCorrection = bp.m_yawCorrection;
    m_yawSignCorrection = bp.m_yawSignCorrection;
    m_upDirWS = bp.m_upDirWS;
    m_rigidBodyForwardDir = bp.m_rigidBodyForwardDir;

    m_flat0DirWS.setCross( m_upDirWS, m_rigidBodyForwardDir );
    m_flat0DirWS.normalize<3>();
    m_flat1DirWS.setCross( m_upDirWS, m_flat0DirWS );

    m_cameraYawAngle.setZero();

    m_set[0] = bp.m_set[0];
    m_set[1] = bp.m_set[1];

    m_velocityRange.setReciprocal<HK_ACC_MID,HK_DIV_SET_HIGH>(m_set[1].m_velocity - m_set[0].m_velocity);
}


hkp1dAngularFollowCam::hkp1dAngularFollowCam(const hkp1dAngularFollowCamCinfo &bp)
{
    reinitialize(bp);
}

hkp1dAngularFollowCam::~hkp1dAngularFollowCam()
{
}


void hkp1dAngularFollowCam::calcYawAngle(hkSimdRealParameter factor, const hkTransform& trans, hkVector4Parameter linearVelocity, hkSimdReal& yaw_angle)
{
// uncomment following if you want the camera to use the local 'up'
// is handy if you are driving in loopings etc.
//#define hkp1dAngularFollowCam_USE_LOCAL_UP
#ifdef hkp1dAngularFollowCam_USE_LOCAL_UP
    m_upDirWS = trans.getColumn<1>();
    m_flat0DirWS.setCross( m_upDirWS, m_rigidBodyForwardDir );
    m_flat0DirWS.normalize<3>();
    m_flat1DirWS.setCross( m_upDirWS, m_flat0DirWS );
#endif

    hkVector4 forwardDirWS;
    forwardDirWS._setRotatedDir(trans.getRotation(), m_rigidBodyForwardDir );

    hkSimdReal speedInfluenceOnCameraDirection;
    speedInfluenceOnCameraDirection.setInterpolate(m_set[0].m_speedInfluenceOnCameraDirection, m_set[1].m_speedInfluenceOnCameraDirection, factor);

    hkVector4 tv;
    tv.setAddMul( forwardDirWS, linearVelocity, hkSimdReal::fromFloat(hkReal(0.01f)) * speedInfluenceOnCameraDirection);

        // calculate new yaw angle
    const hkSimdReal u = tv.dot<3>( m_flat0DirWS );
    const hkSimdReal v = tv.dot<3>( m_flat1DirWS );

    yaw_angle = hkVector4Util::atan2( v, u ) * m_yawSignCorrection - m_yawCorrection + hkSimdReal_PiOver2;
}


void hkp1dAngularFollowCam::calcVelocityFactor(hkVector4Parameter chassisVelocity, hkSimdReal& factor)
{
        // Work out factors based on velocity
    const hkSimdReal  absVelocity = chassisVelocity.length<3>();
    factor.setMul(absVelocity - m_set[0].m_velocity, m_velocityRange);
    factor.setClampedZeroOne(factor); // for using as interpolation factor
}


void hkp1dAngularFollowCam::resetCamera( const hkTransform& trans, hkVector4Parameter linearVelocity, hkVector4Parameter angularVelocity)
{
    hkSimdReal factor; calcVelocityFactor(linearVelocity, factor);
    calcYawAngle(factor, trans, linearVelocity, m_cameraYawAngle);
}

void hkp1dAngularFollowCam::calculateCamera( const CameraInput &in, CameraOutput &out )
{
    hkSimdReal factor; calcVelocityFactor(in.m_linearVelocity, factor);

        // Work out yaw change based on factors and velocity.
    {
        hkSimdReal yawAngle;
        calcYawAngle(factor, in.m_fromTrans, in.m_linearVelocity, yawAngle);

        // now lets see how fast we turn the camera to achieve this target angle.
        hkSimdReal angularRelaxation; angularRelaxation.setInterpolate(m_set[0].m_angularRelaxation, m_set[1].m_angularRelaxation, factor);
        hkSimdReal angularFactor; angularFactor.setClampedZeroOne( angularRelaxation * in.m_deltaTime );

        hkSimdReal k;
        hkSimdReal fullSpin = hkVector4Util::toRange(m_cameraYawAngle - yawAngle, hkSimdReal_Pi, k);

        m_cameraYawAngle.subMul(angularFactor, fullSpin);
    }

    const hkTransform& chassisTransform = in.m_fromTrans;

    hkQuaternion q; q.setAxisAngle(m_upDirWS, m_cameraYawAngle);
    hkTransform r_ws_us; r_ws_us.set(q, chassisTransform.getTranslation());

    {   // calculate camera position
        hkVector4 camPosUS;
        camPosUS.setInterpolate( m_set[0].m_positionUS, m_set[1].m_positionUS, factor);
        out.m_positionWS._setTransformedPos(r_ws_us,camPosUS);
    }


    {   // calculate lookat
        hkVector4 lookAtUS;
        lookAtUS.setInterpolate( m_set[0].m_lookAtUS, m_set[1].m_lookAtUS, factor);
        out.m_lookAtWS._setTransformedPos(chassisTransform,lookAtUS);
    }


    {   // calculate updir
        out.m_upDirWS = m_upDirWS;
    }

    {   // calculate fov
        out.m_fov.setInterpolate(m_set[0].m_fov, m_set[1].m_fov, factor);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
