// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Visualize/hkProcessContext.h>
#include <Physics2012/Dynamics/World/hkpWorldCinfo.h>
#include <Physics2012/Dynamics/World/Listener/hkpWorldDeletionListener.h>
#include <Physics2012/Dynamics/Action/hkpActionListener.h>
#include <Physics2012/Dynamics/Constraint/hkpConstraintListener.h>
#include <Physics2012/Dynamics/Phantom/hkpPhantomListener.h>
#include <Physics2012/Dynamics/Entity/hkpEntityListener.h>

class hkpWorld;
class hkpEntity;
class hkpPhantom;

#define HKP_PHYSICS_CONTEXT_TYPE_STRING "Physics 2012"

/// A simple interface that viewers that want to know
/// when hkWorlds are added and removed from the physics
/// context can implement.
class HK_EXPORT_PHYSICS_2012 hkpPhysicsContextWorldListener
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkpPhysicsContextWorldListener);

        virtual ~hkpPhysicsContextWorldListener() {}
        virtual void worldAddedCallback( hkpWorld* newWorld ) = 0;
        virtual void worldRemovedCallback( hkpWorld* newWorld ) = 0;
};

/// This is the context that processes (here called the older name
/// of 'viewers') can use if they understand physics Worlds. It listens on all
/// added worlds and can trigger deletion and addition callbacks
/// to processes (viewers) if requested. A context itself is just a data store
/// for any information that a viewer will need. In this case it is pointers to
/// hkWorlds, as from that any of the physics viewers can find the parts they are
/// interested in and then register themselves with the world to get the appropriate
/// callbacks to keep their state up to date.
class HK_EXPORT_PHYSICS_2012 hkpPhysicsContext :
    public hkReferencedObject, public hkProcessContext,
    public hkpWorldDeletionListener,
    public hkpEntityListener,
    public hkpPhantomListener,
    public hkpConstraintListener,
    public hkpActionListener
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_VDB);

        hkpPhysicsContext();

            /// Register all Processes (Physics Viewers)
            /// in the Utilities. If you don't call this
            /// or a subset of what it calls, all you will get
            /// is the common viewers in hkVisualize (debug display and stats).
            /// If you don't see the hkpShapeDisplayViewer for instance in the
            /// list of available viewers in the VDB client, it is because
            /// this has not been called.
        static void HK_CALL registerAllPhysicsProcesses();

            /// As there can be any number of different user types
            /// of data contexts, the type is simply identified by string.
        virtual const char* getType() { return HKP_PHYSICS_CONTEXT_TYPE_STRING; }

            /// Register a world. This context is a deletion listener, so will
            /// remove the world when it sees it deleted. It does not reference
            /// the world, just keeps track of it by way of the listeners.
        void addWorld( hkpWorld* newWorld );

            /// Explicitly remove a world from the context. If
            /// you delete a world it will remove itself from
            /// the context anyway as this context is a deletion
            /// listener.
        void removeWorld( hkpWorld* newWorld );

            /// Find the index of the given world, returns -1
            /// if not found in this context.
        int findWorld(hkpWorld* world);

            /// Get number of worlds registered in this context.
        inline int getNumWorlds() { return m_worlds.getSize(); }

            /// Get the i-th world registered in this context.
        inline hkpWorld* getWorld(int i) { return m_worlds[i]; }

            /// So that processes can see all the worlds dynamically,
            /// they can be hkpPhysicsContextWorldListener which simply
            /// get world added to this context events. As such they would
            /// be able to then register themselves as rigidbody added
            /// listeners and so on and for instance be able to create
            /// bodies to display upon those further callbacks.
        void addWorldAddedListener( hkpPhysicsContextWorldListener* cb );

            /// So that processes can see all the worlds dynamically,
            /// they can be hkpPhysicsContextWorldListener which simply
            /// can get world removed from this context, and can update
            /// their state accordingly (remove some display geoms etc).
        void removeWorldAddedListener( hkpPhysicsContextWorldListener* cb );

    public:

            // Physics callbacks we want to track

            /// Raised when the given world has been deleted, default action is
            /// to remove it from the list of registered worlds and notify
            /// any interested viewers that the world has been removed.
        virtual void worldDeletedCallback( hkpWorld* world );

    protected:

            /// As a context must exist at least as long as the VDB, we explicitly
            /// do not allow local variables of it to force the use of 'new' and removeRefernce().
            /// The VDB itself can't add a reference a context is just a abstract interface that
            /// any user data item can implement for their own viewers.
        virtual ~hkpPhysicsContext();

        hkArray<hkpWorld*> m_worlds;
        hkArray<hkpPhysicsContextWorldListener*> m_addListeners;
};

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
