// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Physics2012/Collide/hkpExport.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Physics2012/Dynamics/World/hkpWorld.h>
#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Dynamics/Action/hkpAction.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Physics2012/Vehicle/hkpVehicleInstance.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Vehicle/hkpVehicleViewer.h>

int         hkpVehicleViewer::s_tag             = 0;
const int   hkpVehicleViewer::s_numWheelSegments    = 12;

// A Little macro to convert degrees to radians
#define DEGREES2RADIANS(deg) ((deg)*(0.0174532925f))

void HK_CALL hkpVehicleViewer::registerViewer()
{
    s_tag = hkProcessFactory::getInstance().registerProcess( getName(), create );
}

hkProcess* HK_CALL hkpVehicleViewer::create(const hkArray<hkProcessContext*>& contexts)
{
    return new hkpVehicleViewer(contexts);
}

hkpVehicleViewer::hkpVehicleViewer(const hkArray<hkProcessContext*>& contexts)
: hkpWorldViewerBase(contexts)
{
}

void hkpVehicleViewer::init()
{
    if (m_context)
    {
        for (int i=0; i < m_context->getNumWorlds(); ++i)
        {
            worldAddedCallback( m_context->getWorld(i));
        }
    }
}

hkpVehicleViewer::~hkpVehicleViewer()
{
    if (m_context)
    {
        for (int i=0; i < m_context->getNumWorlds(); ++i)
        {
            worldRemovedCallback( m_context->getWorld(i));
        }
    }
}

void hkpVehicleViewer::worldAddedCallback( hkpWorld* w)
{
    w->markForWrite();
    w->addActionListener(this);
    w->addWorldPostSimulationListener(this);

    // This class is an action listener, and thus will be notified when actions
    // are added to the world. However, since this class can be created by the
    // user in the VDB, there may already be actions added to the world when
    // this class is created. This loops finds any existing actions and passes
    // them to the action added callback.
    hkpPhysicsSystem* system = w->getWorldAsOneSystem();
    const hkArray<hkpAction*>& action = system->getActions();

    for(int i=0; i < action.getSize(); i++)
    {
        actionAddedCallback(action[i]);
    }

    system->removeReference();

    w->unmarkForWrite();
}

void hkpVehicleViewer::worldRemovedCallback( hkpWorld* w)
{
    w->markForWrite();
    w->removeActionListener(this);
    w->removeWorldPostSimulationListener(this);
    w->unmarkForWrite();
}


// HVK-3274
// This function ensures that hkpVehicleViewer will fail to compile if the name
// of hkpVehicleInstance is ever changed. Should this ever happen the string
// below needs to be changed accordingly. This removes the dependency on
// hkvehicle from hkutilities.
static inline const char* getHkVehicleInstanceName(hkpVehicleInstance*) { return "hkpVehicleInstance"; }

void hkpVehicleViewer::actionAddedCallback( hkpAction* action )
{
    // Get the class of the action that was added to the world
    // NB: To get the class we use a string here, rather than
    // hkReflect::getType, so that building hctUtilities does
    // not have a dependency on the vehicle library. See HVK-3274.
    const hkReflect::Type* vehicleClass = hkReflect::typeFromName(getHkVehicleInstanceName(HK_NULL));

    // If the action is a vehicle instance, or it
    // is of a type derived from the vehicle class.
    if ( hkReflect::exactTypeOf(action)->extendsOrEquals(vehicleClass) )
    {
        action->addReference();
        m_vehicles.pushBack(static_cast<hkpVehicleInstance*>(action));
    }
}

void hkpVehicleViewer::actionRemovedCallback( hkpAction* action )
{
    // Search for the action that has just been removed from the
    // world. If the action is in our list of vehicles, remove it
    // from the list.
    const int index = m_vehicles.indexOf((hkpVehicleInstance*)action);

    // If this action is not a member of m_vehicles, we ignore it and return.
    if (index == -1) return;

    m_vehicles.removeAt(index);
    action->removeReference();
}

void hkpVehicleViewer::postSimulationCallback(hkpWorld* world)
{
    HK_TIME_CODE_BLOCK("hkpVehicleViewer::step", HK_NULL);

    const int numVehicles = m_vehicles.getSize();

    static const hkColor::Argb inAirColor       = hkColor::ROSYBROWN;
    static const hkColor::Argb inContactColor   = hkColor::WHITESMOKE;
    static const hkColor::Argb raycastColor     = hkColor::RED;
    static const hkColor::Argb axleColor        = hkColor::YELLOWGREEN;
    static const hkColor::Argb pointColor       = hkColor::YELLOW;
    static const hkReal pointScale              = 0.05f;

    // Loop through each vehicle in the world
    for (int i=0; i<numVehicles; i++)
    {
        const hkpVehicleInstance& vehicle = *m_vehicles[i];

        // Loop through each wheel on the vehicle
        const int numWheels = vehicle.m_wheelsInfo.getSize();
        hkLocalBuffer<hkVector4> centerPoint(numWheels);
        for ( int w = 0; w < numWheels; w++ )
        {

            const hkpVehicleData::WheelComponentParams& wheelParam = vehicle.m_data->m_wheelParams[w];
            hkpVehicleSuspension::SuspensionWheelParameters& suspensionParam = vehicle.m_suspension->m_wheelParams[w];
            const hkpVehicleInstance::WheelInfo& wheel  = vehicle.m_wheelsInfo[w];
            const hkVector4& contactPoint               = wheel.m_contactPoint.getPosition();
            const hkVector4& hardPoint                  = wheel.m_hardPointWs;

            const hkUint64 collidableId = wheel.m_contactBody ? (hkUint64)(wheel.m_contactBody->getCollidable()) : 0;

            centerPoint[w].setAddMul(hardPoint, wheel.m_suspensionDirectionWs, hkSimdReal::fromFloat(suspensionParam.m_length));

            // Draw a ray from hardPoint to center to contactPoint
            m_displayHandler->displayLine(collidableId, hardPoint, centerPoint[w], wheel.m_contactBody ? raycastColor : inAirColor, s_tag);
            m_displayHandler->displayLine(collidableId, centerPoint[w], contactPoint, wheel.m_contactBody ? raycastColor : inAirColor, s_tag);

            // Draw a star at the suspension hardpoint and contactPoint of the wheel.
            m_displayHandler->displayStar( collidableId, hardPoint, pointColor, pointScale, s_tag );
            m_displayHandler->displayStar( collidableId, contactPoint, pointColor, pointScale, s_tag );

            // Draw the wheel rotation line
            {
                hkVector4 spoke = wheel.m_suspensionDirectionWs;
                hkQuaternion spokeRotation; spokeRotation.setAxisAngle(wheel.m_spinAxisWs, -wheel.m_spinAngle);
                spoke.setRotatedDir(spokeRotation, spoke);
                spoke.normalize<3>();
                spoke.mul(hkSimdReal::fromFloat(wheelParam.m_radius));

                hkVector4 start;    start.setAdd(centerPoint[w], spoke);
                hkVector4 end;      end.setSub(centerPoint[w], spoke);

                m_displayHandler->displayLine(collidableId, start, end, wheel.m_contactBody ? inContactColor : inAirColor, s_tag);
            }

            // Draw the wheel
            hkVector4 spokes[s_numWheelSegments];
            spokes[0].setMul(hkSimdReal::fromFloat(wheelParam.m_radius), wheel.m_suspensionDirectionWs);

            // This loop defines a number of points on the circumference of the
            // wheel rim. It then draws a lines to connect all these points.
            hkQuaternion spokeRotation; spokeRotation.setAxisAngle(wheel.m_spinAxisWs, DEGREES2RADIANS(360.0f/(float)s_numWheelSegments));
            for (int s=1; s<s_numWheelSegments+2; s++)
            {
                if (s < s_numWheelSegments)
                {
                    spokes[s].setRotatedDir(spokeRotation, spokes[s-1]);
                }
                if (s <= s_numWheelSegments)
                {
                    spokes[s-1].add(centerPoint[w]);
                }
                if (s>=2)
                {
                    m_displayHandler->displayLine(
                        collidableId,
                        spokes[(s-2)%s_numWheelSegments],
                        spokes[(s-1)%s_numWheelSegments],
                        wheel.m_contactBody ? inContactColor : inAirColor,
                        s_tag );
                }
            }
        }

        // Draw the axles (Do this by drawing a line between all wheels on the
        // same axle).
        const int numAxles  = vehicle.m_data->m_numWheelsPerAxle.getSize();
        for (int axle=0; axle < numAxles; axle++)
        {
            // These two vectors will be used to store the point of connected with
            // the axle of the last two wheels that we have encountered.
            hkVector4 a;
            hkVector4 b;

            // This flag will be used so that we know when we are going through
            // this loop for the first time.
            bool firstTimeThrough = true;
            const int totalWheelsOnThisAxle = vehicle.m_data->m_numWheelsPerAxle[axle];
            int wheelsFoundOnThisAxle  = 0;

            for ( int w = 0; w < numWheels; w++ )
            {
                const hkpVehicleData::WheelComponentParams& wheelParam = vehicle.m_data->m_wheelParams[w];
                const hkpVehicleInstance::WheelInfo& wheel  = vehicle.m_wheelsInfo[w];
                const hkUint64 collidableId = wheel.m_contactBody ? (hkUint64)(wheel.m_contactBody->getCollidable()) : 0;

                // If this wheel is not connected to the axle that we are
                // concerned with, skip it.
                if (wheelParam.m_axle != axle) continue;

                wheelsFoundOnThisAxle++;

                a = b;
                b = centerPoint[w];

                if (firstTimeThrough)
                {
                    // On this first time through this loop, the vector 'a'
                    // would be uninitialized, so we do nothing.
                    firstTimeThrough = false;
                }
                else
                {
                    // Draw a line to connect the last 2 wheels that we
                    // have encountered.
                    m_displayHandler->displayLine(collidableId, a, b, axleColor, s_tag);
                }

                // If we have already dealt with all the wheels on this axle,
                // we can break from the loop.
                if (wheelsFoundOnThisAxle == totalWheelsOnThisAxle) break;
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
