// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Physics2012/Collide/hkpExport.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpPhantomDisplayViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/hkpShapeDisplayBuilder.h>
#include <Common/Visualize/hkDebugDisplayHandler.h>
#include <Common/Visualize/hkProcessFactory.h>

#include <Common/Visualize/Shape/hkDisplayGeometry.h>

#include <Physics2012/Dynamics/World/hkpWorld.h>

#include <Physics2012/Dynamics/Phantom/hkpPhantomType.h>
#include <Physics2012/Dynamics/Phantom/hkpPhantom.h>

#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Object/hkSingleton.h>
#include <Common/Base/Types/Color/hkColor.h>

#define DEBUG_LOG_IDENTIFIER "p12.vdb.phantomviewer"
#include <Common/Base/System/Log/hkLog.hxx>


static const hkColor::Argb HK_DEFAULT_PHANTOM_COLOR = hkColor::rgbFromChars( 240, 200, 0, 200 ); // Orange with Alpha

HK_SINGLETON_IMPLEMENTATION(hkpUserShapePhantomTypeIdentifier);

int hkpPhantomDisplayViewer::s_tag = 0;

void HK_CALL hkpPhantomDisplayViewer::registerViewer()
{
    s_tag = hkProcessFactory::getInstance().registerProcess( getName(), create );
}

hkProcess* HK_CALL hkpPhantomDisplayViewer::create(const hkArray<hkProcessContext*>& contexts)
{
    return new hkpPhantomDisplayViewer(contexts);
}

hkpPhantomDisplayViewer::hkpPhantomDisplayViewer(const hkArray<hkProcessContext*>& contexts)
: hkpWorldViewerBase(contexts)
{

}

void hkpPhantomDisplayViewer::init()
{
    if (m_context)
    {
        for (int i=0; i < m_context->getNumWorlds(); ++i)
        {
            addWorld(m_context->getWorld(i));
        }
    }
}

hkpPhantomDisplayViewer::~hkpPhantomDisplayViewer()
{
    if (m_context)
    {
        for (int i=0; i < m_context->getNumWorlds(); ++i)
        {
            removeWorld(m_context->getWorld(i));
        }
    }
}

void hkpPhantomDisplayViewer::worldRemovedCallback( hkpWorld* world )
{
    removeWorld(world);
}

void hkpPhantomDisplayViewer::worldAddedCallback( hkpWorld* world )
{
    addWorld(world);
}

void hkpPhantomDisplayViewer::removeWorld( hkpWorld* world)
{
    world->markForWrite();

    world->removePhantomListener( this );
    world->removeWorldPostSimulationListener( this );
    const hkArray<hkpPhantom*>& phantoms = world->getPhantoms();

    for(int i = 0; i < phantoms.getSize(); i++)
    {
        phantomRemovedCallback( phantoms[i] );
    }

    world->unmarkForWrite();
}

void hkpPhantomDisplayViewer::addWorld(hkpWorld* world)
{
    world->markForWrite();

    world->addPhantomListener( this );
    world->addWorldPostSimulationListener( this );

    // get all the phantoms from the world and add them
    const hkArray<hkpPhantom*>& phantoms = world->getPhantoms();

    for(int i = 0; i < phantoms.getSize(); i++)
    {
        phantomAddedCallback( phantoms[i] );
    }

    world->unmarkForWrite();
}


void hkpPhantomDisplayViewer::phantomAddedCallback( hkpPhantom* phantom )
{

    const hkpShape* shape = phantom->getCollidable()->getShape();
    const hkpPhantomType type = phantom->getType();

    // For shape phantoms we add and manage a display geometry
    hkArray<hkDisplayGeometry*> displayGeometries;
    bool isShapePhantom = (type == HK_PHANTOM_SIMPLE_SHAPE) || (type == HK_PHANTOM_CACHING_SHAPE);
    if (!isShapePhantom)
    {
        isShapePhantom = hkpUserShapePhantomTypeIdentifier::getInstance().m_shapePhantomTypes.indexOf(type) != -1;
    }

    if (isShapePhantom && (shape != HK_NULL))
    {
        hkpShapeDisplayBuilder::hkpShapeDisplayBuilderEnvironment env;
        hkpShapeDisplayBuilder shapeBuilder(env);
        shapeBuilder.buildDisplayGeometries( shape, displayGeometries);

        // Check the geometries are valid
        for(int i = (displayGeometries.getSize() - 1); i >= 0; i--)
        {
            if( (displayGeometries[i]->getType() == HK_DISPLAY_ANY) &&
                (displayGeometries[i]->getGeometry() == HK_NULL) )
            {
                Log_Error( "Unable to build display geometry from hkpShape geometry data" );
                displayGeometries.removeAt(i);
            }
        }

        // send the display geometeries off to the display handler
        {
            const hkpCollidable* coll = phantom->getCollidable();
            hkUint64 id = (hkUint64)coll;

            // Add to our list of managed phantoms
            m_phantomShapesCreated.pushBack( phantom );

            const hkTransform& trans = coll->getTransform();

            hkColor::Argb color;
            if( phantom->hasProperty( HK_PROPERTY_DEBUG_DISPLAY_COLOR ) )
            {
                color = phantom->getProperty( HK_PROPERTY_DEBUG_DISPLAY_COLOR ).getInt();
            }
            else
            {
                color = HK_DEFAULT_PHANTOM_COLOR;
            }

            m_displayHandler->addGeometryEx(
                id,
                displayGeometries,
                trans,
                color,
                hkDisplayGeometryFlags::DEFAULT,
                s_tag );
        }

        hkReferencedObject::removeReferences( displayGeometries.begin(), displayGeometries.getSize() );
    }
}

void hkpPhantomDisplayViewer::phantomRemovedCallback( hkpPhantom* phantom )
{
    // Check if we are managing a display for this object
    const int index = m_phantomShapesCreated.indexOf(phantom);

    if(index >= 0)
    {
        m_phantomShapesCreated.removeAt(index);

        // remove the geometry from the displayHandler
        hkUint64 id = (hkUint64)phantom->getCollidable();
        m_displayHandler->removeGeometry(id);
    }
}

void hkpPhantomDisplayViewer::postSimulationCallback( hkpWorld* world )
{
    HK_TIMER_BEGIN("hkpPhantomDisplayViewer", this);

    // Create a list of AABBs one for each phantom in the scene
    {
        const hkArray<hkpPhantom*>& phantoms = world->getPhantoms();
        hkLocalArray<hkDisplayAABB> phantomAabbGeometries(phantoms.getSize());
        phantomAabbGeometries.setSize( phantoms.getSize() );

        hkLocalArray<hkDisplayGeometry*> displayGeometries(phantoms.getSize());
        displayGeometries.setSize(phantoms.getSize());

        for(int i = 0; i < phantoms.getSize(); i++)
        {
            hkAabb aabb;
            phantoms[i]->calcAabb(aabb);

            phantomAabbGeometries[i].setExtents(aabb.m_min, aabb.m_max);
            displayGeometries[i] = &phantomAabbGeometries[i];
        }

        // Draw the AABBs for each phantom
        m_displayHandler->displayGeometry(
            0,
            displayGeometries,
            hkTransform::getIdentity(),
            HK_DEFAULT_PHANTOM_COLOR,
            s_tag);
    }

    // Update the transforms for the geometries associated with shape phantoms we are managing
    {
        for(int j = 0; j < m_phantomShapesCreated.getSize(); j++)
        {
            // Send the latest transform for the phantom shape
            const hkpCollidable* coll = m_phantomShapesCreated[j]->getCollidable();
            const hkTransform& trans = m_phantomShapesCreated[j]->getCollidable()->getTransform();
            hkUint64 id = (hkUint64)coll;
            m_displayHandler->updateGeometryTransform(id, trans);
        }
    }

    HK_TIMER_END();

}

void hkpUserShapePhantomTypeIdentifier::registerShapePhantomType( hkpPhantomType t )
{
    m_shapePhantomTypes.pushBack(t);
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
