// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Physics2012/Collide/hkpExport.h>
#include <Common/Base/UnitTest/hkUnitTest.h>

#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics2012/Utilities/Collide/TriggerVolume/hkpTriggerVolume.h>
#include <Physics2012/Collide/Dispatch/hkpAgentRegisterUtil.h>

#include <Physics2012/Utilities/Collide/hkpShapeGenerator.h>

class OverlappingBodiesTestTriggerVolume : public hkpTriggerVolume
{
    public:
        OverlappingBodiesTestTriggerVolume ( hkpRigidBody* triggerBody )
            : hkpTriggerVolume( triggerBody )
        {

        }

        using hkpTriggerVolume::triggerEventCallback;

        virtual void triggerEventCallback( hkpRigidBody* body, EventType type )
        {
            if ( type & ENTERED_EVENT )
            {
                m_bodiesTouched.pushBack( body );
            }
            if ( type & LEFT_EVENT )
            {
                m_bodiesLeft.pushBack( body );
            }
        }

    public:
        hkArray<hkpRigidBody*> m_bodiesTouched;
        hkArray<hkpRigidBody*> m_bodiesLeft;
};

// Creates a rigid body for use in this test.
static hkpRigidBody* createOverlappingBody( hkVector4Parameter boxPosition, hkVector4Parameter fullBoxExtent )
{
    hkpRigidBody* body;

    hkVector4 halfExtents;
    halfExtents.setMul( hkSimdReal_Inv2, fullBoxExtent );

    hkpShape* shape = hkpShapeGenerator::createConvexVerticesBox( halfExtents );

    hkpRigidBodyCinfo info;
    info.m_shape = shape;
    info.m_mass = 1.0f;
    info.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
    info.m_position = boxPosition;
    info.m_rotation.setIdentity();
    info.m_qualityType = HK_COLLIDABLE_QUALITY_CRITICAL;
    info.m_gravityFactor = 0.0f;
    info.m_enableDeactivation = false;

    body = new hkpRigidBody( info );

    shape->removeReference();

    return body;
}


static void testOverlappingCollisions( hkpWorld* world, hkReal deltaTime )
{
    world->lock();
    // Create a set of bodies moving slowly downwards.
    const hkReal heightUnit = 1.0f;
    const hkReal widthUnit = 1.0f;
    const hkReal maxHeight = 4.0f;
    const hkReal heightOfTriggerVolume = heightUnit;
    const hkReal gap = heightUnit;
    const hkReal distance = ( maxHeight * 2.0f ) + ( gap * 2.0f ) + heightOfTriggerVolume;
    const int numSteps = 400;
    const hkReal speed = distance / ( numSteps * deltaTime );

    hkVector4 boxPosition = hkVector4::getZero();
    hkVector4 fullBoxExtent; fullBoxExtent.set( widthUnit, heightUnit, 1.0f );

    hkReal currentHeight = heightUnit;

    hkArray<hkpRigidBody*> overlappingBodies;

    while ( currentHeight <= maxHeight )
    {
        boxPosition(1) = 0.0f;
        while ( boxPosition(1) + currentHeight <= maxHeight )
        {
            // Create the body.
            {
                hkpRigidBody *const body = createOverlappingBody( boxPosition, fullBoxExtent );
                world->addEntity( body );
                hkVector4 velocity;
                velocity.set( 0.0f, speed, 0.0f );
                body->setLinearVelocity( velocity );
                overlappingBodies.pushBack( body );
            }

            boxPosition(0) += 2.0f * widthUnit;
            boxPosition(1) += heightUnit * 0.5f;
        }
        currentHeight += heightUnit;
    }

    // Create a long thin trigger volume that they all pass through.
    hkVector4 triggerVolumeFullExtent;
    triggerVolumeFullExtent.set( boxPosition(0), heightOfTriggerVolume, 1.0f );
    // Let the triggerVolume start at the 'maxHeight + gap' set altitude which all bodies will pass.
    hkVector4 triggerVolumePosition; triggerVolumePosition.set( 0.0f, maxHeight + gap, 0.0f );
    triggerVolumePosition.setAddMul( triggerVolumePosition, triggerVolumeFullExtent, hkSimdReal_Inv2 );

    hkpRigidBody *const triggerBody = createOverlappingBody( triggerVolumePosition, triggerVolumeFullExtent );
    OverlappingBodiesTestTriggerVolume* triggerVolume = new OverlappingBodiesTestTriggerVolume( triggerBody );
    world->addEntity( triggerBody );

    const int numBodies = overlappingBodies.getSize();

    for ( int i = 0; i < numSteps; ++i )
    {
        world->unlock();
        world->stepDeltaTime( deltaTime );
        world->lock();
        for ( int j = 0; j < numBodies; ++j )
        {
            // Confirm the objects are neither deflected nor rotated.
            HK_TEST( hkMath::fabs( overlappingBodies[j]->getLinearVelocity()( 0 ) ) < 0.01f );
            HK_TEST( hkMath::fabs( overlappingBodies[j]->getLinearVelocity()( 1 ) - speed ) < 0.01f );
            HK_TEST( hkMath::fabs( overlappingBodies[j]->getLinearVelocity()( 2 ) ) < 0.01f );
            HK_TEST( overlappingBodies[j]->getAngularVelocity().lengthSquared<3>().getReal() < 0.01f );
        }
    }

    // Check that all bodies got the correct collisions
    for ( int i = 0; i < numBodies; ++i )
    {
        {
            const int indexTouched = triggerVolume->m_bodiesTouched.indexOf( overlappingBodies[i] );
            HK_TEST( indexTouched != -1 );
            if ( indexTouched != -1 )
            {
                triggerVolume->m_bodiesTouched.removeAt( indexTouched );
            }
        }
        {
            const int indexLeft = triggerVolume->m_bodiesLeft.indexOf( overlappingBodies[i] );
            HK_TEST( indexLeft != -1 );
            if ( indexLeft != -1 )
            {
                triggerVolume->m_bodiesLeft.removeAt( indexLeft );
            }
        }
    }
    HK_TEST( triggerVolume->m_bodiesTouched.getSize() == 0 );
    HK_TEST( triggerVolume->m_bodiesLeft.getSize() == 0 );

    // Check our reference count for the bodies in accurate.
    for ( int i = 0; i < numBodies; ++i )
    {
        HK_TEST( overlappingBodies[i]->getReferenceCount() == 2 );
    }

    world->removeEntity( triggerBody );
    triggerVolume->removeReference();
    triggerBody->removeReference();
    for ( int i = 0; i < numBodies; ++i )
    {
        world->removeEntity( overlappingBodies[i] );
        overlappingBodies[i]->removeReference();
    }
    world->unlock();
}

static void testOverlappingBodiesCollisionsWithShapeUpdates( hkpWorld* world, hkReal deltaTime )
{
    // Checks that:
    // [HVK-6268] - Calling hkpRigidBody::updateShape() or hkpRigidBody::setShape() on a body inside
    // a trigger volume should keep the body as 'Overlapping' and should not Assert/Crash.

    world->lock();

    const int numSteps = 100; // Number of 'hkpWorld::stepDeltaTime()' that will be taken in this test.

    const hkReal heightUnit = 1.0f;
    const hkReal widthUnit = 1.0f;
    const hkReal triggerVolumeScale = 20.0f;
    const hkReal distance = -2.5f * triggerVolumeScale * heightUnit; // distance ran by the bodies we'll create.
    const hkReal speed = distance / ( numSteps * deltaTime ); // speed of the bodies we'll create.

    hkVector4 fullBoxExtent; fullBoxExtent.set( widthUnit, heightUnit, 1.0f );

    // Create a set of bodies moving slowly downwards as well as a OverlappingBodiesTestTriggerVolume.
    hkArray<hkpRigidBody*> overlappingBodies;
    OverlappingBodiesTestTriggerVolume* triggerVolume = HK_NULL;
    {
        const hkReal heightDelta = 0.25f;


        hkReal heightIncrement = 0.0f;
        for (hkReal x = -10.0f; x < 10.0f; x += 2.0f)
        {
            hkVector4 boxPosition;
            boxPosition.set(x, 0.5f * triggerVolumeScale * heightUnit + heightIncrement, 0.0f);


            hkpRigidBody *const body = createOverlappingBody( boxPosition, fullBoxExtent );

            world->addEntity( body );
            hkVector4 velocity;
            velocity.set( 0.0f, speed, 0.0f );
            body->setLinearVelocity( velocity );
            overlappingBodies.pushBack( body );
            heightIncrement += heightDelta;
        }

        // Create a long thin trigger volume which they will all pass through.
        hkVector4 triggerVolumeFullExtent;
        triggerVolumeFullExtent.set( triggerVolumeScale * widthUnit, triggerVolumeScale * heightUnit, 1.0f );

        hkVector4 triggerVolumePosition; triggerVolumePosition.set( -10.0f, -10, 0.0f );

        hkpRigidBody *const triggerBody = createOverlappingBody( triggerVolumePosition, triggerVolumeFullExtent );
        triggerVolume = new OverlappingBodiesTestTriggerVolume( triggerBody );

        world->addEntity( triggerBody );
    }

    // Create a number of Shapes that will be used to update the shapes of the different rigid bodies found in overlappingBodies.
    hkVector4 shapeExtents;
    shapeExtents.set(0.5f, 0.5f, 0.5f);

    hkpShape* shapes [] = { hkpShapeGenerator::createConvexShape(shapeExtents, hkpShapeGenerator::BOX, HK_NULL),
                            hkpShapeGenerator::createConvexShape(shapeExtents, hkpShapeGenerator::SPHERE, HK_NULL),
                            hkpShapeGenerator::createConvexShape(shapeExtents, hkpShapeGenerator::CAPSULE, HK_NULL),
                          };

    const int numBodies = overlappingBodies.getSize();
    const int numShapes = sizeof(shapes) / sizeof(shapes[0]);

    int lastShapeUsed = 0; // Used to determine which shape to use next.

    for ( int i = 0; i < numSteps; ++i )
    {
        world->unlock();
        world->stepDeltaTime( deltaTime );
        world->lock();

        for ( int j = 0; j < numBodies; ++j ) // Go through each body to potentially update their shape.
        {
            hkpRigidBody* body = overlappingBodies[j];
            if ( (j & 0x3) == 0) // if j is a multiple of 4.
            {
                if ( (lastShapeUsed & 0x1) == 0) // if lastShapeUsed is an even number
                {
                    const hkpShape* previousShape = body->getCollidable()->getShape();

                    // Handle reference counting here.
                    body->getCollidableRw()->setShape( shapes[lastShapeUsed] );
                    shapes[lastShapeUsed]->addReference();

                    if (previousShape)
                    {
                        previousShape->removeReference();
                    }

                    // Let the hkpRigidBody know that it's shape has changed.
                    body->updateShape();
                }
                else
                {
                    body->setShape( shapes[lastShapeUsed] );
                }

                // Update the lastShape used, rolling back to 0 if we have used them all.
                ++lastShapeUsed;
                if (lastShapeUsed == numShapes)
                {
                    lastShapeUsed = 0;
                }
            }
            // Confirm the objects are neither deflected nor rotated.
            HK_TEST( hkMath::fabs( overlappingBodies[j]->getLinearVelocity()( 0 ) ) < 0.01f );
            HK_TEST( hkMath::fabs( overlappingBodies[j]->getLinearVelocity()( 1 ) - speed ) < 0.01f );
            HK_TEST( hkMath::fabs( overlappingBodies[j]->getLinearVelocity()( 2 ) ) < 0.01f );
            HK_TEST( overlappingBodies[j]->getAngularVelocity().lengthSquared<3>().getReal() < 0.01f );
        }
    }

    // Check that all bodies got the correct collisions (We know they will have all entered and left the TriggerVolume).
    while ( !triggerVolume->m_bodiesTouched.isEmpty() )
    {
        hkpRigidBody* bodyTouched = triggerVolume->m_bodiesTouched.back();

        const int overlappingBodyTouched = overlappingBodies.indexOf( bodyTouched );
        HK_TEST( overlappingBodyTouched != -1 );

        {
            const int indexLeft = triggerVolume->m_bodiesLeft.indexOf( bodyTouched );
            HK_TEST( indexLeft != -1 );
            if ( indexLeft != -1 )
            {
                triggerVolume->m_bodiesLeft.removeAt( indexLeft );
            }
        }
        triggerVolume->m_bodiesTouched.popBack();

    }

    HK_TEST( triggerVolume->m_bodiesTouched.getSize() == 0 );
    HK_TEST( triggerVolume->m_bodiesLeft.getSize() == 0 );

    // Check our reference count for the bodies in accurate.
    for ( int i = 0; i < numBodies; ++i )
    {
        HK_TEST( overlappingBodies[i]->getReferenceCount() == 2 );
    }

    world->removeEntity( triggerVolume->m_triggerBody );
    triggerVolume->removeReference();
    triggerVolume->m_triggerBody->removeReference();

    for ( int i = 0; i < numBodies; ++i )
    {
        world->removeEntity( overlappingBodies[i] );
        overlappingBodies[i]->removeReference();
    }

    for ( int i = 0; i < numShapes; ++i )
    {
        // Check our reference count for the shapes is accurate.
        HK_TEST( shapes[i]->getReferenceCount() == 1 );
        shapes[i]->removeReference();
    }

    world->unlock();
}

static void testOverlappingBodiesCollisionsWithTVShapeUpdates( hkpWorld* world, hkReal deltaTime )
{
    // Checks that:
    // [HVK-6502] - TriggerVolumes correctly report bodies overlap after their shapes are modified
    //              using 'hkpRigidBody::setShape()', or 'hkpRigidBody::updateShape()'.

    world->lock();

    //
    // Create a set of bodies which will be overlapping with our trigger volume.
    //
    hkVector4 unitBoxSize; unitBoxSize.set( 1.0f, 1.0f, 1.0f );
    {
        hkVector4Comparison mask[4];
        mask[0].set( hkVector4ComparisonMask::MASK_NONE );
        mask[1].set( hkVector4ComparisonMask::MASK_X );
        mask[2].set( hkVector4ComparisonMask::MASK_Y );
        mask[3].set( hkVector4ComparisonMask::MASK_Z );
        // Each component of the overlapping bodies positions' will either be
        // boxesAabbMax[i] or boxesAabbMin[i].
        hkVector4 boxesAabbMax; boxesAabbMax.set( 2.5f, 2.5f, 2.5f );
        hkVector4 boxesAabbMin; boxesAabbMin.set(-2.5f,-2.5f,-2.5f );

        for ( int i = 0; i < 4; i++ )
        {
            hkVector4 boxPos;
            boxPos.setSelect( mask[i], boxesAabbMin, boxesAabbMax );
            hkpRigidBody* overlappingBody = createOverlappingBody( boxPos, unitBoxSize );

            // Finally add our body to the world, removing 'our' reference since the world now "owns" it.
            world->addEntity( overlappingBody );
            overlappingBody->removeReference();
        }
    }
    const int numberOfOverlappingBodies = 4;
    //
    // Create our triggerBody and the shapes it will be alternating between.
    //
    hkpRigidBody* triggerBody;
    hkArray<const hkpShape*> triggerVolumeShapes;
    {
        triggerBody = createOverlappingBody( hkVector4::getZero(), unitBoxSize );
        world->addEntity( triggerBody );

        triggerVolumeShapes.pushBack( triggerBody->getCollidable()->getShape() );

        hkVector4 triggerVolumeSize; triggerVolumeSize.setMul( hkSimdReal_5, unitBoxSize );
        triggerVolumeShapes.pushBack( hkpShapeGenerator::createConvexVerticesBox( triggerVolumeSize ) );
        // Add a reference to the triggerBody's shape so we can refer to it throughout this function.
        triggerBody->getCollidable()->getShape()->addReference();
    }
    //
    // Create our triggerVolume and step through the world, performing all the necessary tests.
    //
    {
        OverlappingBodiesTestTriggerVolume* triggerVolume = ( new OverlappingBodiesTestTriggerVolume( triggerBody ) );

        // We will test both changing the triggerVolume's Shape between 'hkpShape::setShape()' and
        // 'hkpShape::updateShape()' methods.
        for ( hkUint32 methodIndex = 0; methodIndex < 2; ++methodIndex )
        {
            // We will go through numberOfIterations worldStep where we will only be changing the shape of our
            // triggerVolume.
            const hkUint32 numberOfIterations = 5;
            triggerBody->setShape( triggerVolumeShapes[0] );
            for ( hkUint32 stepIndex = 0; stepIndex < numberOfIterations; ++stepIndex )
            {
                // Update our trigger volume's shape based on the chosen method.
                {
                    hkpCollidable* collidable = triggerBody->getCollidableRw();

                    const hkpShape* shape = collidable->getShape();
                    const hkpShape* newShape = (shape == triggerVolumeShapes[0]) ? triggerVolumeShapes[1] : triggerVolumeShapes[0];

                    if ( methodIndex == 0)
                    {
                        triggerBody->setShape( newShape );
                    }
                    else
                    {
                        const hkpShape* previousShape = collidable->getShape();

                        // Handle reference counting here.
                        collidable->setShape( newShape );
                        newShape->addReference();
                        previousShape->removeReference();

                        // Let the hkpRigidBody know that it's shape has changed.
                        triggerBody->updateShape();
                    }
                }

                world->stepDeltaTime( deltaTime );

                if ( stepIndex % 2 == 0 )
                {
                    // numberOfOverlappingBodies should have now been added to the trigger volume's
                    // overlapping collidables.
                    HK_TEST1(   triggerVolume->m_bodiesTouched.getSize() ==
                                triggerVolume->m_bodiesLeft.getSize() + numberOfOverlappingBodies,
                                "The TriggerVolume should have reported an overlap.");
                }
                else
                {
                    // All bodies which were marked as overlapping with the trigger volume in the previous step,
                    // should not be overlapping anymore.
                    HK_TEST1( triggerVolume->m_bodiesTouched.getSize() == triggerVolume->m_bodiesLeft.getSize(),
                        "The TriggerVolume should have removed an overlap.");
                }
            }
        }
        triggerVolume->removeReference();
    }
    triggerBody->removeReference();

    // Delete all trigger volume shapes we created.
    hkReferencedObject::removeReferences( triggerVolumeShapes.begin(), triggerVolumeShapes.getSize() );
    world->unlock();
}


static int overlappingCollisions_main()
{
    hkpWorldCinfo info;
    {
        info.m_gravity.setZero();
        info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);
        info.setBroadPhaseWorldSize( 100.0f );
        info.m_enableDeactivation = false;
    }

    // Discrete simulation.
    {
        info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_DISCRETE;

        hkpWorld *const world = new hkpWorld( info );

        const hkReal deltaTime = 1.0f / 60.0f;

        hkpAgentRegisterUtil::registerAllAgents( world->getCollisionDispatcher() );

        testOverlappingCollisions( world, deltaTime );
        testOverlappingBodiesCollisionsWithShapeUpdates( world, deltaTime );
        testOverlappingBodiesCollisionsWithTVShapeUpdates( world, deltaTime );
        world->removeReference();
    }

    // Continuous simulation.
    {
        info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_CONTINUOUS;

        hkpWorld *const world = new hkpWorld( info );

        const hkReal deltaTime = 1.0f / 60.0f;

        hkpAgentRegisterUtil::registerAllAgents( world->getCollisionDispatcher() );

        testOverlappingCollisions( world, deltaTime );
        testOverlappingBodiesCollisionsWithShapeUpdates( world, deltaTime );
        testOverlappingBodiesCollisionsWithTVShapeUpdates( world, deltaTime );
        world->removeReference();
    }

    // Multithreaded simulation.
    {
        info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED;

        hkpWorld *const world = new hkpWorld( info );

        const hkReal deltaTime = 1.0f / 60.0f;

        world->lock();
        hkpAgentRegisterUtil::registerAllAgents( world->getCollisionDispatcher() );
        world->unlock();

        testOverlappingCollisions( world, deltaTime );
        testOverlappingBodiesCollisionsWithShapeUpdates(world, deltaTime);

        world->markForWrite();
        world->removeReference();
    }


    return 0;
}

HK_TEST_REGISTER(overlappingCollisions_main, "Slow", "Physics2012/Test/UnitTest/Utilities/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
