// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
// ------------------------------------------------------TKBMS v1.0

// PCH
#include <Physics2012/Collide/hkpCollide.h>
#include <Physics2012/Collide/hkpExport.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Internal/ConvexHull/hkGeometryUtility.h>
#include <Common/GeometryUtilities/Mesh/hkMeshShape.h>
#include <Common/GeometryUtilities/Mesh/hkMeshVertexBuffer.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshSystem.h>
#include <Common/GeometryUtilities/Mesh/Converters/SceneDataToMesh/hkSceneDataToMeshConverter.h>

#include <Physics2012/Collide/Util/ShapeShrinker/hkpShapeShrinker.h>
#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics2012/Collide/Shape/Convex/Sphere/hkpSphereShape.h>
#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>

#include <Physics2012/Utilities/Collide/ShapeUtils/CreateShape/hkpCreateShapeUtility.h>

#define DEBUG_LOG_IDENTIFIER "p12.collide.util.createshape"
#include <Common/Base/System/Log/hkLog.hxx>


static void _getStridedVertices(const hkArray<hkVector4>& vertices, const hkMatrix4& vertexTransform, hkStridedVertices& stridedVertsOut, hkArray<hkVector4>& tempStorageOut)
{
    stridedVertsOut.m_numVertices   = 0;
    stridedVertsOut.m_striding      = sizeof(hkVector4);

    const hkBool32 vertexTransformIsIdentity = vertexTransform.isApproximatelyIdentity(hkSimdReal::fromFloat(1e-3f));

    // Transform vertices if case
    if (!vertexTransformIsIdentity)
    {
        int numVerts = vertices.getSize();
        tempStorageOut.setSize(numVerts);
        const hkVector4 * pSrc = vertices.begin();
        hkVector4       * pDst = tempStorageOut.begin();

        for(int vi = 0 ; vi < numVerts ; vi++)
        {
            vertexTransform.transformPosition(pSrc[vi], pDst[vi]);
        }

        if (tempStorageOut.getSize() > 0)
        {
            stridedVertsOut.m_numVertices   = tempStorageOut.getSize();
            stridedVertsOut.m_vertices      = &tempStorageOut.begin()[0](0);
        }
    }
    else
    {
        // Can convert directly, no need to copy
        stridedVertsOut.m_numVertices   = vertices.getSize();
        stridedVertsOut.m_vertices      = &vertices.begin()[0](0);
    }
}

//  Computes the AABB from a set of vertices

hkResult HK_CALL hkpCreateShapeUtility::computeAABB(const hkArray<hkVector4>& vertices, const hkMatrix4& vertexTransform, hkReal minBoxExtent, hkVector4& halfExtentsOut, hkTransform& obbTransformOut)
{
    hkArray<hkVector4>  tempStorage;
    hkStridedVertices   stridedVerts;
    _getStridedVertices(vertices, vertexTransform, stridedVerts, tempStorage);

    if (stridedVerts.m_numVertices > 0)
    {
        hkVector4 minBox; minBox.setConstant<HK_QUADREAL_MAX>();
        hkVector4 maxBox; maxBox.setConstant<HK_QUADREAL_MINUS_MAX>();
        {
            const int floatstriding = stridedVerts.m_striding/hkSizeOf(hkReal);
            for( int i = 0; i < stridedVerts.m_numVertices * 3; i++)
            {
                const int vidx = i / 3; // vertex index (0..numVertices)
                const int component = i % 3; // component (0,1,2 == x,y,z)

                hkReal value = stridedVerts.m_vertices[vidx*floatstriding+component];

                maxBox(component) = hkMath::max2(maxBox(component), value);

                minBox(component) = hkMath::min2(minBox(component), value);

            }
        }

        hkVector4 boxCenter; boxCenter.setInterpolate(minBox, maxBox, hkSimdReal_Inv2);

        obbTransformOut.set(hkQuaternion::getIdentity(), boxCenter);
        halfExtentsOut.setSub(maxBox, minBox);

        if (minBoxExtent > hkReal(0))
        {
            hkVector4 minBoxExtentSr; minBoxExtentSr.setAll(minBoxExtent);
            halfExtentsOut.setMax(minBoxExtentSr, halfExtentsOut);
        }

        halfExtentsOut.mul(hkSimdReal_Inv2);

        if (halfExtentsOut.lengthSquared<3>() > hkSimdReal::fromFloat(0.0001f*0.0001f))
        {
            return HK_SUCCESS;
        }
    }

    return HK_FAILURE;
}

//  Computes the OBB of a set of vertices

hkResult HK_CALL hkpCreateShapeUtility::computeOBB(const hkArray<hkVector4>& vertices, const hkMatrix4& vertexTransform, hkReal minBoxExtent, hkVector4& halfExtentsOut, hkTransform& obbTransformOut)
{
    // Use OBB
    hkArray<hkVector4>  tempStorage;
    hkStridedVertices   stridedVerts;
    _getStridedVertices( vertices, vertexTransform, stridedVerts, tempStorage );

    if (stridedVerts.m_numVertices > 0)
    {
        hkBool wasEnabled = hkError::getInstance().isEnabled( 0x34df5494 );
        hkError::getInstance().setEnabled( 0x34df5494, false );

        hkGeometryUtility::calcObb(stridedVerts, halfExtentsOut, obbTransformOut);

        hkError::getInstance().setEnabled( 0x34df5494, wasEnabled );

        if (minBoxExtent > hkReal(0))
        {
            hkVector4 minBoxHalfExtent; minBoxHalfExtent.setAll(minBoxExtent * 0.5f);
            halfExtentsOut.setMax(minBoxHalfExtent, halfExtentsOut);
        }

        if (halfExtentsOut.lengthSquared<3>() > hkSimdReal::fromFloat(0.0001f*0.0001f) )
        {
            return HK_SUCCESS;
        }
    }

    return HK_FAILURE;
}

//  Creates a box shape from a set of vertices

hkResult HK_CALL hkpCreateShapeUtility::createBoxShape(CreateShapeInput& input, ShapeInfoOutput& output)
{
    // Handle the case where the name is null
    const char * szName = input.m_szMeshName ? input.m_szMeshName : "";

    // Compute OBB / AABB
    hkVector4   halfExtents;
    hkTransform geomFromBox;
    hkResult    obbOk;
    {
        if (input.m_bestFittingShapes)
        {
            obbOk = computeOBB (input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }
        else
        {
            obbOk = computeAABB(input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }

        if (obbOk.isFailure())
        {
            HK_WARN_ALWAYS(0x512ba184, "Couldn't calculate OBB / AABB for mesh " << szName);
            return HK_FAILURE;
        }
    }

    // see if we need to add the obb transform to the transform stack
    output.m_extraShapeTransform    = input.m_extraShapeTransform;
    output.m_decomposedWorldT       = input.m_decomposedWorldT;
    if (!geomFromBox.isApproximatelyEqual( hkTransform::getIdentity() ))
    {
        // EXP-1202
        output.m_extraShapeTransform = geomFromBox;

        // add the obbTrans to the decomposed shape trans
        hkTransform res;
        res.setMul(output.m_decomposedWorldT, geomFromBox);
        output.m_decomposedWorldT = res;
    }

    bool automaticShrinkingEnabled  = input.m_enableAutomaticShapeShrinking;
    hkReal extraRadiusOverride      = input.m_extraRadiusOverride; // Override range is between 0 and 1, so we are safe to use this for the time being.
    hkReal extraRadius              = hkReal(0);

    if ( !automaticShrinkingEnabled )
    {
        if ( extraRadiusOverride >= hkReal(0) )
        {
            extraRadius = extraRadiusOverride;
        }
        else
        {
            extraRadius = input.m_defaultConvexRadius;
        }
    }

    hkpBoxShape * boxShape = new hkpBoxShape( halfExtents, extraRadius );

    // Shrink shape if option is enabled (with meaningful values).
    if ( automaticShrinkingEnabled && input.m_relShrinkRadius != hkReal(0) && input.m_maxVertexDisplacement != hkReal(0) )
    {
        hkpShapeShrinker::shrinkBoxShape( boxShape, input.m_relShrinkRadius, input.m_maxVertexDisplacement );
    }

    // Issue some warnings if final convex/extra radius is either 0.0 or rather large.
    {
        hkReal convexRadius = boxShape->getRadius();
        hkVector4 absExtent; absExtent.setAbs(halfExtents); 
        const hkReal meshSize = absExtent.horizontalMax<3>().getReal();

        if ( convexRadius == hkReal(0) )
        {
            HK_WARN_ALWAYS (0xabbabfd2, "'" << szName << "' - Calculated 'extra radius' for box shape is 0. Performance may be affected.");
        }
        else if ( convexRadius > meshSize )
        {
            HK_WARN_ALWAYS (0xabbabad3, "'" << szName << "' - Calculated 'extra radius' for box shape is very large (radius " << convexRadius << " > halfExtent " << meshSize << " )");
        }
    }

    output.m_shape    = boxShape;
    output.m_isConvex = true;

    return HK_SUCCESS;
}

//  Creates a sphere shape from a set of vertices

hkResult HK_CALL hkpCreateShapeUtility::createSphereShape(CreateShapeInput& input, ShapeInfoOutput& output)
{
    // Handle the case where the name is null
    const char * szName = input.m_szMeshName ? input.m_szMeshName : "";

    // Compute OBB / AABB
    hkVector4   halfExtents;
    hkTransform geomFromBox;
    hkResult    obbOk;
    {
        if (input.m_bestFittingShapes)
        {
            obbOk = computeOBB (input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }
        else
        {
            obbOk = computeAABB(input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }

        if (obbOk.isFailure())
        {
            HK_WARN_ALWAYS(0x512ba184, "Couldn't calculate OBB / AABB for mesh " << szName);
            return HK_FAILURE;
        }
    }

    // see if we need to add the obb transform to the transform stack
    output.m_extraShapeTransform    = input.m_extraShapeTransform;
    output.m_decomposedWorldT       = input.m_decomposedWorldT;
    if (!geomFromBox.isApproximatelyEqual( hkTransform::getIdentity() ))
    {
        // EXP-1202
        output.m_extraShapeTransform = geomFromBox;

        // add the obbTrans to the decomposed shape trans
        hkTransform res;
        res.setMul(output.m_decomposedWorldT, geomFromBox);
        output.m_decomposedWorldT = res;
    }

    hkVector4 absExtent; absExtent.setAbs(halfExtents); 
    const hkReal radius = absExtent.horizontalMax<3>().getReal();
    output.m_shape      = new hkpSphereShape (radius);
    output.m_isConvex   = true;

    if ( input.m_enableAutomaticShapeShrinking )
    {
        if ( input.m_relShrinkRadius != hkReal(0) && input.m_maxVertexDisplacement != hkReal(0) )
        {
            Log_Warning( "'{}' - Shape shrinking not supported for sphere shapes.", szName );
        }
    }
    else
    {
        if ( input.m_defaultConvexRadius != 0.0f )
        {
            Log_Warning( "'{}' - Convex radius not supported for sphere shapes.", szName );
        }
    }

    return HK_SUCCESS;
}

//  Creates a capsule shape from a set of vertices

hkResult HK_CALL hkpCreateShapeUtility::createCapsuleShape(CreateShapeInput& input, ShapeInfoOutput& output)
{
    // Handle the case where the name is null
    const char * szName = input.m_szMeshName ? input.m_szMeshName : "";

    // Compute OBB / AABB
    hkVector4   halfExtents;
    hkTransform geomFromBox;
    hkResult    obbOk;
    {
        if (input.m_bestFittingShapes)
        {
            obbOk = computeOBB (input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }
        else
        {
            obbOk = computeAABB(input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }

        if (obbOk.isFailure())
        {
            HK_WARN_ALWAYS(0x512ba184, "Couldn't calculate OBB / AABB for mesh " << szName);
            return HK_FAILURE;
        }
    }

    // Just copy the transforms
    output.m_extraShapeTransform    = input.m_extraShapeTransform;
    output.m_decomposedWorldT       = input.m_decomposedWorldT;

    hkVector4 vecA;
    hkVector4 vecB;
    int majorAxis = halfExtents.getIndexOfMaxAbsComponent<3>();
    vecA.setZero();
    vecA(majorAxis) = -halfExtents(majorAxis);

    vecB.setZero();
    vecB(majorAxis) = halfExtents(majorAxis);

    halfExtents.zeroComponent(majorAxis);
    hkReal radius = halfExtents(halfExtents.getIndexOfMaxAbsComponent<3>());

    // Move in capsule end points
    hkVector4 radiusDelta;
    radiusDelta.setZero();
    radiusDelta(majorAxis) = radius;
    vecA.add(radiusDelta);
    vecB.sub(radiusDelta);

    // Transform end points
    hkVector4 endPointA;
    hkVector4 endPointB;
    endPointA.setTransformedPos(geomFromBox, vecA);
    endPointB.setTransformedPos(geomFromBox, vecB);

    // EXP-386 : Handle degenerate capsules
    if (endPointA.allEqual<3>(endPointB,hkSimdReal::fromFloat(1e-3f)))
    {
        HK_WARN_ALWAYS(0xabba873d, "Degenerate Capsule (" << szName << "): Using Sphere Shape Instead");

        return createSphereShape(input, output);
    }

    output.m_shape  = new hkpCapsuleShape (endPointA, endPointB, radius );
    output.m_isConvex = true;

    if ( input.m_enableAutomaticShapeShrinking )
    {
        if ( input.m_relShrinkRadius != hkReal(0) && input.m_maxVertexDisplacement != hkReal(0) )
        {
            Log_Warning( "'{}' - Shape shrinking not supported for capsule shapes.", szName );
        }
    }
    else
    {
        if ( input.m_defaultConvexRadius != hkReal(0) )
        {
            Log_Warning( "'{}' - Convex radius not supported for capsule shapes.", szName );
        }
    }

    return HK_SUCCESS;
}

//  Creates a cylinder shape from a set of vertices

hkResult HK_CALL hkpCreateShapeUtility::createCylinderShape(CreateShapeInput& input, ShapeInfoOutput& output)
{
    // Handle the case where the name is null
    const char * szName = input.m_szMeshName ? input.m_szMeshName : "";

    // Compute OBB / AABB
    hkVector4   halfExtents;
    hkTransform geomFromBox;
    hkResult    obbOk;
    {
        if (input.m_bestFittingShapes)
        {
            obbOk = computeOBB (input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }
        else
        {
            obbOk = computeAABB(input.m_vertices, input.m_vertexTM, input.m_minBoxExtent, halfExtents, geomFromBox);
        }

        if (obbOk.isFailure())
        {
            HK_WARN_ALWAYS(0x512ba184, "Couldn't calculate OBB / AABB for mesh " << szName);
            return HK_FAILURE;
        }
    }

    // Just copy the transforms
    output.m_extraShapeTransform    = input.m_extraShapeTransform;
    output.m_decomposedWorldT       = input.m_decomposedWorldT;

    hkReal radius;
    hkVector4 pointA;
    hkVector4 pointB;
    hkGeometryUtility::calcBestCylinder( input.m_vertices, halfExtents, geomFromBox, pointA, pointB, radius);

    bool automaticShrinkingEnabled  = input.m_enableAutomaticShapeShrinking;
    hkReal extraRadiusOverride      = input.m_extraRadiusOverride;

    hkReal extraRadius = 0;
    if ( !automaticShrinkingEnabled )
    {
        if ( extraRadiusOverride >= 0 )
        {
            extraRadius = extraRadiusOverride;
        }
        else
        {
            extraRadius = input.m_defaultConvexRadius;
        }
    }

    hkpCylinderShape* cylinderShape = new hkpCylinderShape(pointA, pointB, radius, extraRadius);

    // Shrink shape if option is enabled (with meaningful values).
    if ( automaticShrinkingEnabled && input.m_relShrinkRadius != 0 && input.m_maxVertexDisplacement != 0 )
    {
        hkpShapeShrinker::shrinkCylinderShape( cylinderShape, input.m_relShrinkRadius, input.m_maxVertexDisplacement);
    }

    // Issue some warnings if calculated convex/extra radius is either 0.0 or rather large.
    {
        hkReal convexRadius = cylinderShape->getRadius();
        const hkReal meshSize = halfExtents (halfExtents.getIndexOfMaxAbsComponent<3>());

        if ( convexRadius == 0 )
        {
            HK_WARN_ALWAYS (0xabbabfd4, "'" << szName << "' - Calculated 'extra radius' for cylinder shape is 0. Performance may be affected.");
        }
        else if ( convexRadius > meshSize )
        {
            HK_WARN_ALWAYS (0xabbabad5, "'" << szName << "' - Calculated 'extra radius' for cylinder shape is very large (radius " << convexRadius << " > halfExtent " << meshSize << " )");
        }
    }

    output.m_shape = cylinderShape;
    output.m_isConvex = true;

    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
