// --------------------
// <insert description of file here>
// --------------------

#pragma once		// Include this file only once
#if !defined(PHYSICS_CHARACTER_PROXY_CALLBACKS_H)
#define PHYSICS_CHARACTER_PROXY_CALLBACKS_H

#include <Common/Base/hkBase.h>
#include <Physics2012/Utilities/CharacterControl/CharacterProxy/hkpCharacterProxyListener.h>

// --------------------
//
// Defines/Macros
//
// --------------------

// Maximum size of a deferred event.
#define PHYSICS_CHARACTER_PROXY_DEFERRED_EVENT_MAX_SIZE	(80)

// Contacts with objects with this flag should not be added to the manifold.
#define PCIP_FLAG_IGNORE_CONTACT									(1 << 0)

// Contacts with objects with this flag should not be added to the manifold but should be recorded for
// post-processing.
#define PCIP_FLAG_DEFER_CONTACT									(1 << 1)

// Objects with this flag cannot be pushed by characters.
#define PCIP_FLAG_CANNOT_BE_PUSHED								(1 << 2)

// Objects with this flag are treated as having zero surface velocity to characters.
#define PCIP_FLAG_ZERO_SURFACE_VELOCITY						(1 << 3)

// Objects with this flag are players.
#define PCIP_FLAG_IS_PLAYER										(1 << 4)


// --------------------
//
// Enumerated types
//
// --------------------


// --------------------
//
// Structures/Classes
//
// --------------------

// Forward declarations.
struct hkpCdPoint;
struct hkpCharacterProxyJob;
template <typename TYPE> class hkArraySpu;
class hkpCollidable;
class hkpCpuCharacterProxyCollector;
class hkpFixedBufferCdPointCollector;
class hkpWorldObject;

// A contact to defer for processing after the integration.
//
struct physics_character_proxy_deferred_contact
{
	// The collidable of the character being integrated.
	hkpCollidable const* m_character_collidable;

	// The collidable of the contacted object.
	hkpCollidable const* m_contacted_collidable;

	// The contact point.
	hkContactPoint	m_contact_point;
};
HK_COMPILE_TIME_ASSERT(sizeof(physics_character_proxy_deferred_contact) % 16 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(physics_character_proxy_deferred_contact) <= 
	PHYSICS_CHARACTER_PROXY_DEFERRED_EVENT_MAX_SIZE);

// A character interaction to defer for processing after the integration.
//
struct physics_character_proxy_deferred_character_interaction
{
	// The proxy of the character being integrated.
	hkpCharacterProxy const* m_character_proxy;

	// The proxy of the contacted character.
	hkpCharacterProxy const* m_contacted_proxy;

	// The contact point.
	hkContactPoint m_contact_point;
};
HK_COMPILE_TIME_ASSERT(sizeof(physics_character_proxy_deferred_character_interaction) % 16 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(physics_character_proxy_deferred_character_interaction) <= 
	PHYSICS_CHARACTER_PROXY_DEFERRED_EVENT_MAX_SIZE);

// An object interaction to defer for processing after the integration.
//
struct physics_character_proxy_deferred_object_interaction
{
	// The proxy of the character being integrated.
	hkpCharacterProxy const* m_character_proxy;

	// The event data.
	hkpCharacterObjectInteractionEvent m_event_data;
};
HK_COMPILE_TIME_ASSERT(sizeof(physics_character_proxy_deferred_object_interaction) % 16 == 0);
HK_COMPILE_TIME_ASSERT(sizeof(physics_character_proxy_deferred_object_interaction) <= 
	PHYSICS_CHARACTER_PROXY_DEFERRED_EVENT_MAX_SIZE);

#if !defined (HK_PLATFORM_SPU)

	// Deferred event callback types.
	typedef void (physics_character_proxy_deferred_contact_callback)(physics_character_proxy_deferred_contact const& contact);
	typedef void (physics_character_proxy_deferred_character_interaction_callback)(physics_character_proxy_deferred_character_interaction const& character_interaction);
	typedef void (physics_character_proxy_deferred_object_interaction_callback)(physics_character_proxy_deferred_object_interaction const& object_interaction);

	// Information to initialize the character proxy callbacks system.
	//
	struct physics_character_proxy_callbacks_init_info
	{
		// Callbacks for deferred event processing.
		physics_character_proxy_deferred_contact_callback*						m_deferred_contact_callback;
		physics_character_proxy_deferred_character_interaction_callback*	m_deferred_character_interaction_callback;
		physics_character_proxy_deferred_object_interaction_callback*		m_deferred_object_interaction_callback;

		// Property keys.
		hkUint32	m_character_interaction_key;
		hkUint32	m_object_handle_key;
		hkUint32	m_ignore_handle_key;

		// The invalid object handle value.
		hkUint64 m_invalid_object_handle;
	};

	// A custom listener for the character proxy.
	//
	class physics_custom_character_proxy_listener : public hkpCharacterProxyListener
	{
	public:

		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, physics_custom_character_proxy_listener);

		// Called when the character interacts with another character.
		virtual void characterInteractionCallback (hkpCharacterProxy *proxy, hkpCharacterProxy *otherProxy,
			const hkContactPoint &contact);

		/// Called when the character interacts with another object
		virtual void objectInteractionCallback(hkpCharacterProxy* proxy, 
			const hkpCharacterObjectInteractionEvent& input, hkpCharacterObjectInteractionResult& output);

		// We add in our own custom (per surface) friction model here
		virtual void processConstraintsCallback(const hkpCharacterProxy* proxy,
			const hkArray<hkpRootCdPoint>& manifold, hkSimplexSolverInput& input );
	};

#endif // !defined (HK_PLATFORM_SPU)

// Data packed tightly to control how character proxies interact with an object.
//
struct physics_character_interaction_property {

	// Initialize.
	void initialze()
	{
		m_separation_distance_squared = HK_REAL_MAX;
		m_flags = 0;
	}

	// Get as a 64 bit integer.
	hkUint64 get_as_uint64() const
	{
		hkUint64 data = 0;
		data |= (static_cast<hkUint64 const>(*reinterpret_cast<hkUint32 const*>(&m_separation_distance_squared)) << 32);
		data |= (m_flags << 24);
		return data;
	}

	// Set from a 64 bit integer.
	void set_from_uint64(hkUint64 data)
	{
		hkUint32 temp = static_cast<hkUint32>(data >> 32);
		m_separation_distance_squared = *reinterpret_cast<hkReal*>(&temp);
		m_flags = static_cast<hkUint8>(data >> 24);
	}

	// How close a character must be to begin colliding with this (can allow character to penetrate the collision).
	hkReal	m_separation_distance_squared;

	// Flags.
	hkUint8	m_flags;

	// Padding to 8 bytes.
	hkUint8	m_pad[3];
};
// Must fit in a 64 bit integer!
HK_COMPILE_TIME_ASSERT(sizeof(physics_character_interaction_property) == 8);


// --------------------
//
// Variables
//
// --------------------


// --------------------
//
// Prototypes
//
// --------------------

#if !defined (HK_PLATFORM_SPU)

	// Initialize the character proxy callbacks system.
	//
	// info:	Information to initialize the character proxy callbacks system.
	//
	void physics_character_proxy_callbacks_initialize(physics_character_proxy_callbacks_init_info const& info);

	#if defined (HK_PLATFORM_HAS_SPU)

		// Initialize the character proxy job to provide callback info on SPU.
		//
		// job:	The job holds initialization info.
		//
		void physics_character_proxy_callbacks_initialize_job_for_spu(hkpCharacterProxyJob& job);

	#endif // defined (HK_PLATFORM_HAS_SPU)

#else

	// Initialize the character proxy callbacks system on SPU.
	//
	// job:	The job holds initialization info.
	//
	void physics_character_proxy_callbacks_initialize_spu(hkpCharacterProxyJob const& job);

#endif // !defined (HK_PLATFORM_SPU)

#if !defined (HK_PLATFORM_SPU)

	// Initialize deferred events before integration.
	void physics_character_proxy_initialize_deferred_events();

	// Uninitialize deferred events after integration.
	void physics_character_proxy_uninitialize_deferred_events();

	// Process all the deferred events that were stored up during integration.
	void physics_character_proxy_process_deferred_events();

#endif // !defined (HK_PLATFORM_SPU)

// Add a collision to a character collector.  Does special filtering based on the entities involved.
//
// collector:					(input/output) The collector to potentially add the collision to.
// characters_collidable:	The collidable of the character being integrated.
// cd_point:					The collision information.
//
#if !defined (HK_PLATFORM_SPU)

	void physics_character_proxy_add_cd_point_to_collector(hkpCpuCharacterProxyCollector* collector,
		hkpCollidable const* characters_collidable, hkpCdPoint const& cd_point);

#else

	void physics_character_proxy_add_cd_point_to_collector(hkpCdPoint const& cd_point,
		hkpFixedBufferCdPointCollector* collector);

#endif // !defined (HK_PLATFORM_SPU)

// Called when the character interacts with another character.
//
// proxy:			Character A.
// other_proxy:	Character B.
// contact:			Contact point information.
//
void physics_character_proxy_process_character_interaction(hkpCharacterProxy const* proxy,
	hkpCharacterProxy* other_proxy, hkContactPoint const& contact);

// This is called when a character proxy interacts with a dynamic object.
//
// proxy:	The character proxy.
// input:	The event data.
// output:	(output) The output for the event.
//
void physics_character_proxy_process_object_interaction(hkpCharacterProxy const* proxy,
	hkpCharacterObjectInteractionEvent const& input, hkpCharacterObjectInteractionResult& output);

// We add in our own custom (per surface) friction model here.
//
// proxy:		The character.
// manifold:	The contact manifold.
// input:		Input to the simplex solver.
//
#if !defined (HK_PLATFORM_SPU)

	void physics_character_proxy_process_constraints(hkpCharacterProxy const* proxy,
		hkArray<hkpRootCdPoint> const& manifold, hkSimplexSolverInput& input);

#else

	void physics_character_proxy_process_constraints(hkpCharacterProxy const* proxy,
		hkArraySpu<hkpRootCdPoint> const& manifold, hkpSimplexSolverInput& input); 

#endif // !defined (HK_PLATFORM_SPU)

// Gets an object's character interaction property.
//
// object:	The object.
//
// Returns: The character interaction property.
//
physics_character_interaction_property physics_object_get_character_interaction_property(hkpWorldObject const* object);

#if !defined (HK_PLATFORM_SPU)

	// Sets an object's character interaction property.
	//
	// object:	The object.
	// prop:		The property data to set.
	//
	void physics_object_set_character_interaction_property(hkpWorldObject* object,
		physics_character_interaction_property const& prop);

#endif // !defined (HK_PLATFORM_SPU)

#endif //  !defined(PHYSICS_CHARACTER_PROXY_CALLBACKS_H)
