// --------------------
// <insert description of file here>
// --------------------

#include "physics_character_proxy_callbacks.h"

#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Physics2012/Collide/Agent/Collidable/hkpCdPoint.h>
//#include <Common/Internal/SimplexSolver/hkSimplexSolver.h>
#include <Geometry/Collide/SimplexSolver/hkSimplexSolver.h> // DSFL jfk
#include <Physics2012/Dynamics/Phantom/hkpShapePhantom.h>
#include <Physics2012/Utilities/CharacterControl/CharacterProxy/hkpCharacterProxy.h>
#include <Physics2012/Utilities/CharacterControl/CharacterProxy/Multithreaded/hkpCharacterProxyJobs.h>
#include <Physics2012/Utilities/CharacterControl/CharacterProxy/Multithreaded/Cpu/hkpCpuCharacterProxyCollector.h>

#if defined (HK_PLATFORM_SPU)
	#include <Physics2012/Utilities/CharacterControl/CharacterProxy/Multithreaded/Spu/hkpSpuCharacterProxyConfig.h>
	#include <Physics2012/Utilities/CharacterControl/CharacterProxy/Multithreaded/Spu/hkpSpuCharacterProxyUtil.h>
#endif // defined (HK_PLATFORM_SPU)

// --------------------
//
// Defines/Macros
//
// --------------------

// How far below the proxy an object must be to be considered under it.
#define OBJECT_UNDER_PROXY_Y_THRESHOLD					(-0.25f)

// Maximum contacts to defer for processing after the integration.
#define MAX_DEFERRED_CONTACTS								(500)

// Maximum character interactions to defer for processing after the integration.
#define MAX_DEFERRED_CHARACTER_INTERACTIONS			(200)

// Maximum object interactions to defer for processing after the integration.
#define MAX_DEFERRED_OBJECT_INTERACTIONS				(200)


// --------------------
//
// Enumerated Types
//
// --------------------


// --------------------
//
// Structures/Classes
//
// --------------------


// --------------------
//
// Global Variables
//
// --------------------


// --------------------
//
// Local Variables
//
// --------------------

// Property keys.
static hkUint32 Character_interaction_key = 0;
static hkUint32 Object_handle_key = 0;
static hkUint32 Ignore_handle_key = 0;

// The invalid object handle value.
static hkUint64 Invalid_object_handle = 0;

#if !defined (HK_PLATFORM_SPU)

	// Callbacks for deferred event processing.
	physics_character_proxy_deferred_contact_callback* Deferred_contact_callback = HK_NULL;
	physics_character_proxy_deferred_character_interaction_callback* Deferred_character_interaction_callback = HK_NULL;
	physics_character_proxy_deferred_object_interaction_callback* Deferred_object_interaction_callback = HK_NULL;

	// Buffer sizes for deferred events.
	static hkUint32 const Deferred_contact_buffer_size =
		MAX_DEFERRED_CONTACTS * sizeof(physics_character_proxy_deferred_contact);
	static hkUint32 const Deferred_character_interaction_buffer_size =
		MAX_DEFERRED_CHARACTER_INTERACTIONS * sizeof(physics_character_proxy_deferred_character_interaction);
	static hkUint32 const Deferred_object_interaction_buffer_size =
		MAX_DEFERRED_OBJECT_INTERACTIONS * sizeof(physics_character_proxy_deferred_object_interaction);

	// Deferred contacts, note that these are both 16 byte aligned for SPUs.
	HK_ALIGN16(static physics_character_proxy_deferred_contact* Deferred_contacts) = HK_NULL;
	HK_ALIGN16(static hkUint32 Num_deferred_contacts) = 0;

	// Deferred character interactions, note that these are both 16 byte aligned for SPUs.
	HK_ALIGN16(static physics_character_proxy_deferred_character_interaction* Deferred_character_interactions) = HK_NULL;
	HK_ALIGN16(static hkUint32 Num_deferred_character_interactions) = 0;

	// Deferred object interactions, note that these are both 16 byte aligned for SPUs.
	HK_ALIGN16(static physics_character_proxy_deferred_object_interaction* Deferred_object_interactions) = HK_NULL;
	HK_ALIGN16(static hkUint32 Num_deferred_object_interactions) = 0;

#else

	// Deferred contacts on the PPU.
	static void* Deferred_contacts_on_ppu = HK_NULL;
	static hkUint32* Num_deferred_contacts_on_ppu = HK_NULL;

	// Deferred character interactions on the PPU.
	static void* Deferred_character_interactions_on_ppu = HK_NULL;
	static hkUint32* Num_deferred_character_interactions_on_ppu = HK_NULL;

	// Deferred object interactions on the PPU.
	static void* Deferred_object_interactions_on_ppu = HK_NULL;
	static hkUint32* Num_deferred_object_interactions_on_ppu = HK_NULL;

	// Store an event during DMA.
	static hkUint8 Deferred_event_buffer[PHYSICS_CHARACTER_PROXY_DEFERRED_EVENT_MAX_SIZE];
	HK_COMPILE_TIME_ASSERT(sizeof(Deferred_event_buffer) % 16 == 0);

#endif // !defined (HK_PLATFORM_SPU)

// --------------------
//
// Console Functions
//
// --------------------


// --------------------
//
// Internal Functions
//
// --------------------

// Gets an object's property from its key.
//
// object:			The object.
// key:				The property key.
// missing_value:	The value to return if the key is missing.
//
// Returns:			The property value.
//
static hkUint64 physics_object_get_uint64_property(hkpWorldObject const* object, hkUint32 key,
	hkUint64 missing_value)
{
	// Is it there?
	if ((object == HK_NULL) || (object->hasProperty(key) == false)) {
		return missing_value;
	}

	// Get the real deal.
	hkpPropertyValue pv = object->getProperty(key);
	return pv.getUint64();
}

#if !defined (HK_PLATFORM_SPU)

	// Process all the deferred contacts that were stored up during integration.
	//
	static void physics_character_proxy_process_deferred_contacts()
	{
		// Handle being full.
		hkUint32 const num_contacts = hkMath::min2(Num_deferred_contacts,
			static_cast<hkUint32>(MAX_DEFERRED_CONTACTS));
		// DSFL jfk
// 		HK_ASSERT3(0x0, num_contacts == Num_deferred_contacts,
// 			"Ran short of physics character proxy deferred contacts by " << (Num_deferred_contacts - num_contacts));
		HK_ASSERT(0x0, num_contacts == Num_deferred_contacts,
			"Ran short of physics character proxy deferred contacts by ");

		for (hkUint32 index = 0; index < num_contacts; index++) {

			// Convenience variable.
			physics_character_proxy_deferred_contact const& contact = Deferred_contacts[index];

			// Call the callback.
			(*Deferred_contact_callback)(contact);
		}

		// Clear the list.
		Num_deferred_contacts = 0;
	}

	// Process all the deferred character interactions that were stored up during integration.
	//
	static void physics_character_proxy_process_deferred_character_interactions()
	{
		// Handle being full.
		hkUint32 const num_character_interactions = hkMath::min2(Num_deferred_character_interactions,
			static_cast<hkUint32>(MAX_DEFERRED_CHARACTER_INTERACTIONS));
		// DSFL jfk
// 		HK_ASSERT3(0x0, num_character_interactions == Num_deferred_character_interactions,
// 			"Ran short of physics character proxy deferred character interactions by " <<
// 			(Num_deferred_character_interactions - num_character_interactions));
		HK_ASSERT(0x0, num_character_interactions == Num_deferred_character_interactions,
			"Ran short of physics character proxy deferred character interactions by ");


		for (hkUint32 index = 0; index < num_character_interactions; index++) {

			// Convenience variable.
			physics_character_proxy_deferred_character_interaction const& character_interaction =
				Deferred_character_interactions[index];

			// Call the callback.
			(*Deferred_character_interaction_callback)(character_interaction);
		}

		// Clear the list.
		Num_deferred_character_interactions = 0;
	}

	// Process all the deferred object interactions that were stored up during integration.
	//
	static void physics_character_proxy_process_deferred_object_interactions()
	{
		// Handle being full.
		hkUint32 const num_object_interactions = hkMath::min2(Num_deferred_object_interactions,
			static_cast<hkUint32>(MAX_DEFERRED_OBJECT_INTERACTIONS));
		// DSFL jfk
// 		HK_ASSERT3(0x0, num_object_interactions == Num_deferred_object_interactions,
// 			"Ran short of physics character proxy deferred object interactions by " <<
// 			(Num_deferred_object_interactions - num_object_interactions));
		HK_ASSERT(0x0, num_object_interactions == Num_deferred_object_interactions,
			"Ran short of physics character proxy deferred object interactions by ");

		for (hkUint32 index = 0; index < num_object_interactions; index++) {

			// Convenience variable.
			physics_character_proxy_deferred_object_interaction const& object_interaction =
				Deferred_object_interactions[index];

			// Call the callback.
			(*Deferred_object_interaction_callback)(object_interaction);
		}

		// Clear the list.
		Num_deferred_object_interactions = 0;
	}

#endif // !defined (HK_PLATFORM_SPU)

// Save a deferred event to be processed after integration.
//
// event_buffer:	(input/output) Buffer that the event will be added to.
// num_events:		(input/output) Number of events in the buffer.
// max_events:		Maximum number of events the buffer can hold.
// event_size:		The size of each event.
// event_data:		The event's data.
//
static void physics_character_proxy_add_deferred_event(void* event_buffer, hkUint32* num_events,
	hkUint32 max_events, hkUint32 event_size, void const* event_data)
{
	#if !defined (HK_PLATFORM_SPU)

		// Disabled?
		if (event_buffer == HK_NULL) {
			return;
		}

	#endif // !defined (HK_PLATFORM_SPU)

	// Get a slot index in a thread-safe way.
	#if !defined (HK_PLATFORM_SPU)

		// DSFL jfk
		//hkUint32 slot_index = hkCriticalSection::atomicExchangeAdd(num_events, 1);
		hkUint32 slot_index = hkAtomic::exchangeAdd(num_events, 1);

	#else

		hkUint32 slot_index = hkSpuDmaManager::atomicExchangeAdd(num_events, 1);

	#endif // !defined (HK_PLATFORM_SPU)

	if (slot_index >= max_events) {
		return;
	}

	#if defined (HK_PLATFORM_SPU)

		// Wait for a previous transfer to finish before overwriting the source data.
		hkSpuDmaManager::waitForDmaCompletion(hkSpuCharacterProxyDmaGroups::WRITE_BACK_DATA);

	#endif // defined (HK_PLATFORM_SPU)

	// Record the event.
	void* event_destination = static_cast<hkUint8*>(event_buffer) + slot_index * event_size;

	#if !defined (HK_PLATFORM_SPU)

		// Copy.
		hkMemUtil::memCpy<16>(event_destination, event_data, event_size);

	#else

		// Store the event during DMA.
		hkMemUtil::memCpy<16>(Deferred_event_buffer, event_data, event_size);

		// Transfer to PPU.
		hkSpuDmaManager::putToMainMemory(event_destination, Deferred_event_buffer, event_size,
			hkSpuDmaManager::WRITE_BACK, hkSpuCharacterProxyDmaGroups::WRITE_BACK_DATA);

	#endif // !defined (HK_PLATFORM_SPU)
}


#if !defined (HK_PLATFORM_SPU)

	// -------------------------------------
	// physics_custom_character_proxy_listener member functions
	// -------------------------------------

	// Called when the character interacts with another character.
	//
	// proxy:			Character A.
	// other_proxy:	Character B.
	// contact:			Contact point information.
	//
	void physics_custom_character_proxy_listener::characterInteractionCallback(hkpCharacterProxy* proxy,
		hkpCharacterProxy* other_proxy, hkContactPoint const& contact)
	{
		physics_character_proxy_process_character_interaction(proxy, other_proxy, contact);
	}

	// This is called when a character proxy interacts with a dynamic object.
	//
	// proxy:	The character proxy.
	// input:	The event data.
	// output:	(output) The output for the event.
	//
	void physics_custom_character_proxy_listener::objectInteractionCallback(hkpCharacterProxy* proxy,
		hkpCharacterObjectInteractionEvent const& input, hkpCharacterObjectInteractionResult& output)
	{		
		physics_character_proxy_process_object_interaction(proxy, input, output);
	}

	// We add in our own custom (per surface) friction model here
	//
	// manifold:	The contact manifold.
	// input:		Input to the simplex solver.
	//
	void physics_custom_character_proxy_listener::processConstraintsCallback( const hkpCharacterProxy* proxy,
		const hkArray<hkpRootCdPoint>& manifold, hkSimplexSolverInput& input ) 
	{
		physics_character_proxy_process_constraints(proxy, manifold, input);
	}

#endif // !defined (HK_PLATFORM_SPU)

// --------------------
//
// External Functions
//
// --------------------

#if !defined (HK_PLATFORM_SPU)

	// Initialize the character proxy callbacks system.
	//
	// info:	Information to initialize the character proxy callbacks system.
	//
	void physics_character_proxy_callbacks_initialize(physics_character_proxy_callbacks_init_info const& info)
	{
		Deferred_contact_callback						= info.m_deferred_contact_callback;
		Deferred_character_interaction_callback	= info.m_deferred_character_interaction_callback;
		Deferred_object_interaction_callback		= info.m_deferred_object_interaction_callback;

		Character_interaction_key						= info.m_character_interaction_key;
		Object_handle_key									= info.m_object_handle_key;
		Ignore_handle_key									= info.m_ignore_handle_key;

		Invalid_object_handle							= info.m_invalid_object_handle;
	}

	#if defined (HK_PLATFORM_HAS_SPU)

		// Initialize the character proxy job to provide callback info on SPU.
		//
		// job:	The job holds initialization info.
		//
		void physics_character_proxy_callbacks_initialize_job_for_spu(hkpCharacterProxyJob& job)
		{
			job.m_user_data1 = reinterpret_cast<hkUint32>(Deferred_contacts);
			job.m_user_data2 = reinterpret_cast<hkUint32>(&Num_deferred_contacts);
			job.m_user_data3 = reinterpret_cast<hkUint32>(Deferred_character_interactions);
			job.m_user_data4 = reinterpret_cast<hkUint32>(&Num_deferred_character_interactions);
			job.m_user_data5 = reinterpret_cast<hkUint32>(Deferred_object_interactions);
			job.m_user_data6 = reinterpret_cast<hkUint32>(&Num_deferred_object_interactions);

			job.m_user_data7 = Character_interaction_key;
			job.m_user_data8 = Object_handle_key;
			job.m_user_data9 = Ignore_handle_key;

			job.m_user_data10 = Invalid_object_handle;
		}

	#endif // defined (HK_PLATFORM_HAS_SPU)

#else

	// Initialize the character proxy callbacks system on SPU.
	//
	// job:	The job holds initialization info.
	//
	void physics_character_proxy_callbacks_initialize_spu(hkpCharacterProxyJob const& job)
	{
		Deferred_contacts_on_ppu							= reinterpret_cast<void*>(job.m_user_data1);
		Num_deferred_contacts_on_ppu						= reinterpret_cast<hkUint32*>(job.m_user_data2);
		Deferred_character_interactions_on_ppu			= reinterpret_cast<void*>(job.m_user_data3);
		Num_deferred_character_interactions_on_ppu	= reinterpret_cast<hkUint32*>(job.m_user_data4);
		Deferred_object_interactions_on_ppu				= reinterpret_cast<void*>(job.m_user_data5);
		Num_deferred_object_interactions_on_ppu		= reinterpret_cast<hkUint32*>(job.m_user_data6);

		Character_interaction_key							= job.m_user_data7;
		Object_handle_key										= job.m_user_data8;
		Ignore_handle_key										= job.m_user_data9;

		Invalid_object_handle								= job.m_user_data10;
	}

#endif // !defined (HK_PLATFORM_SPU)

#if !defined (HK_PLATFORM_SPU)

	// Initialize deferred events before integration.
	//
	void physics_character_proxy_initialize_deferred_events()
	{
		// Allocate memory for contacts.
		HK_ASSERT(0x0, Deferred_contacts == HK_NULL, "FAIL");
		Deferred_contacts =
			reinterpret_cast<physics_character_proxy_deferred_contact*>(hkMemSolverBlockAlloc<hkUint8>(Deferred_contact_buffer_size));
		HK_ASSERT(0x0, Deferred_contacts != HK_NULL, "FAIL");
		HK_ASSERT(0x0, (reinterpret_cast<hkUint64>(Deferred_contacts) % 16) == 0, "FAIL");	// HVS_EC  Can't cast 64-bit pointer to 32-bit int on PS4!!

		// Allocate memory for character interactions.
		HK_ASSERT(0x0, Deferred_character_interactions == HK_NULL, "FAIL");
		Deferred_character_interactions =
			reinterpret_cast<physics_character_proxy_deferred_character_interaction*>(hkMemSolverBlockAlloc<hkUint8>(Deferred_character_interaction_buffer_size));
		HK_ASSERT(0x0, Deferred_character_interactions != HK_NULL, "FAIL");
		HK_ASSERT(0x0, (reinterpret_cast<hkUint64>(Deferred_character_interactions) % 16) == 0, "FAIL");	// HVS_EC  Can't cast 64-bit pointer to 32-bit int on PS4!!

		// Allocate memory for object interactions.
		HK_ASSERT(0x0, Deferred_object_interactions == HK_NULL, "FAIL");
		Deferred_object_interactions =
			reinterpret_cast<physics_character_proxy_deferred_object_interaction*>(hkMemSolverBlockAlloc<hkUint8>(Deferred_object_interaction_buffer_size));
		HK_ASSERT(0x0, Deferred_object_interactions != HK_NULL, "FAIL");
		HK_ASSERT(0x0, (reinterpret_cast<hkUint64>(Deferred_object_interactions) % 16) == 0, "FAIL");	// HVS_EC  Can't cast 64-bit pointer to 32-bit int on PS4!!
	}

	// Uninitialize deferred events after integration.
	//
	void physics_character_proxy_uninitialize_deferred_events()
	{
		// Free memory for character interactions.
		hkMemSolverBlockFree<hkUint8>(reinterpret_cast<hkUint8*>(Deferred_contacts),
			Deferred_contact_buffer_size);
		Deferred_contacts = HK_NULL;

		// Free memory for character interactions.
		hkMemSolverBlockFree<hkUint8>(reinterpret_cast<hkUint8*>(Deferred_character_interactions),
			Deferred_character_interaction_buffer_size);
		Deferred_character_interactions = HK_NULL;

		// Free memory for object interactions.
		hkMemSolverBlockFree<hkUint8>(reinterpret_cast<hkUint8*>(Deferred_object_interactions),
			Deferred_object_interaction_buffer_size);
		Deferred_object_interactions = HK_NULL;
	}

	// Process all the deferred events that were stored up during integration.
	//
	void physics_character_proxy_process_deferred_events()
	{
		physics_character_proxy_process_deferred_contacts();
		physics_character_proxy_process_deferred_character_interactions();
		physics_character_proxy_process_deferred_object_interactions();
	}

#endif // !defined (HK_PLATFORM_SPU)

#if defined (HK_PLATFORM_SPU)

	// Create an SPU version of a world object and its properties from its collidable on the PPU.
	//
	// world_object:				(output) The world object of the collidable.
	// property_buffer:			(output) A buffer to hold the world object's properties.
	// property_buffer_on_ppu:	(output) Location on the PPU of the properties.
	// property_buffer_size:	(output) The size (in bytes) of the property buffer.
	// collidable_on_ppu:		Location on the PPU of the collidable we'd like the world object for.
	//
	static void physics_character_proxy_get_world_object_with_properties_from_ppu(hkpWorldObject*& world_object,
		hkpProperty*& property_buffer, hkpProperty const*& property_buffer_on_ppu, hkUint32& property_buffer_size,
		hkpCollidable const* collidable_on_ppu)
	{
		// Get the world object PPU address.
		hkpWorldObject const* world_object_on_ppu =
			reinterpret_cast<hkpWorldObject const*>(reinterpret_cast<hkUint32 const>(collidable_on_ppu) -
			HK_OFFSET_OF(hkpWorldObject, m_collidable));

		// DMA up the world object.
		world_object = hkAllocateStack<hkpWorldObject>(1, "WorldObjectWithProps");
		hkSpuDmaManager::getFromMainMemoryAndWaitForCompletion(world_object, world_object_on_ppu,
			HK_NEXT_MULTIPLE_OF(128, sizeof(hkpWorldObject)), hkSpuDmaManager::READ_COPY);

		// Are there properties?
		property_buffer = HK_NULL;
		property_buffer_size = 0;

		hkUint32 const num_properties = world_object->m_properties.getSize();
		if (num_properties == 0) {
			return;
		}

		// DMA up the properties.
		property_buffer = hkAllocateStack<hkpProperty>(num_properties, "PropBuffer");
		property_buffer_on_ppu = world_object->m_properties.begin();
		property_buffer_size = num_properties * sizeof(hkpProperty);
		hkSpuDmaManager::getFromMainMemoryAndWaitForCompletion(property_buffer, property_buffer_on_ppu,
			property_buffer_size, hkSpuDmaManager::READ_ONLY);

		// Stuff them in the world object.
		world_object->m_properties._setDataUnchecked(property_buffer, property_buffer_size,
			property_buffer_size | hkArray<hkpProperty>::DONT_DEALLOCATE_FLAG);
	}

	// Free an SPU version of a world object and its properties from its collidable on the PPU.
	//
	// world_object:				The world object of the collidable.
	// property_buffer:			A buffer to hold the world object's properties.
	// property_buffer_on_ppu:	Location on the PPU of the properties.
	// property_buffer_size:	The size (in bytes) of the property buffer.
	// collidable_on_ppu:		Location on the PPU of the collidable we'd like the world object for.
	//
	static void physics_character_proxy_free_world_object_with_properties_from_ppu(hkpWorldObject* world_object,
		hkpProperty* property_buffer, hkpProperty const* property_buffer_on_ppu, hkUint32 property_buffer_size,
		hkpCollidable const* collidable_on_ppu)
	{
		// Clean up the properties.
		if (property_buffer != HK_NULL) {
			
			HK_SPU_DMA_PERFORM_FINAL_CHECKS(property_buffer_on_ppu, property_buffer, property_buffer_size);
			hkDeallocateStack(property_buffer);
		}

		// Get the world object PPU address.
		hkpWorldObject const* world_object_on_ppu =
			reinterpret_cast<hkpWorldObject const*>(reinterpret_cast<hkUint32 const>(collidable_on_ppu) -
			HK_OFFSET_OF(hkpWorldObject, m_collidable));
		world_object_on_ppu;

		// Clean up the world object.
		HK_SPU_DMA_PERFORM_FINAL_CHECKS(world_object_on_ppu, world_object,
			HK_NEXT_MULTIPLE_OF(128, sizeof(hkpWorldObject)));
		hkDeallocateStack(world_object);
	}

#endif // defined (HK_PLATFORM_SPU)

// Add a collision to a character collector.  Does special filtering based on the entities involved.
//
// collector:					(input/output) The collector to potentially add the collision to.
// characters_collidable:	The collidable of the character being integrated.
// cd_point:					The collision information.
//
#if !defined (HK_PLATFORM_SPU)

	void physics_character_proxy_add_cd_point_to_collector(hkpCpuCharacterProxyCollector* collector,
		hkpCollidable const* characters_collidable, hkpCdPoint const& cd_point)

#else

	void physics_character_proxy_add_cd_point_to_collector(hkpCdPoint const& cd_point,
		hkpFixedBufferCdPointCollector* collector)

#endif // !defined (HK_PLATFORM_SPU)
{
	#if !defined (HK_PLATFORM_SPU)

		// Not safe to use A as character phantom when using multithreaded proxy util!
		hkpWorldObject* world_object_a = hkpGetWorldObject(characters_collidable);
		hkpCollidable const* root_collidable_b = cd_point.m_cdBodyB.getRootCollidable();
		hkpWorldObject* world_object_b = hkpGetWorldObject(root_collidable_b);

	#else

		hkpCollidable const* characters_collidable = cd_point.m_cdBodyA.getRootCollidable();

		// The collidables in the CD point are SPU pointers.  Fortunately, the PPU pointers are here.
		hkpCollidable const* characters_collidable_on_ppu = g_spuFixedBufferCdPointCollectorCollidableAOnPpu;
		hkpCollidable const* root_collidable_b_on_ppu = g_spuFixedBufferCdPointCollectorCollidableBOnPpu;
		
		// Now make a SPU version of the world object for each with just properties filled in.
		hkpWorldObject* world_object_a;
		hkpProperty* property_buffer_a;
		hkpProperty const* property_buffer_on_ppu_a;
		hkUint32 property_buffer_size_a;
		physics_character_proxy_get_world_object_with_properties_from_ppu(world_object_a, property_buffer_a,
			property_buffer_on_ppu_a, property_buffer_size_a, characters_collidable_on_ppu);

		hkpWorldObject* world_object_b;
		hkpProperty* property_buffer_b;
		hkpProperty const* property_buffer_on_ppu_b;
		hkUint32 property_buffer_size_b;
		physics_character_proxy_get_world_object_with_properties_from_ppu(world_object_b, property_buffer_b,
			property_buffer_on_ppu_b, property_buffer_size_b, root_collidable_b_on_ppu);

	#endif // !defined (HK_PLATFORM_SPU)

	// Get the character interaction property data.
	physics_character_interaction_property const prop_a =
		physics_object_get_character_interaction_property(world_object_a);
	physics_character_interaction_property const prop_b =
		physics_object_get_character_interaction_property(world_object_b);

	// Get some other properties.
	hkUint64 const ignore_handle = physics_object_get_uint64_property(world_object_a, Ignore_handle_key,
		Invalid_object_handle);
	hkUint64 const obj_b_handle = physics_object_get_uint64_property(world_object_b, Object_handle_key,
		Invalid_object_handle);

	#if defined (HK_PLATFORM_SPU)

		// Clean up.
		physics_character_proxy_free_world_object_with_properties_from_ppu(world_object_a, property_buffer_a,
			property_buffer_on_ppu_a, property_buffer_size_a, characters_collidable_on_ppu);
		physics_character_proxy_free_world_object_with_properties_from_ppu(world_object_b, property_buffer_b,
			property_buffer_on_ppu_b, property_buffer_size_b, root_collidable_b_on_ppu);

	#endif // defined (HK_PLATFORM_SPU)

	// Skip contacts with this object?
	if ((prop_b.m_flags & PCIP_FLAG_IGNORE_CONTACT) != 0) {
		return;
	}

	// Defer contacts with this object?
	if ((prop_b.m_flags & PCIP_FLAG_DEFER_CONTACT) != 0) {

		physics_character_proxy_deferred_contact new_contact;
		{
			#if !defined (HK_PLATFORM_SPU)

				new_contact.m_character_collidable = characters_collidable;
				new_contact.m_contacted_collidable = root_collidable_b;

			#else

				new_contact.m_character_collidable = characters_collidable_on_ppu;
				new_contact.m_contacted_collidable = root_collidable_b_on_ppu;

			#endif // !defined (HK_PLATFORM_SPU)

			new_contact.m_contact_point = cd_point.getContact();
		}

		#if !defined (HK_PLATFORM_SPU)

			physics_character_proxy_add_deferred_event(Deferred_contacts, &Num_deferred_contacts,
				MAX_DEFERRED_CONTACTS, sizeof(physics_character_proxy_deferred_contact), &new_contact);

		#else

			physics_character_proxy_add_deferred_event(Deferred_contacts_on_ppu, Num_deferred_contacts_on_ppu,
				MAX_DEFERRED_CONTACTS, sizeof(physics_character_proxy_deferred_contact), &new_contact);

		#endif // !defined (HK_PLATFORM_SPU)

		return;
	}

	// Filter out ignored objects as well.
	if ((ignore_handle != Invalid_object_handle) && (ignore_handle == obj_b_handle)) {
		return;
	}

	// Enforce max separation distance.

	// Chose the larger separation.
#ifdef HK_DEBUG//HVS_DMK
	// DSFL jfk
// 	hkReal distance_allowed_squared = hkMath::max2(prop_a.m_separation_distance_squared,
// 		prop_b.m_separation_distance_squared);
	const hkSimdReal distance_allowed_squared = hkSimdReal::fromFloat(hkMath::max2(prop_a.m_separation_distance_squared,
		prop_b.m_separation_distance_squared));//HVS_DMK
#else
	const hkSimdReal distance_allowed_squared = hkSimdReal::fromFloat(hkMath::max2(prop_a.m_separation_distance_squared,
		prop_b.m_separation_distance_squared));//HVS_DMK
#endif//#ifdef HK_DEBUG//HVS_DMK

	hkVector4 const contact_position = cd_point.getContact().getPosition();
	hkVector4 const character_position = characters_collidable->getTransform().getTranslation();

	// Get the 2D separation distance.
	hkVector4 separation_vec;
	separation_vec.setSub(contact_position, character_position);
	separation_vec(1) = 0.0f;
#ifdef HK_DEBUG//HVS_DMK
	// DSFL jfk
	//hkReal const distance_squared = separation_vec.lengthSquared3();
	hkSimdReal const distance_squared = separation_vec.lengthSquared<3>();
#else
	hkSimdReal const distance_squared = separation_vec.lengthSquared<3>();//HVS_DMK
#endif//#ifdef HK_DEBUG//HVS_DMK

	if (distance_squared > distance_allowed_squared)
	{
		return;
	}

	// Add it to the collector.
	#if !defined (HK_PLATFORM_SPU)

		collector->hkpCpuCharacterProxyCollector::addCdPoint(cd_point);	

	#else
		
		hkpSpuCharacterProxyUtil::customAddCdPointImplementation(cd_point, collector);

	#endif // !defined (HK_PLATFORM_SPU)
}

// Called when the character interacts with another character.
//
// proxy:			Character A.
// other_proxy:	Character B.
// contact:			Contact point information.
//
void physics_character_proxy_process_character_interaction(hkpCharacterProxy const* proxy,
	hkpCharacterProxy* other_proxy, hkContactPoint const& contact)
{
	#if defined (HK_PLATFORM_SPU)

		// Get some data that SPU callbacks don't have access to.
		hkpCharacterProxy const* proxy_on_ppu;
		hkpCollidable const* proxy_collidable_on_ppu;
		hkVector4 proxy_position;
		hk_character_proxy_get_data_for_spu_callbacks(proxy_on_ppu, proxy_collidable_on_ppu, proxy_position);

	#endif // defined (HK_PLATFORM_SPU)

	physics_character_proxy_deferred_character_interaction new_character_interaction;
	{
		#if !defined (HK_PLATFORM_SPU)
	
			new_character_interaction.m_character_proxy = proxy;

		#else

			new_character_interaction.m_character_proxy = proxy_on_ppu;

		#endif // !defined (HK_PLATFORM_SPU)

		new_character_interaction.m_contacted_proxy = other_proxy;
		new_character_interaction.m_contact_point = contact;
	}
	#if !defined (HK_PLATFORM_SPU)

		physics_character_proxy_add_deferred_event(Deferred_character_interactions,
			&Num_deferred_character_interactions, MAX_DEFERRED_CHARACTER_INTERACTIONS,
			sizeof(physics_character_proxy_deferred_character_interaction), &new_character_interaction);

	#else

		physics_character_proxy_add_deferred_event(Deferred_character_interactions_on_ppu,
			Num_deferred_character_interactions_on_ppu, MAX_DEFERRED_CHARACTER_INTERACTIONS,
			sizeof(physics_character_proxy_deferred_character_interaction), &new_character_interaction);

	#endif // !defined (HK_PLATFORM_SPU)
}

// This is called when a character proxy interacts with a dynamic object.
//
// proxy:	The character proxy.
// input:	The event data.
// output:	(output) The output for the event.
//
void physics_character_proxy_process_object_interaction(hkpCharacterProxy const* proxy,
	hkpCharacterObjectInteractionEvent const& input, hkpCharacterObjectInteractionResult& output)
{		
	// Check that we hit a rigid body.
	if (input.m_body == HK_NULL) {
		return;
	}

	#if defined (HK_PLATFORM_SPU)

		// Get some data that SPU callbacks don't have access to.
		hkpCharacterProxy const* proxy_on_ppu;
		hkpCollidable const* proxy_collidable_on_ppu;
		hkVector4 proxy_position;
		hk_character_proxy_get_data_for_spu_callbacks(proxy_on_ppu, proxy_collidable_on_ppu, proxy_position);

	#endif // defined (HK_PLATFORM_SPU)

	// Add the collision event for later processing.
	// NOTE: Most (if not all) high-level game logic should be processed later to avoid problems
	//       with being embedded in a callback.  An example problem is making a ragdoll
	//			inside a callback.  The ragdoll will be created before the character controller
	//			has a chance to resolve any penetrations so the ragdoll can end up embedded in objects.
	physics_character_proxy_deferred_object_interaction new_object_interaction;
	{
		#if !defined (HK_PLATFORM_SPU)

			new_object_interaction.m_character_proxy = proxy;

		#else

			new_object_interaction.m_character_proxy = proxy_on_ppu;

		#endif // !defined (HK_PLATFORM_SPU)

		new_object_interaction.m_event_data = input;
	}
	#if !defined (HK_PLATFORM_SPU)

		physics_character_proxy_add_deferred_event(Deferred_object_interactions,
			&Num_deferred_object_interactions, MAX_DEFERRED_OBJECT_INTERACTIONS,
			sizeof(physics_character_proxy_deferred_object_interaction), &new_object_interaction);

	#else

		physics_character_proxy_add_deferred_event(Deferred_object_interactions_on_ppu,
			Num_deferred_object_interactions_on_ppu, MAX_DEFERRED_OBJECT_INTERACTIONS,
			sizeof(physics_character_proxy_deferred_object_interaction), &new_object_interaction);

	#endif // !defined (HK_PLATFORM_SPU)

	//////////////////////////////////////////////////////////////////////////
	//
	// Do any immediate processing that needs to be done that affect the results of the solver.
	//
	//////////////////////////////////////////////////////////////////////////

	#if !defined (HK_PLATFORM_SPU)

		// Get stuff!
		hkpWorldObject const* world_object_a = proxy->getShapePhantom();
		hkpWorldObject const* world_object_b = input.m_body;
		hkVector4 const& proxy_position = proxy->getPosition();

	#else

		// This should just be pointer math.
		hkpCollidable const* root_collidable_b_on_ppu =
			reinterpret_cast<hkpCollidable const*>(input.m_body->getCollidable());

		// Now make a SPU version of the world object for each with just properties filled in.
		hkpWorldObject* world_object_a;
		hkpProperty* property_buffer_a;
		hkpProperty const* property_buffer_on_ppu_a;
		hkUint32 property_buffer_size_a;
		physics_character_proxy_get_world_object_with_properties_from_ppu(world_object_a, property_buffer_a,
			property_buffer_on_ppu_a, property_buffer_size_a, proxy_collidable_on_ppu);

		hkpWorldObject* world_object_b;
		hkpProperty* property_buffer_b;
		hkpProperty const* property_buffer_on_ppu_b;
		hkUint32 property_buffer_size_b;
		physics_character_proxy_get_world_object_with_properties_from_ppu(world_object_b, property_buffer_b,
			property_buffer_on_ppu_b, property_buffer_size_b, root_collidable_b_on_ppu);

	#endif // !defined (HK_PLATFORM_SPU)

	// Get the character interaction property data.
	physics_character_interaction_property const prop_a =
		physics_object_get_character_interaction_property(world_object_a);
	physics_character_interaction_property const prop_b =
		physics_object_get_character_interaction_property(world_object_b);

	// Get some other properties.
	hkUint64 const obj_b_handle = physics_object_get_uint64_property(world_object_b, Object_handle_key,
		Invalid_object_handle);

	#if defined (HK_PLATFORM_SPU)

		// Clean up.
		physics_character_proxy_free_world_object_with_properties_from_ppu(world_object_a, property_buffer_a,
			property_buffer_on_ppu_a, property_buffer_size_a, proxy_collidable_on_ppu);
		physics_character_proxy_free_world_object_with_properties_from_ppu(world_object_b, property_buffer_b,
			property_buffer_on_ppu_b, property_buffer_size_b, root_collidable_b_on_ppu);

	#endif // defined (HK_PLATFORM_SPU)

	if (obj_b_handle == Invalid_object_handle) {
		return;
	}

	// Don't allow the player to push things they're standing on.
	if ((prop_a.m_flags & PCIP_FLAG_IS_PLAYER) != 0) {

		hkVector4 const& contact_position = input.m_position;

		if ((contact_position(1) - proxy_position(1)) < OBJECT_UNDER_PROXY_Y_THRESHOLD)
		{
			output.m_objectImpulse.setZero();//HVS_DMK
			return;
		}
	}

	// See if the object can be pushed.
	if ((prop_b.m_flags & PCIP_FLAG_CANNOT_BE_PUSHED) != 0)
	{
		output.m_objectImpulse.setZero();//HVS_DMK
	}
}

// We add in our own custom (per surface) friction model here.
//
// proxy:		The character.
// manifold:	The contact manifold.
// input:		Input to the simplex solver.
//
#if !defined (HK_PLATFORM_SPU)

	void physics_character_proxy_process_constraints(hkpCharacterProxy const* proxy,
		hkArray<hkpRootCdPoint> const& manifold, hkSimplexSolverInput& input) 

#else

	void physics_character_proxy_process_constraints(hkpCharacterProxy const* proxy,
		hkArraySpu<hkpRootCdPoint> const& manifold, hkSimplexSolverInput& input) 

#endif // !defined (HK_PLATFORM_SPU)
{
	// Iterate through the constraints and adjust the friction appropriately.  Constraints beyond the manifold
	// size were added artificially and don't need to be processed.
	hkUint8 const num_constraints = static_cast<hkUint8>(hkMath::min2(input.m_numConstraints, manifold.getSize()));

	for (hkUint8 index = 0; index < num_constraints; index++) {

		hkSurfaceConstraintInfo& surface = input.m_constraints[index];

		#if !defined (HK_PLATFORM_SPU)

			// Get the world object of the collidable.
			hkpWorldObject const* world_object_b = hkpGetWorldObject(manifold[index].m_rootCollidableB);

		#else

			hkpCollidable const* root_collidable_b_on_ppu = manifold[index].m_rootCollidableB;

			// Now make a SPU version of the world object with just properties filled in.
			hkpWorldObject* world_object_b;
			hkpProperty* property_buffer_b;
			hkpProperty const* property_buffer_on_ppu_b;
			hkUint32 property_buffer_size_b;
			physics_character_proxy_get_world_object_with_properties_from_ppu(world_object_b, property_buffer_b,
				property_buffer_on_ppu_b, property_buffer_size_b, root_collidable_b_on_ppu);

		#endif // !defined (HK_PLATFORM_SPU)

		// Get the character interaction property data.
		physics_character_interaction_property const prop_b =
			physics_object_get_character_interaction_property(world_object_b);

		#if defined (HK_PLATFORM_SPU)

			// Clean up.
			physics_character_proxy_free_world_object_with_properties_from_ppu(world_object_b, property_buffer_b,
				property_buffer_on_ppu_b, property_buffer_size_b, root_collidable_b_on_ppu);

		#endif // defined (HK_PLATFORM_SPU)

		if ((prop_b.m_flags & PCIP_FLAG_ZERO_SURFACE_VELOCITY) != 0)
		{
			surface.m_velocity.setZero();//HVS_DMK
		}
	}
}

// Gets an object's character interaction property.
//
// object:	The object.
//
// Returns: The character interaction property.
//
physics_character_interaction_property physics_object_get_character_interaction_property(hkpWorldObject const* object)
{
	// If no property is present, return the default.
	physics_character_interaction_property prop;
	prop.initialze();

	// Try to get the actual property.
	hkUint64 prop_uint64 = physics_object_get_uint64_property(object, Character_interaction_key,
		prop.get_as_uint64());

	prop.set_from_uint64(prop_uint64);
	return prop;
}

#if !defined (HK_PLATFORM_SPU)

	// Sets an object's character interaction property.
	//
	// object:	The object.
	// prop:		The property data to set.
	//
	void physics_object_set_character_interaction_property(hkpWorldObject* object,
		physics_character_interaction_property const& prop)
	{
		// Sanity check.
		if (object == HK_NULL) {
			return;
		}

		hkpPropertyValue pv;
		pv.setUint64(prop.get_as_uint64());

		// Add or update it.
		if (object->hasProperty(Character_interaction_key) == false) {
			object->addProperty(Character_interaction_key, pv);
		} else {
			object->editProperty(Character_interaction_key, pv);
		}
	}

#endif // !defined (HK_PLATFORM_SPU)
