// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : PHYSICS+PHYSICS_2012
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/PhysicsMigrationUtils/hkpPhysicsMigrationUtils.h>

#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/System/Io/IArchive/hkIArchive.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/GeometryUtilities/Inertia/hkInertiaTensorComputer.h>

// Physics 2012
#include <Physics2012/Dynamics/hkpDynamics.h>
#include <Physics2012/Dynamics/Motion/hkpMotion.h>
#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics2012/Collide/Shape/Convex/Sphere/hkpSphereShape.h> 
#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics2012/Collide/Shape/Convex/Triangle/hkpTriangleShape.h>
#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppBvTreeShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/ExtendedMeshShape/hkpExtendedMeshShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShape.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Physics2012/Utilities/CharacterControl/CharacterProxy/hkpCharacterProxyCinfo.h>
#include <Physics2012/Utilities/CharacterControl/CharacterRigidBody/hkpCharacterRigidBody.h>
#include <Physics2012/Dynamics/Phantom/hkpShapePhantom.h>
#include <Physics2012/Dynamics/Constraint/Breakable/hkpBreakableConstraintData.h>
#include <Physics2012/Dynamics/Constraint/Malleable/hkpMalleableConstraintData.h>
#include <Physics2012/Internal/Collide/StaticCompound/hkpStaticCompoundShape.h>

// Physics
#include <Physics/Physics/hknpPhysics.h>
#include <Physics/Physics/Collide/Shape/Convex/Sphere/hknpSphereShape.h>
#include <Physics/Physics/Collide/Shape/Convex/Capsule/hknpCapsuleShape.h>
#include <Physics/Physics/Collide/Shape/Convex/Triangle/hknpTriangleShape.h>
#include <Physics/Physics/Collide/Shape/Convex/hknpConvexShapeUtil.h>
#include <Physics/Physics/Collide/Shape/Composite/Mesh/Compressed/hknpCompressedMeshShape.h>
#include <Physics/Physics/Collide/Shape/Composite/Mesh/Compressed/hknpCompressedMeshShapeCinfo.h>
#include <Physics/Physics/Collide/Shape/Composite/Compound/hknpCompoundShape.h>
#include <Physics/Physics/Extensions/PhysicsSystem/hknpPhysicsSceneData.h>
#include <Physics/Physics/Extensions/CharacterControl/Proxy/hknpCharacterProxyCinfo.h>
#include <Physics/Physics/Extensions/CharacterControl/RigidBody/hknpCharacterRigidBody.h>
#include <Physics/Physics/Collide/Shape/Composite/Compound/hknpShapeInstance.h>
#include <Physics/Physics/Dynamics/World/hknpWorld.h>
#include <Physics/Internal/Dynamics/Solver/ConstraintAtom/hknpConstraintAtomSolver.h>
#include <Physics/Physics/Dynamics/Constraint/Breakable/hknpBreakableConstraintData.h>
#include <Physics/Physics/Dynamics/Constraint/Bridge/hknpBridgeConstraintAtom.h>
#include <Physics/Physics/Dynamics/Motion/hknpMotionPropertiesLibrary.h>
#include <Physics/Physics/Dynamics/Material/hknpMaterialLibrary.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "p12.migrationutils"
#include <Common/Base/System/Log/hkLog.hxx>


namespace
{
    hkRefPtr<hknpShape> HK_CALL _processShape(
        const hkpShape& shapeToConvert,
        hkPointerMap<const hkpShape*, hknpShape*>* shapesMap )
    {
        // convert the shape
        hkRefPtr<hknpShape> shape;
        if(shapesMap != HK_NULL)
        {
            shape = shapesMap->getWithDefault( &shapeToConvert, HK_NULL );
        }

        if(!shape)
        {
            shape.setAndDontIncrementRefCount(hkpPhysicsMigrationUtils::convertShape( shapeToConvert ));

            // Conversion failed?
            if(!shape)
            {
                return HK_NULL;
            }

            // Add to dictionary
            if ( shapesMap != HK_NULL )
            {
                shapesMap->insert( &shapeToConvert, shape );
                Log_Debug( "Unique shapes converted: {}", shapesMap->getSize() );
            }
        }

        return shape;
    }

    void HK_CALL _processMaterial( const hkpRigidBody& rigidBody, hknpMaterial& materialOut )
    {
        materialOut.m_dynamicFriction.setReal<true>( rigidBody.getFriction() );
        materialOut.m_staticFriction.setReal<true>( rigidBody.getFriction() );
        materialOut.m_restitution.setReal<true>( rigidBody.getRestitution() );
    }

    void HK_CALL _processDynamicMotionProperties(
        const hkpRigidBody& rigidBody,
        hkVector4Parameter gravity,
        hknpMotionProperties& propsOut )
    {
        // rigidBody.getRigidMotion()->setDeactivationClass(hkpRigidBodyCinfo::SOLVER_DEACTIVATION_OFF);
        hknpMotionProperties dynamicProps;
        dynamicProps.init( hknpMotionPropertiesId::PRESET_DYNAMIC );
        dynamicProps.m_angularDamping   = rigidBody.getAngularDamping();
        dynamicProps.m_linearDamping    = rigidBody.getLinearDamping();
        dynamicProps.m_maxLinearSpeed   = rigidBody.getMaxLinearVelocity();
        dynamicProps.m_maxAngularSpeed  = rigidBody.getMaxAngularVelocity();
        dynamicProps.m_gravityFactor    = rigidBody.getGravityFactor();
        dynamicProps.setSolverStabilization(
            hknpMotionProperties::SolverStabilizationType(const_cast< hkpRigidBody& >( rigidBody ).getMotion()->getDeactivationClass()), 
            gravity.length<3>().getReal() );
        dynamicProps.m_deactivationSettings.init();

        
        propsOut = dynamicProps;
    }

    void HK_CALL _processBodyCinfo(
        const hkpRigidBody& rigidBody,
        hknpShape* shape,
        hknpMaterialId materialId,
        hknpBodyCinfo& bodyCinfoOut )
    {
        bodyCinfoOut.m_shape = shape;
        bodyCinfoOut.m_materialId = materialId;
        bodyCinfoOut.m_position = rigidBody.getPosition();
        bodyCinfoOut.m_orientation = rigidBody.getRotation();
        bodyCinfoOut.m_collisionFilterInfo = rigidBody.getCollisionFilterInfo();
        bodyCinfoOut.m_name = rigidBody.getName();
        if ( rigidBody.m_localFrame != HK_NULL )
        {
            // Make a ref-counted heap copy
            hkArray<char> buffer;
            hkSerializeUtil::save( rigidBody.m_localFrame.val(), hkReflect::getType<hkLocalFrame>(), hkOArchive( buffer ).getStreamWriter() );
            bodyCinfoOut.m_localFrame.setAndDontIncrementRefCount( hkSerializeUtil::loadObject<hkLocalFrame>( hkIArchive( (void*)buffer.begin(), buffer.getSize() ).getStreamReader() ) );
            HK_ASSERT( 0x22440191, bodyCinfoOut.m_localFrame != HK_NULL, "Failure to create an hkLocalFrame copy" );
        }
    }


}   // anonymous namespace


namespace hkpPhysicsMigrationUtils
{
    hknpShape* HK_CALL convertShape( const hkpShape& shapeToConvert )
    {
        hknpConvexShape::BuildConfig configNoShrink;
        configNoShrink.m_shrinkByRadius = false;

        hknpShape::MassConfig massConfig;
        massConfig.m_quality = hknpShape::MassConfig::QUALITY_HIGH;

        hknpShape* shape;
        const hkpShapeContainer* shapeContainer;

        switch ( shapeToConvert.getType() )
        {

        case hkcdShapeType::SPHERE:
            {
                hkpSphereShape* pShape = (hkpSphereShape*)&shapeToConvert;
                hknpSphereShape* newShape = hknpSphereShape::createSphereShape( hkVector4::getZero(), pShape->getRadius());
                newShape->setMassProperties(massConfig);
                shape = newShape;
                break;
            }

        case hkcdShapeType::BOX:
            {
                hkpBoxShape* boxShape = (hkpBoxShape*)&shapeToConvert;
                hkVector4 halfExtents = boxShape->getHalfExtents();
                hkVector4 vertices[8];
                for(int i=0;i<8;++i)
                {
                    vertices[i](0)=halfExtents(0)*(i&1?+1:-1);
                    vertices[i](1)=halfExtents(1)*(i&2?+1:-1);
                    vertices[i](2)=halfExtents(2)*(i&4?+1:-1);
                }
                hknpConvexShape* convexShape = hknpConvexShape::createFromVertices(hkStridedVertices(vertices,8),boxShape->getRadius(), configNoShrink );
                convexShape->setMassProperties(massConfig);
                shape = convexShape;
                break;
            }

        case hkcdShapeType::CYLINDER:
            {
                
                hkpCylinderShape* cylinderShape = (hkpCylinderShape*)&shapeToConvert;
                int numVertices = cylinderShape->getNumCollisionSpheres();
                hkInplaceArray<hkVector4,64> vertices; vertices.setSize( numVertices );
                cylinderShape->getCollisionSpheres( (hkSphere*)vertices.begin() );
                hknpConvexShape* convexShape = hknpConvexShape::createFromVertices(vertices, vertices[0](3), configNoShrink);
                shape = convexShape;
                break;
            }

        case hkcdShapeType::CAPSULE:
            {
                hkpCapsuleShape* capsuleShape = (hkpCapsuleShape*)&shapeToConvert;
                hknpConvexShape* convexShape = hknpCapsuleShape::createCapsuleShape(
                    capsuleShape->getVertex(0), capsuleShape->getVertex(1), capsuleShape->getRadius() );
                convexShape->setMassProperties(massConfig);
                shape = convexShape;
                break;
            }

        case hkcdShapeType::LIST:
            {
                const hkpListShape* listShape = (hkpListShape*)&shapeToConvert;
                int numChildren = listShape->getNumChildShapes();

                hknpCompoundShapeCinfo csCinfo;
                csCinfo.m_capacity = numChildren;
                hknpCompoundShape* compoundShape = new hknpCompoundShape( csCinfo );

                int numInstances = 0;
                for( hkpShapeKey key = listShape->getFirstKey(); key != HK_INVALID_SHAPE_KEY; key = listShape->getNextKey(key) )
                {
                    hkpShapeBuffer buffer;
                    const hkpShape* pChildShape = listShape->getChildShape( key, buffer );
                    const hknpShape* childShape = convertShape( *pChildShape );
                    if ( childShape != HK_NULL )
                    {
                        hknpShapeInstance instance( childShape, hkTransform::getIdentity() );
                        instance.setEnabled( listShape->isChildEnabled(key) );

                        compoundShape->addInstances( &instance, 1 );
                        childShape->removeReference();
                        numInstances++;
                    }
                }
                compoundShape->rebuild();

                HK_ASSERT( 0xf0ffcd23, numInstances != 0, "You cannot convert an empty list shape");
                shape = compoundShape;
                break;
            }

        case hkcdShapeType::CONVEX_VERTICES:
            {
                const hkpConvexVerticesShape* cxShape = (hkpConvexVerticesShape*)&shapeToConvert;
                hkArray<hkVector4> vertices; cxShape->getOriginalVertices(vertices);

                hknpConvexShape* convexShape = hknpConvexShape::createFromVertices( vertices, cxShape->getRadius(), configNoShrink );
                shape = convexShape;
                break;
            }

        case hkcdShapeType::CONVEX_TRANSFORM:
        case hkcdShapeType::CONVEX_TRANSLATE:
            {
                //
                // Convert the child shape and get its transform
                //

                hknpShape* convertedChildShape = HK_NULL;
                hkTransform transform;
                {
                    switch( shapeToConvert.getType() )
                    {
                    case hkcdShapeType::CONVEX_TRANSFORM:
                        {
                            const hkpConvexTransformShape* ctShape = (hkpConvexTransformShape*)&shapeToConvert;
                            convertedChildShape = convertShape( *ctShape->getChildShape() );
                            ctShape->getTransform( &transform );
                            break;
                        }
                    case hkcdShapeType::CONVEX_TRANSLATE:
                        {
                            const hkpConvexTranslateShape* ctShape = (hkpConvexTranslateShape*)&shapeToConvert;
                            convertedChildShape = convertShape( *ctShape->getChildShape() );
                            transform.set( hkQuaternion::getIdentity(), ctShape->getTranslation() );
                            break;
                        }
                    default:
                        break;
                    }
                }

                if( convertedChildShape )
                {
                    //
                    // Bake the transform
                    //

                    const hknpConvexShape* convexShape = convertedChildShape->asConvexShape();
                    if( convexShape )
                    {
                        // Primitives need special casing
                        
                        switch( convexShape->getType() )
                        {
                            case hknpShapeType::SPHERE:
                                {
                                    const hknpSphereShape* ss = static_cast<const hknpSphereShape*>( convexShape );
                                    hkVector4 center;
                                    hkVector4Util::transformPoints( transform, ss->getVertices().begin(), 1, &center );
                                    shape = hknpSphereShape::createSphereShape( center, ss->m_convexRadius );
                                    shape->setMassProperties(massConfig);
                                    break;
                                }
                            case hknpShapeType::CAPSULE:
                                {
                                    const hknpCapsuleShape* cs = static_cast<const hknpCapsuleShape*>( convexShape );
                                    hkVector4 points[2] = { cs->getPointA(), cs->getPointB() };
                                    hkVector4Util::transformPoints( transform, &points[0], 2, points );
                                    shape = hknpCapsuleShape::createCapsuleShape( points[0], points[1], cs->m_convexRadius );
                                    shape->setMassProperties(massConfig);
                                    break;
                                }
                            default:
                                {
                                    const int numVerts = convexShape->getVertices().getSize();
                                    hkArray<hkVector4> vertices( (hkVector4*)convexShape->getVertices().begin(), numVerts, numVerts );
                                    hknpConvexShape::BuildConfig noShrinkWithTransform = configNoShrink;
                                    noShrinkWithTransform.m_extraTransform = &transform;
                                    shape = hknpConvexShape::createFromVertices( vertices, convexShape->m_convexRadius, noShrinkWithTransform );
                                    break;
                                }
                        }
                    }
                    else
                    {
                        // Horrible fallback
                        hkGeometry shapeGeometry;
                        convertedChildShape->buildSurfaceGeometry( hknpShape::CONVEX_RADIUS_DISPLAY_NONE, &shapeGeometry );
                        hknpConvexShape::BuildConfig noShrinkWithTransform = configNoShrink;
                        noShrinkWithTransform.m_extraTransform = &transform;
                        shape = hknpConvexShape::createFromVertices( shapeGeometry.m_vertices, convertedChildShape->m_convexRadius, noShrinkWithTransform );
                    }

                    // destroy the temporary shape
                    convertedChildShape->removeReference();
                }
                else
                {
                    shape = HK_NULL;
                }

                break;
            }

        case hkcdShapeType::BV_COMPRESSED_MESH:
            {
                const hkpBvCompressedMeshShape* compressedMeshShape = (const hkpBvCompressedMeshShape*)&shapeToConvert;
                shapeContainer = compressedMeshShape;
                goto PROCESS_SHAPE_COLLECTION;
            }

        case hkcdShapeType::EXTENDED_MESH:
            {
                shapeContainer = static_cast<const hkpExtendedMeshShape*>(&shapeToConvert);
                goto PROCESS_SHAPE_COLLECTION;
            }

        case hkcdShapeType::MOPP:
            {
                hkpMoppBvTreeShape* moppShape =  (hkpMoppBvTreeShape*)&shapeToConvert;
                shapeContainer = moppShape->getShapeCollection();
            }

PROCESS_SHAPE_COLLECTION:;
            {
                hkGeometry geometry;
                hkArray<hknpShapeInstance> childShapes;
                for ( hkpShapeKey key = shapeContainer->getFirstKey(); key != HK_INVALID_SHAPE_KEY; key = shapeContainer->getNextKey(key))
                {
                    hkpShapeBuffer buffer;
                    const hkpShape* childShape = shapeContainer->getChildShape(key, buffer);
                    if (childShape->m_type == hkcdShapeType::TRIANGLE )
                    {
                        const hkpTriangleShape* tShape = (const hkpTriangleShape*)childShape;
                        hkGeometry::Triangle& t = geometry.m_triangles.expandOne();
                        t.m_a = geometry.m_vertices.getSize(); geometry.m_vertices.pushBack( tShape->getVertex(0));
                        t.m_b = geometry.m_vertices.getSize(); geometry.m_vertices.pushBack( tShape->getVertex(1));
                        t.m_c = geometry.m_vertices.getSize(); geometry.m_vertices.pushBack( tShape->getVertex(2));
                        t.m_material = HKNP_INVALID_SHAPE_TAG;
                    }
                    else
                    {
                        hknpShape* convertedChild = convertShape(*childShape);
                        if(convertedChild)
                        {
                            childShapes.expandOne().setShape(convertedChild);
                            convertedChild->removeReference();
                        }
                    }
                }

                if(geometry.m_triangles.getSize())
                {
                    hknpDefaultCompressedMeshShapeCinfo meshInfo(&geometry);
                    hknpCompressedMeshShape* mesh = new hknpCompressedMeshShape(meshInfo);
                    childShapes.expandOne().setShape(mesh);
                    mesh->removeReference();
                }

                if(childShapes.getSize() > 1)
                {
                    hknpCompoundShapeCinfo csCinfo(childShapes.begin(), childShapes.getSize());
                    shape = new hknpCompoundShape(csCinfo);
                }
                else if(childShapes.getSize() == 1)
                {
                    shape = const_cast<hknpShape*>(childShapes[0].getShape());
                    shape->addReference();
                }
                else
                {
                    shape = HK_NULL;
                }
                break;
            }

        case hkcdShapeType::TRIANGLE:
            {
                hkpTriangleShape* tShape = (hkpTriangleShape*)&shapeToConvert;
                hknpTriangleShape* triangleShape = hknpTriangleShape::createTriangleShape(
                    tShape->getVertex(0), tShape->getVertex(1), tShape->getVertex(2), tShape->getRadius() );
                shape = triangleShape;
                break;
            }

        case hkcdShapeType::STATIC_COMPOUND:
            {
                hkArray<hknpShapeInstance>::Temp instances;
                const hkpStaticCompoundShape* origSCS = static_cast<const hkpStaticCompoundShape*>(&shapeToConvert);
                for(int i = 0; i < origSCS->getInstances().getSize(); i++)
                {
                    const hkQsTransform& qt = origSCS->getInstances()[i].getTransform();
                    hkTransform t; t.setRotation(qt.getRotation()); t.setTranslation(qt.getTranslation());
                    hknpShapeInstance instance(convertShape(*origSCS->getInstances()[i].getShape()), t);
                    instance.setScale(qt.getScale());
                    instances.pushBack(instance);
                }
                hknpCompoundShapeCinfo csCinfo(instances.begin(), instances.getSize());
                shape = new hknpCompoundShape(csCinfo);
                break;
            }

        default:
            {
                HK_WARN_ALWAYS(0x2d1f57c2, "Shape type(" << shapeToConvert.getType() << ") is not supported for conversion.");
                return HK_NULL;
            }

        }

        if ( shape != HK_NULL )
        {
            shape->setMassProperties( massConfig );
        }

        return shape;
    }

    hknpBodyCinfo* HK_CALL convertBody(
        const hkpRigidBody& body,
        hkVector4Parameter gravity,
        hknpPhysicsSystemData& systemDataOut,
        hkPointerMap<const hkpShape*, hknpShape*>* shapesMap )
    {
        // Convert the shape
        if ( body.getCollidable()->getShape() == HK_NULL )
        {
            return HK_NULL;
        }
        hkRefPtr<hknpShape> shape = _processShape( *body.getCollidable()->getShape(), shapesMap );
        if ( shape == HK_NULL )
        {
            return HK_NULL;
        }
        hkDiagonalizedMassProperties mp;
        if ( shape->asConvexShape() && shape->getMassProperties( mp ).isFailure())
        {
            HK_WARN_ALWAYS(0xabba3841," Convex shape of body "<<body.getName()<<" has no mass properties.");
        }
        // Convert the material
        hknpMaterialId materialId;
        {
            materialId = hknpMaterialId( systemDataOut.m_materials.getSize() );
            hknpMaterial& material = systemDataOut.m_materials.expandOne();
            _processMaterial( body, material );
        }

        // Convert the motion properties
        hknpMotionPropertiesId motionPropertiesId = hknpMotionPropertiesId::invalid();
        {
            if ( body.getMotionType() == hkpMotion::MOTION_KEYFRAMED )
            {
                // Invalid motionPropertiesIds get set to hknpMotionPropertiesId::KEYFRAMED in the hknpPhysicsSystem ctor.
                motionPropertiesId = hknpMotionPropertiesId::invalid();
            }
            else if ( !body.isFixedOrKeyframed() )
            {
                // Create a dynamic motionPropertiesId
                motionPropertiesId = hknpMotionPropertiesId( systemDataOut.m_motionProperties.getSize() );
                hknpMotionProperties& motionProperties = systemDataOut.m_motionProperties.expandOne();
                _processDynamicMotionProperties( body, gravity, motionProperties );
            }

            // Fixed bodies will never use a motionPropertiesId, they will be assigned STATIC motion in hknpPhysicsSystem ctor.
        }

        hknpBodyCinfo& bodyCinfo = systemDataOut.m_bodyCinfos.expandOne();
        _processBodyCinfo( body, shape, materialId, bodyCinfo );

        if( body.isFixed() )
        {
            bodyCinfo.m_motionType = hknpMotionType::STATIC;
        }
        else if ( body.getMotionType() == hkpMotion::MOTION_KEYFRAMED )
        {
            bodyCinfo.m_motionType = hknpMotionType::KEYFRAMED;
        }
        else
        {
            bodyCinfo.m_motionType = hknpMotionType::DYNAMIC;
        }


        if( bodyCinfo.m_motionType == hknpMotionType::DYNAMIC )
        {
            bodyCinfo.m_mass = body.getMass();
            bodyCinfo.m_linearVelocity = body.getLinearVelocity();
            bodyCinfo.m_angularVelocity = body.getAngularVelocity();
            bodyCinfo.m_motionPropertiesId = motionPropertiesId;

            hkMatrix3 localInertia;
            body.getRigidMotion()->getInertiaLocal( localInertia );

            hknpRefMassDistribution* massDistrib = new hknpRefMassDistribution;
            massDistrib->m_massDistribution.setFromShape(bodyCinfo.m_shape); // for the shape volume

            hkRotation principleAxisSpace;
            hkInertiaTensorComputer::convertInertiaTensorToPrincipleAxis( localInertia, principleAxisSpace );
            massDistrib->m_massDistribution.m_majorAxisSpace.set(principleAxisSpace);
            massDistrib->m_massDistribution.m_inertiaTensor.setZero();
            massDistrib->m_massDistribution.m_inertiaTensor.addMul(localInertia.getColumn(0), hkVector4::getConstant<HK_QUADREAL_1000>());
            massDistrib->m_massDistribution.m_inertiaTensor.addMul(localInertia.getColumn(1), hkVector4::getConstant<HK_QUADREAL_0100>());
            massDistrib->m_massDistribution.m_inertiaTensor.addMul(localInertia.getColumn(2), hkVector4::getConstant<HK_QUADREAL_0010>());
            massDistrib->m_massDistribution.m_inertiaTensor.mul(hkSimdReal::fromFloat(body.getMassInv()));

            massDistrib->m_massDistribution.m_centerOfMassAndVolume.setXYZ_W(body.getCenterOfMassLocal(), massDistrib->m_massDistribution.m_centerOfMassAndVolume);

            bodyCinfo.m_massDistribution.setAndDontIncrementRefCount(massDistrib);
        }

        return &bodyCinfo;
    }

    void HK_CALL convertConstraint(
        const hkpConstraintInstance& constraint,
        hknpConstraintCinfo& constraintCinfoOut,
        const hkPointerMap<hkpRigidBody*,BodyIdEx>& bodyMap )
    {
        hkpConstraintData* data = constraint.getDataRw();

        if ( data->getType() == hkpConstraintData::CONSTRAINT_TYPE_CONTACT )
        {
            return;
        }

        // Handle wrapped constraint data (breakable and malleable) which are not shared across engines.
        if ( data->getType() == hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE )
        {
            hkpBreakableConstraintData* breakableOld = static_cast<hkpBreakableConstraintData*>(data);
            hkpConstraintData* wrappedData = breakableOld->accessWrappedConstraintData();
            hknpBreakableConstraintData* breakable = new hknpBreakableConstraintData(wrappedData);
            breakable->setBreakingThreshold(breakableOld->getThreshold());
            data = breakable;
        }
        else if (data->getType() == hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE )
        {
            hkpMalleableConstraintData* malleableOld = static_cast<hkpMalleableConstraintData*>(data);
            hkpConstraintData* wrappedData = malleableOld->accessWrappedConstraintData();
            // Defaults to "no runtime" ?
            hknpMalleableConstraintData* malleable = new hknpMalleableConstraintData(wrappedData, false);
            malleable->setStrength(malleableOld->getStrength());
            data = malleable;
        }

        constraintCinfoOut.m_constraintData = data;
        constraintCinfoOut.m_bodyA = bodyMap.getWithDefault( constraint.getRigidBodyA(), hknpBodyId::invalid() );
        constraintCinfoOut.m_bodyB = bodyMap.getWithDefault( constraint.getRigidBodyB(), hknpBodyId::invalid() );
    }

    void HK_CALL convertPhysicsSystem(
        const hkpPhysicsSystem& physicsSystem,
        hknpPhysicsSystemData& systemDataOut,
        const hkVector4* newWorldGravity /*= HK_NULL */,
        hkPointerMap<const hkpShape*, hknpShape*>* shapesMap /*= HK_NULL */ )
    {
        bool internalShapeMape = false;
        if (shapesMap == HK_NULL)
        {
            shapesMap = new hkPointerMap< const hkpShape*, hknpShape*>();
            internalShapeMape = true;
        }

        // Copy name
        systemDataOut.m_name = physicsSystem.getName();

        // Convert bodies
        const hkArray<hkpRigidBody*>& rigidBodies = physicsSystem.getRigidBodies();
        hkPointerMap<hkpRigidBody*,BodyIdEx> bodyMap;
        {
            for( int i = 0; i < rigidBodies.getSize(); i++ )
            {
                // Gravity is only used to set a solver stabilization factor...only it's magnitude matters.
                
                hkVector4 gravity;
                gravity.set( 0.0f, 0.0f, -9.8f, 0.0f );
                bodyMap.insert( rigidBodies[i], systemDataOut.m_bodyCinfos.getSize() ); // since body IDs in hknpPhysicsSystem are actually indices
                convertBody( *rigidBodies[i], (newWorldGravity != HK_NULL) ? *newWorldGravity : gravity, systemDataOut, shapesMap );
            }
        }

        // Convert constraints
        {
            const hkArray<hkpConstraintInstance*>& constraints = physicsSystem.getConstraints();
            for( int cIter = 0; cIter < constraints.getSize(); cIter++ )
            {
                hknpConstraintCinfo& constraintCinfo = systemDataOut.m_constraintCinfos.expandOne();
                convertConstraint( *constraints[cIter], constraintCinfo, bodyMap );
            }
        }

        HK_ASSERT(0x65c12cd1, physicsSystem.getActions().getSize() == 0, "Actions are currently not converted.");
        HK_ASSERT(0x65c12cd1, physicsSystem.getPhantoms().getSize() == 0, "Phantoms are currently not converted.");
        HK_ASSERT(0x65c12cd1, physicsSystem.getUserData() == 0, "User data is currently not converted.");

        if (internalShapeMape)
        {
            delete shapesMap;
        }
    }

    hkReferencedObject* HK_CALL convertCharacterControllerCinfo(
        const hkReferencedObject& characterCinfo,
        hkPointerMap<const hkpShape*, hknpShape*>* shapesMap )
    {
        if ( const hkpCharacterProxyCinfo* oldProxyInfo = hkDynCast<hkpCharacterProxyCinfo>( &characterCinfo ) )
        {
            hknpCharacterProxyCinfo* characterProxyInfo = new hknpCharacterProxyCinfo();
            {
                characterProxyInfo->m_dynamicFriction = oldProxyInfo->m_dynamicFriction;
                characterProxyInfo->m_staticFriction = oldProxyInfo->m_staticFriction;
                characterProxyInfo->m_keepContactTolerance = oldProxyInfo->m_keepContactTolerance;
                characterProxyInfo->m_keepDistance = oldProxyInfo->m_keepDistance;
                characterProxyInfo->m_contactAngleSensitivity = oldProxyInfo->m_contactAngleSensitivity;
                characterProxyInfo->m_userPlanes = oldProxyInfo->m_userPlanes;
                characterProxyInfo->m_maxCharacterSpeedForSolver = oldProxyInfo->m_maxCharacterSpeedForSolver;
                characterProxyInfo->m_characterStrength = oldProxyInfo->m_characterStrength;
                characterProxyInfo->m_characterMass = oldProxyInfo->m_characterMass;
                characterProxyInfo->m_maxSlope = oldProxyInfo->m_maxSlope;
                characterProxyInfo->m_penetrationRecoverySpeed = oldProxyInfo->m_penetrationRecoverySpeed;
                characterProxyInfo->m_maxCastIterations = oldProxyInfo->m_maxCastIterations;
                characterProxyInfo->m_refreshManifoldInCheckSupport = oldProxyInfo->m_refreshManifoldInCheckSupport;

                if ( oldProxyInfo->m_shapePhantom && oldProxyInfo->m_shapePhantom->getCollidable() && oldProxyInfo->m_shapePhantom->getCollidable()->getShape() )
                {
                    hkRefPtr<hknpShape> shape = _processShape( *oldProxyInfo->m_shapePhantom->getCollidable()->getShape(), shapesMap );
                    characterProxyInfo->m_shape = shape;
                }
            }

            return characterProxyInfo;
        }
        else if ( const hkpCharacterRigidBodyCinfo* oldRigidBodyInfo = hkDynCast<hkpCharacterRigidBodyCinfo>( &characterCinfo ) )
        {
            hknpCharacterRigidBodyCinfo* characterRigidBodyInfo = new hknpCharacterRigidBodyCinfo();
            {
                characterRigidBodyInfo->m_mass = oldRigidBodyInfo->m_mass;
                characterRigidBodyInfo->m_maxForce = oldRigidBodyInfo->m_maxForce;

                
                characterRigidBodyInfo->m_dynamicFriction = oldRigidBodyInfo->m_friction;
                characterRigidBodyInfo->m_staticFriction = oldRigidBodyInfo->m_friction;

                characterRigidBodyInfo->m_maxSlope = oldRigidBodyInfo->m_maxSlope;
                characterRigidBodyInfo->m_maxSpeedForSimplexSolver = oldRigidBodyInfo->m_maxSpeedForSimplexSolver;
                characterRigidBodyInfo->m_supportDistance = oldRigidBodyInfo->m_supportDistance;
                characterRigidBodyInfo->m_hardSupportDistance = oldRigidBodyInfo->m_hardSupportDistance;

                
                // oldRigidBodyInfo->m_maxLinearVelocity;

                
                // oldRigidBodyInfo->m_allowedPenetrationDepth;

                
                // oldRigidBodyInfo->m_unweldingHeightOffsetFactor

                if ( oldRigidBodyInfo->m_shape )
                {
                    hkRefPtr<hknpShape> shape = _processShape( *oldRigidBodyInfo->m_shape, shapesMap );
                    characterRigidBodyInfo->m_shape = shape;
                }
            }

            return characterRigidBodyInfo;
        }
        else
        {
            HK_ASSERT( 0x22440537, false, "Unrecognized character controller cinfo" );
            return HK_NULL;
        }
    }

    bool HK_CALL convertRootLevelContainer(
        const hkRootLevelContainer& rootLevelContainer,
        hkRootLevelContainer& rootLevelContainerOut,
        const hkVector4* newWorldGravity,
        hkPointerMap<const hkpShape*, hknpShape*>* shapesMap,
        bool pruneNonConvertedPhysicsData )
    {
        // Note if we make a change of any kind
        bool changed = false;

        bool internalShapeMape = false;
        if (shapesMap == HK_NULL)
        {
            shapesMap = new hkPointerMap< const hkpShape*, hknpShape*>();
            internalShapeMape = true;
        }

        // Iterate over old data, convert as we go
        for( int i = 0; i < rootLevelContainer.m_namedVariants.getSize(); ++i )
        {
            if( rootLevelContainer.m_namedVariants[i].getTypeName() && rootLevelContainer.m_namedVariants[i].get<hkpPhysicsData>() )
            {
                // Grab old data info
                hkpPhysicsData* physicsData = rootLevelContainer.m_namedVariants[i].get<hkpPhysicsData>();
                hkStringPtr physicsDataName;
                if ( &rootLevelContainer == &rootLevelContainerOut )
                {
                    physicsDataName.printf( "_%s_converted_np", rootLevelContainer.m_namedVariants[i].getName() );
                }
                else
                {
                    physicsDataName = rootLevelContainer.m_namedVariants[i].getName();
                }

                // Convert it
                
                
                {
                    hknpPhysicsSceneData* npPhysicsData = new hknpPhysicsSceneData();

                    for ( int systemIter = 0; systemIter < physicsData->getPhysicsSystems().getSize(); systemIter++ )
                    {
                        const hkpPhysicsSystem* physicsSystem = physicsData->getPhysicsSystems()[systemIter];

                        hknpPhysicsSystemData* npPhysicsSystem = new hknpPhysicsSystemData();
                        convertPhysicsSystem( *physicsSystem, *npPhysicsSystem, newWorldGravity, shapesMap );

                        npPhysicsData->m_systemDatas.expandOne().setAndDontIncrementRefCount( npPhysicsSystem );
                    }

                    hkRootLevelContainer::NamedVariant& namedVariant = rootLevelContainerOut.m_namedVariants.expandOne();
                    namedVariant.set( physicsDataName, npPhysicsData );
                    npPhysicsData->removeReference();
                }

                changed = true;
            }
            else if ( &rootLevelContainer != &rootLevelContainerOut )
            {
                // Pass object through to new container
                
                
                
                {
                    rootLevelContainerOut.m_namedVariants.expandOne() = rootLevelContainer.m_namedVariants[i];
                }
            }
        }

        // Prune any physics data if requested
        if ( pruneNonConvertedPhysicsData )
        {
            changed |= pruneDeprecatedClasses( rootLevelContainerOut );
        }

        if (internalShapeMape)
        {
            delete shapesMap;
        }

        return changed;
    }

    hkRootLevelContainer::NamedVariant HK_CALL pruneDeprecatedDestructionClasses( const hkRootLevelContainer::NamedVariant& namedVariant, bool& changed )
    {
        
        if ( hkString::strCmp(namedVariant.getName(), "Resource Data") == 0 )
        {
            hkResourceContainer* resource_Data = reinterpret_cast<hkResourceContainer*>(namedVariant.getObject());
            hkArray<hkResourceContainer*> containers;
            resource_Data->findAllContainersRecursively(containers);

            for (int i = containers.getSize()-1; i >= 0; i--)
            {
                hkResourceContainer* container = containers[i];

                hkArray<hkResourceHandle*> handlesToDelete;
                for (hkResourceHandle* handle = container->findResourceByName(HK_NULL, HK_NULL, HK_NULL); handle;  handle = container->findResourceByName(HK_NULL, HK_NULL, handle))
                {
                    const hkReflect::Type* klass = handle->getObjectType();

                    if (isDeprecatedClass(klass->getName()))
                    {
                        handlesToDelete.pushBack(handle);
                    }
                }

                for (int handleIndex = 0; handleIndex < handlesToDelete.getSize(); handleIndex++)
                {
                    container->destroyResource(handlesToDelete[handleIndex]);
                    changed = true;
                }

                int numHandlesLeft = container->getNumResources();
                if ((numHandlesLeft == 0) && (container->getParent() != HK_NULL) && container->getNumContainers() == 0)
                {
                    container->getParent()->destroyContainer(container);
                    changed = true;
                }
            }
        }

        return namedVariant;
    }

    bool HK_CALL pruneDeprecatedClasses(
        hkRootLevelContainer& rootLevelContainer )
    {
        // monitor if we've made changes
        bool changed = false;

        // Iterate over old data, prune as we go
        for( int i = rootLevelContainer.m_namedVariants.getSize()-1; i >= 0 ; --i )
        {
            // Reference non-physics data in the new container if it's not the same as our source.

            const char* typeName = rootLevelContainer.m_namedVariants[i].getTypeName();
            if ( typeName != HK_NULL )
            {
                // Is the class directly deprecated
                if ( isDeprecatedClass( typeName ) )
                {
                    changed = true;
                    rootLevelContainer.m_namedVariants.removeAt(i);
                    continue;
                }
            }

            // Pass object through to new container
            
            
            
            {
                rootLevelContainer.m_namedVariants[i] = pruneDeprecatedDestructionClasses( rootLevelContainer.m_namedVariants[i], changed );
            }
        }

        return changed;
    }

#define HK_RECORD_TYPE(a, str, cxxstr) #str,
    static const char* EXCEPTION_CLASS_NAMES[] = {
#include <Physics/Constraint/_Auto/hkpConstraintClassList.cxx>
#include <Physics/ConstraintSolver/_Auto/hkpConstraintSolverClassList.cxx>
    };
#undef HK_RECORD_TYPE

    bool HK_CALL isDeprecatedClass( const char* className )
    {
        bool hkdClass = hkString::beginsWith( className, "hkd");
        if (hkdClass)
        {
            return true;
        }

        bool hkpClass = hkString::beginsWith( className, "hkp" );
        if (!hkpClass)
        {
            return false;
        }

        const int exceptionListSize = sizeof(EXCEPTION_CLASS_NAMES) / sizeof(char*);
        for( int classNameIt = 0; classNameIt < exceptionListSize; classNameIt++ )
        {
            const char* exceptionClass = EXCEPTION_CLASS_NAMES[classNameIt];

            if ( hkString::strCmp( className, exceptionClass ) == 0 )
            {
                return false;
            }
        }

        return true;
    }

} // hkpPhysicsMigrationUtils namespace

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
