// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/hkBase.h>

struct hkpBodyVelocity;


/// struct to hold same caches for fast collision resolution.
/// Note: the size is quite big (6*48 = 288 bytes)
/// some cached values, which allow for fast solving of simple 3d constraints
/// (e.g., contact constraints for TOI handling).
struct HK_EXPORT_PHYSICS_2012 hkpSimpleConstraintInfo
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT_SOLVER, hkpSimpleConstraintInfo );

    struct HK_EXPORT_PHYSICS_2012 BodyInfo
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT_SOLVER, hkpSimpleConstraintInfo::BodyInfo );

        hkRotation m_jacT;      // the transposed Jacobians, used to get the velocities.
        hkMatrix3 m_invIjac;    // used to apply forces .
        //hkMatrix3 m_inertia;
        hkVector4 m_invMasses;  // the inertia tensor diagonal in local space.
        hkReal m_mass;

        const hkTransform* m_transform;

        hkReal getMass() const { return m_mass; }
        hkReal getEnergy( const hkpBodyVelocity& vel ) const ;
    };

    HK_ALIGN_REAL(BodyInfo m_bodyInfo[2]);

    /// first column is the normal direction, the other two columns are perpendicular.
    hkRotation m_base;

    /// calculated by J^-1 * invI * J.
    hkMatrix3 m_invMassMatrix;

    /// = m_invMassMatrix^-1
    hkMatrix3 m_massMatrix;

    /// store the inverted determinant of the invMassMatrix without row and column 1 and 2.
    hkReal& getMass00(){ return     m_invMassMatrix.getColumn(0)(3); }
    hkReal getMass00() const{ return    m_invMassMatrix.getColumn(0)(3); }
    hkSimdReal getMass00Sr() const { return     m_invMassMatrix.getElement<3,0>(); }

    /// store the inverted determinant of the invMassMatrix without row and column 1 and 2.
    hkReal& getInvDetM12()          { return    m_invMassMatrix.getColumn(1)(3); }
    hkReal  getInvDetM12() const    { return    m_invMassMatrix.getColumn(1)(3); }
    hkSimdReal  getInvDetM12Sr() const  { return    m_invMassMatrix.getElement<3,1>(); }
};


struct HK_EXPORT_PHYSICS_2012 hkpSimpleConstraintInfoInitInput
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT_SOLVER, hkpSimpleConstraintInfoInitInput );

    hkVector4 m_massRelPos;
    hkMatrix3 m_invInertia;
    hkVector4 m_invMasses;          // the inertia in local space. (3) = mass
    const hkTransform* m_transform; // the transform of the 'rigid body'. This is used mainly by the constraintModifiers
};


struct HK_EXPORT_PHYSICS_2012 hkpSimpleConstraintUtilCollideParams
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT_SOLVER, hkpSimpleConstraintUtilCollideParams );

    hkReal m_friction;
    hkReal m_restitution;
    hkReal m_maxImpulse;

    /// This is the approaching velocity calculated outside this function
    /// (typically the collision velocity).
    /// This allows to make sure that the function applies an impulse even if the
    /// dynamics projected velocities are > 0.
    hkReal m_externalSeperatingVelocity;
    hkReal m_extraSeparatingVelocity;

    // This value will virtually rotate the colliding surface towards the velocity.
    hkReal m_extraSlope;

    //
    //  output
    //
    hkBool m_contactImpulseLimitBreached;   // this is an output variable
};


struct HK_EXPORT_PHYSICS_2012 hkpBodyVelocity
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT_SOLVER, hkpBodyVelocity );

    hkVector4 m_linear; // always in global space
    hkVector4 m_angular; // always in global space
};


extern "C"
{
    /// Initialize a Jacobian cache, directions specifies the direction space, directions.getColumn(0) must be the contact normal.
    HK_EXPORT_PHYSICS_2012 void HK_CALL hkSimpleConstraintUtil_InitInfo(    const hkpSimpleConstraintInfoInitInput& bodyA, const hkpSimpleConstraintInfoInitInput& bodyB,
                                                    const hkMatrix3& directions, hkpSimpleConstraintInfo& infoOut);

    /// Get the point velocity between two objects.
    HK_EXPORT_PHYSICS_2012 void HK_CALL hkSimpleConstraintUtil_getPointVelocity( const hkpSimpleConstraintInfo& info, hkpBodyVelocity& velA, hkpBodyVelocity& velB, hkVector4& out );


    /// Apply an impulse at object a and -impulse at object b, impulse specified in direction space.
    HK_EXPORT_PHYSICS_2012 void HK_CALL hkSimpleConstraintUtil_applyImpulse(    const hkpSimpleConstraintInfo& info, const hkVector4& impulse,
                                                        hkpBodyVelocity& velA, hkpBodyVelocity& velB );

    ///
//  HK_EXPORT_PHYSICS_2012 hkReal HK_CALL hkSimpleConstraintUtil_calcTotalEnergy( const hkpSimpleConstraintInfo& info, hkpBodyVelocity& velA, hkpBodyVelocity& velB );

    /// Collide two objects, returns the impulse in the normal direction.
    HK_EXPORT_PHYSICS_2012 hkReal  HK_CALL hkSimpleConstraintUtil_Collide(  const hkpSimpleConstraintInfo& info, hkpSimpleConstraintUtilCollideParams& params,
                                                    hkpBodyVelocity& velA, hkpBodyVelocity& velB );
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
