// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics2012/Dynamics/Phantom/hkpPhantom.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

struct hkpWorldRayCastInput;
struct hkpWorldRayCastOutput;
class hkpRayHitCollector;
struct hkpLinearCastInput;
class hkpCdPointCollector;
class hkpCdPointCollector;
class hkpCollidable;




/// This phantom simply maintains a list of hkpCollidable objects that overlap with the phantom's AABB.
/// hkAabbPhantoms are very fast as they use only broadphase collision detection. They are recommended
///  - for any triggers / regions you need for game logic for which a detailed shape is not necessary (i.e you can
///    use an AABB). This phantom has no hkpShape associated with it. If you call getShape() on the hkpCollidable
///    stored in this phantom, it will return HK_NULL.
///  - to speed up short-ray casts, which are repeated every frame at roughly the same position
///    (e.g the wheels of the car in the Havok vehicle SDK).
///    See hkpAabbPhantom::castRay() for details.
///
/// Note: This is highly not recommended if you have long or non frame coherent raycasts.
class HK_EXPORT_PHYSICS_2012 hkpAabbPhantom : public hkpPhantom
{
    public:
        HK_DECLARE_CLASS(hkpAabbPhantom, New, Reflect, BypassCtor);

            /// To add a phantom to the world, you use hkpWorld::addEntity( myPhantom ).
        hkpAabbPhantom( const hkAabb& aabb, hkUint32 collisionFilterInfo = 0 );

            /// Gets the hkpPhantom type. For this class the type is HK_PHANTOM_AABB
        virtual hkpPhantomType getType() const;

            /// Gets the list of collidables that are currently overlapping with the phantom.
            /// Although unlikely, it is possible that the order of elements in the array is
            /// nondeterministic. To ensure determinism, proceed the call by a call to
            /// ensureDeterministicOrder().
        inline hkArray<hkpCollidable*>& getOverlappingCollidables();

            /// Casts a ray specified by "input". For each hit found the "collector" object receives
            /// a callback. See the hkpRayHitCollector for details.
            /// Note: You must make sure that the hkAabb for the phantom contains the ray being cast.
            /// call setAabb() with an AABB of the correct size in the correct place if this is not the case.
            /// The phantom castRay collects up all the collidables overlapping with the AABB of the ray, and
            /// calls castRay against these collidables. This is usually much fast than calling hkpWorld::castRay().
            /// If you are calling a long ray cast, you should use
            /// hkpWorld::castRay instead.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RO] [this,HK_ACCESS_RO] );
        void castRay( const hkpWorldRayCastInput& input, hkpRayHitCollector& collector ) const;

            /// Casts a ray specified by "input". This fills out the output structure with the closest
            /// ray hit.
            /// Note: You must make sure that the hkAabb for the phantom contains the ray being cast.
            /// call setAabb() with an AABB of the correct size in the correct place if this is not the case.
            /// The phantom castRay collects up all the collidables overlapping with the AABB of the ray, and
            /// calls castRay against these collidables. If you are calling a long ray cast, you should use
            /// hkpWorld::castRay instead.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RO] [this,HK_ACCESS_RO] );
        void castRay( const hkpWorldRayCastInput& input, hkpWorldRayCastOutput& output ) const;

            /// Casts the collidable "toBeCast" against each of this phantom's overlapping
            /// collidables.
            /// The collision filter of the given collidable is tested against the
            /// collision filter of any collidables overlapping with the phantom, using the
            /// world's collision filter.
            /// This is a convenience function for performing asynchronous queries like
            /// hkpAabbPhantom::castRay, and does not modify the phantom or the collidable
            /// which is cast.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RO] [this,HK_ACCESS_RO] );
        void linearCast(const hkpCollidable* const toBeCast, const hkpLinearCastInput& input, hkpCdPointCollector& castCollector, hkpCdPointCollector* startCollector ) const;


        //
        // hkpPhantom interface
        //

            // hkpPhantom interface implementation
        virtual void calcAabb( hkAabb& aabb );

            // hkpPhantom interface implementation
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RO] [this,HK_ACCESS_RW] );
        virtual void addOverlappingCollidable( hkpCollidable* collidable );

            // hkpPhantom interface implementation
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RO] );
        virtual hkBool isOverlappingCollidableAdded( const hkpCollidable* collidable );

            // hkpPhantom interface implementation
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RO] [this,HK_ACCESS_RW] );
        virtual void removeOverlappingCollidable( hkpCollidable* collidable );

            /// hkpPhantom clone interface
        virtual hkpPhantom* clone() const;

            /// Sets the AABB.
            /// This will also update the broadphase, which may trigger the callbacks addOverlappingCollidable and removeOverlappingCollidable.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RO] [this,HK_ACCESS_RW] );
        void setAabb(const hkAabb& newAabb);

            /// Get the AABB of the phantom.
        inline const hkAabb& getAabb( ) const;

    protected:
        virtual hkMotionState* getMotionState(){ return HK_NULL; }

            /// Phantom batch move utility needs to access the AABB.
        friend class hkpPhantomBatchMoveUtil;

        class hkAabb m_aabb;

        hkArray<hkpCollidable*> m_overlappingCollidables; //+nosave

    public:

        //
        // INTERNAL USE ONLY
        //

            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RO] [this,HK_ACCESS_RW] );
        virtual void deallocateInternalArrays();

    public:
            /// Ensures that collidables returned by getOverlappingCollidables are in deterministic
            /// order.
        virtual void ensureDeterministicOrder();

    public:
            /// False if the array of collidables has changed since we last sorted it.
        hkBool m_orderDirty; //+nosave

            /// Order relation on m_overlappingCollidables.
        class OrderByUid;
};

#include <Physics2012/Dynamics/Phantom/hkpAabbPhantom.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
