// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/Dynamics/hkpDynamics.h>
#include <Physics2012/Dynamics/Motion/SweptTransform/hkSweptTransformUtil.h>

namespace hkSweptTransformUtil
{
    void lerp2( const hkSweptTransform& sweptTrans, hkSimdRealParameter t, hkTransform& transformOut )
    {
        hkSimdReal r = sweptTrans.getInterpolationValue( t );
        _lerp2(sweptTrans, r, transformOut);
    }

    void lerp2( const hkSweptTransform& sweptTrans, hkTime t, hkTransform& transformOut )
    {
        lerp2(sweptTrans, hkSimdReal::fromFloat(hkReal(t)), transformOut);
    }

    void lerp2Ha( const hkSweptTransform& sweptTrans, hkSimdRealParameter t, hkSimdRealParameter tAddOn, hkTransform& transformOut )
    {
        const hkSimdReal r = sweptTrans.getInterpolationValueHiAccuracy( t, tAddOn );
        _lerp2(sweptTrans, r, transformOut);
    }

    void lerp2Ha( const hkSweptTransform& sweptTrans, hkTime t, hkReal tAddOn, hkTransform& transformOut )
    {
        lerp2Ha(sweptTrans, hkSimdReal::fromFloat(hkReal(t)), hkSimdReal::fromFloat(tAddOn), transformOut);
    }

    void lerp2Rel( const hkSweptTransform& sweptTrans, hkSimdRealParameter r, hkTransform& transformOut )
    {
        _lerp2(sweptTrans, r, transformOut);
    }

    void lerp2Rel( const hkSweptTransform& sweptTrans, hkReal r, hkTransform& transformOut )
    {
        _lerp2(sweptTrans, hkSimdReal::fromFloat(r), transformOut);
    }

#if 1 || defined(HK_REAL_IS_FLOAT)

        // only updates t1 and 'hkTransform' of the hkSweptTransform
    void backStepMotionState( hkSimdRealParameter time, hkMotionState& motionState )
    {
        hkSweptTransform& st = motionState.getSweptTransform();
        hkSimdReal t; t.setMax( st.getInterpolationValue( time ), hkSimdReal_Eps );

        _lerp2( st, t, st.m_rotation1 );
        const hkSimdReal newInvDeltaTime = st.getInvDeltaTimeSr() / t;

        st.m_centerOfMass1.setInterpolate( st.m_centerOfMass0, st.m_centerOfMass1, t );
        st.m_centerOfMass1.setComponent<3>(newInvDeltaTime);
        motionState.m_deltaAngle.mul( t );

        calcTransAtT1( st, motionState.getTransform());
    }

    void backStepMotionState( hkTime time, hkMotionState& motionState )
    {
        backStepMotionState(hkSimdReal::fromFloat(time),motionState);
    }

        // resets both t0 and t1 transforms of the hkSweptTransform to the same value and
        // sets invDeltaTime to zero
    void freezeMotionState( hkSimdRealParameter time, hkMotionState& motionState )
    {
        hkSweptTransform& st = motionState.getSweptTransform();
        HK_ASSERT(0xf0ff0082, st.getInvDeltaTime() == hkReal(0) || (( time  * st.getInvDeltaTimeSr() ).getReal() <= ( st.getBaseTime() * st.getInvDeltaTime() ) + hkReal(2) ) , "Inconsistent time in motion state.");

            // we actually freeze the object at the earliest moment (defined by hkSweptTransform.m_startTime) after 'time'
        hkSimdReal maxTime; maxTime.setMax(time, st.getBaseTimeSr());
        const hkSimdReal t = st.getInterpolationValue( maxTime );

        _lerp2( st, t, st.m_rotation1 );
        st.m_rotation0 = st.m_rotation1;

        st.m_centerOfMass1.setInterpolate( st.m_centerOfMass0, st.m_centerOfMass1, t );
        st.m_centerOfMass0 = st.m_centerOfMass1;

        // set time information
        st.m_centerOfMass0.setComponent<3>(maxTime);
        st.m_centerOfMass1.zeroComponent<3>();

        calcTransAtT1( st, motionState.getTransform());
    }

    void freezeMotionState( hkTime time, hkMotionState& motionState )
    {
        freezeMotionState(hkSimdReal::fromFloat(time),motionState);
    }

    #if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    void setTimeInformation( hkTime startTime, hkReal invDeltaTime, hkMotionState& motionState)
    {
        motionState.getSweptTransform().m_centerOfMass0(3) = startTime;
        motionState.getSweptTransform().m_centerOfMass1(3) = invDeltaTime;
    }
    #endif

    void setTimeInformation( hkSimdRealParameter startTime, hkSimdRealParameter invDeltaTime, hkMotionState& motionState)
    {
        motionState.getSweptTransform().m_centerOfMass0.setComponent<3>(startTime);
        motionState.getSweptTransform().m_centerOfMass1.setComponent<3>(invDeltaTime);
    }

    void warpTo( hkVector4Parameter position, hkQuaternionParameter rotation, hkMotionState& ms )
    {
        hkSweptTransform& sweptTransform = ms.getSweptTransform();
        ms.m_deltaAngle.setZero();

        sweptTransform.m_rotation0 = rotation;
        sweptTransform.m_rotation1 = rotation;

        ms.getTransform().setRotation( rotation );
        ms.getTransform().setTranslation( position );

        hkVector4 centerShift;
        centerShift._setRotatedDir( ms.getTransform().getRotation(), sweptTransform.m_centerOfMassLocal );

        const hkSimdReal baseTime = sweptTransform.getBaseTimeSr();

        sweptTransform.m_centerOfMass0.setAdd( position, centerShift );
        sweptTransform.m_centerOfMass1 = sweptTransform.m_centerOfMass0;

        sweptTransform.m_centerOfMass0.setComponent<3>(baseTime);
        sweptTransform.m_centerOfMass1.zeroComponent<3>();  // invDeltaTime
    }

    void warpTo( const hkTransform& transform, hkMotionState& ms )
    {
        hkSweptTransform& sweptTransform = ms.getSweptTransform();
        ms.m_deltaAngle.setZero();

        hkQuaternion rotation; rotation.set( transform.getRotation() );
        ms.getTransform() = transform;

        sweptTransform.m_rotation0 = rotation;
        sweptTransform.m_rotation1 = rotation;

        hkVector4 centerShift;
        centerShift._setRotatedDir( transform.getRotation(), sweptTransform.m_centerOfMassLocal );

        const hkSimdReal baseTime = sweptTransform.getBaseTimeSr();

        sweptTransform.m_centerOfMass0.setAdd( transform.getTranslation(), centerShift );
        sweptTransform.m_centerOfMass1 = sweptTransform.m_centerOfMass0;

        sweptTransform.m_centerOfMass0.setComponent<3>(baseTime);
        sweptTransform.m_centerOfMass1.zeroComponent<3>();  // invDeltaTime
    }

    void warpToPosition( hkVector4Parameter position, hkMotionState& ms )
    {
        const hkRotation& currentRotation = ms.getTransform().getRotation();
        hkSweptTransform& sweptTransform = ms.getSweptTransform();

        ms.m_deltaAngle.setZero();
        ms.getTransform().setTranslation( position );

        hkVector4 centerShift;
        centerShift._setRotatedDir( currentRotation, sweptTransform.m_centerOfMassLocal );

        const hkSimdReal baseTime = sweptTransform.getBaseTimeSr();

        sweptTransform.m_centerOfMass0.setAdd( position, centerShift );
        sweptTransform.m_centerOfMass1 = sweptTransform.m_centerOfMass0;

        sweptTransform.m_rotation0 = sweptTransform.m_rotation1;

        sweptTransform.m_centerOfMass0.setComponent<3>(baseTime);
        sweptTransform.m_centerOfMass1.zeroComponent<3>(); // invDeltaTime
    }

    void warpToRotation( hkQuaternionParameter rotation, hkMotionState& ms )
    {
        warpTo( ms.getTransform().getTranslation(), rotation, ms );
    }

    void keyframeMotionState( const hkStepInfo& stepInfo, hkVector4Parameter pos1, hkQuaternionParameter rot1, hkMotionState& ms )
    {
        hkSweptTransform& sweptTransform = ms.getSweptTransform();

        sweptTransform.m_centerOfMass0 = sweptTransform.m_centerOfMass1;
        sweptTransform.m_rotation0 = sweptTransform.m_rotation1;

        sweptTransform.m_centerOfMass1 = pos1;
        sweptTransform.m_rotation1 = rot1;

        hkQuaternion diff; diff.setMulInverse( rot1, sweptTransform.m_rotation0 );

        hkSimdReal angle = diff.getAngleSr();
        hkVector4 axis;
        if ( diff.hasValidAxis() )
        {
            diff.getAxis(axis);
        }
        else
        {
            axis.setZero();
        }

        ms.m_deltaAngle.setMul( angle, axis );
        ms.m_deltaAngle.setComponent<3>(angle);

        //
        //  Use the angle to calculate redundant information
        //
        /*
        {
            const hkReal angle2 = angle * angle;
            const hkReal angle3 = angle * angle2;
            const hkReal sa = 0.044203f;
            const hkReal sb = 0.002343f;

                // this is:
                // 2.0f * sin( 0.5f * angle ) / angle
                // and can be used as a factor to m_deltaAngle
                // to get m_deltaAngleLower or the maximum projected distance any
                // point on the unit sphere of the object can travel
            const hkReal rel2SinHalfAngle = 1.0f - sa * angle2 + sb * angle3;
            const hkReal collisionToleranceEps = 0.01f * 0.01f;
            ms.m_maxAngularError = collisionToleranceEps + ms.m_objectRadius * rel2SinHalfAngle * angle;
        }
        */

        sweptTransform.m_centerOfMass0(3) = stepInfo.m_startTime;
        sweptTransform.m_centerOfMass1(3) = stepInfo.m_invDeltaTime;

        calcTransAtT1( sweptTransform, ms.getTransform() );
    }

    void setCentreOfRotationLocal( hkVector4Parameter newCenterOfRotation, hkMotionState& motionState)
    {
        hkSweptTransform& st = motionState.getSweptTransform();

        hkVector4 offset; offset.setSub(newCenterOfRotation, st.m_centerOfMassLocal);
        st.m_centerOfMassLocal = newCenterOfRotation;

        hkVector4 offsetWs; offsetWs._setRotatedDir(motionState.getTransform().getRotation(), offset);

        const hkSimdReal t0 = st.m_centerOfMass0.getComponent<3>();
        const hkSimdReal t1 = st.m_centerOfMass1.getComponent<3>();

        st.m_centerOfMass0.add(offsetWs);
        st.m_centerOfMass1.add(offsetWs);

        st.m_centerOfMass0.setComponent<3>(t0);
        st.m_centerOfMass1.setComponent<3>(t1);
    }

#endif // defined(HK_REAL_IS_FLOAT)

    // Has to be outside of inline function as gcc won't inline functions with statics in them.
    const hkQuadReal _stepMotionStateMaxVelf = HK_QUADFLOAT_CONSTANT(1e6f,1e6f,1e6f,1e6f);

} // namespace hkSweptTransformUtil

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
