// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Physics2012/Dynamics/Motion/SweptTransform/hkSweptTransform.h>


    /// An hkMotionState basically is the dynamic state of
    /// a rigid body, which is used for collision detection and basic
    /// integration. It includes basically an hkSweptTransform.
    /// It also holds a transform which caches the position and orientation
    /// of the body at time t1.
class HK_EXPORT_PHYSICS_2012 hkMotionState
{
    public:

        HK_DECLARE_CLASS(hkMotionState, New, Reflect);
        HK_RECORD_ATTR( hk::Version(3) );

            /// Empty constructor, does not initialize anything
        inline hkMotionState();

            /// Initialize with position and rotation
        void initMotionState( const hkVector4& position, const hkQuaternion& rotation );

            /// Get a non-const reference to the transform.
        inline hkTransform& getTransform();

            /// Get a const reference to the transform.
        inline const hkTransform& getTransform() const;

            /// Get a const reference to the swept transform.
        inline const hkSweptTransform& getSweptTransform() const;

            /// Get a non const reference to the transform.
        inline hkSweptTransform& getSweptTransform();

    protected:

        hkTransform m_transform;

            // a swept transform
        class hkSweptTransform m_sweptTransform; //+overridetype(hkVector4[5])

    public:

            /// The angular velocity * dt which was used in the last integration step.
            /// Note: .w = m_deltaAngle.length<3>()
        hkVector4 m_deltaAngle;

            /// A sphere around the center of mass which completely encapsulates the object
        hkReal m_objectRadius;

            /// linear damping
        hkHalf16 m_linearDamping;

            /// angular damping
        hkHalf16 m_angularDamping;

            /// time factor.
        hkHalf16 m_timeFactor;

            /// The maximum linear velocity
        hkUFloat8 m_maxLinearVelocity;

            /// The maximum angular velocity. The units used here are radians per second.
            /// Note: there is also a hard limit in the engine which is 170 degrees/timestep.
        hkUFloat8 m_maxAngularVelocity;

            // Note: the next variables actually belong to hkpMotion, however they can cheaply put here (padding!)
            // The deactivation class of the motion state as used internally
        hkUint8 m_deactivationClass;
};


namespace hkAabbUtil
{
    struct HK_EXPORT_PHYSICS_2012 OffsetAabbInput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_COLLIDE, hkAabbUtil::OffsetAabbInput );

        const hkMotionState* m_motionState;

        // intermediate transforms
        hkTransform m_transforms[4];
        int m_numTransforms;

        // start-end transforms
        hkTransform m_startTransform;
        hkTransform m_endTransformInv;
    };

    /// Sweep a discrete AABB and convert it into a 'continuous' swept AABB.
    HK_EXPORT_PHYSICS_2012 void HK_CALL sweepAabb( const hkMotionState* motionstate, hkReal tolerance, const hkAabb& aabbIn, hkAabb& aabbOut );

    /// Calculate OffsetAabbInput.
    HK_EXPORT_PHYSICS_2012 void HK_CALL initOffsetAabbInput( const hkMotionState* motionstate, OffsetAabbInput& input );

    /// Sweep a discrete AABB which has a huge offset from the mass center and convert it into a 'continuous' swept AABB.
    HK_EXPORT_PHYSICS_2012 void HK_CALL sweepOffsetAabb( const OffsetAabbInput& input, const hkAabb& aabbIn, hkAabb* HK_RESTRICT aabbOut );
}

#include <Physics2012/Dynamics/Motion/MotionState/hkMotionState.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
