// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/Dynamics/hkpDynamics.h>
#include <Physics2012/Dynamics/Motion/MotionState/hkMotionState.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkCheckDeterminismUtil.h>

void hkMotionState::initMotionState( const hkVector4& position, const hkQuaternion& rotation )
{
    m_sweptTransform.initSweptTransform( position, rotation );

    getTransform().set( rotation, position );

    m_deltaAngle.setZero();
    m_objectRadius = 1.0f;
    hkCheckDeterminismUtil::checkMt(0xf0000002, m_objectRadius);

    // Initialize the rest to "invalid" but deterministic values (HVK-6297).
    m_linearDamping.setZero();
    m_angularDamping.setZero();
    m_timeFactor.setZero();
    m_maxLinearVelocity = 0.0f;
    m_maxAngularVelocity = 0.0f;
    m_deactivationClass = 0;    // = hkpSolverInfo::DEACTIVATION_CLASS_INVALID
}

namespace hkAabbUtil
{
    HK_EXPORT_PHYSICS_2012 void HK_CALL initOffsetAabbInput( const hkMotionState* motionState, OffsetAabbInput& input )
    {
        input.m_motionState = motionState;
        input.m_endTransformInv.setInverse( motionState->getTransform() );

        const hkSweptTransform& swept = motionState->getSweptTransform();
        swept._approxTransformAt( swept.getBaseTimeSr(), input.m_startTransform );

        if( swept.getInvDeltaTimeSr().isNotEqualZero() )
        {
            hkSimdReal deltaTime; deltaTime.setReciprocal( swept.getInvDeltaTimeSr() );
            const hkSimdReal deltaAngle = motionState->m_deltaAngle.getComponent<3>();
            const hkSimdReal pi_8 = hkSimdReal::fromFloat( HK_REAL_PI * 0.125f );
            const hkSimdReal pi_4 = hkSimdReal_PiOver4;
            if( deltaAngle <= pi_8 )
            {
                // Just one inter transform
                hkSimdReal invCosApprox; invCosApprox.setReciprocal( hkSimdReal_1 - deltaAngle*deltaAngle*hkSimdReal_Inv2 );
                input.m_transforms[0] = input.m_startTransform;
                input.m_transforms[0].getTranslation().setComponent<3>( invCosApprox );
                input.m_numTransforms = 1;
                return;
            }

            if( deltaAngle <= pi_4 )
            {
                // Just one inter transform
                const hkSimdReal time = swept.getBaseTimeSr() + hkSimdReal_Inv2 * deltaTime;
                motionState->getSweptTransform()._approxTransformAt( time, input.m_transforms[0] );
                // extended arm for the in-between transforms (cos(22.5deg)
                hkSimdReal invCosApprox; invCosApprox.setReciprocal( hkSimdReal_1 - deltaAngle*deltaAngle*hkSimdReal_Inv4*hkSimdReal_Inv2 );
                input.m_transforms[0].getTranslation().setComponent<3>( invCosApprox );
                input.m_numTransforms = 1;
                return;
            }

            {
                const hkSimdReal parts = (deltaAngle + pi_8) / pi_8;
                hkSimdReal partsInv; partsInv.setReciprocal( parts );
                input.m_numTransforms = 0;
                const hkSimdReal cst = hkSimdReal::fromFloat( 1.0824f );
                for( hkSimdReal p = hkSimdReal_1; p < parts; p += hkSimdReal_2 )
                {
                    const hkSimdReal time = swept.getBaseTimeSr() + (p*partsInv) * deltaTime;
                    HK_ASSERT( 0xad7644aa, input.m_numTransforms < 4, "The fixed-capacity Transforms array to small. Make it larger." );
                    hkTransform& t = input.m_transforms[input.m_numTransforms];
                    input.m_numTransforms = input.m_numTransforms + 1;
                    motionState->getSweptTransform()._approxTransformAt( time, t );
                    t.getTranslation().setComponent<3>( cst );
                }
            }
        }
        else
        {
            input.m_numTransforms = 0;
        }
    }

    HK_EXPORT_PHYSICS_2012 void sweepAabb( const hkMotionState* motionState, hkReal tolerance, const hkAabb& aabbIn, hkAabb& aabbOut )
    {
        //
        //  Expand the AABB by the angular part
        //

        hkSimdReal objectRadius; objectRadius.setFromFloat( motionState->m_objectRadius );
        {
            const hkSimdReal radius = motionState->m_deltaAngle.getW() * objectRadius;
            aabbOut.m_min.setSub( aabbIn.m_min, radius );
            aabbOut.m_max.setAdd( aabbIn.m_max, radius );
        }

        //
        // restrict the size of the AABB to the worst case radius size
        //
        {
            hkSimdReal toleranceSr;  toleranceSr.setFromFloat( tolerance );
            hkSimdReal radius4 = objectRadius + toleranceSr;
            hkVector4 maxR; maxR.setAdd( motionState->getSweptTransform().m_centerOfMass1, radius4 );
            hkVector4 minR; minR.setSub( motionState->getSweptTransform().m_centerOfMass1, radius4 );

            HK_ON_DEBUG
            (
                // I consider success when (aabbInOut.m_min <= maxR) and (minR <= aabbInOut.m_max) are true for all XYZ
                hkVector4Comparison a = aabbIn.m_min.lessEqual( maxR );
            hkVector4Comparison b = minR.lessEqual( aabbIn.m_max );
            hkVector4Comparison both; both.setAnd( a, b );
            HK_ASSERT( 0x366ca7b2, both.allAreSet<hkVector4ComparisonMask::MASK_XYZ>(), "Invalid Radius. Did you make changes to a shape or change its center of mass while the owner was added to the world?" );
            );

            aabbOut.m_min.setMax( aabbOut.m_min, minR );
            aabbOut.m_min.setMin( aabbOut.m_min, aabbIn.m_min );

            aabbOut.m_max.setMin( aabbOut.m_max, maxR );
            aabbOut.m_max.setMax( aabbOut.m_max, aabbIn.m_max );
        }

        //
        // Sweep/expand the AABB along the motion path
        //

        hkVector4 invPath; invPath.setSub( motionState->getSweptTransform().m_centerOfMass0, motionState->getSweptTransform().m_centerOfMass1 );
        hkVector4 zero; zero.setZero();
        hkVector4 minPath; minPath.setMin( zero, invPath );
        hkVector4 maxPath; maxPath.setMax( zero, invPath );

        aabbOut.m_min.add( minPath );
        aabbOut.m_max.add( maxPath );

#if defined(HK_DEBUG)
        hkVector4 diffMin; diffMin.setSub( aabbIn.m_min, aabbOut.m_min );
        hkVector4 diffMax; diffMax.setSub( aabbOut.m_max, aabbIn.m_max );
        HK_ASSERT( 0xaf63e413, diffMin.greaterEqualZero().allAreSet<hkVector4ComparisonMask::MASK_XYZ>() && diffMax.greaterEqualZero().allAreSet<hkVector4ComparisonMask::MASK_XYZ>(), "Expanded AABB is smaller than the unexpanded AABB." );
#endif
    }

    HK_EXPORT_PHYSICS_2012 void HK_CALL sweepOffsetAabb( const OffsetAabbInput& input, const hkAabb& aabbIn, hkAabb* HK_RESTRICT aabbOut )
    {
        hkSimdReal half = hkSimdReal_Half;
        hkVector4 aabbHalfSize; aabbHalfSize.setSub( aabbIn.m_max, aabbIn.m_min );  aabbHalfSize.mul( half );
        hkVector4 aabbCenter; aabbCenter.setInterpolate( aabbIn.m_max, aabbIn.m_min, half );
        hkVector4 arm; arm._setTransformedPos( input.m_endTransformInv, aabbCenter );


        hkVector4 min = aabbCenter;
        hkVector4 max = aabbCenter;
        {
            hkVector4 p; p._setTransformedPos( input.m_startTransform, arm );
            min.setMin( min, p );
            max.setMax( max, p );
        }

        // extended arm for the in-between transforms (cos(22.5deg)
        const hkVector4 centerOfMassLocal = input.m_motionState->getSweptTransform().m_centerOfMassLocal;

        for( int i = 0; i < input.m_numTransforms; i++ )
        {
            hkVector4 extendedArm;
            extendedArm.setSub( arm, centerOfMassLocal );
            extendedArm.mul( input.m_transforms[i].getTranslation().getComponent<3>() );
            extendedArm.add( centerOfMassLocal );

            hkVector4 p; p._setTransformedPos( input.m_transforms[i], extendedArm );
            min.setMin( min, p );
            max.setMax( max, p );
        }

        //
        // Expand the AABB due to angular rotation of the shape around the aabbIn's center
        //
        {
            hkSimdReal r = aabbHalfSize.length<3>();
            hkVector4 radius; radius.setAll( r );

            // Limit increase to largest expansion that is actually possible: (sqrt(3)-1) * r
            hkSimdReal limit; limit.setFromFloat( 0.732050808f );

            r.setMin( input.m_motionState->m_deltaAngle.getComponent<3>(), limit * r );

            aabbHalfSize.addMul( r, radius );
            aabbHalfSize.setMin( radius, aabbHalfSize );
        }

        min.setSub( min, aabbHalfSize );
        max.setAdd( max, aabbHalfSize );

        // We need to ensure that in order for our hkAabbUint32 compression to work
        aabbOut->m_min.setMin( aabbIn.m_min, min );
        aabbOut->m_max.setMax( aabbIn.m_max, max );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
