// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/Geometry/LocalFrame/hkLocalFrame.h>
#include <Physics2012/Dynamics/World/hkpWorldObject.h>
#include <Physics2012/Dynamics/Common/hkpMaterial.h>
#include <Physics2012/Dynamics/Common/SmallArray/hkSmallArray.h>
#include <Physics2012/Dynamics/Constraint/hkpConstraintInstance.h>
#include <Physics2012/Dynamics/Motion/Rigid/hkpKeyframedRigidMotion.h>

class hkpEntityListener;
class hkpEntityActivationListener;
class hkpContactListener;
class hkpMotion;
class hkpSimulationIsland;
class hkpWorld;
class hkpConstraintInstance;
class hkpAction;
class hkpDynamicsContactMgr;
class hkSpuCollisionCallbackUtil;
class hkpBreakableBody;
class hkdBreakableBody;



/// This class represents the core "physical object" elements in the dynamics system, such
/// as rigid bodies.
class HK_EXPORT_PHYSICS_2012 hkpEntity : public hkpWorldObject
{
    public:
        // +version(5)

        HK_DECLARE_CLASS(hkpEntity, New, Reflect);

        enum SpuCollisionCallbackEventFilter
        {
            SPU_SEND_NONE                   = 0x00,
            SPU_SEND_CONTACT_POINT_ADDED    = 0x01,
            SPU_SEND_CONTACT_POINT_PROCESS  = 0x02,
            SPU_SEND_CONTACT_POINT_REMOVED  = 0x04,
            SPU_SEND_CONTACT_POINT_ADDED_OR_PROCESS = SPU_SEND_CONTACT_POINT_ADDED|SPU_SEND_CONTACT_POINT_PROCESS
        };

    public:

            // Destructor.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        virtual ~hkpEntity();

            //
            // Event Handling
            //

            /// Adds an entity listener to the entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void addEntityListener( hkpEntityListener* el);

            /// Removes an entity listener from the entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void removeEntityListener( hkpEntityListener* el);

            /// Adds an entity activation listener to the entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void addEntityActivationListener( hkpEntityActivationListener* el);

            /// Removes an entity activation listener from the entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void removeEntityActivationListener( hkpEntityActivationListener* el);

            /// Adds a contact listener to the entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void addContactListener( hkpContactListener* cl );

            /// Removes a contact listener from the entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void removeContactListener( hkpContactListener* cl);

            /// Get const access to the array of entity listeners.
        inline const hkSmallArray<hkpEntityListener*>& getEntityListeners() const;

            /// Get const access to the array of entity activation listeners.
        inline const hkSmallArray<hkpEntityActivationListener*>& getEntityActivationListeners() const;

            /// Get const access to the array of contact listeners.
        inline const hkSmallArray<hkpContactListener*>& getContactListeners() const;

            /// Simple thread safe check free function to see if any contact listener is attached
        inline bool areContactListenersAdded() const;

            /// Gets the contact point callback delay.
        inline hkUint16 getContactPointCallbackDelay() const;

            /// Delays the firing of contactPointCallbacks for all the contact points in the contact
            /// manifold (contactPointCallbacks are always fired for new contact points).
            /// A value of 0 means the callback is called every collision step, whereas a value of 4 means
            /// that a callback is raised every 5th collision step. (When entities are involved in
            /// a collision during continuous physics, there may be more than one collision step
            /// per frame.)
        inline void setContactPointCallbackDelay( hkUint16 delay );


            //
            // Utility functions
            //

            /// Gets the material used by this entity.
            /// If the entity has no collision detection representation,
            /// the material is not used.
        inline hkpMaterial& getMaterial();

            /// Gets the material used by this entity.
            /// If the entity has no collision detection representation,
            /// the material is not used.
        inline const hkpMaterial& getMaterial() const;

            /// A utility function to determine if the entity is fixed.
        inline hkBool isFixed() const;

            /// Checks whether the body's velocity cannot be influenced by physics directly.
            /// Uses a cached variable to avoid referencing hkpMotion object.
        inline hkBool isFixedOrKeyframed() const;

            /// Gets the entity's unique id. The UID is assigned in the entity's constructor and
            /// is also updated when your deserialize objects.
        inline hkUint32 getUid() const;

            /// Find the contact manager between 'this' and the supplied entity.
            ///
            /// Returns HK_NULL if no contact manager exists between 'this' and the supplied entity.
        hkpDynamicsContactMgr* findContactMgrTo(const hkpEntity* entity) const;

            /// Returns a pointer to the attached hkdBreakableBody. Returns HK_NULL if none exists.
        hkdBreakableBody* getBreakableBody() const;


            //
            // Deactivation
            //

            /// Activates the entity and its island.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void activate();

            /// Attempt to deactivate the entity as quickly as possible, rather than waiting for the engine
            /// to consider this entity as inactive some time later. Call this as an optimization, only if
            /// you are sure that the entity will not move during the next few steps.
            /// Deactivation is not guaranteed, as it depends on "connected" entities being inactive too.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RW] [this,HK_ACCESS_RW] );
        void requestDeactivation();

            /// DEPRECATED. Forcibly deactivates the specified entity and its island.
            /// NOTE: This will deactivate connected entities too, which may not be the intent.
            /// Please use requestDeactivation() instead.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RW] [this,HK_ACCESS_RW] );
        void deactivate();


            /// Activates the entity and its island. Uses postponed operations queue if the world is locked for critical operations.
        void activateAsCriticalOperation();

            /// Attempt to deactivate the entity as quickly as possible. Uses postponed operations queue if the world is locked for critical operations.
        void requestDeactivationAsCriticalOperation();

            /// DEPRECATED. Forcibly deactivate the entity and its island. Uses postponed operations queue if the world is locked for critical operations.
            /// Please use requestDeactivationAsCriticalOperation() instead.
        void deactivateAsCriticalOperation();


            /// Returns whether the entity is active. This method returns false if the entity
            /// has not yet been added to a hkpWorld object.
        hkBool isActive() const;


            //
            // Attached action and constraint accessors
            //

            /// Get the number of actions added to the world which reference this entity
        inline int getNumActions() const;

            /// Get the i'th action added to the world which references this entity
        inline hkpAction* getAction(int i);

            /// Returns the number of constraints attached to this entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [m_simulationIsland,HK_ACCESS_RO] );
        int getNumConstraints() const;

            /// Returns the i'th constraint attached to this entity.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [m_simulationIsland,HK_ACCESS_RW] );
        hkpConstraintInstance* getConstraint( int i );

            /// Returns all constraints of the body in a single array. Call sortConstraintsSlavesDeterministically() before this call to ensure a deterministic order of the result.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [m_simulationIsland,HK_ACCESS_RO] );
        void getAllConstraints(hkArray<hkpConstraintInstance*>& constraints);

            /// Returns the i'th constraint attached to this entity (const version). Call sortConstraintsSlavesDeterministically() before this call to ensure a deterministic order of the result.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [m_simulationIsland,HK_ACCESS_RO] );
        const hkpConstraintInstance* getConstraint( int i ) const;

        /// Returns read only access to the internal constraint master list
        inline const hkSmallArray<struct hkConstraintInternal>&  getConstraintMasters() const;

        /// Returns read write access to the internal constraint master list
        inline hkSmallArray<struct hkConstraintInternal>&  getConstraintMastersRw();

        /// Returns read only access to the internal constraint master list. Call sortConstraintsSlavesDeterministically() before this call to ensure a deterministic order of the result.
        inline const hkArray< hkViewPtr<class hkpConstraintInstance> >&  getConstraintSlaves() const;

            /// Constraints for fixed objects might not be deterministically ordered. Call this function to bring the constraints into a deterministic order.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_RW] [m_simulationIsland,HK_ACCESS_RW] );
        void sortConstraintsSlavesDeterministically();

            /// Initialize cached AABB memory and SPU data (if available).
        void setCachedShapeData(const hkpWorld* world, const hkpShape* shape);

            /// Recalculate the cached AABB.
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        void updateCachedAabb();

#if defined (HK_PLATFORM_HAS_SPU)
            /// If you want to receive SPU collision callback events on the PPU, you can use this utility to forward them from SPU to PPU.
            ///
            /// You need to set this utility in at least one of the two colliding
            /// entities. Each utility will receive each event only once, i.e., if two colliding entities share the same utility,
            /// you will only get the event once, whereas two different utilities will both receive this event individually.
            /// Use 'eventFilter' to filter the events sent from SPU to PPU.
            /// Events will only be sent for entities whose 'userFilter' share at least one matching bit.
        void setSpuCollisionCallbackUtil(hkSpuCollisionCallbackUtil* util, SpuCollisionCallbackEventFilter eventFilter = SPU_SEND_CONTACT_POINT_ADDED_OR_PROCESS, hkUint8 userFilter = 0x01);

#endif

    protected:

            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [m_simulationIsland,HK_ACCESS_RO] );
        const   hkSmallArray<struct hkConstraintInternal>&  getConstraintMastersImpl() const;
                    /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [m_simulationIsland,HK_ACCESS_RW] );
                hkSmallArray<struct hkConstraintInternal>&  getConstraintMastersRwImpl();

            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [m_simulationIsland,HK_ACCESS_RO] );
        const   hkArray< hkViewPtr<class hkpConstraintInstance> >&  getConstraintSlavesImpl() const;

        hkpEntity( const hkpShape* shape );

    public:

            //
            // INTERNAL FUNCTIONS
            //

        hkpEntity(hkReflect::BypassCtorFlag f);

        void afterReflectNew();

            // Simulation units use this interface.
        inline hkpMotion* getMotion();
        inline const hkpMotion* getMotion() const;

            // Get the simulation island, is HK_NULL for entities not in simulation.
        inline hkpSimulationIsland* getSimulationIsland() const;

            // Deallocates internal arrays if size 0.
            // Called internal by hkpWorld::removeEntity. Over
            /// ###ACCESS_CHECKS###( [m_world,HK_ACCESS_IGNORE] [this,HK_ACCESS_RW] );
        virtual void deallocateInternalArrays();

        virtual hkMotionState* getMotionState(){ return HK_NULL; }



        //
        // MEMBERS
        //

    protected:

            // The entity's material, only used if the collision detection is enabled.
        class hkpMaterial m_material;

    protected:

        friend class hkpWorldConstraintUtil;

    public:
        // this is just a quick workaround helper class for serialization

    public:
        class SmallArraySerializeOverrideType
        {
            public:
                //+version(1)
                HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkpEntity::SmallArraySerializeOverrideType );
                HK_DECLARE_REFLECTION();

                void* m_data; //+serialized(false)
                hkUint16 m_size;
                hkUint16 m_capacityAndFlags;
        };

    public:
                    /// Use hkpBreakOffPartsUtil::getLimitContactImpulseUtilPtr to access this
                    /// value. If the least significant bit is set, then the default
                    /// implementation will be used when a contact point is created on SPU.
                void* m_limitContactImpulseUtilAndFlag; //+serialized(false)

                    /// A property used by Havok Destruction indicating how much damage an object will cause on another object
                hkReal m_damageMultiplier; //+default(1)

                hkpBreakableBody* m_breakableBody; //+nosave

        // the next three elements store constraint information (note: they are owned by the simulation island
        // offset into the accumulators
    public:     hkUint32                                    m_solverData;                   //+serialized(false)
    public:     hkObjectIndex                               m_storageIndex;                 //+overridetype(hkUint16)
    protected:  hkUint16                                    m_contactPointCallbackDelay;
    protected:  hkSmallArray<struct hkConstraintInternal>   m_constraintsMaster;            //+overridetype(class hkpEntity::SmallArraySerializeOverrideType) +serialized(false)
    protected:  hkArray< hkViewPtr<hkpConstraintInstance> > m_constraintsSlave;             //+serialized(false)

            // ------------------ 2nd CacheLine128 (rarely accessed data ) -------------------------

    protected:

        hkArray<hkUint8> m_constraintRuntime;   //+serialized(false)

        // The entity's simulation island.
        hkpSimulationIsland* m_simulationIsland; //+nosave

    public:

            /// See: hkpRigidBodyCinfo::m_autoRemoveLevel
        hkInt8  m_autoRemoveLevel;

            /// See: hkpRigidBodyCinfo::m_numShapeKeysInContactPointProperties
        hkUint8 m_numShapeKeysInContactPointProperties;

            /// See: hkpRigidBodyCinfo::m_reponseModifierFlags
        hkUint8 m_responseModifierFlags;

            // hkpWorld-unique Id
        hkUint32 m_uid; //+default(0xffffffff)

    public:

        // Deprecated.
        struct SpuCollisionCallback
        {
        public:
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkpEntity::SpuCollisionCallback );
            HK_DECLARE_REFLECTION();
            SpuCollisionCallback():
                m_util(HK_NULL),
                m_capacity(0),
                m_eventFilter(SPU_SEND_CONTACT_POINT_ADDED_OR_PROCESS),
                m_userFilter(0x01)
            {
            }

                // Deprecated.
                // only entities with a callback util will send events from spu to ppu; each event will only be fired once for each util
            hkSpuCollisionCallbackUtil* m_util; // +nosave

                // the maximum buffer size (counted in 16byte blocks) for events to be sent from spu to ppu; this value is set by setSpuCollisionCallbackUtil()
            hkUint16 m_capacity; //+serialized(false)

                // used to filter what events to send from spu to ppu
            hkUint8 m_eventFilter;

                // free to be set by the user; note that events will only be sent from spu to ppu for entities whose userFilter both have at least one matching bit set
            hkUint8 m_userFilter;
        };

            // this class was created to keep the entity size <= 512
        struct ExtendedListeners
        {
            HK_DECLARE_CLASS(ExtendedListeners, New, Reflect);

            hkSmallArray<hkpEntityActivationListener*> m_activationListeners; //+overridetype(class hkpEntity::SmallArraySerializeOverrideType) +serialized(false)
            hkSmallArray<hkpEntityListener*> m_entityListeners; //+overridetype(class hkpEntity::SmallArraySerializeOverrideType) +serialized(false)
        };

    public:

        struct SpuCollisionCallback m_spuCollisionCallback;

            // The motion of the object. May really be any class derived from hkpMotion
        HK_ALIGN_REAL(char m_motion[sizeof(hkpMaxSizeMotion)]) HK_ATTR(hk::Type(hkpMaxSizeMotion));

    protected:

            //
            //  Rarely used members
            //
        friend class hkpEntityCallbackUtil;
        friend class hkpWorldCallbackUtil;
        friend class hkpWorld;
        friend class hkpSimulationIsland;
        friend class hkpWorldOperationUtil;

        hkSmallArray<hkpContactListener*> m_contactListeners; //+overridetype(class hkpEntity::SmallArraySerializeOverrideType) +serialized(false)

        hkSmallArray<hkpAction*> m_actions; //+overridetype(class hkpEntity::SmallArraySerializeOverrideType) +serialized(false)

    public:

            /// A hierarchy of local frames attached to the entity.
        hkRefPtr<hkLocalFrame> m_localFrame;

    protected:

        mutable ExtendedListeners* m_extendedListeners; //+serialized(false)
};

#include <Physics2012/Dynamics/Entity/hkpEntity.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
