// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM       : ALL
// PRODUCT        : PHYSICS_2012
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics2012/Dynamics/Constraint/Chain/hkpConstraintChainInstance.h>
#include <Physics2012/Dynamics/Constraint/Chain/BallSocket/hkpBallSocketChainData.h>
#include <Physics2012/Dynamics/Entity/hkpRigidBodyCinfo.h>

class hkpConstraintChainInstance;
class hkpBallSocketChainData;


class HK_EXPORT_PHYSICS_2012 hkpConstraintChainLengthUtil
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkpConstraintChainLengthUtil);

        struct RopeInfo
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkpConstraintChainLengthUtil::RopeInfo);

            hkVector4 m_settings; // cableCrossSection, materialDensity, segmentLength, materialStretchingConstant
            hkSimdReal m_inertiaMultiplier;

            RopeInfo()
            {
                m_settings.set( 28.0f * 1e-4f,  // [m2]
                                6.7f * 1e3f,    // [kg/m3]
                                0.30f,          // [m]
                                280.0f * 1e6f); // [N]
                m_inertiaMultiplier = hkSimdReal_1;
            }

            HK_INLINE const hkSimdReal getCableCrossSection() const  { return m_settings.getComponent<0>(); }
            HK_INLINE const hkSimdReal getMaterialDensity() const    { return m_settings.getComponent<1>(); }
            HK_INLINE const hkSimdReal getSegmentLength() const      { return m_settings.getComponent<2>(); }
            HK_INLINE const hkSimdReal getMaterialStretching() const { return m_settings.getComponent<3>(); }
        };


        RopeInfo m_ropeInfo;
        hkVector4 m_pivotOnCrane;
        hkpRigidBodyCinfo m_segmentCinfo;
        hkpConstraintChainInstance* m_instance;

        hkSimdReal m_currentUnstretchedLength;

        int m_numLoadBodies;

    private:

        hkpConstraintChainLengthUtil(); // only allow creation via create functions.

    public:

        // Creates rope from hook, to crane pivot. Sets the unstretched length of the rope to the exact distance from load to crane pivot.
        // A length rope of at least two segmentLengths is needed, otherwise it should auto add extra segments.
        hkpConstraintChainLengthUtil(const RopeInfo& info, hkpRigidBody* craneBody, hkVector4Parameter pivotOnCrane, hkVector4Parameter ropeEndInWorld);

        ~hkpConstraintChainLengthUtil();

        void attachHook(hkpRigidBody* hookBody, hkVector4Parameter pivotOnHook);

        void attachLoad(hkpRigidBody* loadBody, hkVector4Parameter pivotOnLoad, hkVector4Parameter pivotOnHook);

        void detachHookOrLoad();

        HK_INLINE const hkSimdReal getUnstretchedLength() { return m_currentUnstretchedLength; }

        void setUnstretchedLength(hkSimdRealParameter length);

        void updatePivotOnStretchedRope(); // just use the stretch above. simple. // consider also multiplying the inertia of the topmost element to compensate for the longer arm.

        void updateChainProperties(hkSimdRealParameter solverStepDeltaTime, int numSolverMicroSteps, hkSimdRealParameter solverTau);

        void getRopeBodies(hkArray<hkpRigidBody*>& ropeBodies);

        hkpBallSocketChainData* getChainData() { return (hkpBallSocketChainData*)m_instance->getData(); }
        const hkpBallSocketChainData* getChainData() const { return (const hkpBallSocketChainData*)m_instance->getData(); }

        const hkVector4 getSegmentAPivot() const { hkVector4 p; p.setMul(hkVector4::getConstant<HK_QUADREAL_0010>(), hkSimdReal_Inv2 * m_ropeInfo.getSegmentLength()); return p; }
        const hkVector4 getSegmentBPivot() const { hkVector4 p; p.setMul(hkVector4::getConstant<HK_QUADREAL_0010>(),-hkSimdReal_Inv2 * m_ropeInfo.getSegmentLength()); return p; }

        int getNumSegments(hkSimdRealParameter length) const;

        const hkVector4 getLastSegmentAPivot() const;

        const hkSimdReal getStretchAtCraneLess1() const;

        const hkSimdReal getStretchAtCrane() const;

    private:

        void updateConstraintInstanceBaseEntities(hkpConstraintChainInstance* chain);
};

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
