// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM       : ALL
// PRODUCT        : PHYSICS_2012
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics2012/Dynamics/Constraint/Chain/hkpConstraintChainInstance.h>
#include <Physics2012/Dynamics/Constraint/Chain/BallSocket/hkpBallSocketChainData.h>
#include <Physics2012/Dynamics/Entity/hkpRigidBodyCinfo.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>

class hkpConstraintChainInstance;
class hkpBallSocketChainData;
class hkpCapsuleShape;
class hkpVelocityConstraintMotor;
class hkpPairCollisionFilter;

/// Simple rope class which has an improved way to interact with user defined cylinders.
/// (Collisions with other objects will be soft and jittery).
class HK_EXPORT_PHYSICS_2012 hkpSegmentedRope: public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO);

        /// Rope info, define the rope global parameters.
        struct HK_EXPORT_PHYSICS_2012 RopeInfo
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkpSegmentedRope::RopeInfo);
            RopeInfo();

            // -------------------------------------------------------------------------------------------------------------
            // Basic parameters
            // -------------------------------------------------------------------------------------------------------------

            /// The total length of the rope.
            /// If you don't set it, it will be automatically calculated from the attachment points.
            /// Note if you have set it you can also set Attachment::m_ropePosition
            hkReal m_ropeLength;

            /// You cannot make the rope shorter than this. This helps to ensure at least a few segments are created
            hkReal m_minRopeLength;

            /// The length of one segment
            hkReal m_segmentLength;

            /// The filter info used for the rope.
            hkUint32 m_filterInfo;

            /// The number of recursive subdivisions a segment can have. Set this to 3 and you get a maximum of 2^3 * m_numSegments
            int m_maxSubdivisions;

            /// optional filter
            hkpPairCollisionFilter* m_pairFilter;

            // -------------------------------------------------------------------------------------------------------------
            // Mass
            // -------------------------------------------------------------------------------------------------------------

            /// The radius
            hkReal m_ropeRadius;

            /// The density of the rope
            hkReal m_materialDensity;

            /// If != 0.0f, this overrides the mass calculated from density and cross section
            hkReal m_massPerMeterOverride;

            // -------------------------------------------------------------------------------------------------------------
            // Inertia
            // -------------------------------------------------------------------------------------------------------------

            /// Just a multiplyer to the inertia for each segment
            hkReal m_segmentInertiaMultiplier;

            /// Should the segment get smaller than this size, the inertia is taken from a segment of that size
            hkReal m_segmentInertiaMinSegmentSize;

            /// parameter to control the eigen frequency of the rope, this is kind of non physical, but without it we
            /// cannot get a stable rope.
            hkReal m_inertiaPerMeter;

            /// The angular damping of each segment multiplied by the segment length.
            /// Which means the longer the segment, the less damping we apply
            hkReal m_angularDampingTimesLength;

            /// linear damping, needed to approximate under water behavior.
            hkReal m_linearDamping;
        };


        /// Internal helper struct to define a cylindrical object,
        struct HK_EXPORT_PHYSICS_2012 Cylinder: public hkReferencedObject
        {
            HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO);
            Cylinder(){ m_body = 0; m_forceCylinderCenter = false; m_cylinderRadius = 1.0f; m_perpendicular.setZero(); m_lockPivots = false; m_checkDetach = false; }

            /// Internal helper class to store information when a rope touches a cylinder
            struct Attachment
            {
                HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, Attachment);
                Attachment(){ m_initialized = false; m_touchPointDirection.setZero(); m_winding2=0; }

                hkBool   m_initialized;
                hkInt8   m_winding;             ///< +1 or -1
                hkInt8   m_winding2;
                hkReal   m_cylinderPosition;
                hkVector4 m_touchPointDirection;    ///< touchpoint - cylinder center, needed to avoid the rope flipping around
            };

            hkVector4 m_cylinderPivots[2];      ///< the pivots in local space
            hkReal    m_cylinderRadius;         ///< radius of the cylinder
            hkReal    m_minRopeLengthUsedForProjection; ///<
            hkBool    m_forceCylinderCenter;    ///< if true, the rope will always be on the center of the cylinder
            hkBool    m_lockPivots;             ///< if set, the cylinder will not be reprojected.
            hkBool    m_checkDetach;            ///< If set, the rope can detach from the cylinder
            hkpRigidBody* m_body;               ///< the body for the cylinder
            hkpVelocityConstraintMotor* m_motor;    ///< optional motor needed to control friction

            // runtime information

            ///< To measure the attachment.m_cylinderPosition we need a perpendicular vector to the cylinder direction in world space, this it used for this
            mutable hkpSegmentedRope* m_rope;
            mutable hkVector4  m_perpendicular;
            mutable Attachment m_attachment[2];         ///< the contact to the rope ends
        };


        /// Internal helper struct to define a cylindrical object,
        struct HK_EXPORT_PHYSICS_2012 Attachment
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkpSegmentedRope::Attachment);
            Attachment(){ m_pivot.setZero(); m_body = HK_NULL; m_cylinder = HK_NULL; m_ropePosition = 0.0f; }
            int attachedNumBodies() const { return m_body? 1:0; }   ///< returns the number of attached bodies, only 0 or 1 possible

            hkVector4     m_pivot;              ///< The position of the attachment, only used if body is set or this structure is used in the constructor
            hkpRigidBody* m_body;               ///< The attached body, or 0 if not attached
            hkReal        m_ropePosition;       ///< the position on the rope between 0 and length
            hkRefPtr<Cylinder> m_cylinder;      ///< An optional attachment to a cylinder
        };

        /// A pivot point
        struct HK_EXPORT_PHYSICS_2012 Pivot
        {
            hkVector4 m_position;
            hkReal m_ropePosition;
            int m_partialId;
            int m_segmentId;
        };

        hkpSegmentedRope( hkpWorld* world, const RopeInfo& ropeInfo, const Attachment& start, const Attachment& end, const hkArray<hkVector4>& path );

        struct HK_EXPORT_PHYSICS_2012 Segment
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkpSegmentedRope::Segment);

            hkVector4 getPivot( const hkSimdReal& side ) const { hkVector4 pos; pos.set( m_segmentLength,0,0,0 ); pos.mul(side * hkSimdReal_Inv2); return pos; }
            hkReal          m_segmentLength;
            int             m_lodLevel;                 ///< the lod level, -1 if dynamic
            hkRefPtr<hkpCapsuleShape> m_privateShape;   ///< the shape of the segment if it is a private shape which can be modified
        };


        ~hkpSegmentedRope();


        struct HK_EXPORT_PHYSICS_2012 RopeEndInfo
        {
            hkReal m_newStartPositionLength; ///< the new length

            hkReal m_ropeEndVelocity; ///< if set we assume that the rope is continuing to move with this velocity

            int m_side; // 0 = start  1 = end

            hkBool m_useEndPivot;
            hkVector4 m_endPivot; ///< the pivot at the end of the rope if m_useEndPivot, else this will be calculated from the attached object
        };

        struct HK_EXPORT_PHYSICS_2012 ProjectOut : public RopeEndInfo
        {
            bool m_isValid;
            hkReal m_angle;
            hkVector4 m_ropeEndProjectedOntoCylinder;
        };

        /// call this before making changes to a running rope.
        void lock();


        /// Shorten or lengthen the rope at the end.
        /// Important note: only for partialId =0 and side = 0 the solver understands the concept of shortening the rope
        /// as a result the behavior will be much better.
        /// So if you build a crane, always change the length of the rope at the very start.
        /// If the rope is attached to a winch cylinder this restriction does not apply.
        void setEndPosition( int partialId, int side, hkReal newStartPosition );

        /// Shorten or lengthen the rope at the end.
        /// Like  setEndPosition(), just more parameters
        void setEndPosition2( int partialId, const RopeEndInfo& startInfo );


        /// Split the rope at a point as close as possible to ropePosition and attach the open ends to the body at the pivot in
        /// local space of that body
        /// Returns the partialId which is split into partialId and partialId+1
        int splitRope( hkReal ropePosition, hkpRigidBody* body, hkVector4Parameter pivotLocal, int partialId = -1 );

        /// Split the rope at a point as close as possible to the cylinder and store a reference to this cylinder.
        /// Returns the partialId which is split into partialId and partialId+1, returns -1 if distance to cylinder is bigger than tolerance
        int splitRope( Cylinder* cylinder, hkReal maxDistanceTolerance = 1000.0f, int partialId = -1 );

        /// merge partial rope partialId and partialId+1
        void combineRope( int partialId );

        /// Get the force applied on both ends of the cylinder
        void getForce( const Cylinder* cylinder, hkVector4& forceAtRopeEndOut, hkVector4& forceAtRopeStartOut ) const;

        /// Iterate over all cylinders and do a correct rope projection
        void doCylinderProjections();

        /// call this after making changes to a running rope.
        void unlock();

        void getPivots( hkArray<Pivot>& pivotsOut );

    public:
        /// internal public functions
        void projectOnCylinder( int partialId, int side, const Cylinder& info, int attachmentIndex, ProjectOut& out );


        struct HK_EXPORT_PHYSICS_2012 PartialRope
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkpSegmentedRope::PartialRope);
            hkVector4 getPivot( hkpSegmentedRope* rope, int segmentIndex, hkSimdRealParameter pivotSide ) const;

            /// this refers to first entity in this rope in the chain constraint.
            /// Note: this includes the attached body.
            int m_chainFirstEntityIndex;

            Attachment m_ends[2];
            hkArray<Segment> m_segments;
        };

    protected:
        void fixupJointDataInChainConstraint( int partialIndex, int segmentIndex );
        void rebuildChainFirstEntityIndex();
        hkpCapsuleShape* createCapsuleShape( hkReal segmentLength );

        void initializeSegmentCinfo( hkReal segmentLength, hkpRigidBodyCinfo* HK_RESTRICT infoOut );
        hkpBallSocketChainData* getChainData() { return (hkpBallSocketChainData*)m_instance->getData(); }
        const hkpBallSocketChainData* getChainData() const { return (const hkpBallSocketChainData*)m_instance->getData(); }
        void updateConstraintInstanceBaseEntities(hkpConstraintChainInstance* chain);
        hkpRigidBody* removeBodyFromChainInstance( int instanceIndexToRemove );
        void addBodyToChainInstance( hkpRigidBody* body, int instanceIndexToAddEntity );

    public:
        struct HK_EXPORT_PHYSICS_2012 SegmentBlueprint
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, hkpSegmentedRope::SegmentBlueprint);

            hkReal m_segmentLength;
            hkRefPtr<const hkpCapsuleShape> m_shape;
            hkpRigidBodyCinfo m_segmentCinfo;
        };

        hkpWorld* m_world;


        RopeInfo m_ropeInfo;                ///< parameters for the rope

        hkArray<SegmentBlueprint> m_segmentBlueprints;  ///< 0 is the largest size

        hkArray<PartialRope*> m_partial;    ///< a part of the rope connecting cylinders or external bodies.

        hkpConstraintChainInstance* m_instance; ///< constraint instance

        hkpBallSocketChainData* m_chainData;    ///< constraint data
};

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
