// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM       : ALL
// PRODUCT        : PHYSICS_2012
// VISIBILITY     : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/Dynamics/hkpDynamics.h>

#include <Physics2012/Dynamics/Constraint/Chain/Rope/hkpSegmentedRope.h>

#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Geometry/Collide/Algorithms/Distance/hkcdDistancePointLine.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Geometry/Collide/Algorithms/ClosestPoint/hkcdClosestPointSegmentSegment.h>

#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics2012/Dynamics/Constraint/Chain/hkpConstraintChainInstance.h>
#include <Physics2012/Dynamics/Constraint/Chain/hkpConstraintChainInstanceAction.h>

#include <Physics2012/Dynamics/Constraint/Chain/BallSocket/hkpBallSocketChainData.h>
#include <Physics2012/Utilities/Constraint/Chain/hkpConstraintChainUtil.h>
#include <Physics2012/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h>
#include <Physics2012/Collide/Filter/Group/hkpGroupFilter.h>
#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics/ConstraintSolver/Jacobian/hkpJacobianSchema.h>
#include <Physics2012/Collide/Agent/hkpProcessCollisionInput.h>
#include <Physics2012/Dynamics/Collide/Filter/Pair/hkpPairCollisionFilter.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>

#include <Geometry/Collide/Algorithms/ClosestPoint/hkcdClosestPointLineLine.h>
#ifdef HK_PLATFORM_RVL
#include <Common/Base/Fwd/hkcstring.h>
#endif


void hkpSegmentedRope_setOrientation( const hkVector4& dir, hkQuaternion* HK_RESTRICT orientation )
{
    hkRotation r;
    r.setColumn<0>(dir);
    hkVector4Util::calculatePerpendicularNormalizedVectors<false>( dir, r.getColumn(1), r.getColumn(2));
    orientation->set(r);
}

hkpSegmentedRope::RopeInfo::RopeInfo()
{
    m_ropeRadius     =  0.05f;
    m_materialDensity =    6.7f * 1e3f;    // [kg/m3]
    //m_materialStretchingConstant = 280.0f * 1e6f; // [N]
    m_inertiaPerMeter = 0.01f;
    m_massPerMeterOverride = 0.0f;
    m_segmentInertiaMinSegmentSize = 0.0f;
    m_ropeLength = 0.0f;
    m_segmentInertiaMultiplier = 1.5f;
    m_angularDampingTimesLength = 10.0f;
    m_linearDamping = 0.0f;
    m_segmentLength = 0.5f;
    m_maxSubdivisions = 1;
    m_minRopeLength = 1.0f;
    m_pairFilter = 0;
    m_filterInfo = 0;
}

hkpCapsuleShape* hkpSegmentedRope::createCapsuleShape( hkReal segmentLength )
{
    hkReal ropeRadius = m_ropeInfo.m_ropeRadius;
    hkVector4 pivotB; pivotB.set( segmentLength * 0.5f,0,0);
    hkVector4 pivotA; pivotA.setNeg<4>( pivotB );
    hkpCapsuleShape* capsule = new hkpCapsuleShape(pivotA, pivotB, ropeRadius );
    return capsule;
}

void hkpSegmentedRope::initializeSegmentCinfo( hkReal segmentLength, hkpRigidBodyCinfo* HK_RESTRICT infoOut )
{
    //hkReal ropeRadius = m_ropeInfo.m_ropeRadius;

    // -------------------------------------------------------------------------------------------------------------
    // Create shape
    // -------------------------------------------------------------------------------------------------------------
    infoOut->m_shape = createCapsuleShape(segmentLength);

    // -------------------------------------------------------------------------------------------------------------
    // Inertia
    // -------------------------------------------------------------------------------------------------------------
    hkReal ropeCrossSection = 3.0f / 4.0f * HK_REAL_PI * m_ropeInfo.m_ropeRadius;
    hkReal segmentMass      = segmentLength * ropeCrossSection * m_ropeInfo.m_materialDensity;
    if ( m_ropeInfo.m_massPerMeterOverride )
    {
        segmentMass = segmentLength * m_ropeInfo.m_massPerMeterOverride;
    }
    hkReal segmentLengthForInertia = hkMath::max2( m_ropeInfo.m_segmentInertiaMinSegmentSize, segmentLength );

    hkMassProperties mp;
    hkInertiaTensorComputer::computeSphereSurfaceMassProperties(segmentLengthForInertia * 0.5f, m_ropeInfo.m_segmentInertiaMultiplier * segmentMass, mp );

    hkReal I = mp.m_inertiaTensor(0,0);
    hkReal minInertia = m_ropeInfo.m_inertiaPerMeter * segmentLength;
    I = hkMath::max2( minInertia, I );

    infoOut->m_inertiaTensor.setMul( hkSimdReal::fromFloat(I), hkMatrix3::getIdentity() );
    infoOut->m_mass = segmentMass;

    //          // Greatly multiply inertia around rope axis to prevent rotation
    //          infoOut->m_inertiaTensor(2,2) *= 1000.0f;

    // -------------------------------------------------------------------------------------------------------------
    // Other, e.g. Dampen angular velocity to eliminate rotation around rope axis.
    // -------------------------------------------------------------------------------------------------------------
    infoOut->m_angularDamping = m_ropeInfo.m_angularDampingTimesLength / segmentLength;
    infoOut->m_linearDamping = m_ropeInfo.m_linearDamping;
    infoOut->m_maxAngularVelocity = 100.0f;
    infoOut->m_maxLinearVelocity  = 100.0f;

    // Disable solver deactivation, which may freeze slow moving bodies that are not in prefect balance.
    infoOut->m_solverDeactivation = hkpRigidBodyCinfo::SOLVER_DEACTIVATION_OFF;

    // Disable collisions with itself
    infoOut->m_collisionFilterInfo = m_ropeInfo.m_filterInfo;
}


hkpSegmentedRope::hkpSegmentedRope( hkpWorld* world, const RopeInfo& info, const Attachment& start, const Attachment& end, const hkArray<hkVector4>& path )
{
    m_ropeInfo = info;
    m_world = world;

    hkVector4 startPos = start.m_pivot;
    if ( start.m_body )
    {
        startPos.setTransformedPos( start.m_body->getTransform(), startPos );
    }

    hkVector4 endPos = end.m_pivot;
    if ( end.m_body )
    {
        endPos.setTransformedPos( end.m_body->getTransform(), endPos );
    }

    hkVector4 dir;
    dir.setSub( endPos, startPos );
    hkReal distanceBetweenPivots = dir.normalizeWithLength<3>().getReal();
    if ( path.getSize() )
    {
        distanceBetweenPivots = startPos.distanceTo( path[0] ).getReal();
        for (int i=0; i < path.getSize()-1; i++)
        {
            distanceBetweenPivots += path[i].distanceTo( path[i+1] ).getReal();
        }
        distanceBetweenPivots += path.back().distanceTo( endPos ).getReal();
        dir.setSub( path[0], startPos );
        dir.setNormalizedEnsureUnitLength<3>(dir);
    }


    // -------------------------------------------------------------------------------------------------------------
    // Create segment blueprints.
    // -------------------------------------------------------------------------------------------------------------
    {
        hkReal segmentLength = m_ropeInfo.m_segmentLength;
        m_segmentBlueprints.setSize(info.m_maxSubdivisions);
        for (int i =0; i < info.m_maxSubdivisions; i++)
        {
            SegmentBlueprint* HK_RESTRICT sp = &m_segmentBlueprints[i];

            initializeSegmentCinfo(segmentLength, &sp->m_segmentCinfo);
            sp->m_segmentLength = segmentLength;
            sp->m_shape.setAndDontIncrementRefCount( (const hkpCapsuleShape*) sp->m_segmentCinfo.m_shape );

            m_ropeInfo.m_segmentInertiaMinSegmentSize = segmentLength;  // use the finest detail LOD to define min segment size
            segmentLength *= 0.5f;
        }
    }

    // -------------------------------------------------------------------------------------------------------------
    // Create rope bodies and the chain constraint.
    // -------------------------------------------------------------------------------------------------------------
    hkpBallSocketChainData* chainData = new hkpBallSocketChainData();
    m_instance = new hkpConstraintChainInstance( chainData );
    m_chainData = chainData;
    chainData->removeReference();
    chainData->useStabilizedCode(true);

    PartialRope* rope = new PartialRope;
    m_partial.pushBack(rope);
    (Attachment&)(rope->m_ends[0]) = start;
    (Attachment&)(rope->m_ends[1])   = end;

    if ( m_ropeInfo.m_ropeLength == 0)
    {
        m_ropeInfo.m_ropeLength = distanceBetweenPivots + start.m_ropePosition;
    }
    HK_ASSERT( 0xf054fe56, distanceBetweenPivots <= m_ropeInfo.m_ropeLength * ( 1+HK_REAL_EPSILON*2 ), "Your specified rope length is too short" );

    rope->m_ends[1].m_ropePosition = distanceBetweenPivots + start.m_ropePosition;

    HK_ASSERT( 0xf054fe56, rope->m_ends[1].m_ropePosition <= m_ropeInfo.m_ropeLength, "Your specified end position is longer than the rope length" );


    int lodLevel = 0;
    const SegmentBlueprint& blueprint = m_segmentBlueprints[lodLevel];

    hkpRigidBodyCinfo cinfo = blueprint.m_segmentCinfo;
    hkpSegmentedRope_setOrientation(dir, &cinfo.m_rotation);
    cinfo.m_position = startPos;

    // -------------------------------------------------------------------------------------------------------------
    // Add start of rope
    // -------------------------------------------------------------------------------------------------------------
    if ( start.m_body )
    {
        m_instance->addEntity(start.m_body);
    }


    // -------------------------------------------------------------------------------------------------------------
    // Add bodies and create segments
    // -------------------------------------------------------------------------------------------------------------

    hkReal ropePosition = rope->m_ends[0].m_ropePosition;
    int numSegments = HK_INT32_MAX; // just used as a safety net, the real abort criteria is in the loop
    int currentPathIndex = 0;
    for (int si = 0; si <= numSegments; si++)
    {
        hkReal segmentLength = blueprint.m_segmentLength;

        // first segment
        hkpCapsuleShape* privateShape = HK_NULL;
        if ( si == 0 )
        {
            privateShape = createCapsuleShape(segmentLength);
            cinfo.m_shape = privateShape;
        }
        // last segment
        else if ( ropePosition > rope->m_ends[1].m_ropePosition - blueprint.m_segmentLength*1.1f )
        {
            segmentLength =  rope->m_ends[1].m_ropePosition - ropePosition;
            privateShape = createCapsuleShape(segmentLength);
            numSegments = 0;    // end while loop
            cinfo.m_shape = privateShape;
        }
        else
        {
            cinfo.m_shape = blueprint.m_shape;  // reset private shape
        }

        Segment& s = rope->m_segments.expandOne();
        s.m_lodLevel = lodLevel;
        s.m_privateShape.setAndDontIncrementRefCount(privateShape);
        s.m_segmentLength = segmentLength;

        cinfo.m_position.addMul( dir, hkSimdReal::fromFloat(.5f * segmentLength) );
        hkpRigidBody* body = new hkpRigidBody(cinfo);
        world->addEntity(body);
        m_instance->addEntity(body);
        HK_SET_OBJECT_COLOR( hkUint64(body->getCollidable()), 0xff339933 ^ (si << 15));

        body->removeReference();

        cinfo.m_position.addMul( dir, hkSimdReal::fromFloat(.5f * segmentLength) );

        if ( currentPathIndex < path.getSize() )
        {
            // check if we walked passed the next control point
            hkVector4 diff; diff.setSub( path[currentPathIndex], cinfo.m_position );
            hkSimdReal dot = diff.dot<3>(dir);
            if( dot.isLessEqualZero() )
            {
                currentPathIndex++;
                hkVector4 endPoint = ( currentPathIndex < path.getSize()) ? path[currentPathIndex] : endPos;
                dir.setSub( endPoint, cinfo.m_position );
                dir.setNormalizedEnsureUnitLength<3>(dir);
                hkpSegmentedRope_setOrientation(dir, &cinfo.m_rotation);
            }
        }
        ropePosition += segmentLength;
    }
    rope->m_chainFirstEntityIndex = 0;
    if ( end.m_body )
    {
        m_instance->addEntity(end.m_body);
    }

    // -------------------------------------------------------------------------------------------------------------
    // Add segments
    // -------------------------------------------------------------------------------------------------------------
    if ( start.m_body )
    {
        chainData->addConstraintInfoInBodySpace( start.m_pivot, rope->m_segments[0].getPivot( hkSimdReal_Minus1 ) );
    }

    for (int si = 0; si < rope->m_segments.getSize()-1; si++)
    {
        const Segment& a = rope->m_segments[si];
        const Segment& b = rope->m_segments[si+1];
        chainData->addConstraintInfoInBodySpace( a.getPivot( hkSimdReal_1 ), b.getPivot( hkSimdReal_Minus1 ) );
    }

    if ( end.m_body )
    {
        chainData->addConstraintInfoInBodySpace(rope->m_segments.back().getPivot( hkSimdReal_1 ), end.m_pivot );
    }

    if (m_ropeInfo.m_pairFilter )
    {
        const hkArray<hkpEntity*>& bodies = m_instance->m_chainedEntities;
        for (int i =0; i < bodies.getSize()-1; i++ )
        {
            m_ropeInfo.m_pairFilter->disableCollisionsBetween( bodies[i], bodies[i+1] );
        }
    }
}

hkpSegmentedRope::~hkpSegmentedRope()
{
    for(int i = 0; i < m_partial.getSize(); i++)
    {
        delete m_partial[i];
    }

    m_instance->removeReference();
}


hkpRigidBody* hkpSegmentedRope::removeBodyFromChainInstance( int instanceIndexToRemove )
{
    if (m_ropeInfo.m_pairFilter )
    {
        const hkArray<hkpEntity*>& bodies = m_instance->m_chainedEntities;
        if ( instanceIndexToRemove>0 )
        {
            m_ropeInfo.m_pairFilter->enableCollisionsBetween( bodies[instanceIndexToRemove-1], bodies[instanceIndexToRemove] );
            if ( instanceIndexToRemove<bodies.getSize()-1 )
            {
                m_ropeInfo.m_pairFilter->disableCollisionsBetween( bodies[instanceIndexToRemove-1], bodies[instanceIndexToRemove+1] );
            }
        }
        if ( instanceIndexToRemove<bodies.getSize()-1 )
        {
            m_ropeInfo.m_pairFilter->enableCollisionsBetween( bodies[instanceIndexToRemove+1], bodies[instanceIndexToRemove] );
        }
    }
    hkpRigidBody* segmentBodyToBeRemoved = (hkpRigidBody*)m_instance->removeEntity( instanceIndexToRemove );
    return segmentBodyToBeRemoved;
}


void hkpSegmentedRope::addBodyToChainInstance( hkpRigidBody* body, int instanceIndexToAddEntity )
{
    m_instance->insertEntityAtFront(body,instanceIndexToAddEntity);
    if (m_ropeInfo.m_pairFilter )
    {
        const hkArray<hkpEntity*>& bodies = m_instance->m_chainedEntities;
        if ( instanceIndexToAddEntity>0 )
        {
            m_ropeInfo.m_pairFilter->disableCollisionsBetween( bodies[instanceIndexToAddEntity-1], bodies[instanceIndexToAddEntity] );
            if ( instanceIndexToAddEntity<bodies.getSize()-1 )
            {
                m_ropeInfo.m_pairFilter->enableCollisionsBetween( bodies[instanceIndexToAddEntity-1], bodies[instanceIndexToAddEntity+1] );
            }
        }
        if ( instanceIndexToAddEntity<bodies.getSize()-1 )
        {
            m_ropeInfo.m_pairFilter->disableCollisionsBetween( bodies[instanceIndexToAddEntity+1], bodies[instanceIndexToAddEntity] );
        }
    }
}



void hkpSegmentedRope::fixupJointDataInChainConstraint( int partialIndex, int segmentIndex )
{
    PartialRope* rope = m_partial[partialIndex];
    Segment& segment = rope->m_segments[segmentIndex];
    int segmentInChainIndex = rope->m_chainFirstEntityIndex + segmentIndex + rope->m_ends[0].attachedNumBodies();
    if ( segmentInChainIndex > 0 )
    {
        hkpBallSocketChainData::ConstraintInfo& info = m_chainData->m_infos[segmentInChainIndex-1];
        info.m_pivotInB = segment.getPivot(hkSimdReal_Minus1);
    }
    if ( segmentInChainIndex < m_chainData->m_infos.getSize() )
    {
        hkpBallSocketChainData::ConstraintInfo& info = m_chainData->m_infos[segmentInChainIndex];
        info.m_pivotInA   = segment.getPivot(hkSimdReal_1);
    }
}

void hkpSegmentedRope::setEndPosition( int partialId, int side, hkReal newStartPosition )
{
    RopeEndInfo si;
    si.m_newStartPositionLength = newStartPosition;

    PartialRope* rope = m_partial[partialId];
    hkReal lengthChange = rope->m_ends[side].m_ropePosition - newStartPosition;
    hkReal invDt = m_world->m_dynamicsStepInfo.m_stepInfo.m_invDeltaTime;

    si.m_ropeEndVelocity = lengthChange * invDt;
    si.m_useEndPivot = false;
    si.m_side = side;
    setEndPosition2( partialId, si );
}

hkVector4 hkpSegmentedRope::PartialRope::getPivot( hkpSegmentedRope* rope, int segmentIndex, hkSimdRealParameter pivotSide ) const
{
    const Segment& segment = m_segments[segmentIndex];
    hkVector4 pivot = segment.getPivot( pivotSide );
    int bodyIndex = m_chainFirstEntityIndex + m_ends[0].attachedNumBodies() + segmentIndex;
    const hkpRigidBody* body = (const hkpRigidBody*)rope->m_instance->m_chainedEntities[bodyIndex];
    hkVector4 pivotWs; pivotWs.setTransformedPos( body->getTransform(), pivot );
    return pivotWs;
}

void hkpSegmentedRope::projectOnCylinder( int partialId, int side, const Cylinder& info, int attachmentIndex, ProjectOut& out )
{

    // -------------------------------------------------------------------------------------------------------------
    // Calculate basic cylinder properties
    // -------------------------------------------------------------------------------------------------------------
    hkVector4 cylA = info.m_cylinderPivots[0];
    hkVector4 cylB = info.m_cylinderPivots[1];

    if ( info.m_body )
    {
        cylA.setTransformedPos( info.m_body->getTransform(), cylA );
        cylB.setTransformedPos( info.m_body->getTransform(), cylB );
    }
    hkVector4 cylinderDiff; cylinderDiff.setSub( cylB, cylA );
    hkVector4 cylinderDir = cylinderDiff; cylinderDir.normalize<3>();

    // -------------------------------------------------------------------------------------------------------------
    // Basic rope parameters
    // -------------------------------------------------------------------------------------------------------------
    int forward = 1-side*2; // to move forward on the rope
    hkSimdReal forwardSr = hkSimdReal::fromInt32(forward);

    // take start segment and project onto cylinder
    PartialRope* rope    = m_partial[partialId];

    int firstSegmentIdx = side * (rope->m_segments.getSize()-1);
    int indexOfFirstSegmentBody = rope->m_chainFirstEntityIndex + rope->m_ends[0].attachedNumBodies() + firstSegmentIdx;

    hkpBallSocketChainData::ConstraintInfo& ci = m_chainData->m_infos[indexOfFirstSegmentBody-side];
    ci.m_flags |= (side) ? hkpPoweredBallSocketChainFlags::EXTRAPOLATE_ANGULAR_VELOCITY_B : hkpPoweredBallSocketChainFlags::EXTRAPOLATE_ANGULAR_VELOCITY_A;

    // -------------------------------------------------------------------------------------------------------------
    // Project rope onto the cylinder using an iterative algorithm
    // -------------------------------------------------------------------------------------------------------------
    Cylinder::Attachment& ca = info.m_attachment[attachmentIndex];

    // first find a point on the rope which is a good observer

    hkReal ropeDistanceRopeEndToObserver;
    bool observerValid = false;
    hkVector4 observer;
    hkVector4 toPoint;
    {
        ropeDistanceRopeEndToObserver = rope->m_segments[ firstSegmentIdx ].m_segmentLength;
        toPoint  = rope->getPivot( this, firstSegmentIdx, -forwardSr  );
        observer = rope->getPivot( this, firstSegmentIdx,  forwardSr  );

        // we check if we are using the next segment
        for ( int segmentOffst = 0; true;  )
        {
            int numSeg = rope->m_segments.getSize();
            hkVector4 diffTo; diffTo.setSub( toPoint, cylA );
            hkVector4 det;    det.setCross( diffTo, cylinderDir );
            hkSimdReal toPointDistSqrd = det.lengthSquared<3>();

            diffTo.setSub( observer, cylA );
            det.setCross( diffTo, cylinderDir );
            hkSimdReal observerDistSqrd = det.lengthSquared<3>();
            hkReal cylRadius2 = info.m_cylinderRadius * info.m_cylinderRadius;



            // check for valid observer
            if ( observerDistSqrd.getReal() > cylRadius2  * 1.0001f )
            {
                if ( observerDistSqrd < toPointDistSqrd  )
                {
                    toPoint = observer;
                    observerValid = false;
                }
                else
                {
                    observerValid = true;
                    hkReal distanceObserver2ToPointSqrd = observer.distanceToSquared( toPoint ).getReal();
                    if ( distanceObserver2ToPointSqrd > info.m_minRopeLengthUsedForProjection * info.m_minRopeLengthUsedForProjection )
                    {
                        break;
                    }
                }
            }

            if ( segmentOffst == (numSeg-1) || -segmentOffst == (numSeg-1))
            {
                break;  // can't walk any more
            }
            // lets walk to the next segment and give it a try
            segmentOffst += forward;

            Segment& segment = rope->m_segments[ firstSegmentIdx + segmentOffst ];
            observer = rope->getPivot( this, firstSegmentIdx + segmentOffst,  forwardSr  );
            ropeDistanceRopeEndToObserver += segment.m_segmentLength;
        }
        {
            hkVector4 a = observer; a(1) += 0.1f;
            hkVector4 b = toPoint; b(1) += 0.1f;
            b.sub(a);       HK_DISPLAY_ARROW( a, b, hkColor::ORANGE );
        }
    }

    if ( !observerValid )
    {
        out.m_isValid = false;
        return;
    }

    hkVector4 ropeDirection;
    hkVector4 ropeEndProjectedOntoCylinder = toPoint;
    {
        hkVector4 cylPos;
        for (int i=0; i < 3; i++)   // small iterative algorithm to get onto the cylinder
        {
            ropeDirection.setSub( ropeEndProjectedOntoCylinder, observer );
            ropeDirection.normalize<3>();

            hkSimdReal u, v;
            hkcdClosestPointLineLine( observer, ropeDirection, cylA, cylinderDiff, u, v );
            u.setMax( u, hkSimdReal_0 ); // enforce positive

            ropeEndProjectedOntoCylinder.setAddMul( observer, ropeDirection, u );   // note that u should be +

            hkSimdReal newv = ( info.m_forceCylinderCenter ) ? hkSimdReal_Inv2: v;
            newv.setClampedZeroOne( newv );

            ropeEndProjectedOntoCylinder.addMul( cylinderDiff, newv - v);   // shift point to be on the cylinder

            // project onto the cylinder surface by ke
            cylPos.setAddMul( cylA, cylinderDiff, newv );

            hkSimdReal dist = ropeEndProjectedOntoCylinder.distanceTo<HK_ACC_MID, HK_SQRT_IGNORE>( cylPos );
            hkSimdReal radius = hkSimdReal::fromFloat( info.m_cylinderRadius );
            ropeEndProjectedOntoCylinder.setInterpolate( cylPos, ropeEndProjectedOntoCylinder, radius/dist );

            // check side
            {
                hkVector4 newTouchPointDirection; newTouchPointDirection.setSub( ropeEndProjectedOntoCylinder, cylPos );
                hkVector4 cross; cross.setCross(newTouchPointDirection, cylinderDiff );
                hkVector4 newDir; newDir.setSub( ropeEndProjectedOntoCylinder, observer );
                hkSimdReal dot = cross.dot<3>( newDir );
                if ( !ca.m_initialized )
                {
                    ca.m_winding2 = dot.isGreaterEqualZero() ? 1 : -1;
                }
                else
                {
                    if ( dot.getReal() * ca.m_winding2 < 0.0f )
                    {
                        ropeEndProjectedOntoCylinder.subMul( newTouchPointDirection, hkSimdReal_2 );
                    }
                }
            }

            hkVector4 a = observer; a(1) += 0.1f;
            hkVector4 b = ropeEndProjectedOntoCylinder; b(1) += 0.1f;
            b.sub(a);       HK_DISPLAY_ARROW( a, b, hkColor::MAGENTA );
        }

        {
            hkVector4 newTouchPointDirection; newTouchPointDirection.setSub( ropeEndProjectedOntoCylinder, cylPos );
            hkSimdReal interpolation = ( !ca.m_initialized ) ? hkSimdReal_1 : hkSimdReal_Inv_15;
            ca.m_touchPointDirection.setInterpolate( ca.m_touchPointDirection, newTouchPointDirection, interpolation );

            HK_DISPLAY_ARROW( cylPos, ca.m_touchPointDirection, hkColor::YELLOW );
            HK_DISPLAY_STAR( ropeEndProjectedOntoCylinder, 0.2f, hkColor::RED );

            //hkReal dist = newTouchPointDirection.distanceTo( ca.m_touchPointDirection ).getReal();
            //newTouchPointDirection = ca.m_touchPointDirection;
            //ropeEndProjectedOntoCylinder.setAdd( cylPos, newTouchPointDirection );
        }
    }


    // work out the length change of the rope by calculating the angle in local space
    hkReal lengthChange;
    {
        hkVector4 cylinderDirCylSpace;
        cylinderDirCylSpace.setSub( info.m_cylinderPivots[1], info.m_cylinderPivots[0] );
        cylinderDirCylSpace.normalize<3,HK_ACC_FULL, HK_SQRT_IGNORE>();

        hkVector4 perpendicularB;
        if ( info.m_perpendicular.allExactlyEqualZero<4>() )
        {
            hkVector4Util::calculatePerpendicularVector( cylinderDirCylSpace, info.m_perpendicular );
        }
        perpendicularB.setCross( cylinderDirCylSpace, info.m_perpendicular );
        hkVector4 endPosInCylSpace; endPosInCylSpace._setTransformedInversePos( info.m_body->getTransform(), ropeEndProjectedOntoCylinder );

        hkVector4 endPosMinusCylinderCenter; endPosMinusCylinderCenter.setSub( endPosInCylSpace, info.m_cylinderPivots[0] );
        hkSimdReal u = endPosMinusCylinderCenter.dot<3>( info.m_perpendicular );
        hkSimdReal v = endPosMinusCylinderCenter.dot<3>( perpendicularB );
        hkReal angle = hkMath::atan2( u.getReal(), v.getReal() );
        out.m_angle = angle;
        hkReal pos = angle * info.m_cylinderRadius;
        if ( ca.m_initialized )
        {
            pos *= ca.m_winding;
            lengthChange = pos - ca.m_cylinderPosition;
            hkReal halfCircum = info.m_cylinderRadius * HK_REAL_PI;
            if ( lengthChange > halfCircum  ){  lengthChange -= 2.0f*halfCircum;        }
            if ( lengthChange < -halfCircum ){  lengthChange += 2.0f*halfCircum;        }
        }
        else
        {
            hkReal distanceTouchPointToObserver = observer.distanceTo( ropeEndProjectedOntoCylinder ).getReal();
            lengthChange = distanceTouchPointToObserver - ropeDistanceRopeEndToObserver;
            ca.m_initialized = true;
            hkVector4 ropeDirectionCylSpace; ropeDirectionCylSpace.setRotatedInverseDir( info.m_body->getRotation(), ropeDirection );
            hkSimdReal u2 = ropeDirectionCylSpace.dot<3>( info.m_perpendicular );
            hkSimdReal v2 = ropeDirectionCylSpace.dot<3>( perpendicularB );
            hkReal angle2 = hkMath::atan2( u2.getReal(), v2.getReal() );
            while  ( angle2 > angle+HK_REAL_PI ) { angle2 -= 2.0f*HK_REAL_PI; }
            while  ( angle2 < angle-HK_REAL_PI ) { angle2 += 2.0f*HK_REAL_PI; }
            ca.m_winding = ( angle2 > angle) ? 1 : -1;
            pos *= ca.m_winding;
        }
        ca.m_cylinderPosition = pos;
        //Log_Info( "Length change {}", lengthChange );
    }


    hkReal newStartPosition = rope->m_ends[side].m_ropePosition - forward * lengthChange;

//  if ( 0 && checkForRunningOutOfRope )
//  {
//      if ( newStartPosition <= 0.0f )
//      {
//          if ( rope->m_ends[side].m_ropePosition == .0f )
//          {
//              return; // nothing to do, out of rope
//          }
//          newStartPosition = 0.0f;
//      }
//
//      if ( newStartPosition > m_ropeInfo.m_ropeLength )
//      {
//          if ( rope->m_ends[side].m_ropePosition == m_ropeInfo.m_ropeLength )
//          {
//              return; // nothing to do, out of rope
//          }
//          newStartPosition = m_ropeInfo.m_ropeLength;
//      }
//  }


    // fix pivot on cylinder
    {
        hkVector4 cylPosLocal; cylPosLocal.setTransformedInversePos( rope->m_ends[side].m_body->getTransform(), ropeEndProjectedOntoCylinder );
        if ( side == 0)
        {
            hkpBallSocketChainData::ConstraintInfo& cinfo = m_chainData->m_infos[indexOfFirstSegmentBody-1];
            cinfo.m_pivotInA = cylPosLocal;
        }
        else
        {
            hkpBallSocketChainData::ConstraintInfo& cinfo = m_chainData->m_infos[indexOfFirstSegmentBody];
            cinfo.m_pivotInB = cylPosLocal;
        }
    }

    out.m_isValid = true;
    out.m_ropeEndProjectedOntoCylinder = ropeEndProjectedOntoCylinder;

    RopeEndInfo& si = out;
    si.m_newStartPositionLength = newStartPosition;

    si.m_ropeEndVelocity = 0.0f;
    si.m_side = side;

    si.m_useEndPivot = true;
    si.m_endPivot = ropeEndProjectedOntoCylinder;
}

void hkpSegmentedRope::lock()
{
    m_world->markForWrite();
    m_world->removeConstraint( m_instance );
}

void hkpSegmentedRope::unlock()
{
    m_world->addConstraint( m_instance );
    m_world->unmarkForWrite();
}

void hkpSegmentedRope::setEndPosition2( int partialId, const RopeEndInfo& startInfo )
{
    int side = startInfo.m_side;
    int forward = 1-side*2; // to move forward on the rope
    hkSimdReal forwardSr = hkSimdReal::fromInt32(forward);

    // check if we can simply just update the shape of the body
    PartialRope* rope = m_partial[partialId];
    Attachment& ropeEnd = rope->m_ends[side];
    const SegmentBlueprint& sb = m_segmentBlueprints[0];

    hkReal newStartPosition = startInfo.m_newStartPositionLength;
    if ( side == 0)
    {
        newStartPosition = hkMath::min2( newStartPosition, rope->m_ends[1].m_ropePosition - m_ropeInfo.m_minRopeLength );
        newStartPosition = hkMath::max2( 0.0f, newStartPosition );  // cannot make rope position negative
    }
    else
    {
        newStartPosition = hkMath::max2( newStartPosition, m_ropeInfo.m_minRopeLength - rope->m_ends[0].m_ropePosition );
        newStartPosition = hkMath::max2( 0.0f, newStartPosition );  // cannot make rope position negative
    }

    hkReal lengthChange = forward * (ropeEnd.m_ropePosition - newStartPosition);
    if ( lengthChange == 0.0f )
    {
        m_chainData->m_link0PivotBVelocity.setZero();
        return;
    }
    ropeEnd.m_ropePosition = newStartPosition;

    int firstSegmentIdx = side * (rope->m_segments.getSize()-1);

    hkReal newStartSegmentLength = rope->m_segments[firstSegmentIdx].m_segmentLength + lengthChange;

    int indexOfFirstSegmentBody = rope->m_chainFirstEntityIndex + rope->m_ends[0].attachedNumBodies() + firstSegmentIdx;

    const hkpRigidBody* oldEndBody = (hkpRigidBody*)m_instance->m_chainedEntities[ indexOfFirstSegmentBody ];

    hkVector4 newSecondPivot = rope->m_segments[firstSegmentIdx].getPivot( forwardSr );
    newSecondPivot.setTransformedPos( oldEndBody->getTransform(), newSecondPivot );

    // calculate direction from the attachment point to the newSecondPivot
    hkVector4 dir;  // the direction of the last segment in direction of the rope
    hkVector4 startPos;
    {
        if ( startInfo.m_useEndPivot )
        {
            startPos = startInfo.m_endPivot;
        }
        else if ( ropeEnd.m_body )
        {
            startPos.setTransformedPos( ropeEnd.m_body->getTransform(), ropeEnd.m_pivot );
        }
        else
        {
            startPos  = rope->m_segments[firstSegmentIdx].getPivot( forwardSr * hkSimdReal_Minus1 );
        }
        dir.setSub( newSecondPivot, startPos );
        dir.setNormalizedEnsureUnitLength<3,HK_ACC_MID>( dir );
        dir.mul( forwardSr );
    }

    // remove segments if needed, update dir
    while ( newStartSegmentLength < 0.1f * sb.m_segmentLength )
    {
        if ( rope->m_segments.getSize()>1)
        {
            // kill second segment
            int secondSegmentIdx = firstSegmentIdx + forward;
            int instanceIndexToRemove = indexOfFirstSegmentBody+forward;
            hkpRigidBody* secondSegmentBodyToBeRemoved = removeBodyFromChainInstance(instanceIndexToRemove);

            m_chainData->m_infos.removeAtAndCopy(indexOfFirstSegmentBody-side);

            {
                Segment& segmentToRemove = rope->m_segments[secondSegmentIdx];
                newStartSegmentLength += segmentToRemove.m_segmentLength;

                newSecondPivot = segmentToRemove.getPivot( forwardSr );
                newSecondPivot.setTransformedPos( secondSegmentBodyToBeRemoved->getTransform(), newSecondPivot );

                rope->m_segments.removeAtAndCopy(secondSegmentIdx);
            }

            dir.setSub( newSecondPivot, startPos );
            dir.setNormalizedEnsureUnitLength<3,HK_ACC_MID>( dir );
            dir.mul( forwardSr );

            m_world->removeEntity(secondSegmentBodyToBeRemoved);

            indexOfFirstSegmentBody -= side;
            firstSegmentIdx         -= side;
            secondSegmentIdx        -= side;

            if ( rope->m_segments.getSize()>=2 )    // fixup attached segment
            {
                fixupJointDataInChainConstraint( partialId, secondSegmentIdx ); // this will not fix up the first segment, but this will be done later
            }
        }
        else
        {
            newStartSegmentLength = 0.1f * sb.m_segmentLength;
        }
    }

    // add segments
    while( newStartSegmentLength > 1.1f * sb.m_segmentLength )
    {
        // get the world space position of pivotA segment 1
        int lodLevel = 0;
        hkpRigidBodyCinfo cinfo = sb.m_segmentCinfo;
        hkpSegmentedRope_setOrientation(dir, &cinfo.m_rotation);
        cinfo.m_position.setAddMul( newSecondPivot, dir, forwardSr * hkSimdReal::fromFloat(-0.5f * sb.m_segmentLength));

        int newSegmentIdx = firstSegmentIdx + 1-side;
        Segment& s = *rope->m_segments.expandAt(newSegmentIdx,1);
        s.m_lodLevel = lodLevel;
        s.m_privateShape  = HK_NULL;
        s.m_segmentLength = sb.m_segmentLength;

        if ( rope->m_segments.getSize() == 2 )  // add a private shape
        {
            s.m_privateShape.setAndDontIncrementRefCount(createCapsuleShape(s.m_segmentLength));
            cinfo.m_shape = s.m_privateShape;
        }

        firstSegmentIdx  += side;

        m_chainData->addConstraintInfoInBodySpace( hkVector4::getZero(), hkVector4::getZero(), indexOfFirstSegmentBody );
        fixupJointDataInChainConstraint( partialId, newSegmentIdx ); // this will not fix up the first segment, but this will be done later

        hkpRigidBody* body = new hkpRigidBody(cinfo);
        m_world->addEntity(body);
        addBodyToChainInstance(body, indexOfFirstSegmentBody+1-side);
        indexOfFirstSegmentBody += side;

        body->removeReference();

        newSecondPivot.addMul( dir, hkSimdReal::fromFloat(-sb.m_segmentLength) * forwardSr );
        newStartSegmentLength -= sb.m_segmentLength;
    }

    hkReal firstSegmentOldLength = 0.0f;
    // update last segment by using information from the new ropeStart (oldBody)
    {
        hkpRigidBody* startBody = (hkpRigidBody*)m_instance->m_chainedEntities[ indexOfFirstSegmentBody ];
        // update velocity information
        if ( startBody != oldEndBody )
        {
            hkVector4 linVel = oldEndBody->getLinearVelocity();
            startBody->setLinearVelocity( linVel );
        }
        {
            hkVector4 newPos; newPos.setAddMul( newSecondPivot, dir, forwardSr * hkSimdReal::fromFloat(- 0.5f * newStartSegmentLength ));
            hkQuaternion q; hkpSegmentedRope_setOrientation( dir, &q );
            startBody->setPositionAndRotation(newPos,q);
        }

        // update capsule and pivots
        hkVector4 a; a.set( -0.5f * newStartSegmentLength,0,0);
        hkVector4 b; b.set(  0.5f * newStartSegmentLength,0,0);
        Segment& startSegment = rope->m_segments[firstSegmentIdx];
        firstSegmentOldLength = startSegment.m_segmentLength;
        startSegment.m_privateShape->setVertex<0>( a );
        startSegment.m_privateShape->setVertex<1>( b );

        startBody->setShape( startSegment.m_privateShape ); // just to update the graphics
        startSegment.m_segmentLength = newStartSegmentLength;

        fixupJointDataInChainConstraint( partialId, firstSegmentIdx );
    }
    if (partialId==0 && side == 0)
    {
        // tell the solver that we have a smooth shortening the ropes
        m_chainData->m_link0PivotBVelocity.setMul( dir, hkSimdReal::fromFloat(-startInfo.m_ropeEndVelocity));
    }

    // fixup other segment offsets
    rebuildChainFirstEntityIndex();
}

int hkpSegmentedRope::splitRope( hkReal ropePosition, hkpRigidBody* body, hkVector4Parameter bodyPivot, int partialId )
{
    //HK_ASSERT_NO_MSG( 0xf0dfedfd, m_instance->m_owner == HK_NULL );
    if ( partialId < 0 )
    {
        // search partial
        partialId = 0;
        while ( partialId < m_partial.getSize()-1 && ropePosition > m_partial[partialId]->m_ends[0].m_ropePosition )
        {
            partialId++;
        }
    }
    PartialRope* rope = m_partial[partialId];

    hkReal actualRopePosition = rope->m_ends[0].m_ropePosition;
    int segmentIndex = rope->m_segments.getSize()-1;
    {
        hkReal restLen = ropePosition - rope->m_ends[0].m_ropePosition;
        for (int i =0; i < rope->m_segments.getSize()-1;i++)
        {
            if ( restLen <= HK_REAL_EPSILON * 100.0f )
            {
                segmentIndex = i;
                break;
            }
            hkReal segmentLength = rope->m_segments[i].m_segmentLength;
            actualRopePosition += segmentLength;
            restLen -= segmentLength;
        }
    }

    // clip the ends
    int numTotalSegments = rope->m_segments.getSize();
    segmentIndex = hkMath::max2( segmentIndex, 1 );
    segmentIndex = hkMath::min2( segmentIndex, numTotalSegments-1 );
    hkReal minRopePos = rope->m_ends[0].m_ropePosition + rope->m_segments[0].m_segmentLength;
    hkReal maxRopePos = rope->m_ends[1].m_ropePosition - rope->m_segments[numTotalSegments-1].m_segmentLength;
    actualRopePosition = hkMath::clamp( actualRopePosition, minRopePos, maxRopePos );


    PartialRope* rope2 = new PartialRope;
    // create rope
    {
        m_partial.insertAt(partialId+1, rope2);
        rope2->m_ends[1] = rope->m_ends[1];

        rope2->m_ends[0].m_body = body;
        rope2->m_ends[0].m_ropePosition = actualRopePosition;
        rope2->m_ends[0].m_pivot = bodyPivot;

        rope ->m_ends[1].m_body = body;
        rope ->m_ends[1].m_ropePosition = actualRopePosition;
        rope ->m_ends[1].m_pivot = bodyPivot;

    }

    // split segments ( no need to create new ones, just create private shapes)
    {
        rope2->m_segments.append( rope->m_segments.begin() + segmentIndex, numTotalSegments-segmentIndex );
        rope->m_segments.setSize( segmentIndex );
        rebuildChainFirstEntityIndex();
    }

    // inject the body
    // update the chain constraints
    int indexOfNewBody = rope->m_chainFirstEntityIndex + rope->m_ends[0].attachedNumBodies() + segmentIndex;
    m_chainData->addConstraintInfoInBodySpace( hkVector4::getZero(), hkVector4::getZero(), indexOfNewBody );
    m_chainData->m_infos[indexOfNewBody-1].m_pivotInB = bodyPivot;
    m_chainData->m_infos[indexOfNewBody  ].m_pivotInA = bodyPivot;

    fixupJointDataInChainConstraint( partialId, segmentIndex-1 );
    fixupJointDataInChainConstraint( partialId+1, 0 );

    addBodyToChainInstance( body, indexOfNewBody );

    // create private shapes at the ends
    {
        {
            Segment& ls = rope2->m_segments[0];
            ls.m_privateShape.setAndDontIncrementRefCount(createCapsuleShape(ls.m_segmentLength));
            m_instance->m_chainedEntities[indexOfNewBody-1]->setShape(ls.m_privateShape);
        }
        {
            Segment& ls = rope->m_segments.back();
            ls.m_privateShape.setAndDontIncrementRefCount(createCapsuleShape(ls.m_segmentLength));
            m_instance->m_chainedEntities[indexOfNewBody+1]->setShape(ls.m_privateShape);
        }
    }

    // set the lengths precisely.
    setEndPosition( partialId,   1, ropePosition );
    setEndPosition( partialId+1, 0, ropePosition );

    return partialId;
}

void hkpSegmentedRope::combineRope( int partialId )
{
    PartialRope* rope = m_partial[partialId];
    PartialRope* rope2 = m_partial[partialId+1];

    // take the rope position from the longer rope, this is typically more stable
    if ( rope->m_segments.getSize() > rope2->m_segments.getSize() )
    {
        setEndPosition( partialId+1, 0, rope->m_ends[1].m_ropePosition );
    }
    else
    {
        setEndPosition( partialId, 1, rope2->m_ends[0].m_ropePosition );
    }
    m_partial.removeAtAndCopy(partialId+1);

    int constraintChainMiddleBodyIndex = rope2->m_chainFirstEntityIndex;
    removeBodyFromChainInstance( constraintChainMiddleBodyIndex );
    m_chainData->m_infos.removeAtAndCopy( constraintChainMiddleBodyIndex );

    int segmentIndex = rope->m_segments.getSize();
    rope->m_segments.append( rope2->m_segments.begin(), rope2->m_segments.getSize() );
    rope->m_ends[1] = rope2->m_ends[1];

    rope2->m_segments.setSize(0);

    fixupJointDataInChainConstraint( partialId, segmentIndex-1 );
    fixupJointDataInChainConstraint( partialId, segmentIndex );

    delete rope2;
}

/// Split the rope at a point as close as possible to the cylinder and store a reference to this cylinder.
/// returns the partial id if success, < 0 on failure
int hkpSegmentedRope::splitRope( Cylinder* cylinder, hkReal maxDistanceTolerance, int partialId )
{
    hkLocalArray<Pivot> pivots(200);
    getPivots(pivots);

    hkVector4 cylA; cylA.setTransformedPos(cylinder->m_body->getTransform(), cylinder->m_cylinderPivots[0] );
    hkVector4 cylB; cylB.setTransformedPos(cylinder->m_body->getTransform(), cylinder->m_cylinderPivots[1] );

    hkVector4 cylinderCenter; cylinderCenter.setInterpolate( cylA, cylB, hkSimdReal_Inv2 );

    int closestPivot = -1;
    hkVector4 projectedPoint;
    {
        // find closest segment
        hkSimdReal tolerance2 = hkSimdReal::fromFloat( maxDistanceTolerance + cylinder->m_cylinderRadius );
        tolerance2 = tolerance2 * tolerance2;
        for (int i =0; i < pivots.getSize()-1; i++ )
        {
            hkVector4 pos = pivots[i].m_position;
            hkVector4 pos2 = pivots[i+1].m_position;

            hkVector4 diff; diff.setSub( pos2, pos );
            hkVector4 cylDiff; cylDiff.setSub( cylB, cylA );

            hkSimdReal t,u; hkcdClosestPointSegmentSegment(cylA,  cylDiff, pos, diff, t,u);
            hkVector4 A; A.setAddMul ( cylA, cylDiff, t );
            hkVector4 B; B.setAddMul ( pos, diff, u );
            //HK_DISPLAY_LINE( A, B, hkColor::YELLOW );

            hkSimdReal d2 = A.distanceToSquared(B);
            if ( d2 < tolerance2 )
            {
                tolerance2 = d2;
                closestPivot = i;
                projectedPoint = B;
            }
        }
    }
    if ( /*maxDistanceTolerance < 100.0f ||*/ closestPivot < 0 )
    {
        return -1;
    }
    hkVector4 projectedPointLs; projectedPointLs.setTransformedInversePos( cylinder->m_body->getTransform(), projectedPoint );

    const Pivot& p = pivots[closestPivot];
    hkReal partialSegmentLength = p.m_position.distanceTo( projectedPoint ).getReal();
    hkReal ropePosition = p.m_ropePosition + partialSegmentLength;
    splitRope( ropePosition, cylinder->m_body, projectedPointLs, p.m_partialId );

    m_partial[p.m_partialId]->m_ends[1].m_cylinder   = cylinder;
    m_partial[p.m_partialId+1]->m_ends[0].m_cylinder = cylinder;
    cylinder->m_rope = this;
    return p.m_partialId;
}

void hkpSegmentedRope::getForce( const Cylinder* cylinder, hkVector4& forceAtRopeEndOut, hkVector4& forceAtRopeStartOut ) const
{
    forceAtRopeStartOut.setZero();
    forceAtRopeEndOut.setZero();
    const hkpBallSocketChainData::Runtime* runtime = hkpBallSocketChainData::getRuntime(m_instance->getRuntime());

    for (int i =0; i < m_partial.getSize(); i++ )
    {
        const PartialRope* rope = m_partial[i];
        if ( rope->m_ends[0].m_cylinder == cylinder )
        {
            int si = rope->m_chainFirstEntityIndex;
            const hkpSolverResults* sr = runtime->getSolverResults() + si*3;
            hkReal x = sr[0].m_impulseApplied;
            hkReal y = sr[1].m_impulseApplied;
            hkReal z = sr[2].m_impulseApplied;
            forceAtRopeStartOut.set(x,y,z);
        }

        if ( rope->m_ends[1].m_cylinder == cylinder )
        {
            int si = rope->m_chainFirstEntityIndex + rope->m_segments.getSize()-1 + rope->m_ends[0].attachedNumBodies();
            const hkpSolverResults* sr = runtime->getSolverResults() + si*3;
            hkReal x = sr[0].m_impulseApplied;
            hkReal y = sr[1].m_impulseApplied;
            hkReal z = sr[2].m_impulseApplied;
            forceAtRopeEndOut.set(x,y,z);
        }
    }
    hkSimdReal dt = hkSimdReal::fromFloat(m_world->getCollisionInput()->m_stepInfo.m_invDeltaTime);
    forceAtRopeStartOut.mul( dt );
    forceAtRopeEndOut.mul( dt );
}

void hkpSegmentedRope::doCylinderProjections()
{
    // check for cylinder pairs
    for (int p=0; p < m_partial.getSize()-1; p++ )
    {
        PartialRope* rope0 = m_partial[p];
        PartialRope* rope1 = m_partial[p+1];
        Cylinder* cyl  = rope0->m_ends[1].m_cylinder;
        if ( cyl && !cyl->m_lockPivots)
        {
            //PartialRope* rope1 = m_partial[p+1];
            ProjectOut result;  projectOnCylinder( p,   1, *cyl, 0, result );
            ProjectOut result2; projectOnCylinder( p+1, 0, *cyl, 1, result2 );


            // work out the error
            hkReal p0 = rope0->m_ends[1].m_ropePosition;
            hkReal p1 = rope1->m_ends[0].m_ropePosition;
            if( result.m_isValid && result2.m_isValid )
            {
                hkReal angleDiff = result2.m_angle - result.m_angle;
                angleDiff *= cyl->m_attachment[0].m_winding;
                if ( angleDiff < -HK_REAL_PI*.5f  ) angleDiff += HK_REAL_PI*2.0f;
                if ( angleDiff > HK_REAL_PI*1.5f )  angleDiff -= HK_REAL_PI*2.0f;
                hkReal d = angleDiff * cyl->m_cylinderRadius;
                //Log_Info( "d={}   diff={}", d, (p1-p0)-d );

                hkReal fixup = (p1-p0)-d;
                hkReal erp = fixup * 0.01f;
                result.m_newStartPositionLength  += erp;
                result2.m_newStartPositionLength -= erp;
            }

            if ( cyl->m_checkDetach && p0 > p1 )
            {
                combineRope( p );
                p--;
                continue;
            }

            if( result.m_isValid )
            {
                setEndPosition2( p, result );
            }
            if( result2.m_isValid )
            {
                setEndPosition2( p+1, result2 );
            }
        }
    }

    // check ends
    {
        hkpSegmentedRope::Cylinder* cyl = m_partial[0]->m_ends[0].m_cylinder;
        if ( cyl && !cyl->m_lockPivots )
        {
            int p = 0;
            HK_ASSERT_NO_MSG( 0xf0456567, cyl->m_body == m_partial[p]->m_ends[0].m_body );
            ProjectOut result;  projectOnCylinder( p, 0, *cyl, 0, result );
            if ( result.m_isValid )
            {
                setEndPosition2( p, result );
            }
        }
    }

    {
        int e = m_partial.getSize()-1;
        hkpSegmentedRope::Cylinder* cyl = m_partial[e]->m_ends[1].m_cylinder;
        if ( cyl && !cyl->m_lockPivots )
        {
            HK_ASSERT_NO_MSG( 0xf0456567, cyl->m_body == m_partial[e]->m_ends[1].m_body );
            ProjectOut result;  projectOnCylinder( e, 1, *cyl, 1, result );
            if ( result.m_isValid )
            {
                setEndPosition2( e, result );
            }
        }
    }
}

void hkpSegmentedRope::rebuildChainFirstEntityIndex()
{
    int chainFirstEntityIndex = 0;
    for (int i = 0; i< m_partial.getSize(); i++ )
    {
        PartialRope* rope = m_partial[i];
        rope->m_chainFirstEntityIndex = chainFirstEntityIndex;
        chainFirstEntityIndex += rope->m_segments.getSize() + rope->m_ends[1].attachedNumBodies();
    }
}

void hkpSegmentedRope::getPivots( hkArray<Pivot>& pivotsOut )
{
    for (int p = 0; p< m_partial.getSize(); p++ )
    {
        PartialRope* rope = m_partial[p];
        int bi = rope->m_chainFirstEntityIndex + rope->m_ends[0].attachedNumBodies();
        hkReal ropePosition = rope->m_ends[0].m_ropePosition;
        for (int s = 0; s < rope->m_segments.getSize(); s++ )
        {
            Segment& segment = rope->m_segments[s];
            hkpRigidBody* body = (hkpRigidBody*)m_instance->m_chainedEntities[ bi + s ];
            Pivot& pivot = pivotsOut.expandOne();
            pivot.m_position.setTransformedPos( body->getTransform(), segment.getPivot(hkSimdReal_Minus1 ));
            pivot.m_partialId = p;
            pivot.m_segmentId = s;
            pivot.m_ropePosition = ropePosition;

            ropePosition += segment.m_segmentLength;
        }
    }

    {
        PartialRope* rope = m_partial.back();
        hkpRigidBody* body = (hkpRigidBody*)m_instance->m_chainedEntities[ rope->m_chainFirstEntityIndex + rope->m_ends[0].attachedNumBodies() + rope->m_segments.getSize()-1 ];
        Pivot& p = pivotsOut.expandOne();
        p.m_position.setTransformedPos( body->getTransform(), rope->m_segments.back().getPivot(hkSimdReal_1 ));
        p.m_partialId = m_partial.getSize()-1;
        p.m_segmentId = rope->m_segments.getSize()-1;
        hkReal ropePosition = rope->m_ends[1].m_ropePosition;
        p.m_ropePosition = ropePosition;
    }
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
