// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/Dynamics/hkpDynamics.h>
#include <Physics2012/Dynamics/Constraint/Breakable/hkpBreakableConstraintData.h>
#include <Physics2012/Dynamics/Constraint/Breakable/hkpBreakableListener.h>
#include <Physics2012/Dynamics/Constraint/Util/hkpConstraintCallbackUtil.h>
#include <Physics/ConstraintSolver/Constraint/Bilateral/hkp1dBilateralConstraintInfo.h>
#include <Physics2012/Internal/Solver/Atom/hkpBuildJacobianFromAtoms.h>
#include <Physics/ConstraintSolver/Constraint/hkpConstraintQueryIn.h>
#include <Physics/ConstraintSolver/Accumulator/hkpVelocityAccumulator.h>
#include <Physics2012/Dynamics/World/hkpSimulationIsland.h>
#include <Physics2012/Dynamics/World/hkpWorld.h>
#include <Physics2012/Dynamics/World/Util/hkpWorldCallbackUtil.h>
#include <Physics/ConstraintSolver/Constraint/hkpConstraintQueryOut.h>
#include <Physics/Constraint/Data/hkpConstraintDataUtils.h>


hkpBreakableConstraintData::hkpBreakableConstraintData( hkpConstraintData* constraintData )
: hkpWrappedConstraintData(constraintData),
  m_solverResultLimit(10),
  m_removeWhenBroken(false)
{
    m_revertBackVelocityOnBreak = false;

    hkpConstraintData::RuntimeInfo info;
    m_constraintData->getRuntimeInfo( true, info );
    m_childRuntimeSize      = hkUint16(info.m_sizeOfExternalRuntime);
    m_childNumSolverResults = hkUint16(info.m_numSolverResults);

    m_atoms.m_bridgeAtom.init( this );
}

//
//  Post finish serialization constructor

void hkpBreakableConstraintData::afterReflectNew()
{
    m_childRuntimeSize      = 0;
    m_childNumSolverResults = 0;

    if ( m_constraintData )
    {
        hkpConstraintData::RuntimeInfo info;
        m_constraintData->getRuntimeInfo( true, info );
        m_childRuntimeSize      = hkUint16(info.m_sizeOfExternalRuntime);
        m_childNumSolverResults = hkUint16(info.m_numSolverResults);
    }
    m_atoms.m_bridgeAtom.init( this );
}

void hkpBreakableConstraintData::buildJacobianCallback( const hkpConstraintQueryIn &in, const hkpConstraintQueryOut& out )
{
   // Determine if reaction forces from previous solution should cause a break
    hkpSolverResults* results = reinterpret_cast<hkpSolverResults*>( out.m_constraintRuntime );
    int numResults = m_childNumSolverResults;
    HK_ASSERT( 0xf06521fe, out.m_constraintRuntime , "The constraint which is wrapped by the hkpBreakableConstraintData does not support breaking" );

    Runtime* runtime = getRuntime( out.m_constraintRuntime );
    if ( !runtime->m_isBroken )
    {
        hkSimdReal sumReactionForce; sumReactionForce.setZero();
        for(int j = 0; j < numResults; j++)
        {
            hkSimdReal impulse; impulse.setFromFloat( results[j].m_impulseApplied );
            sumReactionForce.addMul(impulse, impulse);
        }

        hkSimdReal solverResultLimit; solverResultLimit.setFromFloat(m_solverResultLimit);
        if(sumReactionForce >  solverResultLimit * solverResultLimit)
        {
            const hkSimdReal rootSumReactionForce = sumReactionForce.sqrt();
            setBroken( in.m_constraintInstance, true, rootSumReactionForce.getReal() );

            if ( this->m_revertBackVelocityOnBreak )
            {
                // revert back the velocities
                const hkSimdReal f = solverResultLimit / rootSumReactionForce;

                hkVector4 linA; linA.load<3,HK_IO_NATIVE_ALIGNED>(runtime->m_linearVelcityA);
                hkVector4 linB; linB.load<3,HK_IO_NATIVE_ALIGNED>(runtime->m_linearVelcityB);
                hkVector4 angA; angA.load<3,HK_IO_NATIVE_ALIGNED>(runtime->m_angularVelcityA);
                hkVector4 angB; angB.load<3,HK_IO_NATIVE_ALIGNED>(runtime->m_angularVelcityB);

                hkpVelocityAccumulator* bodyA = const_cast<hkpVelocityAccumulator*>(in.m_bodyA);
                hkpVelocityAccumulator* bodyB = const_cast<hkpVelocityAccumulator*>(in.m_bodyB);
                { hkVector4& d = bodyA->m_linearVel; d.setInterpolate( linA, d, f ); }
                { hkVector4& d = bodyB->m_linearVel; d.setInterpolate( linB, d, f ); }
                { hkVector4& d = bodyA->m_angularVel; d.setInterpolate( angA, d, f ); }
                { hkVector4& d = bodyB->m_angularVel; d.setInterpolate( angB, d, f ); }
            }
        }
    }

    // zero results at the end so we can use recursive breakable constraints
    for(int j = 0; j < numResults; j++)
    {
        results[j].m_impulseApplied = hkReal(0);
    }
}


void hkpBreakableConstraintData::setBroken (hkpConstraintInstance* instance, hkBool broken, hkReal currentForce )
{
    HK_ASSERT(0x34322884, instance != HK_NULL, "Null instance pointer passed to setBroken");
    Runtime* runtime = getRuntime( instance->getRuntime() );
    HK_ASSERT(0x34322883, runtime != HK_NULL, "Runtime must be non-null. This constraint may not be added to the world.");

    if ( broken == runtime->m_isBroken)
    {
        return;
    }

    runtime->m_isBroken = broken;

    if ( !broken )
    {
        instance->m_internal->m_callbackRequest |= hkpConstraintAtom::CALLBACK_REQUEST_SETUP_CALLBACK;
    }
    else
    {
        instance->m_internal->m_callbackRequest &= ~hkpConstraintAtom::CALLBACK_REQUEST_SETUP_CALLBACK;
    }

    {
        hkpSimulationIsland* island = static_cast<hkpSimulationIsland*>(instance->getOwner() ); // we are not on the spu, so this should work
        hkpWorld* world = island->getWorld();

        if ( broken )
        {
            hkpConstraintBrokenEvent be( world, instance, hkReflect::getType<hkpBreakableConstraintData>());
            be.m_actualImpulse = currentForce;
            be.m_impulseLimit  = m_solverResultLimit;
            if(instance->m_listeners.getSize())
            {
                hkpConstraintCallbackUtil::fireConstraintBroken( be );
            }
            hkpWorldCallbackUtil::fireConstraintBroken( world,be);
        }
        else
        {
            hkpConstraintRepairedEvent be( world, instance, hkReflect::getType<hkpBreakableConstraintData>());
            if(instance->m_listeners.getSize())
            {
                hkpConstraintCallbackUtil::fireConstraintRepaired( be );
            }
            hkpWorldCallbackUtil::fireConstraintRepaired( world, be);
        }
    }
}


void hkpBreakableConstraintData::buildJacobian( const hkpConstraintQueryIn &in, hkpConstraintQueryOut &out )
{
   // Determine if reaction forces from previous solution should cause a break
    //  hkpSolverResults* results = static_cast<hkpSolverResults*>( out.m_constraintRuntime );
    //  int numResults = m_childNumSolverResults;
    HK_ASSERT( 0xf06521fe, out.m_constraintRuntime , "The constraint which is wrapped by the hkpBreakableConstraintData does not support breaking" );

    Runtime* runtime = getRuntime( out.m_constraintRuntime );

    if(!runtime->m_isBroken)
    {
        const hkpVelocityAccumulator* bodyA = in.m_bodyA;
        const hkpVelocityAccumulator* bodyB = in.m_bodyB;
        // Save the velocity accumulators
        bodyA->m_linearVel.store<3,HK_IO_NATIVE_ALIGNED>(runtime->m_linearVelcityA);
        bodyB->m_linearVel.store<3,HK_IO_NATIVE_ALIGNED>(runtime->m_linearVelcityB);
        bodyA->m_angularVel.store<3,HK_IO_NATIVE_ALIGNED>(runtime->m_angularVelcityA);
        bodyB->m_angularVel.store<3,HK_IO_NATIVE_ALIGNED>(runtime->m_angularVelcityB);

        hkpConstraintData::ConstraintInfo info; m_constraintData->getConstraintInfo(info);
        hkSolverBuildJacobianFromAtoms( info.m_atoms, info.m_sizeOfAllAtoms, in, out );
    }
    else
    {
        // insert a nop statement into the solver
        buildNopJacobian( in, out );
        if ( m_removeWhenBroken )
        {
            hkpConstraintInstance* constraint = in.m_constraintInstance;
            hkpWorld* world = constraint->getEntityA()->getWorld();
            world->removeConstraint(constraint);
        }
    }
}


void hkpBreakableConstraintData::buildNopJacobian( const hkpConstraintQueryIn& in, hkpConstraintQueryOut& out )
{
    in.beginConstraints( out, HK_NULL, sizeof(hkpSolverResults) );
    hkEndConstraints();
}



void hkpBreakableConstraintData::getConstraintInfo( hkpConstraintData::ConstraintInfo& info ) const
{
    m_constraintData->getConstraintInfo( info);
    info.m_atoms = const_cast<hkpConstraintAtom*>(m_atoms.getAtoms());
    info.m_sizeOfAllAtoms = m_atoms.getSizeOfAllAtoms();
}


void hkpBreakableConstraintData::getRuntimeInfo( hkBool wantRuntime, hkpConstraintData::RuntimeInfo& infoOut ) const
{
    infoOut.m_numSolverResults = m_childNumSolverResults;
    infoOut.m_sizeOfExternalRuntime = m_childRuntimeSize + sizeof(Runtime);
}


int hkpBreakableConstraintData::getType() const
{
    return hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE;
}

#if !defined(HK_PLATFORM_SPU)

hkpWrappedConstraintData* hkpBreakableConstraintData::deepClone() const
{
    // Clone wrapped constraint data
    const hkpConstraintData* srcChildData = getWrappedConstraintData();
    hkpConstraintData* dstChildData = hkpConstraintDataUtils::deepClone(srcChildData);

    // Clone breakable constraint
    hkpBreakableConstraintData* dstData = new hkpBreakableConstraintData(dstChildData);
    dstChildData->removeReference();

    // Copy members
    dstData->setThreshold( getThreshold() );
    dstData->setRemoveWhenBroken( getRemoveWhenBroken() );
    dstData->setRevertBackVelocityOnBreak( getRevertBackVelocityOnBreak() );
    return dstData;
}

#endif

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
