// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/Collide/hkpCollide.h>
#include <Physics2012/Collide/Util/Deprecated/ConvexHull/hkGeomConvexHullBuilder.h>
#include <Physics2012/Collide/Util/Deprecated/ConvexHull/hkGeomConvexHullTester.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>

// Include these to use the debug display of the convex hull.
//
//#include <hkutilities/visualdebugger/hkDebugDisplay.h>
//#include <hkdemoframework/hkTextDisplay.h>

//#define HK_DEBUG_PLANARHULL_DEMO
//#define HK_DEBUG_CONVEX_HULL
//#define HK_DEBUG_D2

//
// Uncomment this to display the convex hull.  Requires some of the includes
// above to be uncommented.
//
//#define HK_DEBUG_DISPLAY_CONVEX_HULL // calling draw should draw..
//#define HK_DEBUG_DISPLAY_PLANE_EQUATIONS


#if defined(HK_DEBUG_CONVEX_HULL) || defined(HK_DEBUG_D2)
#include <Demos/DemoCommon/DemoFramework/hkDemoConsole.h>
#endif

#include <Common/Visualize/hkDebugDisplay.h>

const hkBool HK_DEBUG_CONVEX_HULL_USEDEBUG = false;



void HK_CALL hkGeomConvexHullBuilder::generateConvexHull( const hkVector4* verts, int numVertices,
                                              hkGeomHull& hullOut, hkArray<hkVector4>& usedVerticesOut, hkGeomConvexHullMode mode )
{
////#ifdef HK_DEBUG_D2
//  if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
//  {
//      hkprintf("\n\n -- starting here with %i vertices -- \n\n", numVertices);
//      hkprintf(" Starting set:\n");
//      for ( int i = 0 ; i < numVertices ; i++ )
//      {
//          hkprintf("\t%.20ff, %.20ff, %.20ff,  // %d \n", verts[i](0), verts[i](1), verts[i](2), i );
//      }
//  }
////#endif

    // generate the hull without any filtering

    hkGeomConvexHullTolerances tolerances;
    tolerances.m_postFilter = false;
    tolerances.m_degenerate_tolerance = 4e-6f;

    tolerances.m_accurateButSlow = ( mode == HK_GEOM_CONVEXHULL_MODE_ACCURATE_BUT_SLOW );
    tolerances.m_runConvertToUnitCube = false;

    // sort out the vertices
    hkInplaceArrayAligned16<hkVector4, 64> filteredVertices;
    {

        for ( int iter = 0 ; iter < numVertices ; iter++ )
        {
            filteredVertices.pushBack( verts[iter] );
        }

        hkSort(filteredVertices.begin(), filteredVertices.getSize(), vectorLessAndMergeCoordinates);

        int newNumVertices;
        weldXsortedVertices( tolerances.m_weld_tolerance, filteredVertices, newNumVertices );

        //preFilterVertices( filteredVertices, 0.001f );
    }


    hkInplaceArrayAligned16<hkVector4, 64> planeEquations;

    // Check if we can generate a convex hull really quickly
    hkBool isValidHull = false;
//  {
//      approximateHullFromVerticesWithKDop( filteredVertices.begin(), filteredVertices.getSize(), planeEquations, usedVerticesOut);
//      buildConvexHull( tolerances, usedVerticesOut.begin(), usedVerticesOut.getSize(), hullOut, usedVerticesOut );
//
//      isValidHull = hkGeomConvexHullTester::isValidHull( tolerances, filteredVertices.begin(), filteredVertices.getSize(), hullOut, usedVerticesOut );
//  }

    if ( !isValidHull )
    {
        usedVerticesOut.clear();
        buildConvexHull( tolerances, filteredVertices.begin(), filteredVertices.getSize(), hullOut, usedVerticesOut );
        isValidHull = hkGeomConvexHullTester::isValidHull( tolerances, filteredVertices.begin(), filteredVertices.getSize(), hullOut, usedVerticesOut );
    }

    // If we are dealing with a planar shape, then use the generateHullFromPlanarPoints algorithm

    if ( !isValidHull && tolerances.m_accurateButSlow )
    {
        planeEquations.clear();
        hkBool isPlanar = false;
        hkInplaceArrayAligned16<PlaneAndPoints, 64> tangentPlanes;
        hkVector4 planarPlaneEquation;
        buildPlaneEquations( tolerances, hullOut, usedVerticesOut, planarPlaneEquation, isPlanar, planeEquations, tangentPlanes );

        if ( isPlanar )
        {
            usedVerticesOut.clear();

            removeCollinearVertices( filteredVertices, 0.001f );

            //int numPoints;
            hkInplaceArrayAligned16< hkVector4, 64 > unusedPlaneEquations;
            generateHullFromPlanarPoints( planarPlaneEquation, filteredVertices.begin(), filteredVertices.getSize(), usedVerticesOut, unusedPlaneEquations );
            //numPoints = usedVerticesOut.getSize();

            isValidHull = hkGeomConvexHullTester::isValidHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );

            //buildConvexHull( tolerances, usedVerticesOut.begin(), usedVerticesOut.getSize(), hullOut, usedVerticesOut );

            //isValidHull = hkGeomConvexHullTester::isValidHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );

            //if ( !isValidHull )
            //{
            //  int bla; bla++;
            //}
            //if ( numPoints != usedVerticesOut.getSize() )
            //{
            //  int bla; bla++;
            //}
            //HK_ASSERT_NO_MSG(0x462f484b, numPoints == usedVerticesOut.getSize() );
        }
    }

    if ( tolerances.m_accurateButSlow )
    {
        return;
    }

    tolerances.m_runConvertToUnitCube = true;

    if ( !isValidHull )
    {
        tolerances.m_postFilter = true;

        hkGeomConvexHullBuilder::buildConvexHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );
        isValidHull = hkGeomConvexHullTester::isValidHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );

        if ( ! isValidHull )
        {
            tolerances.m_degenerate_tolerance = 2e-7f;

            hkGeomConvexHullBuilder::buildConvexHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );
            isValidHull = hkGeomConvexHullTester::isValidHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );

            if ( ! isValidHull )
            {
                //hkprintf(" invalid hull generated. ");

                // what do we do now?
                // Set some of the tolerances to crazy values....
                // Weld more vertices together... they must be too far apart

                tolerances.m__tol3 = 1e-6f;

                hkGeomConvexHullBuilder::buildConvexHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );

                isValidHull = hkGeomConvexHullTester::isValidHull( tolerances, verts, numVertices, hullOut, usedVerticesOut );

                if ( ! isValidHull )
                {
                    //HK_WARN(0x52401f80, "The convex hull generated may not be convex.");
                    //hkprintf(" really bad hull... ");

                    // save the vertices to a file
//                  if (0)
//                  {
//                      hkOstream out("robverts.txt");
//
//                      if(out.isOk())
//                      {
//                          char* outText[128];
//
//                          //hkString::sprintf( outText, "\n\nenum { NUM_VERTICES = %d }; \n\n", numVertices );
//                          out << outText;
//                          out << "float r_vertices[] = {\n";
//
//                          for ( int i = 0 ; i < numVertices ; i++ )
//                          {
//                              hkString::sprintf( outText, "\t %.20ff, %.20ff, %.20ff,\n", verts[i](0), verts[i](1), verts[i](2) );
//                              out << outText;
//                          }
//                      }
//
//                      out << "\n";
//                      out.flush();
//                  }
//
//                  // call a modified version of the approximatePlanesFromVerticesWithKDop function here.
                }
            }
        }
    }
}


hkSimdReal hkGeomConvexHullBuilder::getPseudoAngle( hkSimdRealParameter cosTheta, hkSimdRealParameter sinTheta )
{
    //  Returns a value that can be used to compare the size of angles.
    //  This value is given below, according to where which quadrant in the
    //  unit sphere the angle is.
    //
    //          |
    //      2-s |  s
    //     -----------
    //      2-s | 4-s
    //          |
    //
    // (s: sinTheta)

    hkSimdReal retVal;

    // There is an discontinuity near 0 that
    // should be dealt with more carefully.
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    hkSimdReal chooseSin;
    chooseSin.setSelect(sinTheta.greaterZero(), sinTheta, hkSimdReal_4 - sinTheta);
    retVal.setSelect(cosTheta.lessZero(), hkSimdReal_2 - sinTheta, chooseSin);
#else
    if (cosTheta.isLessZero())
    {
        retVal = hkSimdReal_2-sinTheta;
    }
    else
    {
        if (sinTheta.isGreaterZero())
            retVal = sinTheta;
        else
            retVal = hkSimdReal_4-sinTheta;
    }
#endif
    return retVal;
}


//
// Given a set of input points, and the plane they are all on, this function finds those
// vertices that are on the convex hull of the vertices.
//
// Cut'n'paste this code into a demo and define HK_DEBUG_PLANARHULL_DEMO to see it working.
//
// The walking algorithm in this function is also needed to generate the bevel plane equations for
// planar hulls.

void hkGeomConvexHullBuilder::generateHullFromPlanarPoints(const hkVector4& planeEqn, const hkVector4 *verticesin, int numverts, hkArray<hkVector4>& usedVertices, hkArray<hkVector4>& planeEquationsOut)
{
    // Add the two planarPlaneEquations
    {
        planeEquationsOut.clear();
        hkVector4 reversePlaneEqn;
        reversePlaneEqn.setNeg<4>( planeEqn );

        planeEquationsOut.pushBack( planeEqn );
        planeEquationsOut.pushBack( reversePlaneEqn );

#ifdef HK_DEBUG_PLANARHULL_DEMO
        HK_DISPLAY_PLANE( planeEqn, verticesin[0], 1.0f, hkColor::RED );
#endif

#ifdef HK_DEBUG_PLANARHULL_DEMO
        HK_DISPLAY_PLANE( reversePlaneEqn, verticesin[0], 1.0f, hkColor::RED );
#endif
    }


#ifdef HK_DEBUG_PLANARHULL_DEMO
    static int pointNumber = 0;
    pointNumber = 0;
    char* pointName = hkAllocate<char>(128, HK_MEMORY_CLASS_STRING);
#endif
    //
    // This is what we will be using
    // to test for termination.
    // (But what if we visit the same point, from a different direction?)
    //

    hkArray< hkBool32 > alreadyVisited(numverts, false);

    //
    // Get the starting vertex & direction.
    //

#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    hkIntVector counter;
    const hkIntVector one = hkIntVector::getConstant<HK_QUADINT_1>();
#endif

    hkVector4 supportDir;
    int startVertexIndex = -1;
    {
        hkSimdReal absEqn; absEqn.setAbs(planeEqn.getComponent<2>());
        hkSimdReal absInvEqn; absInvEqn.setAbs(hkSimdReal_1 - absEqn);
        if ( absInvEqn < hkSimdReal::fromFloat(1e-6f) )
        {
            supportDir.setCross(hkVector4::getConstant<HK_QUADREAL_1000>(), planeEqn);
        }
        else
        {
            supportDir.setCross( planeEqn, hkVector4::getConstant<HK_QUADREAL_0010>() );
        }
        {
            hkSimdReal maxDot = hkSimdReal_MinusMax;
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
            hkIntVector startVertexI; startVertexI.splatImmediate32<-1>();
            counter.setZero();
#endif
            for( int i = 0 ; i < numverts; i++ )
            {
                const hkSimdReal curDot = verticesin[i].dot<3>(supportDir);
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
                hkVector4Comparison gt = curDot.greater(maxDot);
                maxDot.setSelect(gt, curDot, maxDot);
                startVertexI.setSelect(gt, counter, startVertexI);
                counter.setAddS32(counter, one);
#else
                if ( curDot > maxDot )
                {
                    maxDot = curDot;
                    startVertexIndex = i;
                }
#endif
            }
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
            startVertexI.store<1, HK_IO_NATIVE_ALIGNED>((hkUint32*)&startVertexIndex);
#endif
        }

        alreadyVisited[startVertexIndex] = true;
        usedVertices.pushBack( verticesin[startVertexIndex] );
    }

    // Get the starting direction.
    hkVector4 currentDir;
    int currentVertexIndex;
    {
        currentDir.setCross( supportDir, planeEqn ); currentDir.add( supportDir );
        currentVertexIndex = startVertexIndex;
    }

#ifdef HK_DEBUG_PLANARHULL_DEMO
    HK_DISPLAY_PLANE( planeEqn, verticesin[startVertexIndex], 1.0f, hkColor::WHITE );
#endif
    //
    // Until we get to a point we have already visited, we continue wrapping the starting
    // direction around the vertices.
    //

    //HK_ASSERT_NO_MSG(0xb5d35ee, currentDir.length<3>() > 1e-16f );

    int previousVertexIndex = startVertexIndex;
    int firstVertexIndex=-1, closestVertexIndex=-1;

    bool finished = false;
    while ( !finished )
    {
        hkSimdReal currentSine = -hkSimdReal_2;

        // Check if calculations are correct.
        {
            currentDir.normalize<3>();

            hkVector4 endPoint;
            endPoint.setAdd( verticesin[currentVertexIndex], currentDir );
            endPoint.setW(hkSimdReal_1);

#ifdef HK_DEBUG_PLANARHULL_DEMO

            HK_ASSERT_NO_MSG(0x504566b0, hkMath::fabs( endPoint.dot<4>( planeEqn ) ) < 1e-4f );

            // Arrow pointing in currentDir, from the current point
            HK_DISPLAY_ARROW( verticesin[currentVertexIndex], currentDir, hkColor::GREEN);

            hkString::sprintf( pointName, "%d", pointNumber++ );
            //m_env->m_textDisplay->outputText3D( pointName, endPoint(0), endPoint(1), endPoint(2));
#endif
        }

        // Find the vertex that makes the minimum angle with the current
        // direction line from the current vertex.
        hkSimdReal minAngle = hkSimdReal_Max;

        hkVector4 toVertexV, crossToVertexV;

#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
        hkIntVector closestVertexI; closestVertexI.splatImmediate32<-1>();
        counter.setZero();
#endif
        for ( int v = 0 ; v < numverts; v++ )
        {
            if ( v != currentVertexIndex )
            {
                toVertexV.setSub( verticesin[v], verticesin[currentVertexIndex] );
                crossToVertexV.setCross( currentDir, toVertexV );
                hkSimdReal dotToVertexV = currentDir.dot<3>( toVertexV );

                hkSimdReal cosTheta, sinTheta;
                {
                    const hkSimdReal lenToV = toVertexV.lengthInverse<3>();
                    const hkSimdReal lenCrossToV = crossToVertexV.length<3>();

                    sinTheta = lenCrossToV * lenToV;
                    cosTheta = dotToVertexV * lenToV;

                    sinTheta.setFlipSign(sinTheta, crossToVertexV.dot<3>( planeEqn ).greaterZero());
                }

                hkSimdReal angleToV = getPseudoAngle( cosTheta, sinTheta );
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
                hkVector4Comparison lt = angleToV.less(minAngle);
                minAngle.setSelect(lt, angleToV, minAngle);
                closestVertexI.setSelect(lt, counter, closestVertexI);
                currentSine.setSelect(lt, sinTheta, currentSine);
                counter.setAddS32(counter, one);
#else
                if ( angleToV < minAngle )
                {
                    minAngle = angleToV;
                    closestVertexIndex = v;
                    currentSine = sinTheta;
                }
#endif
            }
        }
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
        closestVertexI.store<1, HK_IO_NATIVE_ALIGNED>((hkUint32*)&closestVertexIndex);
#endif


        // At this point, we have an edge on the hull and so can create a plane
        // on it's edge (perpendicular to planeEqn).
        if ( currentVertexIndex != startVertexIndex )
        {
            hkVector4 edge; edge.setSub( verticesin[currentVertexIndex], verticesin[previousVertexIndex] );
            hkVector4 awayDir; awayDir.setSub( verticesin[currentVertexIndex], verticesin[closestVertexIndex] );

            hkVector4& newBevelPlane = planeEquationsOut.expandOne();
            {
                hkVector4 planarPlane;
                if ( previousVertexIndex != closestVertexIndex )
                {
                    planarPlane.setCross( edge, awayDir );
                }
                else
                {
                    planarPlane = planeEqn;
                }


                // Note that this planarPlane should be parallel or anti-parallel to the planeEqn

                newBevelPlane.setCross( edge, planarPlane );
                newBevelPlane.setFlipSign(newBevelPlane, newBevelPlane.dot<3>( awayDir ).less(hkSimdReal::fromFloat(1e-6f)));

                newBevelPlane.normalize<3>();

                newBevelPlane.setW( -newBevelPlane.dot<3>( verticesin[currentVertexIndex] ) );
            }

#ifdef HK_DEBUG_PLANARHULL_DEMO
            HK_DISPLAY_PLANE( newBevelPlane, verticesin[currentVertexIndex], 1.0f, hkColor::CYAN );
#endif

        }
        else
        {
            firstVertexIndex = closestVertexIndex;
        }


        // Terminating condition: check if this vertex has already
        // been visited.
        if ( alreadyVisited[closestVertexIndex] )
        {
            finished = true;
        }
        else
        {
            // This new vertex is on the convex hull of this shape.
            alreadyVisited[closestVertexIndex] = true;
            usedVertices.pushBack( verticesin[closestVertexIndex] );

            // Arrow pointing closestVertex, from the current point
            hkVector4 toClosest;
            {
                toClosest.setSub( verticesin[closestVertexIndex], verticesin[currentVertexIndex] );
                toClosest.normalize<3>();
                //toClosest(0) -= 0.1f; toClosest(1) -= 0.1f; toClosest(2) -= 0.1f;

#ifdef HK_DEBUG_PLANARHULL_DEMO

                // Arrow from current vertex to the next vertex
                HK_DISPLAY_ARROW( verticesin[currentVertexIndex], toClosest, hkColor::BLUE);

                hkString::sprintf( pointName, "%d", pointNumber );
                hkVector4 endPoint; endPoint.setAdd( verticesin[currentVertexIndex], toClosest );
    //          m_env->m_textDisplay->outputText3D( pointName, endPoint(0), endPoint(1), endPoint(2));
#endif
            }

            if ( currentSine.isLess(hkSimdReal::fromFloat(1e-4f)) )
            {
                //hkVector4 newDir; newDir.setCross( currentDir, toClosest );
                //currentDir.setCross( newDir, currentDir );
                //HK_DISPLAY_ARROW( verticesin[currentVertexIndex], currentDir, hkColor::RED);

                currentDir = toClosest;
            }
            else
            {
                currentDir.add( toClosest );
            }

            //HK_ASSERT_NO_MSG(0x13ac33a7, currentDir.length<3>() > 1e-16f );

            previousVertexIndex = currentVertexIndex;
            currentVertexIndex = closestVertexIndex;
        }
    }


    // We are finished - there is one more plane equation that must be added.
    {
        hkVector4 edge; edge.setSub( verticesin[closestVertexIndex], verticesin[currentVertexIndex] );
        hkVector4 awayDir; awayDir.setSub( verticesin[closestVertexIndex], verticesin[firstVertexIndex] );

        hkVector4& newBevelPlane = planeEquationsOut.expandOne();
        {
            newBevelPlane.setCross( edge, planeEquationsOut[0] );
            newBevelPlane.setFlipSign(newBevelPlane, newBevelPlane.dot<3>( awayDir ).less(hkSimdReal::fromFloat(1e-6f)));
            newBevelPlane.normalize<3>();

            newBevelPlane.setW( -newBevelPlane.dot<3>( verticesin[closestVertexIndex] ) );
        }

#ifdef HK_DEBUG_PLANARHULL_DEMO
        HK_DISPLAY_PLANE( newBevelPlane, verticesin[closestVertexIndex], 1.0f, hkColor::GREEN );
#endif
    }



#ifdef HK_DEBUG_PLANARHULL_DEMO
    hkDeallocate(pointName);
#endif
}

static hkBool weightedNeighbourLess( const struct hkGeomConvexHullBuilder::WeightedNeighbour& w1, const struct hkGeomConvexHullBuilder::WeightedNeighbour& w2);

static hkBool weightedLineLess( const struct hkGeomConvexHullBuilder::WeightedLine& w1, const struct hkGeomConvexHullBuilder::WeightedLine& w2);



void hkGeomConvexHullBuilder::getAabb( const hkArray<hkVector4>& verts, hkAabb& aabb )
{
    if ( verts.getSize() > 0 )
    {
        aabb.setFromPoint(verts[0]);

        for ( int i = 1 ; i < verts.getSize() ; i++ )
        {
            aabb.includePoint(verts[i]);
        }
    }
}




void hkGeomConvexHullBuilder::weldXsortedVertices( hkReal weldTolerance, hkArray<hkVector4>& verts, int& numVertices)
{
    // vertices are considered close if they are within a sphere of radius epsilon
    // of each other.

    hkVector4* s = verts.begin();   // source pointer
    hkVector4* d = verts.begin();   // dest point

//  hkReal maxVerticesExtent = getMaxVertexExtent( verts );
//  if ( maxVerticesExtent > 1.0f )
//      {
//          weldTolerance *= 10*( maxVerticesExtent );
//      }

    for ( int i = verts.getSize()-1 ; i>=0; i-- )
    {
        //
        //  Check our d backwards, whether we find the same point
        //
        for ( hkVector4* e = d-1; ; e-- )
        {
            if ( e < verts.begin() || e[0](0) < s[0](0) - 0.01f )
            {
                // not found, use it
                *(d++) = *s;
                break;
            }

            hkVector4 difference; difference.setSub( *e, *s );
            if ( difference.lengthSquared<3>().getReal() < weldTolerance ) // this must use lengthSquared4() in order to be used for plane equations
            {
                // found, ignore it

                //
                // Check ahead to ensure there are no more points within this range.
                // This is to ensure that a line cannot "collapse", i.e. start and end
                // points being more than weld_tolerance distance away, but each of the
                // intermediate points being within weld_tolerance of each other.
                //
                for ( int j = i-1 ; j >= 0 ; j-- )
                {
                    hkVector4 forwardDifference; forwardDifference.setSub( *e, *(s+1) );
                    if ( forwardDifference.lengthSquared<3>().getReal() < weldTolerance )
                    {
                        s++;
                        i--;
                    }
                    else
                    {
                        break;
                    }
                }
                break;
            }
        }
        s++;
    }

    numVertices = (int)(d - verts.begin());

    verts.setSize( numVertices );
}



void hkGeomConvexHullBuilder::printEdges( hkGeomHull& hull )
{
    for (int i = 0; i < hull.m_edges.getSize(); i++)
    {
        //hkGeomEdge& e= hull.m_edges[i];
        //hkprintf("\tEdge %i: next:%i mirror:%i  vertex:%i\n", i, e.m_next, e.m_mirror, e.m_vertex);
    }
}

void hkGeomConvexHullBuilder::buildGeometry( hkGeomHull& hull, hkGeometry& geometryOut )
{
    // build up the geometry from the hull edges
    hkGeomEdge* edgeBase = hull.m_edges.begin();

    for ( int ie = 0 ; ie < hull.m_edges.getSize() ; ie++ )
    {
        // look at the two next edges
        hkGeomEdge edge0 = hull.m_edges[ie];
        int  edge1Index = edge0.m_next;
        int edge2Index  = edge0.getNext( edgeBase )->m_next;

        if ( ( ie < edge1Index ) && ( ie < edge2Index ) )
        {
            // we haven't visited this triangle yet - add it to the geometry
            int vertexIndex0 = edge0.m_vertex;
            int vertexIndex1 = edge0.getNext( edgeBase )->m_vertex;
            int vertexIndex2 = edge0.getNext( edgeBase )->getNext( edgeBase )->m_vertex;

            // ensure the winding is correct
            hkGeometry::Triangle& tri = *geometryOut.m_triangles.expandBy(1);
            tri.set( vertexIndex0, vertexIndex2, vertexIndex1 );
        }
    }
}



hkBool hkGeomConvexHullBuilder::buildPlaneEquations( const hkGeomConvexHullTolerances&  tolerances, hkGeomHull& hull, const hkArray<hkVector4>& usedVertices, hkArray<hkVector4>& planeEquationsOut )
{
    hkBool isPlanar;
    hkArray<PlaneAndPoints> tangentPlanes;

    hkVector4 planarPlaneEquation;
    buildPlaneEquations( tolerances, hull, usedVertices, planarPlaneEquation, isPlanar, planeEquationsOut, tangentPlanes );

    return isPlanar;
}

hkReal hkGeomConvexHullBuilder::getMaximumDistance( hkGeomHull& hull, const hkArray<hkVector4>& usedVertices, const hkArray<hkVector4>& set)
{
    hkGeomConvexHullTolerances  tolerances;
    hkArray<hkVector4>          planes;
    hkSimdReal                  maxDistance=hkSimdReal_MinusMax;
    buildPlaneEquations(tolerances,hull,usedVertices,planes);
    for(int i=0,ni=set.getSize();i<ni;++i)
    {
        const hkVector4&    p=set[i];
        for(int j=0,nj=planes.getSize();j<nj;++j)
        {
            const hkSimdReal d = planes[j].dot<3>(p) + planes[j].getComponent<3>();
            maxDistance.setMax(d,maxDistance);
        }
    }
    return maxDistance.getReal();
}


void hkGeomConvexHullBuilder::removeCollinearVertices( hkArray< hkVector4>& vertices, hkReal degenerateToleranceR )
{
    {
        // reset all the w-components
        {
            for ( int i = 0 ; i < vertices.getSize() ; ++i )
            {
                vertices[i].zeroComponent<3>();
            }
        }

        {
            const hkSimdReal one = hkSimdReal_1;
            const hkSimdReal degenerateTolerance = hkSimdReal::fromFloat(degenerateToleranceR);
            for ( int i = 0 ; i < vertices.getSize() ; ++i )
            {
                for ( int j = i+1 ; j < vertices.getSize() ; ++j )
                {
                    for ( int k = j+1 ; k < vertices.getSize() ; ++k )
                    {
                        if ( vertices[i].getW().isNotEqual(one) && vertices[j].getW().isNotEqual(one) && vertices[k].getW().isNotEqual(one) )
                        {
                            hkVector4 ji; ji.setSub( vertices[i], vertices[j] ); ji.normalize<3>();
                            hkVector4 ij; ij.setNeg<4>(ji);

                            hkVector4 ki; ki.setSub( vertices[i], vertices[k] ); ki.normalize<3>();
                            hkVector4 ik; ik.setNeg<4>(ki);

                            hkVector4 jk; jk.setSub( vertices[k], vertices[j] ); jk.normalize<3>();
                            hkVector4 kj; kj.setNeg<4>(jk);

                            hkVector4 planeEqn;
                            hkSimdReal planeEqnLength;
                            {
                                planeEqn.setCross( ji, ki );
                                planeEqnLength = planeEqn.lengthSquared<3>();

                                if (( planeEqnLength < degenerateTolerance ) && ji.dot<3>(ki).isLessZero() )
                                {
#ifdef HK_DEBUG_D2
                                    hkprintf("PlaneEqnLength %f, removing vertex %d of (%d,%d,%d) ", planeEqnLength.getReal(), i, i, j, k);
#endif
                                    vertices[i].setW(hkSimdReal_1);
#ifdef HK_DEBUG_CONVEXPIECEMESH

                                    printf( " Removing vertex (%f,%f,%f)\n  [ others: (%f,%f,%f), (%f,%f,%f) ]\n\n ",
                                        vertices[i](0), vertices[i](1), vertices[i](2),
                                        vertices[j](0), vertices[j](1), vertices[j](2),
                                        vertices[k](0), vertices[k](1), vertices[k](2)
                                        );
#endif
                                    continue;
                                }
                            }

                            {
                                planeEqn.setCross( jk, ik );
                                planeEqnLength = planeEqn.lengthSquared<3>();

                                if (( planeEqnLength < degenerateTolerance ) && jk.dot<3>(ik).isLessZero() )
                                {
#ifdef HK_DEBUG_D2
                                    hkprintf("PlaneEqnLength %f, removing vertex %d of (%d,%d,%d) ", planeEqnLength.getReal(), k, i, j, k);
#endif
                                    vertices[k].setW(hkSimdReal_1);

#ifdef HK_DEBUG_CONVEXPIECEMESH
                                    printf( " Removing vertex (%f,%f,%f)\n [ others: (%f,%f,%f), (%f,%f,%f) ]\n ",
                                        vertices[k](0), vertices[k](1), vertices[k](2),
                                        vertices[j](0), vertices[j](1), vertices[j](2),
                                        vertices[i](0), vertices[i](1), vertices[i](2)
                                        );
#endif
                                    continue;
                                }
                            }

                            {
                                planeEqn.setCross( ij, kj );
                                planeEqnLength = planeEqn.lengthSquared<3>();

                                if (( planeEqnLength < degenerateTolerance ) && ij.dot<3>(kj).isLessZero() )
                                {

#ifdef HK_DEBUG_D2
                                    hkprintf("PlaneEqnLength %f, removing vertex %d of (%d,%d,%d) ", planeEqnLength.getReal(), j, i, j, k);
#endif
                                    vertices[j].setW(hkSimdReal_1);

#ifdef HK_DEBUG_CONVEXPIECEMESH
                                    printf( " Removing vertex (%f,%f,%f)\n  [ others: (%f,%f,%f), (%f,%f,%f) ]\n ",
                                        vertices[j](0), vertices[j](1), vertices[j](2),
                                        vertices[k](0), vertices[k](1), vertices[k](2),
                                        vertices[i](0), vertices[i](1), vertices[i](2)
                                        );
#endif
                                    continue;
                                }
                            }
                        }
                    }
                }
            }
        }

        removeFlaggedVertices( vertices );
    }
}



void hkGeomConvexHullBuilder::postFilterVertices( hkGeomHull& hull, int startVertex, int endVertex,
                                                 const hkGeomConvexHullTolerances&  tolerances,
                                                 hkBool& vertsHaveChanged )
{
    {
        vertsHaveChanged = false;

        if ( tolerances.m_postFilter == false )
        {
            return;
        }

        hkGeomEdge* edgeBase = hull.m_edges.begin();
        hkVector4* vertexBase = hull.m_vertexBase;


        // Identifies collinear triangles using a degenerate_tolerance (where area of triangle < degenerate_tolerance).
        // For each degenerate triangle found, the interior point is removed.

        {
            // clear the edge visited flag

            for ( int jn = 0 ; jn < hull.m_edges.getSize() ; jn++ )
            {
                hull.m_edges[jn].m_info = 0;
            }

            for ( int ie = 0 ; ie < hull.m_edges.getSize() ; ie++ )
            {
                // take an edge and it's mirror - try to find a vertex that is collinear with it.
                if ( hull.m_edges[ie].m_info != 1 )
                {
                    hkGeomEdge& edge = hull.m_edges[ie];
                    hkGeomEdge& mirror = *edge.getMirror( edgeBase );
                    edge.m_info = 1;
                    mirror.m_info = 1;

                    int v0 = edge.m_vertex;
                    int v1 = mirror.m_vertex;


                    hkVector4& vv0 = vertexBase[v0];
                    hkVector4& vv1 = vertexBase[v1];


                    hkVector4 d1; d1.setSub( vv1, vv0 );
                    //d1.normalize_23BitAccurate<3>();

                    hkArray<int> possibles;
                    possibles.clear();

                    int min_index = v0;
                    int max_index = v1;

                    hkReal min_proj = 1e-6f;
                    hkReal max_proj = d1.dot<3>( d1 ).getReal();

                    if ( (v1 != v0 ) && (vv0(3) == 0) && (vv1(3) == 0))
                    {
                        for ( int v2 = startVertex ; v2 < endVertex+1 ; v2++ )
                        {
                            hkVector4& vv2 = vertexBase[v2];
                            if ((v0 != v2) && (v1 != v2) && (vv2(3) == 0))
                            {
                                hkVector4 tangentPlaneEquation;
                                {
                                    hkVector4 d2; d2.setSub( vv2, vv0 );

                                    hkReal proj = d1.dot<3>( d2 ).getReal();

                                    tangentPlaneEquation.setCross( d1, d2 );

                                    if ( tangentPlaneEquation.lengthSquared<3>().getReal() < tolerances.m_degenerate_tolerance )
                                    {
                                        // compare distance
                                        if(proj < min_proj)
                                        {
                                            possibles.pushBack(min_index);
                                            min_index = v2;
                                            min_proj = proj;
                                        }
                                        else if(proj > max_proj)
                                        {
                                            possibles.pushBack(max_index);
                                            max_index = v2;
                                            max_proj = proj;
                                        }
                                        else
                                        {
                                            possibles.pushBack(v2);
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // Now we've got a (possibly empty) list of possibles. Push back all but the min and max
                    for(int ip = 0; ip < possibles.getSize(); ip++)
                    {
                        if((possibles[ip] != min_index) && (possibles[ip] != max_index))
                        {
                            vertexBase[possibles[ip]](3) = 1;
                            vertsHaveChanged = true;
                        }
                    }
                }
            }
        }


        // Remove vertices which are redundant due to coplanarity. For each vertex, all adjacent faces are examined.
        // If all the normals are all within an angle determined by coplanar_vertices_tolerance of the vertex normal
        // (determined by the average of all face normals), the vertex is removed.
        {

            // clear the edge visited flag
            for ( int jn = 0 ; jn < hull.m_edges.getSize() ; jn++ )
            {
                hull.m_edges[jn].m_info = 0;
            }


            if ( (endVertex - startVertex > 2) && (hull.m_edges.getSize() > 2) )
            {
                for ( int iv = startVertex; iv < endVertex+1 ; iv++ )
                {
                    if ( vertexBase[iv](3) == 0 )
                    {
                        hkArray<hkVector4> normals;

                        for ( int ie = 0 ; ie < hull.m_edges.getSize() ; ie++ )
                        {
                            if (( hull.m_edges[ie].m_info != 1 ) && ( hull.m_edges[ie].m_vertex == iv ))
                            {
                                hull.m_edges[ie].m_info = 1;
                                hkGeomEdge* startNeighbour = hull.m_edges[ie].getMirror( edgeBase );
                                hkGeomEdge* currentNeighbour = startNeighbour;
                                do {
                                    currentNeighbour->getNext( edgeBase )->m_info = 1;

                                    const hkVector4& u0 = *(currentNeighbour->getVertex( vertexBase ));
                                    const hkVector4& u1 = *(currentNeighbour->getNext( edgeBase )->getVertex( vertexBase ));
                                    const hkVector4& u2 = *(currentNeighbour->getNext( edgeBase )->getNext( edgeBase )->getVertex( vertexBase ));

                                    //if ( ( u0(3) != 1 ) && ( u1(3) != 1 ) && ( u2(3) != 1 ) )
                                    {
                                        hkVector4 tangentDir; tangentDir.setSub( u0, u1 );
                                        hkVector4 other;      other.setSub( u2, u1 );
                                        hkVector4& n = *normals.expandBy(1);
                                        n.setCross( tangentDir, other );

                                        // there are no more degenerate triangles at this point
                                        n.normalize<3>();
                                    }

                                    currentNeighbour = currentNeighbour->getNext( edgeBase )->getMirror( edgeBase );

                                } while( currentNeighbour != startNeighbour );
                            }
                        }

                        if(normals.getSize() > 0)
                        {
                            // Get normal
                            int k;
                            hkVector4 normal; normal.setZero();
                            for(k = 0; k < normals.getSize(); k++)
                            {
                                normal.add( normals[k] );
                            }

                            if ( normal.lengthSquared<3>().getReal() > tolerances.m_coplanar_vertices_tolerance )
                            {
                                normal.normalize<3>();
                                hkBool valid = false;

                                for(k = 0; k < normals.getSize(); k++)
                                {
                                    hkReal dot = normals[k].dot<3>(normal).getReal();
                                    if(dot < 1 - tolerances.m_coplanar_vertices_tolerance)
                                    {
                                        valid = true;
                                        break;
                                    }
                                }

                                // If invalid, remove
                                if(!valid)
                                {
                                    vertexBase[iv](3) = 1;
                                    vertsHaveChanged = true;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}



hkBool hkGeomConvexHullBuilder::buildPlaneEquations( const hkGeomConvexHullTolerances&  tolerances,
                                                  hkGeomHull& hull, const hkArray<hkVector4>& usedVertices,
                                                  hkVector4& planarPlaneEquationOut, hkBool& isPlanarOut, hkArray<hkVector4>& planeEquationsOut,
                                                  hkArray<PlaneAndPoints>& tangentPlanesOut )
{

    hkGeomEdge* edgeBase = hull.m_edges.begin();
    hkVector4* vertexBase = hull.m_vertexBase;

    // calculate the tangentPlanesOut
    {
        for ( int ie = 0 ; ie < hull.m_edges.getSize() ; ie++ )
        {
            hkGeomEdge edge0 = hull.m_edges[ie];
            int  edge1Index = edge0.m_next;
            int edge2Index  = edge0.getNext( edgeBase )->m_next;

            if ( (ie < edge1Index) && (ie < edge2Index) )
            {
                // add the plane equation for this triangle to the plane equations
                hkVector4& tangentPlaneEquation = *planeEquationsOut.expandBy(1);
                {
                    hkVector4 v0 = *edge0.getVertex( hull.m_vertexBase );
                    hkVector4 v1 = *edgeBase[ edge1Index ].getVertex( hull.m_vertexBase );
                    hkVector4 v2 = *edgeBase[ edge2Index ].getVertex( hull.m_vertexBase );

                    hkVector4 tangentDir; tangentDir.setSub( v0, v1 );

                    // Get the tangent plane equation.
                    // Sum all 3 calcs to avoid numerical inaccuracy in any of them.
                    {
                        hkVector4 edges[3];
                        edges[0].setSub( v0, v1 );
                        edges[0].normalize<3>();
                        edges[1].setSub( v1, v2 );
                        edges[1].normalize<3>();
                        edges[2].setSub( v2, v0 );
                        edges[2].normalize<3>();

                        hkVector4 planeEqs[3];
                        planeEqs[0].setCross( edges[1], edges[0] );
                        planeEqs[1].setCross( edges[2], edges[1] );
                        planeEqs[2].setCross( edges[0], edges[2] );

                        tangentPlaneEquation.setZero();
                        tangentPlaneEquation.add( planeEqs[0] );
                        tangentPlaneEquation.add( planeEqs[1] );
                        tangentPlaneEquation.add( planeEqs[2] );
                    }

                    //HK_ASSERT_NO_MSG( 0xabc123, tangentPlaneEquation.lengthSquared<3>() > tolerances.m_degenerate_tolerance );
                    if ( tangentPlaneEquation.lengthSquared<3>().getReal() < tolerances.m_degenerate_tolerance )
                    {
                        // degenerate plane equation.
                        // this will be removed in the postFilter

                        planeEquationsOut.setSize( planeEquationsOut.getSize() - 1 );
                    }
                    else
                    {
                        tangentPlaneEquation.normalize<3>();
                        tangentPlaneEquation.setW( - tangentPlaneEquation.dot<3>( v0 ) );

                        // draw the plane equation around the centre of the triangle
                        hkVector4 centre;
                        {
                            hkVector4 e1; e1.setSub( v1, v0 ); e1.setMul( hkSimdReal_Half, e1 );
                            hkVector4 e2; e2.setSub( v2, v0 ); e2.setMul( hkSimdReal_Half, e2 );

                            centre.setAdd( e1, e2 );
                            centre.add( v0 );
                        }

#                       ifdef HK_DEBUG_DISPLAY_PLANE_EQUATIONS
                        drawPlane( tangentPlaneEquation, centre, hkColor::WHITE );
#                       endif

                        PlaneAndPoints& tp = *tangentPlanesOut.expandBy(1);
                        {
                            tp.m_planeEquation = tangentPlaneEquation;
                            tp.m_v0 = &hull.m_edges[ie];
                            tp.m_v1 = &edgeBase[ edge1Index ];
                            tp.m_v2 = &edgeBase[ edge2Index ];
                            tp.sort();
                        }
                    }
                }
            }
        }
    }

    {
        /*
        // if there is a single, degenerate triangle in the hull, then
        // consider the hull to be planar.
        if ( planeEquationsOut.getSize() == 0 )
        {
            isPlanar = true;
        }
        else
        {*/
            isPlanarOut = false;
        //}
    }


    // add bevel planes for nearly opposite planes and set nearly opposite planes
    // to be exactly opposite

    {
        for ( int i = 0 ; i < tangentPlanesOut.getSize() ; i++ )
        {
            for ( int j = i+1 ; j < tangentPlanesOut.getSize() ; j++ )
            {
                PlaneAndPoints pp1 = tangentPlanesOut[i];
                PlaneAndPoints pp2 = tangentPlanesOut[j];

                // check if the plane equations are nearly exactly opposite
                hkVector4 diff; diff.setAdd( pp1.m_planeEquation, pp2.m_planeEquation );
                if ( diff.dot<4>( diff ) < hkSimdReal::fromFloat(tolerances.m_oppositeNormal_tolerance) )
                {
                    isPlanarOut = true;

                    // set the two plane equations to be exactly opposite.
                    pp1.m_planeEquation.setNeg<3>(pp2.m_planeEquation);

                    // check if the planes have an edge in common
                    {
                        // there are only 7 possible combinations

                        if ( PlaneAndPoints::findPair(edgeBase, pp1.m_v0, pp2.m_v0, pp1.m_v1, pp2.m_v1, pp1, pp2 ) || ( PlaneAndPoints::findPair(edgeBase, pp1.m_v0, pp2.m_v0, pp1.m_v1, pp2.m_v2, pp1, pp2 ) ) )
                            {
                                isPlanarOut = true;
                                createBevelPlane( pp1.m_planeEquation, vertexBase[pp1.m_v0->m_vertex], vertexBase[pp1.m_v1->m_vertex], vertexBase[pp1.m_v2->m_vertex], planeEquationsOut );
                            }

                        if ( PlaneAndPoints::findPair(edgeBase, pp1.m_v0, pp2.m_v0, pp1.m_v2, pp2.m_v1, pp1, pp2 ) || PlaneAndPoints::findPair(edgeBase, pp1.m_v0, pp2.m_v0, pp1.m_v2, pp2.m_v2, pp1, pp2 ) )
                            {
                                isPlanarOut = true;
                                createBevelPlane( pp1.m_planeEquation, vertexBase[pp1.m_v0->m_vertex], vertexBase[pp1.m_v2->m_vertex], vertexBase[pp1.m_v1->m_vertex], planeEquationsOut );
                            }

                        if ( PlaneAndPoints::findPair(edgeBase, pp1.m_v1, pp2.m_v0, pp1.m_v2, pp2.m_v1, pp1, pp2 ) ||
                             PlaneAndPoints::findPair(edgeBase, pp1.m_v1, pp2.m_v0, pp1.m_v2, pp2.m_v2, pp1, pp2 ) ||
                             PlaneAndPoints::findPair(edgeBase, pp1.m_v1, pp2.m_v1, pp1.m_v2, pp2.m_v2, pp1, pp2 ) )
                            {
                                isPlanarOut = true;
                                createBevelPlane( pp1.m_planeEquation, vertexBase[pp1.m_v1->m_vertex], vertexBase[pp1.m_v2->m_vertex], vertexBase[pp1.m_v0->m_vertex], planeEquationsOut );
                            }
                    }
                }
            }
        }
    }


    if ( isPlanarOut )
    {
        HK_ASSERT_NO_MSG(0x55da3a41, usedVertices.getSize() >= 3 );

        hkVector4 edge1; edge1.setSub( usedVertices[0], usedVertices[1] );
        hkVector4 edge2; edge2.setSub( usedVertices[0], usedVertices[2] );
        planarPlaneEquationOut.setCross( edge1, edge2 );

        planarPlaneEquationOut.normalize<3>();

        planarPlaneEquationOut.setW( -planarPlaneEquationOut.dot<3>( usedVertices[0] ) );
    }


    // remove any duplicate plane equations
    {
        // reuse the vertices weld
        //hkprintf("\n sorting %d plane equations (1). ", planeEquationsOut.getSize());
        hkSort(planeEquationsOut.begin(), planeEquationsOut.getSize(), vectorLessAndMergeCoordinates);

        int newNumPlanes;
        weldXsortedVertices( tolerances.m_coplanar_plane_tolerance, planeEquationsOut, newNumPlanes );
    }


    //
    // It may still be necessary to add more bevel planes - in the case where we have a planar
    // object and the plane equations being used are around vertices that are not necessarily on the
    // same edge.
    {
        if ( planeEquationsOut.getSize() < 2 )
        {
            //
            // we must return something...
            //
            hkVector4 v0, v1, v0v1;
            if ( hull.m_edges.getSize() == 1 )
            {
                v0 = *hull.m_edges[0].getVertex( hull.m_vertexBase );
                v1.setAdd( v0, hkVector4::getConstant<HK_QUADREAL_1000>() );
            }
            else
            {
                v0 = *hull.m_edges[0].getVertex( hull.m_vertexBase );
                v1 = *hull.m_edges[0].getMirror( hull.m_edges.begin() )->getVertex( hull.m_vertexBase );
            }

            v0v1.setSub( v0, v1 );

            // Find which axis has the lowest dot-product with v0v1
            hkVector4 minDir; minDir.setZero();
            hkSimdReal minDot = hkSimdReal_Max;
            for ( int i = 0; i < 3 ; i++ )
            {
                const hkVector4& axis = hkVector4::getConstant((hkVectorConstant)(HK_QUADREAL_1000 + i));
                hkSimdReal dot; dot.setAbs(v0v1.dot<3>( axis ));
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
                hkVector4Comparison lt = dot.less(minDot);
                minDot.setSelect(lt, dot, minDot);
                minDir.setSelect(lt, axis, minDir);
#else
                if ( dot < minDot )
                {
                    minDot = dot;
                    minDir = axis;
                }
#endif
            }

            // Get the normals of 4 planes that intersect at these points.
            int startSize = planeEquationsOut.getSize();
            planeEquationsOut.setSize( startSize + 6 );

            hkVector4& p0 = planeEquationsOut[startSize];
            p0.setCross( v0v1, minDir );
            p0.normalize<3>();
            p0.setW( -p0.dot<3>( v0 ) );

            hkVector4& p1 = planeEquationsOut[startSize + 1];
            p1.setCross( v0v1, p0 );
            p1.normalize<3>();
            p1.setW( -p1.dot<3>( v0 ) );

            hkVector4& p2 = planeEquationsOut[startSize + 2];
            p2.setNeg<4>( p0 );
            p2.setW( -p2.dot<3>( v0 ) );

            hkVector4& p3 = planeEquationsOut[startSize + 3];
            p3.setNeg<4>( p1 );
            p3.setW( -p3.dot<3>( v0 ) );


            hkVector4& p4 = planeEquationsOut[startSize + 4];
            p4 = v0v1;
            p4.normalize<3>();
            p4.setW( -p4.dot<3>( v0 ) );

            hkVector4& p5 = planeEquationsOut[startSize + 5];
            p5.setNeg<4>( p4 );
            p5.normalize<3>();
            p5.setW( (hull.m_edges.getSize() == 1) ? -p5.dot<3>( v0 ) : -p5.dot<3>( v1 ) );
        }

        if ( isPlanarOut && tolerances.m_accurateButSlow )
        {
            // We need to run the generateHullFromPlanarPoints algorithm to find the planeEquations
            // that bound this shape.
            planeEquationsOut.clear();
            hkArray< hkVector4 > planarHullVertices;
            generateHullFromPlanarPoints( planarPlaneEquationOut, usedVertices.begin(), usedVertices.getSize(), planarHullVertices, planeEquationsOut);
        }
    }


    // remove any duplicate plane equations
    {
        // reuse the vertices weld
        //hkprintf("\n sorting %d plane equations (2). ", planeEquationsOut.getSize());
        hkSort(planeEquationsOut.begin(), planeEquationsOut.getSize(), vectorLessAndMergeCoordinates);

        int newNumPlanes;
        weldXsortedVertices( tolerances.m_coplanar_plane_tolerance, planeEquationsOut, newNumPlanes );
    }

    return true;
}



void hkGeomConvexHullBuilder::createBevelPlane( const hkVector4& planeNormal, const hkVector4& vertex0, const hkVector4& vertex1, const hkVector4& vertex2, hkArray<hkVector4>& planeEquationsOut )
{
    //
    // Returns the planeEquation of a plane that is perpendicular to the input plane,
    // contains the first two vertices and has a normal pointing away from the
    // third vertex.
    //

    hkVector4& bevelPlaneEqn = *planeEquationsOut.expandBy(1);
    {
        hkVector4 lineDir; lineDir.setSub( vertex0, vertex1 );
        bevelPlaneEqn.setCross( planeNormal, lineDir );

        hkVector4 awayDir; awayDir.setSub( vertex2, vertex1 );
        hkSimdReal awayValue = bevelPlaneEqn.dot<3>( awayDir );

        // which direction should the normal point in?
        bevelPlaneEqn.setFlipSign(bevelPlaneEqn, awayValue.greater(hkSimdReal::fromFloat(1e-6f)));

        if ( bevelPlaneEqn.lengthSquared<3>().getReal() > 1e-4f )
        {
            bevelPlaneEqn.normalize<3,HK_ACC_23_BIT,HK_SQRT_IGNORE>();
            bevelPlaneEqn.setW( - bevelPlaneEqn.dot<3>( vertex0 ) );


            // draw the plane equation around the centre of the edge
            hkVector4 centre; centre.setInterpolate( vertex0, vertex1, hkSimdReal_Half );

#           ifdef HK_DEBUG_DISPLAY_PLANE_EQUATIONS
            drawPlane( bevelPlaneEqn, centre, hkColor::CYAN );
#           endif

        }
        else
        {
            planeEquationsOut.setSize( planeEquationsOut.getSize() - 1 );
        }
    }
}



void hkGeomConvexHullBuilder::convertToUnitCube( hkArray<hkVector4>& usedVerticesOut, hkVector4& extents, hkVector4& aabbCenter )
{
    // the AABB of the vertices
    hkAabb vertsAabb; vertsAabb.setEmpty();
    getAabb( usedVerticesOut, vertsAabb );

    // the centre of AABB - this will be subtracted from every vertex
    aabbCenter.setAdd ( vertsAabb.m_max, vertsAabb.m_min );
    aabbCenter.mul( hkSimdReal::getConstant<HK_QUADREAL_INV_2>() );

    extents.setSub( vertsAabb.m_max, vertsAabb.m_min );

    hkMatrix3 scalingMatrix;
    {
        hkVector4 invExtents; invExtents.setReciprocal(extents);
        hkVector4 diag; diag.setSelect(extents.greater(hkVector4::getConstant<HK_QUADREAL_EPS>()), invExtents, hkVector4::getConstant<HK_QUADREAL_1>());
        hkMatrix3Util::_setDiagonal(diag, scalingMatrix);
    }

    hkMatrix3 vertexMatrix;

    // transform then scale
    // (transform: subtract the centre of the AABB from the vertex,
    //  scale: divide by the extents of the AABB )
    for ( int i = 0 ; i < usedVerticesOut.getSize() ; i++ )
    {
        hkVector4 transformedVert; transformedVert.setSub( usedVerticesOut[i], aabbCenter );

        vertexMatrix.setRows( transformedVert, hkVector4::getZero(), hkVector4::getZero() );
        vertexMatrix.mul( scalingMatrix );
        vertexMatrix.getRow<0>( usedVerticesOut[i] );
    }
}

// Transform the vertices from the unit cube back to their original shape
void hkGeomConvexHullBuilder::convertFromUnitCube( hkArray<hkVector4>& usedVerticesOut, hkVector4& extents, hkVector4& aabbCenter )
{
    hkMatrix3 scalingMatrix; hkMatrix3Util::_setDiagonal( extents, scalingMatrix );
    hkMatrix3 vertexMatrix;

    // transform then scale
    // (transform: add the centre of the AABB to the vertex,
    //  scale: multiply by the extents of the AABB )
    for ( int i = 0 ; i < usedVerticesOut.getSize() ; i++ )
    {
        vertexMatrix.setRows( usedVerticesOut[i], hkVector4::getZero(), hkVector4::getZero() );
        vertexMatrix.mul( scalingMatrix );
        vertexMatrix.getRow<0>( usedVerticesOut[i] );

        usedVerticesOut[i].add( aabbCenter );
    }
}




void hkGeomConvexHullBuilder::buildConvexHull( const hkGeomConvexHullTolerances&  tolerances, const hkVector4* verts, int numVertices,
                                              hkGeomHull& hullOut, hkArray<hkVector4>& usedVerticesOut )
{
#   ifdef HK_DEBUG_CONVEX_HULL
    if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
    {
        hkprintf("\n\n***************************** Build Convex Hull ***************** \n\n");
    }
#   endif

    usedVerticesOut.clear();

    for ( int iter = 0 ; iter < numVertices ; iter++ )
    {
        usedVerticesOut.pushBack( verts[iter] );
    }

    // Transform the vertices to be in a unit cube
    hkVector4 extents, aabbCenter;
    aabbCenter.setZero();
    extents.setZero();
    if ( tolerances.m_runConvertToUnitCube )
    {
        convertToUnitCube( usedVerticesOut, extents, aabbCenter );
    }

    hkSort(usedVerticesOut.begin(), usedVerticesOut.getSize(), vectorLessAndMergeCoordinates);

    int newNumVertices;
    weldXsortedVertices( tolerances.m_weld_tolerance, usedVerticesOut, newNumVertices );

#ifdef HK_DEBUG_D2
    if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
    {
        hkprintf("\n\n -- starting here with %i vertices -- \n\n", usedVerticesOut.getSize());
        hkprintf(" Starting set:\n");
        for ( int i = 0 ; i < usedVerticesOut.getSize() ; i++ )
        {
            hkprintf("\t%.20ff, %.20ff, %.20ff,  // %d \n", usedVerticesOut[i](0), usedVerticesOut[i](1), usedVerticesOut[i](2), i );
        }
    }
#endif

    if ( tolerances.m_accurateButSlow && (newNumVertices < 300) )
    {
        //hkprintf(" newNumVertices: %d\n", newNumVertices);
        removeCollinearVertices( usedVerticesOut, 0.001f );
    }


#   ifdef HK_DEBUG_CONVEX_HULL
    if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
    {
        hkprintf(" And providing tonight's entertainment will be:\n", usedVerticesOut.getSize() );
        for ( int i = 0 ; i < usedVerticesOut.getSize() ; i++ )
        {
            hkprintf(" vertex%d:\t (%.20ff, %.20ff, %.20ff )\n", i, usedVerticesOut[i](0), usedVerticesOut[i](1), usedVerticesOut[i](2) );
        }
    }
#   endif

    HK_ASSERT_NO_MSG(0x227e1b95, numVertices > 0 );

    {
        hkBool vertsHaveChanged = true;
        hkBool filteredVertices = false;

        while ( (!filteredVertices) || vertsHaveChanged )
        {
            filteredVertices = !vertsHaveChanged;

            {
                // clear the 'w' component of all the vertices
                // This will be used to determine if vertices are to be kept.
                {
                    for ( int iv = 0 ; iv < usedVerticesOut.getSize() ; iv++ )
                    {
                        usedVerticesOut[iv](3) = 0;
                    }
                }

                hullOut.m_edges.clear();
            }

            buildConvexSubHull( tolerances, usedVerticesOut, 0, usedVerticesOut.getSize()-1, hullOut);

            removeUnusedVertices( hullOut, usedVerticesOut );
            removeFlaggedVertices( usedVerticesOut );

#           ifdef HK_DEBUG_CONVEX_HULL
            if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
            {
                hkprintf("\n running postFilterVertices \n");
            }
#           endif

            postFilterVertices( hullOut, 0, usedVerticesOut.getSize()-1, tolerances, vertsHaveChanged );

            // remove all vertices that have been flagged.
            removeFlaggedVertices( usedVerticesOut );
        }
    }


#ifdef HK_DEBUG
    hullOut.isValidTopology();
#endif

    if ( tolerances.m_runConvertToUnitCube )
    {
        convertFromUnitCube( usedVerticesOut, extents, aabbCenter );
    }


#   ifdef HK_DEBUG_CONVEX_HULL
    if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
    {
        hkprintf("\n\n***************************** End Convex Hull ***************** \n\n");
    }
#   endif

}


void hkGeomConvexHullBuilder::draw( hkGeomHull& hull, hkColor::Argb color, hkTextDisplay* textDisplay )
{
    // draw edges
    {
        char* edgeNum = hkAllocate<char>(128, HK_MEMORY_CLASS_STRING);

        for (int e = 0; e < hull.m_edges.getSize(); e++)
        {
            hkGeomEdge* edge = &hull.m_edges[e];
            hkGeomEdge* next = edge->getNext( hull.m_edges.begin() );
            hkGeomEdge* nn   = next->getNext( hull.m_edges.begin() );
            const hkVector4* a = edge->getVertex( hull.m_vertexBase );
            const hkVector4* b = next->getVertex( hull.m_vertexBase );
            const hkVector4* c =   nn->getVertex( hull.m_vertexBase );
            hkVector4 diff; diff.setSub( *b, *a );
            hkSimdReal d; d.setFromFloat(0.3f);
            diff.mul( d );
            hkSimdReal t; t.setFromFloat(0.1f);
            hkVector4 start; start.setInterpolate( *a, *c, t );

            {
                hkString::sprintf(edgeNum, "%d(%d)", e, hull.m_edges[e].m_next );

                hkVector4 arrowPoint; arrowPoint.setAdd( start, diff );
                if ( textDisplay != HK_NULL )
                {
                    //textDisplay->outputText3D(edgeNum, arrowPoint(0), arrowPoint(1), arrowPoint(2));
                }
            }

#           ifdef HK_DEBUG_DISPLAY_CONVEX_HULL
            HK_DISPLAY_ARROW( start, diff, hkColor::BLUE );
            HK_DISPLAY_LINE( *a, *b, color );
#           endif
        }
        hkDeallocate(edgeNum);
    }
}

void hkGeomConvexHullBuilder::drawPlane( const hkVector4& planeEqn, const hkVector4& centrePoint, hkColor::Argb color )
{
#   ifdef HK_DEBUG_DISPLAY_CONVEX_HULL
    hkVector4 offset; offset.setMul( planeEqn(3), planeEqn );
    offset.setSub( centrePoint, offset );
    HK_DISPLAY_PLANE( planeEqn, offset, 2.0f, color );
#   endif
}


void hkGeomConvexHullBuilder::buildConvexSubHull(const hkGeomConvexHullTolerances&  tolerances, hkArray<hkVector4>& xSortedVerts, int startVertex, int endVertex,
                                                 hkGeomHull& hullOut)
{
    //int quantity = endVertex - startVertex + 1;

    int quantity = 0;
    {
        // quantity is the number of vertices that haven't been marked for deletion.
        for ( int iv = startVertex ; iv < endVertex+1 ; iv++ )
        {
            quantity += ( xSortedVerts[iv](3) == 0 ) ? 1 : 0;
        }
    }


    if ( quantity > 3 )
    {
        //
        // find the convex hull of the divided set
        //

        int mid = (endVertex + startVertex)/2;

        hkGeomHull lhull;
        lhull.m_vertexBase = xSortedVerts.begin();

        hkGeomHull rhull;
        rhull.m_vertexBase = xSortedVerts.begin();

//      {
//          buildConvexSubHull( tolerances, xSortedVerts, startVertex, mid, lhull );
//          buildConvexSubHull( tolerances, xSortedVerts, mid+1, endVertex, rhull );
//      }

        {
            hkBool vertsHaveChanged = true;
            while ( vertsHaveChanged )
            {
                while ( vertsHaveChanged )
                {
                    lhull.m_edges.clear();
                    buildConvexSubHull( tolerances, xSortedVerts, startVertex, mid, lhull );
                    postFilterVertices( lhull, startVertex, mid, tolerances, vertsHaveChanged );
                }

                vertsHaveChanged = true;
                while ( vertsHaveChanged )
                {
                    rhull.m_edges.clear();
                    buildConvexSubHull( tolerances, xSortedVerts, mid+1, endVertex, rhull );
                    postFilterVertices( rhull, mid+1, endVertex, tolerances, vertsHaveChanged );
                }

                hkBool leftVertsHaveChanged = true;
                postFilterVertices( lhull, startVertex, endVertex, tolerances, leftVertsHaveChanged );
                hkBool rightVertsHaveChanged = true;
                postFilterVertices( rhull, startVertex, endVertex, tolerances, rightVertsHaveChanged );

                vertsHaveChanged = leftVertsHaveChanged || rightVertsHaveChanged;
            }
        }

#   ifdef HK_DEBUG_CONVEX_HULL
        static int q = 0;
        if ( ++q == 471)
        {
            q = q;
            for (int i = startVertex; i <= mid; i++)
            {
                const hkVector4& v = xSortedVerts[i];
                hkprintf("\t\tverts.expandBy(1)->set( %ff, %ff, %ff ) ;\n",
                    v(0), v(1), v(2) );
            }
        }
#endif

        //
        // merge the hulls
        //

        {
            if ( lhull.m_edges.getSize() == 0 )
            {
                // hullOut is rHull
                for ( int ir = 0 ; ir < rhull.m_edges.getSize() ; ir++ )
                {
                    hullOut.m_edges.pushBack( rhull.m_edges[ir] );
                }
            }
            else if ( rhull.m_edges.getSize() == 0 )
            {
                // hullOut is rHull
                for ( int il = 0 ; il < lhull.m_edges.getSize() ; il++ )
                {
                    hullOut.m_edges.pushBack( lhull.m_edges[il] );
                }
            }
            else
            {
                mergeHulls( tolerances, lhull, rhull, hullOut);
            }
        }

#       ifdef HK_DEBUG_DISPLAY_PLANE_EQUATIONS
        draw( lhull, hkColor::GREEN, HK_NULL );
        draw( rhull, hkColor::CYAN, HK_NULL );
#       endif

        if( hullOut.m_edges.getSize() == 0)
        {
            hullOut.m_edges = lhull.m_edges;
        }
    }
    else
    {
        hullOut.m_vertexBase = xSortedVerts.begin();

        // find the non-deleted vertices.
        hkInplaceArray<int, 3> vertices;
        {
            for ( int iq = startVertex ; iq < endVertex+1 ; iq++ )
            {
                if ( xSortedVerts[iq](3) == 0 )
                {
                    vertices.pushBack( iq );
                }
            }
        }

        switch (quantity)
        {
        case 0:
            break;
        case 1:
            hullOut.initializeWithVertex( vertices[0] );
            break;
        case 2:
            hullOut.initializeWithEdge( vertices[0], vertices[1] );
            break;
        default:
            hullOut.initializeWithTriangle( vertices[0], vertices[1], vertices[2] );
            break;
        }
    }
}



hkResult hkGeomConvexHullBuilder::mergeHulls( const hkGeomConvexHullTolerances&  tolerances, hkGeomHull& lhull, hkGeomHull& rhull, hkGeomHull& hullOut)
{
    hullOut.m_vertexBase = lhull.m_vertexBase;

    if ( isSingleLine( tolerances.m_degenerate_tolerance, lhull, rhull, hullOut ) )
    {
        return HK_SUCCESS;
    }

    //
    // Construct the wrapping plane.
    // ----------------------------
    //
    // First: Get a supporting plane for both lhull and rhull.
    //

    const hkVector4* vertexBase = lhull.m_vertexBase;
    // find a valid wrapping line for each hull


    hkVector4 startPlaneEquation;
    hkInplaceArray<WeightedLine, 1> firstTangentLines;
    {
        WeightedLine& startLine = *firstTangentLines.expandByUnchecked(1);
        getCommonTangent( lhull, rhull, startLine, startPlaneEquation );

//#ifdef HK_DEBUG_CONVEX_HULL
//      drawPlane( startPlaneEquation, *startLine.m_leftEdge->getVertex( vertexBase ));
//#endif
    }
    hkVector4 previousPlaneEquation = startPlaneEquation;

    typedef hkInplaceArray<WeightedLine,16> TangentArray;

    hkInplaceArray<TangentArray*, 64> allTangentArrays;

    int maximumExpectedWrappingLength = lhull.m_edges.getSize() + rhull.m_edges.getSize() + 2;

    hkpGeomConvexHullConfig config;
    config.m_maxAngle = tolerances.m__maxAngle;
    config.m_maxIterations = maximumExpectedWrappingLength * 3;

#   ifdef HK_DEBUG_CONVEX_HULL
    if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
    {
        hkprintf("\n\t left hull:\n");
        printEdges(lhull);

        hkprintf("\n\t right hull:\n");
        printEdges(rhull);
    }
#   endif

    hkInplaceArray<WrappingLine,128> wrapping;
    hkArray<WeightedLine>* currentTangentLines = &firstTangentLines;
#   ifdef HK_DEBUG_CONVEX_HULL
    if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
    {
        hkprintf("\n\n\n************ merge %i %i\n\n", lhull.m_edges.getSize(), rhull.m_edges.getSize());
    }
#   endif


    int wrappingLengthWithHighAngle = 0;
    for ( int wrappingLength = 0 ; wrappingLengthWithHighAngle < config.m_maxIterations; wrappingLength++ )
    {
        if ( wrappingLength > maximumExpectedWrappingLength )
        {
            config.m_maxAngle *= 1.1f;
            if ( config.m_maxAngle > 1.0f)
            {
                maximumExpectedWrappingLength++;
            }
        }

#       ifdef HK_DEBUG_CONVEX_HULL
        if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
        {
            hkprintf("\t\nWrappingLength %i\n\n", wrappingLength );
        }
#       endif

        {
            TangentArray* nextTangentLines = hkAllocateStack<TangentArray>(1);
            new ( nextTangentLines ) TangentArray();
            allTangentArrays.pushBack( nextTangentLines );

            for ( int tanIter = 0; tanIter < currentTangentLines->getSize() ; tanIter++ )
            {
                WeightedLine* sourceLine = &(*currentTangentLines)[tanIter];

                hkInplaceArray<WeightedNeighbour, 64> leftNeighbours;
                hkInplaceArray<WeightedNeighbour, 64> rightNeighbours;

                const hkVector4* tangentStart = sourceLine->m_leftEdge->getVertex( vertexBase );
                const hkVector4* tangentEnd = sourceLine->m_rightEdge->getVertex( vertexBase );


                //
                //  build our tangent
                //
                hkVector4 tangentPlaneEquation;
                if ( sourceLine->m_source )
                {
                    const hkVector4* a = sourceLine->m_source->m_leftEdge->getVertex( vertexBase );
                    const hkVector4* b = sourceLine->m_source->m_rightEdge->getVertex( vertexBase );
                    const hkVector4* c = ( a == tangentStart ) ? tangentEnd : tangentStart;
                    hkVector4 tangentDir; tangentDir.setSub( *b, *a); tangentDir.normalize<3>();
                    hkVector4 other; other.setSub( *c, *a ); other.normalize<3>();
                    tangentPlaneEquation.setCross( tangentDir, other );

                    hkReal planeEqnMinLength = 1e-6f;

                    if ( tangentPlaneEquation.lengthSquared<3>().getReal() < planeEqnMinLength )
                    {
                        other.setSub( *c, *b ); other.normalize<3>();
                        tangentPlaneEquation.setCross( tangentDir, other );
                        if ( tangentPlaneEquation.lengthSquared<3>().getReal() < planeEqnMinLength )
                        {

                            // find ANY plane equation that contains a and b
                            hkVector4 newOther; newOther.setAdd( tangentDir, other );
                            hkVector4 newTangentDir; newTangentDir.setSub( tangentDir, other );

                            if ( ( newTangentDir.lengthSquared<3>().getReal() < planeEqnMinLength ) ||
                                 ( newOther.lengthSquared<3>().getReal() < planeEqnMinLength ) )
                            {
                                // the lines are collinear - use the previous plane equation
                                tangentPlaneEquation = previousPlaneEquation;
                            }
                            else
                            {
                                newOther.normalize<3>();
                                newTangentDir.normalize<3>();
                                tangentPlaneEquation.setCross( newTangentDir, newOther );
                                HK_ASSERT_NO_MSG(0x123456, tangentPlaneEquation.lengthSquared<3>().getReal() > planeEqnMinLength );
                            }

                        }
                    }
                    tangentPlaneEquation.normalize<3>();
                }
                else
                {
                    tangentPlaneEquation = startPlaneEquation;
                }


                previousPlaneEquation = tangentPlaneEquation;
#           ifdef HK_DEBUG_CONVEX_HULL
                if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
                {
                    hkprintf("\n\t Starting line %i - %i (%f, %f, %f, %f) \n", sourceLine->m_leftEdge->m_vertex, sourceLine->m_rightEdge->m_vertex, tangentPlaneEquation(0), tangentPlaneEquation(1), tangentPlaneEquation(2), tangentPlaneEquation(3));
                }
#           endif

                findWeightedNeighbours( tolerances, lhull, tangentPlaneEquation, (hkUint16)sourceLine->m_lastVertex, sourceLine->m_leftEdge, tangentStart, tangentEnd, leftNeighbours );
                findWeightedNeighbours( tolerances, rhull, tangentPlaneEquation, (hkUint16)sourceLine->m_lastVertex, sourceLine->m_rightEdge, tangentStart, tangentEnd, rightNeighbours );

                // If the lowest neighbour of each of these two is identical, then and additional check can be done to remove
                // coplanarity.

                validateNeighbours( tolerances, rhull.m_vertexBase, tangentPlaneEquation, (hkUint16)sourceLine->m_lastVertex, sourceLine, tangentStart, tangentEnd, leftNeighbours, rightNeighbours );

                addWrappingLines( config, sourceLine, leftNeighbours, rightNeighbours, *nextTangentLines );
            }
            currentTangentLines = nextTangentLines;
        }

        hkSort(currentTangentLines->begin(), currentTangentLines->getSize(), weightedLineLess);

        findWrapping( lhull, rhull, *currentTangentLines, wrapping );

        if ( wrapping.getSize() != 0 )
        {
            break;
        }
    }



    //
    // Cleanup memory (in reverse order)
    //
    {
        for (int i = allTangentArrays.getSize()-1; i >=0 ; i--)
        {
            allTangentArrays[i]->~TangentArray();
            hkDeallocateStack<TangentArray>(allTangentArrays[i], 1);
        }
        allTangentArrays.clear();
    }


    if ( wrapping.getSize() != 0 )
    {
        stitchHulls( lhull, rhull, wrapping, hullOut );
        return HK_SUCCESS;
    }

    
    return HK_FAILURE;
}


hkBool hkGeomConvexHullBuilder::isSingleLine( hkReal degenerateTolerance, hkGeomHull& lhull, hkGeomHull& rhull, hkGeomHull& hullOut )
{
    hkVector4* vertexBase = lhull.m_vertexBase;

    if ( ( lhull.m_edges.getSize() < 3 ) && ( rhull.m_edges.getSize() < 3 ) )
    {
        int v1=0, v2=0, v3=0, v4=0;
        int numVerts = 3;

        if ( lhull.m_edges.getSize() == 2 )
        {
            //v1 = *lhull.m_edges[0].getVertex( vertexBase );
            //v2 = *lhull.m_edges[0].getMirror( lhull.m_edges.begin() )->getVertex( vertexBase );
            v1 = lhull.m_edges[0].m_vertex;
            v2 = lhull.m_edges[0].getMirror( lhull.m_edges.begin() )->m_vertex;

            // ensure the ordering still holds
            if ( v2 < v1 )
            {
                hkAlgorithm::swap( v1, v2 );
            }

            v3 = rhull.m_edges[0].m_vertex;

            if ( rhull.m_edges.getSize() == 2 )
            {
                v4 = rhull.m_edges[0].getMirror( rhull.m_edges.begin() )->m_vertex;

                // ensure the ordering still holds
                if ( v4 < v3 )
                {
                    hkAlgorithm::swap( v4, v3 );
                }

                numVerts = 4;
            }
        }
        else if ( rhull.m_edges.getSize() == 2 )
        {
            v1 = lhull.m_edges[0].m_vertex;
            v2 = rhull.m_edges[0].getMirror( rhull.m_edges.begin() )->m_vertex;
            v3 = rhull.m_edges[0].m_vertex;

            // ensure the ordering still holds
            if ( v2 < v1 )
            {
                hkAlgorithm::swap( v1, v2 );
            }
        }
        else
        {
            hullOut.initializeWithEdge( lhull.m_edges[0].m_vertex, rhull.m_edges[0].m_vertex );
            return true;
        }


        {
            hkSimdReal lenT1, lenT2, lenT3;
            hkVector4 t1; t1.setSub( vertexBase[v2], vertexBase[v1] );
            hkVector4 t2; t2.setSub( vertexBase[v3], vertexBase[v2] );
            hkVector4 t3; t3.setSub( vertexBase[v1], vertexBase[v3] );

            lenT1 = t1.normalizeWithLength<3>();
            lenT2 = t2.normalizeWithLength<3>();
            lenT3 = t3.normalizeWithLength<3>();


            hkVector4 difft2t1; difft2t1.setSub( t2, t1 );
            const hkSimdReal degen = hkSimdReal::fromFloat(degenerateTolerance);
            if ( difft2t1.lengthSquared<3>() < degen )
            {
                if ( numVerts == 4 )
                {
                    hkVector4 t4; t4.setSub( vertexBase[v4], vertexBase[v2] );
                    const hkSimdReal lenT4 = t4.normalizeWithLength<3>();

                    hkVector4 difft2t4; difft2t4.setSub( t2, t4 );
                    if ( difft2t4.lengthSquared<3>() < degen )
                    {
                        if ( lenT4 > lenT2 )
                        {
                            v3 = v4;
                            t2 = t4;
                            t3.setSub( vertexBase[v1], vertexBase[v4] );
                            lenT3 = t3.lengthSquared<3>();
                        }
                    }
                    else
                    {
                        return false;
                    }
                }

                // The points are collinear, so
                // determine which points are the endpoints.

                hkSimdReal maxT2T3; maxT2T3.setMax( lenT2, lenT3 );
                hkSimdReal maxLen; maxLen.setMax( lenT1, maxT2T3 );

                if ( lenT1.isEqual(maxLen) )
                {
                    hullOut.initializeWithEdge( v1, v2 );
                }
                else if ( lenT2.isEqual(maxLen) )
                {
                    hullOut.initializeWithEdge( v2, v3 );
                }
                else if ( lenT3.isEqual(maxLen) )
                {
                    hullOut.initializeWithEdge( v3, v1 );
                }
            }
            else
            {
                return false;
            }
        }
    }
    else
    {
        return false;
    }

    return true;
}


void hkGeomConvexHullBuilder::removeFlaggedVertices( hkArray<hkVector4>& vertices )
{
    hkVector4* s = vertices.begin();    // source pointer
    hkVector4* d = vertices.begin();    // dest point

    for ( int i = vertices.getSize()-1 ; i>=0; i-- )
    {
        if ( s->getComponent<3>().isEqualZero() )
        {
            *(d++) = *s;
        }
        s++;
    }
    vertices.setSize( static_cast<int>( d - vertices.begin() ) );
}


void hkGeomConvexHullBuilder::removeUnusedVertices( hkGeomHull& hull, hkArray<hkVector4>& vertices )
{
    // set the 'w' component if the vertex is used in the hull
    {
        for ( int ie = 0 ; ie < hull.m_edges.getSize() ; ie++ )
        {
            int vertexIndex = hull.m_edges[ie].m_vertex;
            vertices[ vertexIndex ].setW(hkSimdReal_2);
        }
    }


    // compress the vertex data and reindex the edges
    {
        hkLocalArray<int> verticesReindexMap(vertices.getSize());
        verticesReindexMap.setSizeUnchecked( vertices.getSize() );

        int d = 0;
        for ( int s = 0; s < vertices.getSize(); s++ )
        {
            const hkVector4& test = vertices[s];
            if ( test.getW().isEqual(hkSimdReal_2) )
            {
                vertices[d] = test;
                verticesReindexMap[s] = d;
                d++;
            }
            else
            {
                verticesReindexMap[s] = -1;
            }
        }
        vertices.setSize(d);

        // reindex the edges
        for ( int ie = 0 ; ie < hull.m_edges.getSize() ; ie++ )
        {
            hkGeomEdge& edge = hull.m_edges[ie];
            edge.m_vertex = hkUint16(verticesReindexMap[ edge.m_vertex ]);
        }
    }


    // reset the 'w' component to 0
    {
        for ( int ie = 0 ; ie < hull.m_edges.getSize() ; ie++ )
        {
            int vertexIndex = hull.m_edges[ie].m_vertex;
            vertices[ vertexIndex ].zeroComponent<3>();
        }
    }
}


void hkGeomConvexHullBuilder::appendAndReindexEdges( hkGeomHull& hull, hkBool usingLeftWrapping, hkGeomHull& hullOut, hkArray<hkUint16>& hullReindexMap )
{
    hullReindexMap.setSize( hull.m_edges.getSize() );
    int startOfHullOut = hullOut.m_edges.getSize();

    for ( int i = 0 ; i < hull.m_edges.getSize() ; i++ )
    {
        hkGeomEdge& edge = hull.m_edges[i];
        if ( edge.m_info == VISITED_VISIBLE || edge.m_info == VISITED_BOUNDARY )
        {
            hullReindexMap[i] = hkUint16(hullOut.m_edges.getSize());
            hullOut.m_edges.pushBack( edge );
        }
        else
        {
            hullReindexMap[i] = 0xffff;
        }
    }

    {
        for ( int ie = startOfHullOut ; ie < hullOut.m_edges.getSize() ; ie++ )
        {
            hkGeomEdge& edge = hullOut.m_edges[ie];
            edge.m_mirror = hullReindexMap[ edge.m_mirror ];
            edge.m_next = hullReindexMap[ edge.m_next ];
        }
    }
}



void hkGeomConvexHullBuilder::stitchHulls( hkGeomHull& lhull, hkGeomHull& rhull, hkArray<WrappingLine>& wrapping, hkGeomHull& hullOut )
{
    hullOut.m_edges.clear();

    hkLocalArray<hkUint16> lhullReindexMap( lhull.m_edges.getSize() );
    appendAndReindexEdges( lhull, true, hullOut, lhullReindexMap );

    hkLocalArray<hkUint16> rhullReindexMap( rhull.m_edges.getSize() );
    appendAndReindexEdges( rhull, false, hullOut, rhullReindexMap );

    hullOut.m_edges.reserveExactly( hullOut.m_edges.getSize() + 3 * wrapping.getSize() );

    //
    //  Relink and left mirror the wrapping
    //
    {
        for (int i = 0; i < wrapping.getSize(); i++)
        {
            WrappingLine& w = wrapping[i];

            w.m_leftVertex = w.m_leftEdge->m_vertex;
            w.m_leftNextEdgeVertex = w.m_leftEdge->getNext( lhull.m_edges.begin() )->m_vertex;

            // because we mirror it within the next code block
            //w.m_rightVertex = w.m_rightEdge->getNext( rhull.m_edges.begin() )->m_vertex;
            w.m_rightNextEdgeVertex = w.m_rightEdge->m_vertex;

            int leftIndex = static_cast<int>( w.m_leftEdge - lhull.m_edges.begin() );
            hkGeomEdge* rightEdgeBase = rhull.m_edges.begin();
            int rightIndex = static_cast<int>( w.m_rightEdge->getMirror( rightEdgeBase ) - rightEdgeBase);

            int newLeftIndex = lhullReindexMap[ leftIndex ];
            w.m_leftEdge = ( newLeftIndex == 0xffff ) ? HK_NULL : &hullOut.m_edges[ newLeftIndex ];

            int newRightIndex = rhullReindexMap[ rightIndex ];
            w.m_rightEdge = ( newRightIndex == 0xffff ) ? HK_NULL : &hullOut.m_edges[ newRightIndex ];
        }
    }

    {
        hkGeomEdge* edgeBase = hullOut.m_edges.begin();

        WrappingLine* lastLine = &wrapping[0];
        hkGeomEdge* lastEdge = HK_NULL;
        int lastEdgeIndex = 0xffff;
        int firstEdgeIndex = hullOut.m_edges.getSize();

        // run through the wrappings in the order they have been pushed back
        for (int i = wrapping.getSize()-1; i >=0 ; i--)
        {

            WrappingLine& w = wrapping[i];

            int e0Index = hullOut.m_edges.getSize();
            hkGeomEdge* e0 = hullOut.m_edges.expandByUnchecked(1);

            int e2Index = hullOut.m_edges.getSize();
            hkGeomEdge* e2 = hullOut.m_edges.expandByUnchecked(1);

            if ( w.m_leftEdge == lastLine->m_leftEdge )
            {
                hkBool rightSideIsLine = w.m_rightEdge->m_info == VISITED_BOUNDARY;
                int e1Index    = (rightSideIsLine) ? w.m_rightEdge->m_mirror           : hullOut.m_edges.getSize();
                hkGeomEdge* e1 = (rightSideIsLine) ? w.m_rightEdge->getMirror(edgeBase): hullOut.m_edges.expandByUnchecked(1);

                e0->m_vertex = w.m_rightEdge->m_vertex;
                e0->m_next = hkInt16(e2Index);
                e0->m_mirror = hkInt16(lastEdgeIndex);

                if ( lastEdge )
                {
                    lastEdge->m_mirror = hkInt16(e0Index);
                }

                e1->m_vertex = w.m_rightNextEdgeVertex;
                e1->m_next = hkInt16(e0Index);
                e1->m_mirror = hkInt16(w.m_rightEdge - edgeBase);
                w.m_rightEdge->m_mirror = hkInt16(e1Index);

                e2->m_vertex = w.m_leftVertex;
                e2->m_next = hkInt16(e1Index);
            }
            else
            {
                hkBool leftSideIsLine = w.m_leftEdge->m_info == VISITED_BOUNDARY;
                int e1Index    = (leftSideIsLine) ? w.m_leftEdge->m_mirror           : hullOut.m_edges.getSize();
                hkGeomEdge* e1 = (leftSideIsLine) ? w.m_leftEdge->getMirror(edgeBase): hullOut.m_edges.expandByUnchecked(1);
                HK_ASSERT_NO_MSG(0x68ed085c, w.m_rightEdge == lastLine->m_rightEdge );

                e0->m_vertex = w.m_rightNextEdgeVertex;
                e0->m_next = hkInt16(e1Index);
                e0->m_mirror = hkInt16( lastEdgeIndex );

                if ( lastEdge )
                {
                    lastEdge->m_mirror = hkInt16(e0Index);
                }

                e1->m_vertex = w.m_leftNextEdgeVertex;
                e1->m_next = hkInt16(e2Index);
                e1->m_mirror = hkInt16( w.m_leftEdge - edgeBase );
                w.m_leftEdge->m_mirror = hkInt16(e1Index);

                e2->m_vertex = w.m_leftEdge->m_vertex;
                e2->m_next = hkInt16(e0Index);
            }
            lastEdgeIndex = e2Index;
            lastEdge = e2;
            lastLine = &w;
        }

        lastEdge->m_mirror = hkInt16(firstEdgeIndex);
        edgeBase[ firstEdgeIndex ].m_mirror = hkInt16(lastEdgeIndex);
    }
}


// get the plane equation parallel to the z-axis
void hkGeomConvexHullBuilder::getPlaneEquationZaxis( const hkVector4& a, const hkVector4& b, hkVector4& planeEquationOut)
{
    hkVector4 planeGeneratorTolerance; planeGeneratorTolerance.setAll(1e-6f);
    hkVector4 aTob; aTob.setSub( a, b );

    // This should obviously be true because weldXSortedVertices has been called already.
    HK_ASSERT_NO_MSG(0x3f310cd5, aTob.lengthSquared<3>().getReal() > HK_REAL_EPSILON );

    // if the vectors are equal in the xy-plane, then return
    // a plane parallel to the x-axis.
    hkVector4 absAB; absAB.setAbs(aTob);
    if (absAB.less(planeGeneratorTolerance).allAreSet<hkVector4ComparisonMask::MASK_XY>())
    {
        //planeEquationOut.set( 0.0f, -1.0f, 0.0f );
        planeEquationOut.setCross( aTob, hkVector4::getConstant<HK_QUADREAL_1000>() );
        planeEquationOut.normalize<3>();
    }
    else
    {
        aTob.normalize<3>();
        planeEquationOut.setCross( aTob, hkVector4::getConstant<HK_QUADREAL_0100>() );

        if ( planeEquationOut.lengthSquared<3>().isLess(hkSimdReal_Eps) )
        {
            planeEquationOut.setCross( aTob, hkVector4::getConstant<HK_QUADREAL_0010>() );
        }

        planeEquationOut.normalize<3>();
    }
}


void hkGeomConvexHullBuilder::getCommonTangent( hkGeomHull& lhull, hkGeomHull& rhull, WeightedLine& weightedLineOut, hkVector4& tangentPlaneEquationOut)
{
    const hkVector4* vertexBase = lhull.m_vertexBase;
    HK_ASSERT_NO_MSG(0x3f310cb5, lhull.m_vertexBase == rhull.m_vertexBase );

    weightedLineOut.m_leftEdge =  &(lhull.m_edges[0]);
    weightedLineOut.m_rightEdge = &(rhull.m_edges[0]);
    weightedLineOut.m_source = HK_NULL;
    weightedLineOut.m_weight = 0;
    weightedLineOut.m_lastVertex = unsigned(-1);

    // get the plane equation of a plane containing lvert and rvert, parallel to the z-axis
    const hkVector4* lhullTangentPoint = weightedLineOut.m_leftEdge->getVertex( vertexBase );
    const hkVector4* rhullTangentPoint = weightedLineOut.m_rightEdge->getVertex( vertexBase );

    getPlaneEquationZaxis( *lhullTangentPoint, *rhullTangentPoint, tangentPlaneEquationOut );

    hkBool changed;
    int maxIterations = 2*(lhull.m_edges.getSize() + rhull.m_edges.getSize());

    for (int iter = 0; iter < maxIterations; iter++)
    {
        changed = false;

        // lhs
        {
            for( int i = lhull.m_edges.getSize() -1 ; i >= 0; i--)
            {
                // check if lhull.verts[i] is inside the lct
                hkGeomEdge* edge = &lhull.m_edges[i];
                const hkVector4* newVertex = edge->getVertex( vertexBase );
                hkVector4 delta; delta.setSub( *newVertex,  *lhullTangentPoint );
                hkReal dist = tangentPlaneEquationOut.dot<3>( delta ).getReal();
                if ( dist > 1e-7f ) {
                    weightedLineOut.m_leftEdge = edge;
                    lhullTangentPoint = newVertex;
                    getPlaneEquationZaxis( *lhullTangentPoint, *rhullTangentPoint, tangentPlaneEquationOut );
                    changed = true;
                }
            }
        }
        {
            for( int i = rhull.m_edges.getSize()-1 ; i >= 0; i--)
            {
                hkGeomEdge* edge = &rhull.m_edges[i];
                const hkVector4* newVertex = edge->getVertex( vertexBase );
                hkVector4 delta; delta.setSub( *newVertex,  *rhullTangentPoint );
                hkReal dist = tangentPlaneEquationOut.dot<3>( delta ).getReal();
                if ( dist > 1e-7f ) {
                    weightedLineOut.m_rightEdge = edge;
                    rhullTangentPoint = newVertex;
                    getPlaneEquationZaxis( *lhullTangentPoint, *rhullTangentPoint, tangentPlaneEquationOut );
                    changed = true;
                }
            }
        }
        if(!changed)
        {
            break;
        }
        getPlaneEquationZaxis( *lhullTangentPoint, *rhullTangentPoint, tangentPlaneEquationOut );
    }
}



void hkGeomConvexHullBuilder::findWeightedNeighbours( const hkGeomConvexHullTolerances&  tolerances, hkGeomHull& hull, const hkVector4& tangentPlaneEquation, const hkUint16 lastVertex, const hkGeomEdge* startPoint, const hkVector4* tangentStart, const hkVector4* tangentEnd, hkArray<WeightedNeighbour>& neighboursOut )
{
#   ifdef HK_DEBUG_CONVEX_HULL
    if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
    {
        hkprintf("\t weights of neighbours of %i -->\n", startPoint->m_vertex);
    }
#   endif

    if ( hull.m_edges.getSize() == 1 ) { return; }

    // Traverse through the set of neighbours of startPoint.
    hkGeomEdge* edgeBase = hull.m_edges.begin();
    const hkVector4* vertexBase = hull.m_vertexBase;

    hkGeomEdge* startNeighbour = startPoint->getMirror( edgeBase );
    hkGeomEdge* currentNeighbour = startNeighbour;

    do {
        const hkVector4& vertex = *(currentNeighbour->getVertex( vertexBase ));

        hkReal angleI;
        if ( currentNeighbour->m_vertex == lastVertex )
        {
            angleI = 4.0f;
        }
        else
        {
            angleI = getAngleBetweenVertexAndPlane( tolerances, vertex, tangentPlaneEquation, *tangentStart, *tangentEnd );
        }

        WeightedNeighbour& wn = *neighboursOut.expandBy(1);
        wn.m_edge = currentNeighbour;
        wn.m_weight = angleI;

#       ifdef HK_DEBUG_CONVEX_HULL
        if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
        {
            hkprintf( "\t\t angle with %i: %f\n", currentNeighbour->m_vertex, angleI );
        }
#       endif

        currentNeighbour = currentNeighbour->getNext( edgeBase )->getMirror( edgeBase );

    } while( currentNeighbour != startNeighbour );


    hkSort(neighboursOut.begin(), neighboursOut.getSize(), weightedNeighbourLess);

    removeCoPlanarNeighbours( tolerances, hull, tangentPlaneEquation, startPoint, tangentStart, tangentEnd, neighboursOut );
}


void hkGeomConvexHullBuilder::removeCoPlanarNeighbours( const hkGeomConvexHullTolerances& tolerances, hkGeomHull& hull, const hkVector4& tangentPlaneEquation, const hkGeomEdge* startPoint, const hkVector4* tangentStart, const hkVector4* tangentEnd, hkArray<WeightedNeighbour>& neighboursOut )
{
    hkVector4* vertexBase = hull.m_vertexBase;

    hkReal epsilon = tolerances.m_coplanar_vertices_tolerance;

    // if the top few neighbours have the same weight, then they are coplanar.
    if ( ( neighboursOut.getSize() > 1 ) && ( ( neighboursOut[1].m_weight - neighboursOut[0].m_weight ) < epsilon ) )
    {
        hkInplaceArray<WeightedNeighbour, 64> coplanarNeighbours;
        hkReal lowestWeight = neighboursOut[0].m_weight;
        {
            for ( int i = 0 ; i < neighboursOut.getSize(); i++ )
            {
                if ( (neighboursOut[i].m_weight - lowestWeight) > epsilon ) { break; }
                coplanarNeighbours.pushBack( neighboursOut[i] );
            }
        }

        calculateNewNeighbours( vertexBase, tolerances, lowestWeight, tangentPlaneEquation, startPoint, tangentStart, tangentEnd, coplanarNeighbours );


#       ifdef HK_DEBUG_CONVEX_HULL
        if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
        {
            hkprintf( "\t\t replacing neighbours to %i: %f with %f\n", coplanarNeighbours[0].m_edge->m_vertex, coplanarNeighbours[0].m_weight, lowestWeight );
        }
#       endif

        neighboursOut[0] = coplanarNeighbours[0];
        // reset the weight
        neighboursOut[0].m_weight = lowestWeight;
        neighboursOut.setSize(1);

    }
}


void hkGeomConvexHullBuilder::calculateNewNeighbours( const hkVector4* vertexBase, const hkGeomConvexHullTolerances& tolerances, hkReal lowestWeight,
                            const hkVector4& tangentPlaneEquation, const hkGeomEdge* startPoint, const hkVector4* tangentStart, const hkVector4* tangentEnd, hkArray<WeightedNeighbour>& neighboursOut )
{
    // we want to create new weights for these neighbors, which are calculated by rotation the
    // tangentPlaneEquation about a new axis.
    // This new axis is

    hkReal epsilon = tolerances.m_coplanar_vertices_tolerance;
    {
        const hkVector4* startVertex = startPoint->getVertex( vertexBase );
        hkVector4 tangentEquation( tangentPlaneEquation );

        hkVector4 newTangentStart, newTangentEnd;
        {
            hkVector4 newOtherTangentPoint; newOtherTangentPoint.setSub( *tangentStart, *tangentEnd );

            // change the tangentPlaneEquation in the special case where the points are on the tangent
            if (( lowestWeight < epsilon || lowestWeight > 8 - epsilon ) )//|| ( hkMath::fabs( lowestWeight - 4 ) < epsilon ))
            {
                tangentEquation.setCross( tangentEquation, newOtherTangentPoint );
                tangentEquation.normalize<3>();
            }
            else if ( hkMath::fabs( lowestWeight - 4 ) < epsilon )
            {
                tangentEquation.setCross( newOtherTangentPoint, tangentEquation);
                tangentEquation.normalize<3>();
            }


            newOtherTangentPoint.setCross( tangentEquation, newOtherTangentPoint );
            newOtherTangentPoint.add( *startVertex );

            if ( startVertex == tangentStart )
            {
                newTangentStart = newOtherTangentPoint;
            }
            else
            {
                newTangentStart = *startVertex;
            }

            if ( startVertex == tangentStart )
            {
                newTangentEnd = *startVertex;
            }
            else
            {
                newTangentEnd = newOtherTangentPoint;
            }
        }

        //hkVector4 otherTangentPoint = ( startVertex == tangentStart ) ? *tangentEnd : *tangentStart;

        // calculate the weights using this new pivot axis
        for ( int j = 0 ; j < neighboursOut.getSize() ; j++ )
        {
            const hkVector4& vertex = *(neighboursOut[j].m_edge->getVertex( vertexBase ));

            hkReal angleI = getAngleBetweenVertexAndPlane( tolerances, vertex, tangentEquation, newTangentStart, newTangentEnd );

            neighboursOut[j].m_weight = angleI;
        }
    }

    hkSort(neighboursOut.begin(), neighboursOut.getSize(), weightedNeighbourLess);
}


void hkGeomConvexHullBuilder::validateNeighbours( const hkGeomConvexHullTolerances& tolerances, const hkVector4* vertexBase, const hkVector4& tangentPlaneEquation, const hkUint16 lastVertex, WeightedLine* sourceLine, const hkVector4* tangentStart, const hkVector4* tangentEnd, hkArray<WeightedNeighbour>& leftNeighboursOut, hkArray<WeightedNeighbour>& rightNeighboursOut )
{
    // if the lowest weight of neighbours on each hull is identical, then we should try doing the same trick
    // as above to predict which neighbour is more likely to be the one we actually want.
    // What we will do is to cheat - since the ordering of each is checked separately, we need to specify explicitly
    // that the neighbour on the side that is lowest is lower than the other one.
    // The best way to do this could be to set the weight of this neighbour to be half of it's original value.
    


    if ( ( leftNeighboursOut.getSize() == 1 )&& ( rightNeighboursOut.getSize() == 1 ) )
    {
        // We want to keep rotating around the same vertex (why?)
        hkGeomEdge* startPoint = sourceLine->m_leftEdge;
        if ( sourceLine->m_source != HK_NULL )
        {
            if ( ( lastVertex != hkUint16(-1) ) && ( sourceLine->m_source->m_rightEdge->m_vertex == lastVertex ) )
            {
                startPoint = sourceLine->m_rightEdge;
            }
        }

        hkReal epsilon = tolerances.m_coplanar_vertices_tolerance;

        // if first neighbours of each side are close, then they are coplanar.
        if ( hkMath::fabs( leftNeighboursOut[0].m_weight - rightNeighboursOut[0].m_weight ) < epsilon )
        {
            hkInplaceArray<WeightedNeighbour, 2> coplanarNeighbours;
            hkReal lowestWeight = rightNeighboursOut[0].m_weight;

            coplanarNeighbours.pushBack( leftNeighboursOut[0] );
            coplanarNeighbours.pushBack( rightNeighboursOut[0] );

            calculateNewNeighbours( vertexBase, tolerances, lowestWeight, tangentPlaneEquation, startPoint, tangentStart, tangentEnd, coplanarNeighbours );

            // Find which side the lower neighbour is...

            if ( coplanarNeighbours[0].m_edge == leftNeighboursOut[0].m_edge )
            {
                // and set it's weight to be half the previous value.
                leftNeighboursOut[0].m_weight *= 0.5f;

#       ifdef HK_DEBUG_CONVEX_HULL
                if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
                {
                    hkprintf( "\t\t modifying weight of left neighbour %i from %f\n", leftNeighboursOut[0].m_edge->m_vertex, leftNeighboursOut[0].m_weight );
                }
#       endif

            }
            else
            {
                HK_ASSERT_NO_MSG(0x704c7f1f, coplanarNeighbours[0].m_edge == rightNeighboursOut[0].m_edge );
                rightNeighboursOut[0].m_weight *= 0.5f;
#       ifdef HK_DEBUG_CONVEX_HULL
                if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
                {
                    hkprintf( "\t\t modifying weight of right neighbour %i from %f\n", rightNeighboursOut[0].m_edge->m_vertex, leftNeighboursOut[0].m_weight );
                }
#       endif

            }
        }
    }
}


void hkGeomConvexHullBuilder::addWrappingLines( const hkpGeomConvexHullConfig& config, WeightedLine* sourceLine, hkArray<WeightedNeighbour>& leftN, hkArray<WeightedNeighbour>& rightN, hkArray<WeightedLine>& wrappingLinesOut )
{
    hkReal minAngle;
    if ( leftN.getSize() == 0 )
    {
        minAngle = rightN[0].m_weight;
    }
    else if ( rightN.getSize() == 0 )
    {
        minAngle = leftN[0].m_weight;
    }
    else
    {
        minAngle = hkMath::min2( leftN[0].m_weight, rightN[0].m_weight );
    }

    {
        hkGeomEdge* rightEdge = sourceLine->m_rightEdge;

        for ( int il = 0 ; il < leftN.getSize() ; il++)
        {
            hkReal diffAngle = ( leftN[il].m_weight ) - minAngle;

            if ( diffAngle > config.m_maxAngle )
            {
                break;
            }

            WeightedLine wrappingLine;
            wrappingLine.m_weight = diffAngle + sourceLine->m_weight;
            wrappingLine.m_leftEdge = leftN[il].m_edge;
            wrappingLine.m_rightEdge = rightEdge;
            wrappingLine.m_source = sourceLine;
            wrappingLine.m_lastVertex = sourceLine->m_leftEdge->m_vertex;

            hkBool foundWrappingLine = false;
            for ( int iw = 0 ; iw < wrappingLinesOut.getSize() ; iw++ )
            {
                if ( wrappingLinesOut[iw].vertsAndVertexEqual(wrappingLine ) )
                {
                    if ( wrappingLinesOut[iw].m_weight < wrappingLine.m_weight )
                    {
                        wrappingLinesOut[iw] = wrappingLine;
                    }
                    foundWrappingLine = true;
                    break;
                }
            }

            if ( ! foundWrappingLine )
            {
                wrappingLinesOut.pushBack( wrappingLine );
            }
        }
    }
    {
        hkGeomEdge* leftEdge = sourceLine->m_leftEdge;
        for ( int ir = 0 ; ir < rightN.getSize() ; ir++)
        {
            hkReal diffAngle = ( rightN[ir].m_weight ) - minAngle;

            if ( diffAngle > config.m_maxAngle )
            {
                break;
            }
            WeightedLine wrappingLine;
            wrappingLine.m_weight = diffAngle + sourceLine->m_weight;
            wrappingLine.m_leftEdge = leftEdge;
            wrappingLine.m_rightEdge = rightN[ir].m_edge;
            wrappingLine.m_source = sourceLine;
            wrappingLine.m_lastVertex = sourceLine->m_rightEdge->m_vertex;

            hkBool foundWrappingLine = false;
            for ( int iw = 0 ; iw < wrappingLinesOut.getSize() ; iw++ )
            {
                if ( wrappingLinesOut[iw].vertsAndVertexEqual(wrappingLine ) )
                {
                    if ( wrappingLinesOut[iw].m_weight < wrappingLine.m_weight )
                    {
                        wrappingLinesOut[iw] = wrappingLine;
                    }
                    foundWrappingLine = true;
                    break;
                }
            }

            if ( ! foundWrappingLine )
            {
                wrappingLinesOut.pushBack( wrappingLine );
            }
        }
    }
}

void hkGeomConvexHullBuilder::findWrapping( hkGeomHull& lhull, hkGeomHull& rhull, const hkArray<WeightedLine>& lastTangents, hkArray<WrappingLine>& wrappingOut )
{
    // For each tangent, we check whether we can find a closed loop
    for ( int lti = 0 ; lti < lastTangents.getSize() ; lti++ )
    {
        wrappingOut.clear();
        const WeightedLine* lastTangent = &lastTangents[lti];

#       ifdef HK_DEBUG_CONVEX_HULL
            if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
            {
                hkprintf("\t\tStartingFrom %i %i (%i)  e%i e%i [%f]\n", lastTangent->m_leftEdge->m_vertex, lastTangent->m_rightEdge->m_vertex, lastTangent->m_lastVertex,lastTangent->m_leftEdge - lhull.m_edges.begin(), lastTangent->m_rightEdge - rhull.m_edges.begin(), lastTangent->m_weight );
            }
#       endif
        {
            WrappingLine& wl = *wrappingOut.expandBy(1);
            wl.m_leftEdge = lastTangent->m_leftEdge;
            wl.m_rightEdge = lastTangent->m_rightEdge;
        }

        for ( WeightedLine* currentSource = lastTangent->m_source ; currentSource != HK_NULL ; currentSource = currentSource->m_source )
        {
#           ifdef HK_DEBUG_CONVEX_HULL
            if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
            {
                hkprintf("\t\t\tCheck %i %i (%i)  e%i e%i [%i]\n", currentSource->m_leftEdge->m_vertex, currentSource->m_rightEdge->m_vertex, currentSource->m_lastVertex, currentSource->m_leftEdge - lhull.m_edges.begin(), currentSource->m_rightEdge - rhull.m_edges.begin(), currentSource->m_weight );
            }
#           endif
            if ( currentSource->edgesAndVertexEqual( *lastTangent ))
            {
                // we've found a wrapping
                hkBool leftIsValid = isValidWrapping( lhull, rhull.m_edges.begin(), true, wrappingOut );
                hkBool rightIsValid = isValidWrapping( rhull, lhull.m_edges.begin(), false, wrappingOut );
#               ifdef HK_DEBUG_CONVEX_HULL
                if ( HK_DEBUG_CONVEX_HULL_USEDEBUG )
                {
                    hkprintf("\t\t\tValidate %i %i\n", leftIsValid, rightIsValid );
                }
#               endif
                if ( leftIsValid && rightIsValid )
                {
                    return;
                }
            }
            WrappingLine& wl = *wrappingOut.expandBy(1);
            wl.m_leftEdge = currentSource->m_leftEdge;
            wl.m_rightEdge = currentSource->m_rightEdge;
        }
    }

    wrappingOut.clear();
    return;
}


//
//      Flood fill the hull (which is NOT_VISITED) with edgeInfo
//      If there is a conflict return false
//      If edges are VISITED_BOUNDARY and edgeInfo == VISITED_VISIBLE set them to VISITED_VISIBLE
//
hkBool hkGeomConvexHullBuilder::isValidEdgeCheck( hkGeomHull& hull, hkGeomEdge* startEdge, int edgeInfo )
{
    hkGeomEdge* edgeBase = hull.m_edges.begin();

    hkInplaceArray<hkGeomEdge*, 128> openEdges;
    openEdges.pushBackUnchecked( startEdge );

    while ( openEdges.getSize())
    {
        hkGeomEdge* edge = openEdges.end()[-1];
        openEdges.popBack();

        //
        //  Run through all 3 edges
        //
        {
            hkGeomEdge* nextEdge = edge;
            do
            {
                nextEdge = nextEdge->getNext( edgeBase );
                switch ( nextEdge->m_info )
                {
                case VISITED_BOUNDARY:
                    if ( edgeInfo == VISITED_HIDDEN )
                    {
                        continue;
                    }
                    break;
                case NOT_VISITED:
                    break;
                case VISITED_VISIBLE:
                case VISITED_HIDDEN:
                    if ( nextEdge->m_info  != edgeInfo )
                    {
                        return false;
                    }
                }
                nextEdge->m_info = edgeInfo;
            }
            while ( nextEdge != edge);
        }

        //
        //  Recurse the 3 mirrors
        //
        {
            hkGeomEdge* nextEdge = edge;
            do
            {
                nextEdge = nextEdge->getNext( edgeBase );
                hkGeomEdge* mirrorEdge = nextEdge->getMirror( edgeBase );
                if ( mirrorEdge->m_info == NOT_VISITED )
                {
                    openEdges.pushBack( mirrorEdge );
                }
            }
            while ( nextEdge != edge);
        }
    }
    return true;
}

//
// A wrapping is considered _valid_ iff
//
//
//  Rules:
//      - every edge is visited at max once
//      - HIDDEN and VISIBLE are read only (don't touch)
//      - BOUNDARIES are only attached to HIDDED areas (that means VISIBLE overrides BOUNDARY)
//

hkBool hkGeomConvexHullBuilder::isValidWrapping( hkGeomHull& hull, hkGeomEdge* otherEdgeBase, hkBool usingLeftWrapping, hkArray<WrappingLine>& wrapping )
{
    hkGeomEdge* edgeBase = hull.m_edges.begin();

    // clear the info about having visited edges
    {
        for ( int ihe = 0 ; ihe < hull.m_edges.getSize(); ihe++ )
        {
            hull.m_edges[ihe].m_info = NOT_VISITED;
        }
    }

    //
    //  First set the mirrored edges to be hidden and check for duplicated entries
    //
    {
        int edgeCount = 0;
        hkGeomEdge* lastEdge = ( usingLeftWrapping ) ? wrapping[0].m_leftEdge : wrapping[0].m_rightEdge;
        for ( int iw = wrapping.getSize()-1 ; iw >=0  ; iw-- )
        {
            hkGeomEdge* edge = ( usingLeftWrapping ) ? wrapping[iw].m_leftEdge : wrapping[iw].m_rightEdge;
            if ( edge == lastEdge )     {   continue;   }
            lastEdge = edge;
            edgeCount++;

            // the edgeToDelete is the edge we need to delete
            hkGeomEdge* edgeToDelete = ( !usingLeftWrapping ) ? edge : edge->getMirror( edgeBase );
            if ( edgeToDelete->m_info  == VISITED_HIDDEN)
            {
                return false;
            }
            edgeToDelete->m_info = VISITED_HIDDEN;
        }

        if ( edgeCount == 0)    // we only have a single vertex, no edges used
        {
            return true;
        }
    }

    //
    //  Set everything to be visible or boundary
    //
    {
        hkGeomEdge* lastEdge = ( usingLeftWrapping ) ? wrapping[0].m_leftEdge : wrapping[0].m_rightEdge;
        for ( int iw = wrapping.getSize()-1; iw >= 0; iw-- )
        {
            hkGeomEdge* edge = ( usingLeftWrapping ) ? wrapping[iw].m_leftEdge : wrapping[iw].m_rightEdge;
            if ( edge == lastEdge )     {   continue;   }
            lastEdge = edge;

            hkGeomEdge* edgeToKeep = ( usingLeftWrapping ) ? edge : edge->getMirror( edgeBase );

            switch ( edgeToKeep->m_info )
            {
                case VISITED_VISIBLE:
                    return false;
                case NOT_VISITED:
                    edgeToKeep->m_info = VISITED_VISIBLE;
                    break;
                case VISITED_HIDDEN:
                    edgeToKeep->m_info = VISITED_BOUNDARY;
            }
        }
    }
    hkBool isValidTopology = true;

    //
    // start the flood fill on each side of the edges for clear hidden/visible
    //
    {
        hkGeomEdge* lastEdge = ( usingLeftWrapping ) ? wrapping[0].m_leftEdge : wrapping[0].m_rightEdge;
        for ( int iw = wrapping.getSize()-1; iw >= 0; iw-- )
        {
            hkGeomEdge* edge = ( usingLeftWrapping ) ? wrapping[iw].m_leftEdge : wrapping[iw].m_rightEdge;
            if ( edge == lastEdge ){    continue;   }
            lastEdge = edge;


            //
            //  Flood fill front side and set it to VISITED_VISIBLE (can override VISITED_BOUNDARY flags)
            //
            hkGeomEdge* edgeToKeep = ( usingLeftWrapping ) ? edge : edge->getMirror( edgeBase );
            if ( edgeToKeep->m_info == VISITED_VISIBLE  )
            {
                isValidTopology = isValidTopology && isValidEdgeCheck( hull, edgeToKeep, VISITED_VISIBLE );
            }

            //
            //  Flood fill back side (except VISITED_BOUNDARY) and set it to VISITED_HIDDEN
            //
            hkGeomEdge* edgeToDelete = edgeToKeep->getMirror( edgeBase );
            if ( edgeToDelete->m_info == VISITED_HIDDEN )
            {
                isValidTopology = isValidTopology && isValidEdgeCheck( hull, edgeToDelete, VISITED_HIDDEN );
            }
        }
    }

    //
    // flood fill boundary edges
    //
    {
        hkGeomEdge* lastEdge = ( usingLeftWrapping ) ? wrapping[0].m_leftEdge : wrapping[0].m_rightEdge;
        for ( int iw = wrapping.getSize()-1; iw >= 0; iw-- )
        {
            hkGeomEdge* edge = ( usingLeftWrapping ) ? wrapping[iw].m_leftEdge : wrapping[iw].m_rightEdge;
            if ( edge == lastEdge ){    continue;   }
            lastEdge = edge;

            hkGeomEdge* edgeToKeep = ( usingLeftWrapping ) ? edge : edge->getMirror( edgeBase );

            //
            //  Assume that remaining boundaries are connected to hidden areas,
            //  so flood fill all boundaries with the hidden flag. This does not override the
            //  boundary flag
            //
            if ( edgeToKeep->m_info == VISITED_BOUNDARY )
            {
                // If there are any other edges that aren't boundary edges, then
                // the other flood fills will take care of the remaining edges.

                hkBool onlyBoundaryEdgesInWrapping = true;
                hkGeomEdge* jEdge;
                {
                    hkGeomEdge* jlastEdge = ( usingLeftWrapping ) ? wrapping[0].m_leftEdge : wrapping[0].m_rightEdge;
                    for ( int jw = wrapping.getSize()-1; jw >= 0; jw-- )
                    {
                        jEdge = ( usingLeftWrapping ) ? wrapping[jw].m_leftEdge : wrapping[jw].m_rightEdge;
                        if ( jEdge == jlastEdge ){  continue;   }
                        jlastEdge = jEdge;

                        if ( jEdge->m_info != VISITED_BOUNDARY )
                        {
                            onlyBoundaryEdgesInWrapping = false;
                            break;
                        }
                    }
                }

                hkBool otherUnvisitedEdges = false;
                {
                    for ( int ei = 0 ; ei < hull.m_edges.getSize() ; ei++)
                    {
                        if( hull.m_edges[ ei ].m_info == NOT_VISITED )
                        {
                            otherUnvisitedEdges = true;
                            break;
                        }
                    }
                }

                //  ... otherwise, we have a problem.
                //  All other edges are visible or invisible, but we don't know which.
                //
                if ( onlyBoundaryEdgesInWrapping && otherUnvisitedEdges )
                {
                    // the flood-fill will always be hidden, since the case where the shape
                    // is flat and every other vertex is outside never occurs.
                    isValidTopology = isValidTopology && isValidEdgeCheck( hull, edgeToKeep, VISITED_HIDDEN );
                }
            }
        }
    }

    //
    // check that every point has been visited.
    //
    {
        for ( int ihe = 0 ; ihe < hull.m_edges.getSize(); ihe++ )
        {
            isValidTopology = isValidTopology && ( hull.m_edges[ihe].m_info != NOT_VISITED );
        }
    }

    return isValidTopology;

}


//
// Returns the angle that the plane given by planeEquation has to rotate
// (about the axes given by points 'tangentStart' and 'tangentEnd') before it hits point 'verta'.
//
// This function has potential numerical accuracy problems - more investigation is needed.
//
//

hkReal hkGeomConvexHullBuilder::getAngleBetweenVertexAndPlane( const hkGeomConvexHullTolerances& tolerances, const hkVector4& vertex, const hkVector4& planeEquation, const hkVector4& tangentStart, const hkVector4& tangentEnd )
{
//#     ifdef HK_DEBUG_CONVEX_HULL
//          HK_DISPLAY_STAR( vertex, 0.1f, hkColor::BLUE );
//          HK_DISPLAY_STAR( tangentStart, 0.1f, hkColor::CYAN );
//          HK_DISPLAY_STAR( tangentEnd, 0.1f, hkColor::GREEN );
//
//          hkVector4 centre; vector4Sub( tangentEnd, tangentStart, centre ); centre.setMul( 0.5f, centre ); centre.add( tangentStart );
//          drawPlane( planeEquation, centre, hkColor::WHITE );
//#     endif

    hkVector4 relP; relP.setSub( vertex, tangentStart );

    if ( relP.lengthSquared<3>().getReal() < tolerances.m_degenerate_tolerance )
    {
        return 0.0f;
    }
    relP.normalize<3>();

    hkReal sinTheta = - planeEquation.dot<3>( relP ).getReal();

    hkVector4 edge; edge.setSub( tangentStart, tangentEnd );
    hkVector4 vInPlane; vInPlane.setCross( edge, planeEquation);

    // we can normalize here, because we initially removed duplicated points.
    vInPlane.normalize<3>();

    hkReal cosTheta = vInPlane.dot<3>( relP ).getReal();

    const hkReal DEGENERATE_TRIANGLE_EPS = tolerances.m_degenerate_tolerance;

    if ( sinTheta * sinTheta + cosTheta * cosTheta < DEGENERATE_TRIANGLE_EPS )
    {
        return HK_REAL_MAX * 0.5f; 
    }

    hkReal angle;

    // return a value between 0 and 8

    hkReal tol = 1e-6f;
    hkReal tol2 = 1e-7f;
    hkReal tol3 = tolerances.m__tol3;

    if ( (hkMath::fabs(sinTheta) < tol ) && (hkMath::fabs(cosTheta) < tol2 ))
    {
        angle = ( cosTheta < 0.0f ) ? 4.f : 0.f;
    }
    else if ( hkMath::fabs(sinTheta) > hkMath::fabs(cosTheta) )
    {
        angle = 2.0f - cosTheta / sinTheta;
        angle += ( sinTheta < -tol ) ? 4.f : 0.f;
    }
    else
    {
        angle = sinTheta / cosTheta;

        if ((cosTheta < (tol - 1)) && ( hkMath::fabs(sinTheta) < tol ))
        {
            angle = 4.0f;
        }
        else
        {
            angle += ( cosTheta < -tol ) ? 4.f : 0.f;
        }

        if ( hkMath::fabs( angle ) < tol3 )
        {
            angle = 0.0f;
        }
        else
        {
            angle += ( (cosTheta > tol ) && (hkMath::fabs(sinTheta) < tol2 ) ) ? 8.0f : 0.0f;
        }
    }

    // sanity checks
    if ( angle < -tol3 )
    {
        angle += 8.0f;
    }
    if ( angle > 8.0f )
    {
        angle = 0.0f;
    }
    //HK_ASSERT_NO_MSG( 0x234567, ( angle < 4.5f ) );
    return angle;
}

void hkGeomConvexHullBuilder::PlaneAndPoints::sort()
{
    if ( m_v0->m_vertex > m_v1->m_vertex ) hkAlgorithm::swap( m_v0, m_v1 );
    if ( m_v1->m_vertex > m_v2->m_vertex ) hkAlgorithm::swap( m_v1, m_v2 );
    if ( m_v0->m_vertex > m_v1->m_vertex ) hkAlgorithm::swap( m_v0, m_v1 );
}

hkBool hkGeomConvexHullBuilder::PlaneAndPoints::findPair( hkGeomEdge* edgeBase, hkGeomEdge* p11, hkGeomEdge* p21, hkGeomEdge* p12, hkGeomEdge* p22, PlaneAndPoints& p1, PlaneAndPoints& p2 )
{
    hkBool firstPairOk =
            (( p11 != p21 ) &&
            ( p11->m_vertex == p21->m_vertex ) &&
            ( p11->getMirror( edgeBase )->m_vertex != p21->getMirror( edgeBase )->m_vertex ));

    hkBool secondPairOk =
            (( p12 != p22 ) &&
            ( p12->m_vertex == p22->m_vertex ) &&
            ( p12->getMirror( edgeBase )->m_vertex != p22->getMirror( edgeBase )->m_vertex ));

    hkBool isOk = firstPairOk && secondPairOk;

    if ( isOk )
    {
        // check that every other edge doesn't conflict with both these ones.
        hkBool p22ok = true;
        {
            p22ok = p22ok && EDGE_OK( p11, p22);
            p22ok = p22ok && EDGE_OK( p12, p22 );
            p22ok = p22ok && MIRROR_OK( p11, p22 );
            p22ok = p22ok && MIRROR_OK( p12, p22 );
        }

        hkBool p21ok = true;
        {
            p21ok = p21ok && EDGE_OK( p11, p21);
            p21ok = p21ok && EDGE_OK( p12, p21 );
            p21ok = p21ok && MIRROR_OK( p11, p21 );
            p21ok = p21ok && MIRROR_OK( p12, p21 );
        }

        isOk = p22ok && p21ok;
    }

    return isOk;
}

hkBool hkGeomConvexHullBuilder::vectorLessAndMergeCoordinates( hkVector4& v1, hkVector4& v2 )
{
    /*

    // compare the vectors by coordinates

    // If the two coordinates are close enough, consider them similar.
    // Keep the smaller of the two values.
    hkReal tolerance = 1e-8f;

    if ( hkMath::fabs( v1(0) - v2(0) ) < tolerance )
    {
        if ( v1(0) < v2(0) )    v2(0) = v1(0);
        else                    v1(0) = v2(0);
    }

    if ( v1(0) < v2(0) )
    {
        return true;
    }

    if ( v1(0) == v2(0) )
    {
        if ( hkMath::fabs( v1(1) - v2(1) ) < tolerance )
        {
            if ( v1(1) < v2(1) )    v2(1) = v1(1);
            else                    v1(1) = v2(1);
        }
    }
    else
    {
        return false;
    }


    if ( (v1(1) < v2(1)) )
    {
        return true;
    }

    if ( v1(1) == v2(1) )
    {
        if ( hkMath::fabs( v1(2) - v2(2) ) < tolerance )
        {
            if ( v1(2) < v2(2) )    v2(2) = v1(2);
            else                    v1(2) = v2(2);
        }
    }
    else
    {
        return false;
    }


    if (v1(2) < v2(2))
    {
        return true;
    }
    else
    {
        return false;
    }

    */

    // compare the vectors by coordinates
    if ( v1(0) < v2(0) )
    {
        return true;
    }
    else if ( (v1(0) == v2(0)) && (v1(1) < v2(1)) )
    {
        return true;
    }
    else if ( (v1(0) == v2(0)) && (v1(1) == v2(1)) && (v1(2) < v2(2)) )
    {
        return true;
    }
    else
    {
        return false;
    }
}



static hkBool weightedNeighbourLess( const struct hkGeomConvexHullBuilder::WeightedNeighbour& w1, const struct hkGeomConvexHullBuilder::WeightedNeighbour& w2)
{
    return (w1.m_weight < w2.m_weight);
}

static hkBool weightedLineLess( const struct hkGeomConvexHullBuilder::WeightedLine& w1, const struct hkGeomConvexHullBuilder::WeightedLine& w2)
{
    return (w1.m_weight < w2.m_weight);
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
