// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics2012/Collide/Shape/hkpShape.h>
#include <Physics2012/Collide/Shape/hkpShapeContainer.h>
#include <Physics2012/Collide/Shape/Compound/Collection/hkpShapeCollection.h>
#include <Common/Base/Types/Geometry/Sphere/hkSphere.h>

class hkpShapeCollection;
struct hkpAabbCastCollector;



    // The maximum number of keys returned by a single queryAabb query. Must be a power of 2
#if !defined(HK_PLATFORM_HAS_SPU)
    enum { HK_MAX_NUM_HITS_PER_AABB_QUERY = 2048 };
#else
    enum { HK_MAX_NUM_HITS_PER_AABB_QUERY = 1024 };
#endif


    /// An hkpBvTreeShape adds bounding volume tree information to an hkpShapeCollection, such as an hkpMeshShape.
    /// This is an abstract base class. See hkpMoppBvTreeShape for an implementation.
    ///
    /// What does the bounding volume tree do?
    ///
    /// A bounding volume tree is useful in situations where you need to check for collisions between a moving object
    /// and a large static geometry, such as a landscape.
    ///
    /// The shapes that make up the landscape are hierarchically grouped in
    /// a binary bounding volume tree.
    /// At every node in the tree there exists a bounding polytope, which encapsulates all
    /// of its children. The top-level bounding volume contains the entire landscape, while
    /// the nodes on the leaf levels encapsulate one geometric primitive, normally a
    /// triangle. The fit of this bounding volume can be perfect (as in some AABB trees), or can
    /// have an extra margin/tolerance built in (e.g. MOPP). For more information please see the
    /// section on Creating Shapes in the Collision Detection chapter of the Physics manual.
    ///
    /// Instead of checking whether the moving object is colliding with each of the triangles in the landscape in turn,
    /// which would be extremely time-consuming, the bounding box of the moving object
    /// is checked against the bounding volume tree - first, whether it is intersecting with the top-level bounding volume, then with
    /// any of its child bounding volumes, and so on until the check reaches the leaf nodes. A list of any potentially colliding triangles
    /// is then passed to the narrowphase collision detection. You can think of
    /// the bounding volume tree as a filter to the narrowphase collision
    /// detection system.
class HK_EXPORT_PHYSICS_2012 hkpBvTreeShape: public hkpShape
{
    // +version(1)

    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SHAPE);
        HK_DECLARE_REFLECTION();
        HKCD_DECLARE_SHAPE_TYPE(hkcdShapeType::BV_TREE);

        enum BvTreeType
        {
            BVTREE_MOPP,
            BVTREE_TRISAMPLED_HEIGHTFIELD,
            BVTREE_STATIC_COMPOUND,
            BVTREE_COMPRESSED_MESH,
            BVTREE_USER,
            BVTREE_MAX
        };

            /// Empty constructor, to be called by the shape vtable util on SPU
        HK_INLINE hkpBvTreeShape() {}

            /// Creates an hkpBvTreeShape with the specified hkpShapeCollection.
        inline hkpBvTreeShape( ShapeType type, BvTreeType bvType );

    public:

            /// Returns the hkpShapeKey for all shapes in the hkpShapeCollection that intersect with the AABB. The initial contents of the
            /// hits array are preserved, query results are appended at the end.
        virtual void queryAabb( const hkAabb& aabb, hkArray<hkpShapeKey>& hits ) const = 0;

            /// Populates the preallocated hits buffer with shape keys, returns the number of actual hits which may be greater than maxNumKeys.
            /// The hits array should be able to hold maxNumKeys keys. On SPU this is no more than HK_MAX_NUM_HITS_PER_AABB_QUERY
        HK_INLINE hkUint32 queryAabb( const hkAabb& aabb, hkpShapeKey* hits, int maxNumKeys ) const;

            /// Adds a hit to the collector for every shape in the hkpShapeCollection whose aabb overlaps with the
            /// casted one at any point on its path.
            /// \param from Cast aabb in its initial position in the hkpBvTreeShape space
            /// \param to Final position of the aabb center in the hkpBvTreeShape space
            /// \param collector Collector used to gather the cast results
        HK_INLINE void castAabb( const hkAabb& from, hkVector4Parameter to, hkpAabbCastCollector& collector ) const;

            /// Default implementation for the castAabb method. Call castAabb to make sure the optimal implementation
            /// for the object's dynamic type is used.
        HKP_SHAPE_VIRTUAL void castAabbImpl( HKP_SHAPE_VIRTUAL_THIS const hkAabb& from, hkVector4Parameter to, hkpAabbCastCollector& collector ) HKP_SHAPE_VIRTUAL_CONST;

            // Set up the bv-tree specific tables on the SPU
        static void HK_CALL registerBvTreeCollideFunctions();
        static void HK_CALL registerBvTreeCollideFunctions_StaticCompound();
        static void HK_CALL registerBvTreeCollideQueryFunctions();

#if !defined(HK_PLATFORM_SPU)

        void afterReflectNew();

        virtual const hkpShapeContainer* getContainer() const = 0;

        HKP_SHAPE_VIRTUAL hkUint32 queryAabbImpl(HKP_SHAPE_VIRTUAL_THIS const hkAabb& aabb, hkpShapeKey* hits, int maxNumKeys ) HKP_SHAPE_VIRTUAL_CONST = 0;

#else

        // On the CPU/PPU, there is no need for these calls as the methods are virtual
        // But on the SPU we don't have a proper vtable, so we do the dispatching ourselves.

        HKP_SHAPE_VIRTUAL HK_SHAPE_CONTAINER* getContainerImpl(const hkpShape* shape, hkpShapeBuffer& buffer);

        typedef hkUint32 (HK_CALL *QueryAabbFunc)        (HKP_SHAPE_VIRTUAL_THIS const hkAabb& aabb, hkpShapeKey* hits, int maxNumKeys ) HKP_SHAPE_VIRTUAL_CONST;

        typedef void (HK_CALL *CastAabbFunc) (HKP_SHAPE_VIRTUAL_THIS const hkAabb& from, hkVector4Parameter to, hkpAabbCastCollector& collector) HKP_SHAPE_VIRTUAL_CONST;

        // GetAabbFunc and CastRayFunc are already defined in hkpShape

        struct BvTreeFuncs
        {
            QueryAabbFunc               m_queryAabbFunc;
            CastAabbFunc                m_castAabbFunc;
        };

        static BvTreeFuncs s_bvTreeFunctions[ BVTREE_MAX ];

#endif

    public:

        hkEnum<BvTreeType, hkUint8> m_bvTreeType; //+default(hkpBvTreeShape::BVTREE_USER)
};

HK_REFLECT_ENUM(HK_EXPORT_PHYSICS_2012, hkpBvTreeShape::BvTreeType);

#include <Physics2012/Collide/Shape/Compound/Tree/hkpBvTreeShape.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
