// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

// MOPP Virtual Machine implementation

// include all default MOPP headers
#include <Physics2012/Collide/hkpCollide.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Machine/hkpMoppLongRayVirtualMachine.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Code/hkpMoppCommands.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Machine/hkpMoppVirtualMachineMacros.h>
#include <Physics2012/Collide/Shape/Compound/Collection/hkpShapeCollection.h>
#include <Physics2012/Collide/Shape/Query/hkpRayShapeCollectionFilter.h>
#include <Physics2012/Collide/Shape/Query/hkpRayHitCollector.h>
#include <Common/Base/Math/Vector/hkIntVector.h>

#if defined(HK_PLATFORM_SPU)
#   include <Physics2012/Collide/Filter/Spu/hkpSpuCollisionFilterUtil.h>
#endif

//For PPu on PlayStation(R)3 and Xbox 360 using a lookup table eliminated LHS when converting form int to float
#if defined(HK_PLATFORM_XBOX360) || ( defined(HK_PLATFORM_PS3_PPU) && !defined(HK_PLATFORM_SPU) )
#define HK_UINT8_TO_REAL( x ) hkReal( hkUInt8ToFloat32[ (x) ] )
#else
#define HK_UINT8_TO_REAL( x ) hkReal ( (x) )
#endif

// Virtual Machine command definitions
struct hkpMoppLongRayVirtualMachine::QueryInt
{
    //the offset of the all previous scales are accumulated here (in B space)
    hkVector4 m_FtoBoffset;

    // pc
    //const unsigned char* m_pc;

    //the shifts from all previous scale commands are accumulated here
    int m_shift;

    // this converts  floating point space into form byte space
    hkReal m_FtoBScale;

    // the current offset for the primitives
    unsigned int m_primitiveOffset;
    unsigned int m_properties[hkpMoppCode::MAX_PRIMITIVE_PROPERTIES];
};

struct hkpMoppLongRayVirtualMachine::QueryFloat
{
    // this is the ray in local int coordinate space
    hkVector4 m_rayEnds[2];
};


void hkpMoppLongRayVirtualMachine::addHit(unsigned int key, const unsigned int properties[hkpMoppCode::MAX_PRIMITIVE_PROPERTIES])
{
#ifdef HK_MOPP_DEBUGGER_ENABLED
    if ( hkpMoppDebugger::getInstance().find() )
    {
        hkprintf("Adding correct triangle as %i %i\n", key, properties[0]);
    }
#endif


#if !defined(HK_PLATFORM_SPU)
    if ( m_ray.m_rayShapeCollectionFilter == HK_NULL || m_ray.m_rayShapeCollectionFilter->isCollisionEnabled( m_ray, *m_collection, key ) )
#else
    if ( m_ray.m_rayShapeCollectionFilter == HK_NULL || hkpSpuCollisionFilterUtil::s_rayShapeContainerIsCollisionEnabled( m_ray, *m_collection, key) )
#endif
    {
        // Allocate on Havok stack to avoid blowing the program stack, since this function gets inlined into the recursive query function
        hkpShapeBuffer* shapeBuffer = hkAllocateStack<hkpShapeBuffer>(1);
        const hkpShape* childShape = m_collection->getChildShape( key, *shapeBuffer );

        if ( m_rayResultPtr )
        {
            //
            //  data driven query
            //
            m_rayResultPtr->changeLevel(1);

            hkBool hitFound = childShape->castRay( m_ray, *m_rayResultPtr);

            m_rayResultPtr->changeLevel(-1);
            if( hitFound )
            {
                m_hitFound = hitFound;
                m_earlyOutHitFraction = m_rayResultPtr->m_hitFraction;
                m_rayResultPtr->setKey( key );
            }
        }
        else
        {
            //
            //  callback driven query
            //
            hkpCdBody childBody( m_body );
            childBody.setShape( childShape, key );
            childShape->castRayWithCollector( m_ray, childBody, *m_collector );
            m_earlyOutHitFraction = m_collector->m_earlyOutHitFraction;
        }

        hkDeallocateStack(shapeBuffer, 1);
    }

}

#define SPLIT2( POSL, POSH, NEG )                                   \
            foffseth = float( (PC1_float * 2.0f) - (255.0f * NEG) );        \
            foffsetl = float( (PC2_float * 2.0f) - (255.0f * NEG) );        \
    positionl = float(POSL);                                        \
    positionh = float(POSH);                                        \
    goto do_compareN4

#define SPLIT3( POSL, POSH, NEG )                                   \
    foffseth = float( (PC1_float * 3.0f) - (255.0f * NEG) );            \
    foffsetl = float( (PC2_float * 3.0f) - (255.0f * NEG) );            \
    positionl = float(POSL);                                        \
    positionh = float(POSH);                                        \
    goto do_compareN4


//
//  The spaces:
//  - original floating point space:  O-space (not used in this class)
//  - the relative MOPP space:        F-space  = O-space - m_code->m_info.m_offset
//  - the root level integer space:   I-space  = F-space * m_code->m_info.m_scale = F-space / m_ItoFScale
//  - the MOPP byte space:            B-space  = F-space * quary->m_FtoBScale - query->m_FtoBoffset]
//
#if defined(HK_PLATFORM_SPU)
static inline const hkUint8* recheckCache( int chunkOffset, const hkUint8* originalBaseAddress, int dmaGroup )
{
    // Fetch the chunk
    const unsigned char* chunkAddress = originalBaseAddress + chunkOffset;
    unsigned char* chunkBase = (unsigned char*)g_SpuMoppCache->getFromMainMemoryInlined( chunkAddress, HK_MOPP_CHUNK_SIZE, dmaGroup, true );
    return chunkBase;
}
#endif


void hkpMoppLongRayVirtualMachine::queryRayOnTree( const hkpMoppLongRayVirtualMachine::QueryInt* query, const unsigned char* PC,hkpMoppLongRayVirtualMachine::QueryFloat* const fQuery, int chunkOffset)
{
    hkpMoppLongRayVirtualMachine::QueryInt scaledQuery; // for fast scale commands
    while (1)
    {
        HK_MOPP_LOAD_PC();

        float positionh,positionl;

        //a temporary direction ( used when subtractions are involved)
        int tdir = 999;
        int offsetRB;
        int offsetLB;

        //these need to be changed later
        float foffseth;
        float foffsetl;

        int offsetl;
        int offseth;

        switch (command)
        {
#define RS fQuery->m_rayEnds[0]
#define RE fQuery->m_rayEnds[1]

        case HK_MOPP_SPLIT_YZ:      SPLIT2( (RS(1) + RS(2)),    (RE(1) + RE(2)), 0);
        case HK_MOPP_SPLIT_YMZ:     SPLIT2( (RS(1) - RS(2)),    (RE(1) - RE(2)), 1);
        case HK_MOPP_SPLIT_XZ:      SPLIT2( (RS(0) + RS(2)),    (RE(0) + RE(2)), 0);
        case HK_MOPP_SPLIT_XMZ:     SPLIT2( (RS(0) - RS(2)),    (RE(0) - RE(2)), 1);
        case HK_MOPP_SPLIT_XY:      SPLIT2( (RS(0) + RS(1)),    (RE(0) + RE(1)), 0);
        case HK_MOPP_SPLIT_XMY:     SPLIT2( (RS(0) - RS(1)),    (RE(0) - RE(1)), 1);

        case HK_MOPP_SPLIT_XYZ:     SPLIT3( (RS(0) + RS(1) + RS(2)),    (RE(0) + RE(1) + RE(2)), 0);
        case HK_MOPP_SPLIT_XYMZ:    SPLIT3( (RS(0) + RS(1) - RS(2)),    (RE(0) + RE(1) - RE(2)), 1);
        case HK_MOPP_SPLIT_XMYZ:    SPLIT3( (RS(0) - RS(1) + RS(2)),    (RE(0) - RE(1) + RE(2)), 1);
        case HK_MOPP_SPLIT_XMYMZ:   SPLIT3( (RS(0) - RS(1) - RS(2)),    (RE(0) - RE(1) - RE(2)), 2);
#undef RS
#undef RE
        case HK_MOPP_SINGLE_SPLIT_Z:
        case HK_MOPP_SINGLE_SPLIT_Y:
        case HK_MOPP_SINGLE_SPLIT_X:
            tdir = command - HK_MOPP_SINGLE_SPLIT_X;
            foffsetl = float(PC1_float);
            foffseth = foffsetl + 1.0f;

            positionl = float((&fQuery->m_rayEnds[0](0))[tdir]);
            positionh = float((&fQuery->m_rayEnds[1](0))[tdir]);
            offsetLB = 0;
            offsetRB = PC2;
            PC += 3;
            goto do_compare;
        case HK_MOPP_SPLIT_JUMP_Z:
        case HK_MOPP_SPLIT_JUMP_Y:
        case HK_MOPP_SPLIT_JUMP_X:
            tdir = command - HK_MOPP_SPLIT_JUMP_X;
            foffseth = float(PC1_float);
            foffsetl = float(PC2_float);
            positionl = float((&fQuery->m_rayEnds[0](0))[tdir]);
            positionh = float((&fQuery->m_rayEnds[1](0))[tdir]);
            offsetLB = (PC3 << 8) + (PC4);
            offsetRB = (PC5 << 8) + (PC6);
            PC += 7;
            goto do_compare;

        case HK_MOPP_SPLIT_Z:
        case HK_MOPP_SPLIT_Y:
        case HK_MOPP_SPLIT_X:
            tdir = command - HK_MOPP_SPLIT_X;
            foffseth = float(PC1_float);
            foffsetl = float(PC2_float);
            positionl = float((&fQuery->m_rayEnds[0](0))[tdir]);
            positionh = float((&fQuery->m_rayEnds[1](0))[tdir]);
do_compareN4:
            offsetLB = 0;
            offsetRB = PC3;
            PC += 4;
do_compare:
            {
                // not in right branch -> traverse left only
                if ( (positionh < foffsetl) && (positionl < foffsetl) )
                {
                    HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectRight() );
                    HK_QVM_DBG( hkpMoppDebugger::getInstance().recurseLeft() );
                    PC += offsetLB;
                    //the PC is ready for the left branch
                    continue;
                }

                //move to the right branch
                PC += offsetRB;

                // now we have to check for traversing the right
                if ((positionl > foffseth) && (positionh > foffseth))
                {
                    HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectLeft() );
                    HK_QVM_DBG( hkpMoppDebugger::getInstance().recurseRight() );
                    continue;
                }

                //the ray may cross the plane
                {
                    //the distance from the start and end to the planePos
                    //these values are for clipping purposes
                    const float la = positionl - foffseth;
                    const float ha = positionl - foffsetl;
                    const float lb = positionh - foffseth;
                    const float hb = positionh - foffsetl;

                    //if the ray goes from a low number to a high number,
                    //then we go left first, otherwise we go right first
                    //now that the ray must be split, we actually create 2 smaller rays
                    hkpMoppLongRayVirtualMachine::QueryFloat leftFloat;

                    leftFloat.m_rayEnds[0] = fQuery->m_rayEnds[0];
                    leftFloat.m_rayEnds[1] = fQuery->m_rayEnds[1];

                    if(la < lb)
                    {

                        //they should have different signs
                        if( (la * lb) < 0.0f)
                        {
                            const float ratio = la / (la-lb);
                            leftFloat.m_rayEnds[1].setInterpolate( fQuery->m_rayEnds[0], fQuery->m_rayEnds[1], hkSimdReal::fromFloat(ratio));
                        }

                        HK_QVM_DBG2( deep, hkpMoppDebugger::getInstance().recurseLeft() );
                        //move to the left branch
                        queryRayOnTree( query,  PC - offsetRB + offsetLB,&leftFloat, chunkOffset);
#if defined ( HK_PLATFORM_SPU )
                        const hkUint8* spuAddress = recheckCache( chunkOffset, m_originalBaseAddress, hkSpuShapeRayCastDmaGroups::GET_MOPP_CHUNK );

                        // We may have been given a different way in the cache so our local PC is potentially incorrect
                        // Since each chunk is aligned we can use the bottom bits and the new address
                        PC = (const unsigned char*) ( (hkUlong)spuAddress | ((hkUlong)PC & HK_MOPP_CHUNK_MASK ) );
#endif

                        //they should have different signs
                        if( (ha * hb) < 0.0f)
                        {
                            const float ratio = ha / (ha-hb);
                            fQuery->m_rayEnds[0].setInterpolate( fQuery->m_rayEnds[0], fQuery->m_rayEnds[1], hkSimdReal::fromFloat(ratio));
                        }

                        //we have recursed through one branch, which may have
                        //caused a hit - we must now check if this branch needs
                        //to be traversed at all
                        if( m_earlyOutHitFraction < 1.0f )
                        {
                            hkVector4 hitpoint;
                            hkReal fraction = m_earlyOutHitFraction;
                            hitpoint.setInterpolate( m_ray.m_from, m_ray.m_to, hkSimdReal::fromFloat(fraction));

                            fQuery->m_rayEnds[1].setSub( hitpoint, m_code->m_info.m_offset);
                            fQuery->m_rayEnds[1].mul(hkSimdReal::fromFloat(query->m_FtoBScale) );
                            fQuery->m_rayEnds[1].sub( query->m_FtoBoffset );

                            //if the point is below the low clipping plane, then
                            //there are definitely no hits on the other branch
                            if( (tdir < 3) && ((&fQuery->m_rayEnds[1](0))[tdir] < foffsetl))
                            {
                                HK_QVM_DBG( hkpMoppDebugger::getInstance().pop( deep ));
                                HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectRight() );
                                return;
                            }
                        }
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().pop( deep ));
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().recurseRight() );
                        continue;
                    }
                    else
                    {
                        //they should have different signs
                        if( (ha * hb) < 0.0f)
                        {
                            const float ratio = ha / (ha-hb);
                            leftFloat.m_rayEnds[1].setInterpolate( fQuery->m_rayEnds[0], fQuery->m_rayEnds[1], hkSimdReal::fromFloat(ratio));
                        }

                        HK_QVM_DBG2( deep, hkpMoppDebugger::getInstance().recurseRight() );
                        //move to the right branch
                        queryRayOnTree( query, PC,&leftFloat, chunkOffset);
#if defined( HK_PLATFORM_SPU )
                        const hkUint8* spuAddress = recheckCache( chunkOffset, m_originalBaseAddress, hkSpuShapeRayCastDmaGroups::GET_MOPP_CHUNK );

                        // We may have been given a different way in the cache so our local PC is potentially incorrect
                        // Since each chunk is aligned we can use the bottom bits and the new address
                        PC = (const unsigned char*) ( (hkUlong)spuAddress | ((hkUlong)PC & HK_MOPP_CHUNK_MASK ) );
#endif
                        //they should have different signs
                        if( (la * lb) < 0.0f)
                        {
                            const float ratio = la / (la-lb);
                            fQuery->m_rayEnds[0].setInterpolate( fQuery->m_rayEnds[0], fQuery->m_rayEnds[1], hkSimdReal::fromFloat(ratio));
                        }
                        //we have recursed through one branch, which may have
                        //caused a hit - we must now check if this branch needs
                        //to be traversed at all
                        if(m_earlyOutHitFraction < 1.0f)
                        {
                            hkVector4 hitpoint;
                            hkReal fraction = m_earlyOutHitFraction;
                            hitpoint.setInterpolate(m_ray.m_from, m_ray.m_to, hkSimdReal::fromFloat(fraction));

                            fQuery->m_rayEnds[1].setSub( hitpoint, m_code->m_info.m_offset);
                            fQuery->m_rayEnds[1].mul(hkSimdReal::fromFloat(query->m_FtoBScale) );
                            fQuery->m_rayEnds[1].sub( query->m_FtoBoffset );
                            //if the point is below the low clipping plane, then
                            //there are definitely no hits on the other branch

                            if( (tdir < 3) && ((&fQuery->m_rayEnds[1](0))[tdir] > foffseth))
                            {
                                HK_QVM_DBG( hkpMoppDebugger::getInstance().pop( deep ));
                                HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectLeft() );
                                return;
                            }
                        }
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().pop( deep ));
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().recurseLeft() );
                        //left branch
                        PC = PC - offsetRB + offsetLB;
                        continue;
                    }
                }
            }
        case HK_MOPP_DOUBLE_CUT24_X:
        case HK_MOPP_DOUBLE_CUT24_Y:
        case HK_MOPP_DOUBLE_CUT24_Z:
            {
                tdir = command - HK_MOPP_DOUBLE_CUT24_X;

                foffsetl = float( ((PC1<<16) + (PC2<<8) + PC3) * m_ItoFScale );
                foffseth = float( ((PC4<<16) + (PC5<<8) + PC6) * m_ItoFScale );

                foffsetl *= float(query->m_FtoBScale);
                foffseth *= float(query->m_FtoBScale);

                foffsetl -= float(query->m_FtoBoffset(tdir));
                foffseth -= float(query->m_FtoBoffset(tdir));
                PC += 7;
                goto doCut;
            }

        case HK_MOPP_DOUBLE_CUT_X:
        case HK_MOPP_DOUBLE_CUT_Y:
        case HK_MOPP_DOUBLE_CUT_Z:
            {
                tdir = command - HK_MOPP_DOUBLE_CUT_X;
                foffsetl = float(PC1_float);
                foffseth = float(PC2_float);
                PC += 3;
            }
doCut:
            {
                positionl = float(fQuery->m_rayEnds[0](tdir));
                positionh = float(fQuery->m_rayEnds[1](tdir));

                int sign;
                if ( positionl < positionh)
                {
                    if ( (positionh < foffsetl) ||  (positionl > foffseth)  )
                    {
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectRight() );
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectLeft() );
                        goto end_of_function;
                    }
                    sign = 0;
                }
                else
                {
                    if ( (positionl < foffsetl) ||  (positionh > foffseth)  )
                    {
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectRight() );
                        HK_QVM_DBG( hkpMoppDebugger::getInstance().rejectLeft() );
                        goto end_of_function;
                    }
                    sign = 1;
                }

                // Note: we actually have to copy this vectors
                const hkVector4 rayStartPoint = fQuery->m_rayEnds[0];
                const hkVector4 rayEndPoint = fQuery->m_rayEnds[1];

                //
                //  clip start of ray
                //
                {
                    const float R0 = positionl - foffseth;
                    const float R1 = positionh - foffseth;
                    if ( R0 * R1 < 0.0f)
                    {
                        const float ratio = R0 / (R0-R1);
                        fQuery->m_rayEnds[ 1-sign ].setInterpolate( rayStartPoint, rayEndPoint, hkSimdReal::fromFloat(ratio));
                    }
                }

                //
                //  clip end of ray
                //
                {
                    const float L0 = positionl - foffsetl;
                    const float L1 = positionh - foffsetl;
                    if ( L0 * L1 < 0.0f)
                    {
                        const float ratio = L0 / (L0-L1);
                        fQuery->m_rayEnds[ sign ].setInterpolate( rayStartPoint, rayEndPoint, hkSimdReal::fromFloat(ratio));
                    }
                }
                continue;
            }




        HK_MOPP_JUMP_MACRO

        case HK_MOPP_JUMP_CHUNK:
        {
            const int chunkId = ( PC1 << 8 ) |  PC2;
            chunkOffset = HK_MOPP_CHUNK_SIZE * chunkId;
            goto jumpRecheckCache;

        }

        case HK_MOPP_JUMP_CHUNK32:
            {
                chunkOffset = (PC1<<24) + (PC2<<16) + (PC3<<8) + PC4;
jumpRecheckCache:

#if defined(HK_PLATFORM_SPU)
                const hkUint8* spuAddress = recheckCache( chunkOffset, m_originalBaseAddress, hkSpuShapeRayCastDmaGroups::GET_MOPP_CHUNK );
                PC = spuAddress;
#else
                const unsigned char* jumpAddress = m_code->m_data.begin() + chunkOffset;
                PC = jumpAddress;
#endif
                continue;
            }

        case HK_MOPP_DATA_OFFSET:
            {
                // Skip over the command
                PC+=5;
                continue;
            }

        case HK_MOPP_SCALE1:
        case HK_MOPP_SCALE2:
        case HK_MOPP_SCALE3:
        case HK_MOPP_SCALE4:
            {
                const unsigned int shift = command - HK_MOPP_SCALE0;

                hkVector4 extraOffset; extraOffset.set( PC1_float, PC2_float, PC3_float);

                hkSimdReal f; f.setFromInt32(1<<shift);
                fQuery->m_rayEnds[0].sub( extraOffset );
                fQuery->m_rayEnds[1].sub( extraOffset );
                fQuery->m_rayEnds[0].mul( f );
                fQuery->m_rayEnds[1].mul( f );

                //accumulate the offset in root node space
                scaledQuery.m_FtoBoffset.setAdd( query->m_FtoBoffset, extraOffset );
                scaledQuery.m_FtoBoffset.mul( f );

                //the shift has been accumulating
                scaledQuery.m_shift = query->m_shift + shift;
                scaledQuery.m_FtoBScale = query->m_FtoBScale * f.getReal();

                //no need to copy anything except for the primitive toffset and the properties
                for(int p = 0; p < hkpMoppCode::MAX_PRIMITIVE_PROPERTIES;p++)
                {
                    scaledQuery.m_properties[p] = query->m_properties[p];
                }

                scaledQuery.m_primitiveOffset = query->m_primitiveOffset;
                //no need to copy anything except for the primitive toffset
                query = &scaledQuery;

                PC += 4;
                continue;
            }

        HK_MOPP_CHUNK_TERMINAL_MACRO

        HK_MOPP_REOFFSET_MACRO

        HK_MOPP_PROPERTY_MACRO

        HK_MOPP_DEFAULT_MACRO
        }
    }

end_of_function:
    return;
}


void hkpMoppLongRayVirtualMachine::queryLongRaySub(const hkpMoppCode* code,   const hkpShapeRayCastInput& input) // true on successful execution
{
    //this will be handy for accessing code properties later
    m_code = code;

    const unsigned char* programCounter = &m_code->m_data[0];

    hkpMoppLongRayVirtualMachine::QueryFloat fQuery;
    hkpMoppLongRayVirtualMachine::QueryInt   query;

    query.m_FtoBoffset.setZero();


    m_ItoFScale = 1.0f / float(m_code->m_info.getScale());
    query.m_FtoBScale = m_code->m_info.getScale() / ( 1 << 0x10 );


    //***************FLOAT QUERY SETUP
    fQuery.m_rayEnds[0].setSub( input.m_from, code->m_info.m_offset);
    fQuery.m_rayEnds[1].setSub( input.m_to,   code->m_info.m_offset);
    const hkSimdReal FtoBScale = hkSimdReal::fromFloat(query.m_FtoBScale);
    fQuery.m_rayEnds[0].mul( FtoBScale );
    fQuery.m_rayEnds[1].mul( FtoBScale );

    m_ray = input;

    //no hits yet
    m_hitFound = false;

    //any re-offsetting will occur in the tree
    query.m_primitiveOffset = 0;
    query.m_shift = 0;

    for(int p = 0; p < hkpMoppCode::MAX_PRIMITIVE_PROPERTIES; p++)
    {
        query.m_properties[p] = 0;
    }

    // If the MOPP code contains terminal data then we need to reindex the terminals
    m_reindexingMask = (*programCounter == HK_MOPP_DATA_OFFSET) ? 0xffffffff : 0x00000000;

    //call the query with the left-of-decimal place part
    HK_QVM_DBG( hkpMoppDebugger::getInstance().startRecurse() );
    queryRayOnTree( &query, programCounter, &fQuery, 0 );
}


hkBool hkpMoppLongRayVirtualMachine::queryLongRay(const HK_SHAPE_CONTAINER* collection, const hkpMoppCode* code,
                                               const hkpShapeRayCastInput& input, hkpShapeRayCastOutput& rayResult) // true on successful execution
{
    m_collection = collection;
    m_rayResultPtr = &rayResult;
    m_earlyOutHitFraction = 1.0f;

#ifdef HK_PLATFORM_SPU
    m_originalBaseAddress = code->m_data.begin();
    const int moppBytes = hkMath::min2( HK_MOPP_CHUNK_SIZE, HK_NEXT_MULTIPLE_OF(16, code->m_data.getSize()) );
    const hkpMoppChunk* spuCode = (const hkpMoppChunk*)g_SpuMoppCache->getFromMainMemoryInlined( m_originalBaseAddress, moppBytes, hkSpuShapeRayCastDmaGroups::GET_MOPP_CHUNK, true );

    HK_DECLARE_ALIGNED_LOCAL_PTR( hkpMoppCode, spuMoppCode, 16 );
    spuMoppCode->initialize( code->m_info, (const hkUint8*)spuCode, moppBytes );
    code = spuMoppCode;
#endif

    queryLongRaySub( code, input );

    return hkBool(m_hitFound);
}


void hkpMoppLongRayVirtualMachine::queryLongRay(const HK_SHAPE_CONTAINER* collection, const hkpMoppCode* code,
                                               const hkpShapeRayCastInput& input, const hkpCdBody& body, hkpRayHitCollector& collector) // true on successful execution
{
    m_collection = collection;
    m_rayResultPtr = HK_NULL;
    m_body = &body;
    m_collector = &collector;

    m_earlyOutHitFraction = m_collector->m_earlyOutHitFraction;

#ifdef HK_PLATFORM_SPU
    m_originalBaseAddress = code->m_data.begin();
    const int moppBytes = hkMath::min2( HK_MOPP_CHUNK_SIZE, HK_NEXT_MULTIPLE_OF(16, code->m_data.getSize()) );
    const hkpMoppChunk* spuCode = (const hkpMoppChunk*)g_SpuMoppCache->getFromMainMemoryInlined( m_originalBaseAddress, moppBytes, hkSpuShapeRayCastDmaGroups::GET_MOPP_CHUNK, true );

    HK_DECLARE_ALIGNED_LOCAL_PTR( hkpMoppCode, spuMoppCode, 16 );
    spuMoppCode->initialize( code->m_info, (const hkUint8*)spuCode, moppBytes );
    code = spuMoppCode;
#endif


    queryLongRaySub( code, input );
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
