// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics2012/Collide/hkpCollide.h>

// hkpMoppDefaultSplitter implementation

// include all default MOPP headers
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/hkbuilder.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Splitter/hkpMoppSplitter.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Splitter/hkpMoppDefaultSplitter.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>


// Havok Memory Optimised Partial Polytope Tree - Implementation
// This constructor builds the BVtree
hkpMoppDefaultSplitter::hkpMoppDefaultSplitter()
{
    m_compileParams = HK_NULL;
    m_maxList = HK_NULL;
}

hkpMoppDefaultSplitter::~hkpMoppDefaultSplitter()
{
}


// checkplaneLeftPos
void hkpMoppDefaultSplitter::checkplaneLeftPos(int minIndex, hkpMoppBasicNode* node,
                               int& bestMaxIndex, hkpMoppCostFunction::hkpPlanesParams& planeInfo, const hkpMoppPrimitiveArray& params)
{
    hkpMoppCostFunction* costFunction = m_costFunction;
    const int minRangeMaxListCheck = m_compileParams->m_minRangeMaxListCheck;
    const int checkAllEveryN = m_compileParams->m_checkAllEveryN;
    const hkReal costPlaneRight = costFunction->costPlaneRight(planeInfo);
    int searchMinimalCountDown = checkAllEveryN;
    hkReal bestMaxListCost = 10000000.0f;
    int maxIndex = 0;
    int maxIndexStart = 0;
    int maxIndexEnd = 0;
    const int halfRange = (minRangeMaxListCheck >> 1);

    if (m_maxList->size() == 0) // check for gaps
    {
         // if the element exists, set the second splitting plane position to it
        if (m_maxList->getLastRemovedElement())
        {
            planeInfo.m_planeLeftPosition = m_maxList->getLastRemovedElement()->m_extent.m_max;
            goto TRYGAP;
        }
        else // we are currently handling the first primitive
        {
            goto ADDPREVPRIM;
        }
    }
    else // normal execution, planeRight < planeLeft
    {
        // check if an improvement in cost is possible
        if ( costPlaneRight > node->m_bestOverallCost)
        {
            goto ADDPREVPRIM; // no improvement possible
        }

        // only check a certaind range in the maxList about the last best maxIndex
        if (searchMinimalCountDown-- > 0)
        {
            maxIndexEnd = bestMaxIndex + halfRange;

            // make sure maxIndex is always pointing into the list
            // check if end is outside the list
            if ( maxIndexEnd > m_maxList->size())
            {
                maxIndexEnd = m_maxList->size();
            }

            maxIndexStart = bestMaxIndex - halfRange;
            if ( maxIndexStart < 0)
            {
                maxIndexStart = 0;
            }

        }
        else // check all elements in the maxList
        {
            // reset
            searchMinimalCountDown = checkAllEveryN;
            // check the entire list
            maxIndexEnd = m_maxList->size();
        }
    }
    //maxIndexStart = 0;
    for (maxIndex = maxIndexEnd-1; maxIndex >= maxIndexStart; maxIndex--)
    {
        planeInfo.m_planeLeftPosition = m_maxList->elementAt(maxIndex)->m_extent.m_max;
TRYGAP:
        // load extra cost info for planeLeft
        planeInfo.m_numSplitPrimitives = maxIndex;
        planeInfo.m_currentNumPrimitivesLeft = params.m_numPrimitives - minIndex + maxIndex;

        // calculate extra cost due to planeLeft
        const hkReal totalCost = costPlaneRight + costFunction->extraCostPlaneLeft(planeInfo);

        // check if this cost is lower than best max list cost so far
        if (totalCost < bestMaxListCost)
        {
            bestMaxListCost = totalCost;
            bestMaxIndex = maxIndex;

            // check if this cost is lower than the total best so far and
            if (totalCost < node->m_bestOverallCost)
            {
                // set splitting plane positions and save number of primitives on each side
                // min < max and ignore back of planeRight and front of planeLeft
                node->m_bestOverallCost = totalCost;
                node->m_plane = planeInfo.m_plane;
                node->m_planeRightPosition = planeInfo.m_planeRightPosition;
                node->m_planeLeftPosition = planeInfo.m_planeLeftPosition;
                // search more elements with less cuts
                if ( maxIndex - halfRange < maxIndexStart)
                {
                    maxIndexStart = maxIndex - halfRange;
                    if ( maxIndexStart < 0)
                    {
                        maxIndexStart = 0;
                    }
                }

#ifdef MOPP_DEBUG_COSTS
                costFunction->debugCosts( planeInfo, node->m_costInfo );
#endif
            }
        }
    }
    bestMaxIndex--;
ADDPREVPRIM:
    // add maximum of current primitive
    m_maxList->addElement(&params.m_primitives[minIndex]);
}


// findSplittingPlanePositions
void hkpMoppDefaultSplitter::findSplittingPlanePositions(hkpMoppBasicNode* node, const hkpMoppSplittingPlaneDirection* plane,
                                                        const hkpMoppPrimitiveArray& params, const hkpMoppExtent& extents, int treeDepth)
{
    int bestMaxIndex = 0;
    int maxSplits = params.m_maxPrimitiveSplits;
    hkpMoppCostFunction::hkpPlanesParams planesInfo;
    planesInfo.m_currentDepth = treeDepth;
    planesInfo.m_optimalDepth = m_optimalDepth;
    planesInfo.m_absoluteMin = extents.m_min;
    planesInfo.m_absoluteMax = extents.m_max;
    planesInfo.m_numPrimitives = params.m_numPrimitives;
    planesInfo.m_numPrimitivesInv = 1.0f / params.m_numPrimitives;
    planesInfo.m_plane = plane;

    planesInfo.m_minPrimitiveId =  params.m_primitives[0].m_primitiveID;
    planesInfo.m_maxPrimitiveId =  params.m_primitives[0].m_primitiveID;

    const hkReal size = extents.m_max - extents.m_min;
    if ( size < 0.001f)
    {
        // useless anyway
        planesInfo.m_extentsInv = 1.0f;
    }
    else
    {
        planesInfo.m_extentsInv = 1.0f / size;
    }

    // calculate maximum number of primitive splits allowed
    if (maxSplits > m_compileParams->m_maxPrimitiveSplitsPerNode)
    {
        maxSplits = m_compileParams->m_maxPrimitiveSplitsPerNode;
    }

    // check all positions for first(right subtree) splitting plane
    for (int minIndex = 0; minIndex < params.m_numPrimitives; minIndex++)
    {
        planesInfo.m_currentNumPrimitivesRight = minIndex;
        hkpMoppCompilerPrimitive& cp = params.m_primitives[minIndex];
        planesInfo.m_planeRightPosition = cp.m_extent.m_min;

        // update id spread
        planesInfo.m_minPrimitiveId = hkMath::min2( planesInfo.m_minPrimitiveId, cp.m_primitiveID);
        planesInfo.m_maxPrimitiveId = hkMath::max2( planesInfo.m_maxPrimitiveId, cp.m_primitiveID);


        // remove primitives already handled from the maxList
        // max list is ascending order and the index captures # of split primitives
        m_maxList->removeElementsLessThan(planesInfo.m_planeRightPosition);

        // remove primitives which cannot be handled due to space limitations
        m_maxList->removeOverflowElements(maxSplits);

        // check (all allowed) positions for second splitting plane
        checkplaneLeftPos(minIndex, node, bestMaxIndex, planesInfo, params);
    }

    // cleanup
    m_maxList->removeAllElements();
}




// findPosition
enum {
    HK_MOPP_POS_ANY = 0,
    HK_MOPP_POS_LEFT = 1,
    HK_MOPP_POS_RIGHT  = 2,
    HK_MOPP_POS_LEFT_AND_RIGHT = 3
};

int hkpMoppDefaultSplitter::findPosition(hkpMoppBasicNode* bestNode, hkpMoppCompilerPrimitive* current)
{
    int position = HK_MOPP_POS_ANY;
    // Distance < 0 means in front, > 0 behind, = 0 in plane
    // if minimum vertex is behind plane 1, add to back
    if ( current->m_extent.m_min < bestNode->m_planeRightPosition)
    {
        if ( current->m_extent.m_max <= bestNode->m_planeLeftPosition )
        {
            position  = HK_MOPP_POS_LEFT;
        }
        else
        {
            position = HK_MOPP_POS_RIGHT | HK_MOPP_POS_LEFT;
        }
    }
    else if ( current->m_extent.m_max > bestNode->m_planeLeftPosition )
    {
        position = HK_MOPP_POS_RIGHT;
    }

    return position;
}


//  Resort one input element into three sections: left and right and middle==unknown.
//  If a shapeKey is flagged as HK_MOPP_POS_LEFT_AND_RIGHT it gets duplicated into the left and the right list
//  If the shapeKey is flagged as HK_MOPP_POS_ANY it gets into the middle list
//
//  Initially all elements are in the left list and the right and middle lists are empty
//
//  Input:
//      bestNode is the node which drives the split
//      position *beginUnsorted is the node which needs to be sorted
//
//  Working Array:
//      0               // elements of the left list
//      ...             // elements of the left list
//
//      beginUnsorted   // unsorted elements
//      ...             // unsorted elements
//
//      endUnsorted     // free elements
//      ...             // free elements
//
//      beginRight      // elements of the right list
//      ...             // elements of the right list
//
//      beginMiddle     // elements of the middle list
//      ....            // elements of the middle list
//
//      beginMiddle+maxSplits // end of list
//
//

void hkpMoppDefaultSplitter::sortLeftAndRight(int switchPosition, hkpMoppBasicNode* bestNode, int depth, int& maxSplits, int& numSplits,
                      hkpMoppCompilerPrimitive*& beginUnsorted, hkpMoppCompilerPrimitive*& endUnsorted,
                      hkpMoppCompilerPrimitive*& beginRight, hkpMoppCompilerPrimitive*& beginMiddle)
{
    // front elements stay, back and middle list grows from end towards beginUnsorted, (middle elements at end)
    switch (switchPosition)
    {

    case HK_MOPP_POS_LEFT:
        // front stays, so just move to next unsorted element
        beginUnsorted++;
        // beginUnsorted now points at next unsorted element
        break;

    case HK_MOPP_POS_RIGHT:
        {
            // move to next empty space
            beginRight--;
            // move to last (unsorted) front element
            endUnsorted--;
            // check if we are not at last element
            if (beginUnsorted != beginRight)
            {
                // save last (unsorted) front element
                hkpMoppCompilerPrimitive ef = *endUnsorted;
                // copy back element
                *beginRight = *beginUnsorted;
                // copy unsorted element to beginUnsorted
                *beginUnsorted   = ef;
            }
        }
        // beginUnsorted now points at next unsorted element
        break;

    case HK_MOPP_POS_LEFT_AND_RIGHT:
        {
            HK_ASSERT_NO_MSG(0x4b373ba2,  maxSplits > 0 );
            maxSplits--;
            // increase number of splits
            numSplits++;


            // move to next empty space
            beginRight--;
            const hkVector4& direction = bestNode->getPlaneNormal();
            hkReal position  = 0.5f * ( bestNode->m_planeRightPosition + bestNode->m_planeLeftPosition );

            // copy element to the right
            m_mediator->splitPrimitive( *beginUnsorted,  direction, position, depth, beginRight );
            // copy element to the left
            hkVector4 negd;
            negd.setNeg<4>(direction);

            m_mediator->splitPrimitive( *beginUnsorted, negd, -position, depth, beginUnsorted );
            // front stays, so just move to next unsorted element
            beginUnsorted++;
        }
        // beginUnsorted now points at next unsorted element
        break;

    case HK_MOPP_POS_ANY:
        {
            // move to next empty splace
            beginRight--;
            // move to first right element
            beginMiddle--;
            // move to last (unsorted) front element
            endUnsorted--;

            // save middle element
            hkpMoppCompilerPrimitive s = *beginUnsorted;
            // save last (unsorted) front element
            hkpMoppCompilerPrimitive ef = *endUnsorted;
            // copy first back element to end of back list (to make room for middle element)
            *beginRight = *beginMiddle;
            // copy middle element
            *beginMiddle = s;
            // check if we are not at last element
            if (beginUnsorted != endUnsorted)
                // copy unsorted element to beginUnsorted
                *beginUnsorted = ef;
        }
        // beginUnsorted now points at next unsorted element
        break;
    }
}


// distributeMiddle
// distribute (cleverly) the ones in the middle
void hkpMoppDefaultSplitter::distributeMiddle(hkpMoppCompilerPrimitive*& startLeft, hkpMoppCompilerPrimitive*& endLeft,
                     hkpMoppCompilerPrimitive*& startRight, hkpMoppCompilerPrimitive*& endRight,
                     hkpMoppCompilerPrimitive*& startMiddle, hkpMoppCompilerPrimitive*& endMiddle,
                     const hkpMoppBasicNode *bestNode)
{
    if ( (endMiddle - startMiddle) == 0 )
    {
        return;
    }

    // find an axis, which is perpendicular to our current axis
    int direction = 0;
    {
        float maxExt = 0.0f;
        for (int i = 0; i<3; i++)
        {
            if ( i == bestNode->m_plane->m_index )
            {
                continue;
            }
            float diff = float(bestNode->m_extents.m_extent[i].m_max - bestNode->m_extents.m_extent[i].m_min);
            if ( diff > maxExt )
            {
                maxExt = diff;
                direction = i;
            }
        }
    }
    { // project and sort middle list
        hkpMoppExtent extents;
        int numPrimitives = static_cast<int>( endMiddle - startMiddle );
        m_mediator->projectPrimitives( m_planeDirections[direction].m_direction, direction, startMiddle, numPrimitives, &extents.m_min, &extents.m_max);

        // Sort using Quicksort algorithm
        hkSort( startMiddle, numPrimitives );
    }

    hkBool favor_left;
    if ((endLeft - startLeft) < (endRight - startRight))
    {
        favor_left = true;
    }else{
        favor_left = false;
    }


    while (startMiddle < endMiddle)
    {
        int sizeLeft = static_cast<int>( endLeft - startLeft ); // ptr->ulong->int, downsize to int (2GB limit)
        int sizeRight = static_cast<int>( endRight - startRight );
        if ( favor_left )
        {
            sizeRight *= 4;
        }
        else
        {
            sizeLeft *= 4;
        }

        // add to left
        if (sizeLeft < sizeRight)
        {
                hkpMoppCompilerPrimitive middle = endMiddle[-1];
                endMiddle[-1] = *startMiddle;

                // remove one middle
                hkpMoppCompilerPrimitive sb = *startRight;
                *endLeft = middle;
                *startMiddle = sb;
                endLeft++;
                startRight ++;
                startMiddle++;
                // should be equal
                endRight++;
                favor_left = true;
        }
        else // add to right
        {
                HK_ASSERT_NO_MSG(0x25248f1b,  endRight == startMiddle ); //*endRight = *startMiddle;
                endRight++;
                startMiddle++;
                favor_left = false;
        }
    }
}


// groupTriangles
// into two groups, i.e. into triangles that are in front of splitting plane and ones that are behind splitting plane.
// Note this function uses the *original* unsorted list of primitives to index into the sorted list of mapped lengths
// which is in turn used to choose the side (front or back) to which to add the primitive
void hkpMoppDefaultSplitter::groupPrimitives(const hkpMoppPrimitiveArray& in, hkpMoppBasicNode* bestNode, int depth, hkpMoppPrimitiveArray& leftOut, hkpMoppPrimitiveArray& rightOut)
{
    int maxSplits = in.m_maxPrimitiveSplits;
    int numSplits = 0;
    hkpMoppCompilerPrimitive* startLeft = &in.m_primitives[0];
    hkpMoppCompilerPrimitive* endLeft = &in.m_primitives[in.m_numPrimitives];

    hkpMoppCompilerPrimitive* startRight = &in.m_primitives[in.m_numPrimitives + in.m_maxPrimitiveSplits];
    hkpMoppCompilerPrimitive* startMiddle = startRight;
    hkpMoppCompilerPrimitive* endMiddle   = startRight;

    //int position;
    {
        hkpMoppCompilerPrimitive* start = &in.m_primitives[0];
        while (start < endLeft)
        {
            int position = findPosition(bestNode, start);
            sortLeftAndRight(position, bestNode, depth, maxSplits, numSplits, start, endLeft, startRight, startMiddle);
        }
    }

    hkpMoppCompilerPrimitive*   endRight = startMiddle;

    distributeMiddle(startLeft, endLeft, startRight, endRight, startMiddle, endMiddle, bestNode);
    // distribute the middle

    // set up out parameters, allocating numPrimitveSplits according to number of elements in each list
    leftOut.m_numPrimitives = static_cast<int>( (hkUlong)(endLeft - startLeft));
    leftOut.m_maxPrimitiveSplits = hkMath::hkFloatToInt((float(leftOut.m_numPrimitives) * float(maxSplits))/ float(in.m_numPrimitives)) ;
    rightOut.m_numPrimitives = static_cast<int>( (hkUlong)(endRight - startRight));
    rightOut.m_maxPrimitiveSplits = maxSplits - leftOut.m_maxPrimitiveSplits;
    leftOut.m_primitives = startLeft;
    rightOut.m_primitives = endLeft + leftOut.m_maxPrimitiveSplits;

    // move back list into correct position
    if (rightOut.m_maxPrimitiveSplits )
    {
        hkpMoppCompilerPrimitive *newStartRight = rightOut.m_primitives;
        while (startRight < endRight)
        {
            *newStartRight = *startRight;
            newStartRight++;
            startRight++;
        }
    }

    // sanity checks
    HK_ASSERT_NO_MSG(0x17dcd4ed, leftOut.m_numPrimitives > 0);
    HK_ASSERT_NO_MSG(0x715d41c2, rightOut.m_numPrimitives > 0);
    HK_ASSERT_NO_MSG(0x57b506be, leftOut.m_numPrimitives < in.m_numPrimitives);
    HK_ASSERT_NO_MSG(0x3e863625, rightOut.m_numPrimitives < in.m_numPrimitives);
    HK_ASSERT_NO_MSG(0x3bfa8e03, leftOut.m_numPrimitives + rightOut.m_numPrimitives - numSplits == in.m_numPrimitives);
    HK_ASSERT_NO_MSG(0x1acf30ac, (leftOut.m_numPrimitives + rightOut.m_numPrimitives) <= (in.m_numPrimitives + in.m_maxPrimitiveSplits));
    HK_ASSERT_NO_MSG(0x620f4bd2, rightOut.m_primitives + rightOut.m_numPrimitives + rightOut.m_maxPrimitiveSplits
               <= in.m_primitives + in.m_numPrimitives + in.m_maxPrimitiveSplits );
    HK_ASSERT_NO_MSG(0x523643a5, leftOut.m_primitives + leftOut.m_numPrimitives + leftOut.m_maxPrimitiveSplits
               <= in.m_primitives + in.m_numPrimitives + in.m_maxPrimitiveSplits );
    HK_ASSERT_NO_MSG(0x56d6bab1,  leftOut.m_primitives + leftOut.m_numPrimitives + leftOut.m_maxPrimitiveSplits  <= rightOut.m_primitives );
}

void hkpMoppDefaultSplitter::calculateMaxMinId( const hkpMoppPrimitiveArray& params, hkpMoppTreeNode* node)
{
    hkpMoppCompilerPrimitive* primitive = params.m_primitives;
    hkpMoppPrimitiveId mn = primitive->m_primitiveID;
    hkpMoppPrimitiveId mx = primitive->m_primitiveID;

    {   // reset min max PropertyValue
        node->m_numProperties = 0;
        for(int j = 0; j < hkpMoppCode::MAX_PRIMITIVE_PROPERTIES; ++j)
        {
            node->m_minPropertyValue[j] = hkUint32(hkpMoppCode::MAX_PROPERTY_VALUE);
            node->m_maxPropertyValue[j] = hkUint32(hkpMoppCode::MIN_PROPERTY_VALUE);
        }
    }

    for (int i = params.m_numPrimitives-1; i>=0; i--)
    {
        hkpMoppPrimitiveId id = primitive->m_primitiveID;
        if ( id > mx )
        {
            mx = id;
        }

        if (id < mn )
        {
            mn = id;
        }

        hkpPrimitiveProperty tempProperties[hkpMoppCode::MAX_PRIMITIVE_PROPERTIES];
        const int numProps = m_mediator->getPrimitiveProperties(*primitive,tempProperties);

        if ( numProps > node->m_numProperties )
        {
            node->m_numProperties = numProps;
        }

        for(int j = 0; j < numProps; j++)
        {
            if (tempProperties[j] < node->m_minPropertyValue[j])
            {
                //this is the new offset
                node->m_minPropertyValue[j] = tempProperties[j];
            }

            if (tempProperties[j] > node->m_maxPropertyValue[j])
            {
                node->m_maxPropertyValue[j] = tempProperties[j];
            }
        }

        primitive++;
    }

    node->m_numPrimitives = params.m_numPrimitives;
    node->m_minPrimitiveId = mn;
    node->m_maxPrimitiveId  = mx;
}

    // allocates a new node from our node array
hkpMoppTreeTerminal*    hkpMoppDefaultSplitter::createTerminal(hkpMoppTreeInternalNode* parentNode, const hkpMoppPrimitiveArray& params)
{
    hkpMoppTreeTerminal* term = m_freeTerminals.getElem();

    term->init();
    term->m_isTerminal = true;
    term->m_parent = parentNode;
    term->m_primitive = &params.m_primitives[0];

    calculateMaxMinId( params, term);

    for (int currentDirection = 0; currentDirection < 3; currentDirection++)
    {
        const hkVector4& currentPlaneNormal = m_planeDirections[currentDirection].m_direction;
        hkpMoppExtent& extents = term->m_extents.m_extent[currentDirection];
        m_mediator->findExtents( currentPlaneNormal, currentDirection, params.m_primitives, params.m_numPrimitives, &extents.m_min, &extents.m_max);
    }

    return term;
}


hkpMoppTreeInternalNode* hkpMoppDefaultSplitter::createNode()
{
    hkpMoppTreeInternalNode* node = m_freeNodes.getElem();
    node->init();

    node->m_isTerminal = false;
    // make sure initial cost is very high
    node->m_bestOverallCost = 1000000.0f;
    node->m_planeRightPosition = 1e10f;

    node->m_leftBranch = HK_NULL;
    node->m_rightBranch = HK_NULL;
    node->m_plane = HK_NULL;
    return ( node );
}



inline hkReal HK_CALL hkpMoppDefaultSplitter::calcAxisLengthCost( hkReal maxExtents, hkReal extents)
{
    const hkReal d = maxExtents - extents;
    const hkReal c = d / (maxExtents);
    const hkReal e = c * c * c * 16.0f;
    return e;
}



// resort the axis: the axis with the highest likelihood of getting the best hit is put
// first into the list, the other direction get a very small extra cost
// the cost are using a formula:  0.02f * (maxSize-size)/(maxSize+1.0f)
void hkpMoppDefaultSplitter::resortAxis( const hkpMoppTreeInternalNode* parentNode, int* directionsOut, hkReal* extraCostsOut)
{
    if (parentNode)
    {
        const hkReal weight = 0.05f;
        hkReal size[3];

        hkReal maxSize = 0.0f;
        int maxIndex = 0;

        for( int i = 0; i < 3; i++)
        {
            size[i] = parentNode->m_extents.m_extent[i].m_max - parentNode->m_extents.m_extent[i].m_min;

            //
            //  estimate the reduced size
            //
            if ( parentNode->m_plane == &this->m_planeDirections[i] )
            {
                size[i] *= 0.66f;
            }
            if ( size[i] > maxSize )
            {
                maxSize = size[i];
                maxIndex = i;
            }

        };

        const int a = (maxIndex + 1)%3;
        const int b = (maxIndex + 2)%3;

        directionsOut[0] = maxIndex;
        extraCostsOut[0] = 0.0f;
        if ( size[a] < size[b] )
        {
            directionsOut[1] = b;
            extraCostsOut[1] = calcAxisLengthCost(maxSize,size[b]) * weight;
            directionsOut[2] = a;
            extraCostsOut[2] = calcAxisLengthCost(maxSize,size[a]) * weight;
        }
        else
        {
            directionsOut[1] = a;
            extraCostsOut[1] = calcAxisLengthCost(maxSize,size[a]) * weight;
            directionsOut[2] = b;
            extraCostsOut[2] = calcAxisLengthCost(maxSize,size[b]) * weight;
        }
    }
    else
    {
        directionsOut[0] = 0;
        directionsOut[1] = 1;
        directionsOut[2] = 2;
        extraCostsOut[0] = 0.0f;
        extraCostsOut[1] = 0.0f;
        extraCostsOut[2] = 0.0f;
    }


}

// Split the space
hkpMoppTreeNode* hkpMoppDefaultSplitter::split(hkpMoppTreeInternalNode* parentNode, const hkpMoppPrimitiveArray& params, hkpMoppDsSide side,  int depth)
{

    HK_ASSERT(0x1a2ed314,  params.m_numPrimitives > 0, "No primitives to build MOPP around. This normally happens when the hkpShapeCollection used to build the MOPP returns no leaf nodes, either because the hkpShapeCollection is empty or it contains only degenerate triangles e.g. triangles that all fail the degeneracy tests in hkpTriangleUtil::isDegenerate()." );
    if ( params.m_numPrimitives == 1 )
    {
        if ( m_compileParams->m_interleavedBuildingEnabled ) {
            if ( m_freeTerminals.getNumFreeElems() == 0 )
            {
                m_assembler->assemble( m_rootNode, this, (HK_MOPP_ENABLED_INTERLEAVED_BUILDING_SPLITTER_MAX_NODES>>1) );
            }
        }
        hkpMoppTreeTerminal* term = createTerminal(parentNode,params  );
        return term;
    }

    // set up new node
    if ( m_compileParams->m_interleavedBuildingEnabled )
    {
        if ( m_freeNodes.getNumFreeElems() == 0 )
        {
            m_assembler->assemble( m_rootNode, this, (HK_MOPP_ENABLED_INTERLEAVED_BUILDING_SPLITTER_MAX_NODES>>1) );
        }
    }
    hkpMoppTreeInternalNode* bestNode = createNode();

    bestNode->m_numPrimitives = params.m_numPrimitives;
    bestNode->m_parent = parentNode;

    if ( !parentNode )
    {
        this->m_rootNode = bestNode;
    }
    else
    {
        if ( side == HK_MOPP_DS_LEFT )
        {
            parentNode->m_leftBranch = bestNode;
        }
        else
        {
            parentNode->m_rightBranch = bestNode;
        }
        bestNode->m_extents = parentNode->m_extents;
    }

    // the axis which the values are sorted and set
    const hkpMoppSplittingPlaneDirection* lastSortedPlane = HK_NULL;
    {
        // do the first three axis separately (some optimizations possible)
        int i;
        hkReal extraCosts[3];
        int resortedDirections[3];

        resortAxis( parentNode, resortedDirections, extraCosts);

        for (i = 0; i < 3; i++)
        {
            const int currentDirection = resortedDirections[i];
            const hkpMoppSplittingPlaneDirection* plane = &m_planeDirections[currentDirection];
            const hkVector4 &currentPlaneNormal = plane->m_direction;
            const hkReal currentDirectionCost     = plane->m_cost + extraCosts[i];

            // check if an improvement is actually possible
            if ( currentDirectionCost > bestNode->m_bestOverallCost )
            {
                break; // no improvement possible
            }

            // step1: retrieve the (unsorted) projected primitives' minimum and maximum vertices
            hkpMoppExtent& extents = bestNode->m_extents.m_extent[currentDirection];
            m_mediator->projectPrimitives( currentPlaneNormal, currentDirection, params.m_primitives, params.m_numPrimitives, &extents.m_min, &extents.m_max);

            // step2: Sort using Quicksort algorithm
            hkSort( params.m_primitives, params.m_numPrimitives );
            lastSortedPlane = plane;

            // step3: Find the splitting plane.
            findSplittingPlanePositions(bestNode, plane, params, extents, depth);


            if( !bestNode->m_plane )
            {
                HK_WARN_ALWAYS(0xabba2344, "Could not find splitting plane for child" );
                m_freeNodes.addElem( bestNode );
                return HK_NULL;
            }
        }

        // fill in the extents for the last axis
        for ( /*i from break above*/; i < 3; i++)
        {
            const int currentDirection = resortedDirections[i];
            const hkpMoppSplittingPlaneDirection* plane = &m_planeDirections[currentDirection];
            const hkVector4 &currentPlaneNormal = plane->m_direction;
            hkpMoppExtent& extents = bestNode->m_extents.m_extent[currentDirection];
            m_mediator->findExtents( currentPlaneNormal, currentDirection, params.m_primitives, params.m_numPrimitives, &extents.m_min, &extents.m_max);
        }

    }
    {
        // check all REMAINING splitting plane directions
        for (int currentDirection = 3; currentDirection < m_numPlaneDirections; currentDirection++)
        {
            const hkpMoppSplittingPlaneDirection* plane = &m_planeDirections[currentDirection];
            const hkReal currentDirectionCost     = plane->m_cost;

            // check if an improvement is actually possible
            if ( currentDirectionCost > bestNode->m_bestOverallCost )
            {
                break; // no improvement possible
            }
            const hkVector4& currentPlaneNormal = plane->m_direction;

            hkpMoppExtent extents;
            m_mediator->projectPrimitives( currentPlaneNormal, currentDirection, params.m_primitives, params.m_numPrimitives, &extents.m_min, &extents.m_max);
            hkSort( params.m_primitives, params.m_numPrimitives );
            lastSortedPlane = plane;
            findSplittingPlanePositions(bestNode, plane, params, extents, depth);
        }
    }

    {   // calc max min Id
        calculateMaxMinId( params, bestNode);
    }
    // step 3: group primitives of best direction evenly
    hkpMoppPrimitiveArray leftParams, rightParams;
    {
        hkReal absoluteMin, absoluteMax;
        // only reproject primitives if necessary
        if ( lastSortedPlane != bestNode->m_plane)
        {
            m_mediator->projectPrimitives(bestNode->getPlaneNormal(), bestNode->m_plane->m_index, params.m_primitives, params.m_numPrimitives, &absoluteMin, &absoluteMax);
        }
        // destroys the min max values
        groupPrimitives(params, bestNode, depth, leftParams, rightParams);
    }


    bestNode->m_rightBranch = split( bestNode, rightParams, HK_MOPP_DS_RIGHT, depth+1 );    // right first due to assembler works right first also
    bestNode->m_leftBranch  = split( bestNode, leftParams,  HK_MOPP_DS_LEFT,  depth+1 );

    if( bestNode->m_rightBranch == HK_NULL || bestNode->m_leftBranch == HK_NULL )
    {
        m_freeNodes.addElem( bestNode );
        return HK_NULL;
    }

    // split successful
    return bestNode;
}

void hkpMoppDefaultSplitter::releaseNode( class hkpMoppTreeNode* nodeToRelease )
{
    if ( nodeToRelease == HK_NULL )
    {
        return;
    }

    if ( nodeToRelease->m_isTerminal )
    {
        m_freeTerminals.addElem( nodeToRelease->toTerminal() );
    }
    else
    {
        m_freeNodes.addElem( nodeToRelease->toNode() );
    }
}

int hkpMoppDefaultSplitter::getFreeNodes()
{
    return m_freeNodes.getNumFreeElems() + m_freeTerminals.getNumFreeElems();
}


// build tree
hkpMoppTreeNode* hkpMoppDefaultSplitter::buildTree(hkpMoppMediator* mediator, hkpMoppCostFunction* costFunction, hkpMoppAssembler* assembler, const hkpMoppSplitParams& in, hkpMoppScratchArea& temp)
{
    int depth = 0;

    m_compileParams = &in;

    m_mediator = mediator;
    m_costFunction = costFunction;
    m_assembler = assembler;

    hkpMoppPrimitiveArray params;
    params.m_numPrimitives       = mediator->getNumPrimitives();
    params.m_maxPrimitiveSplits  = in.m_maxPrimitiveSplits;
    params.m_primitives          = temp.m_primitives;

    { // calc m_optimalDepth
        m_optimalDepth = 0;
        int n = params.m_numPrimitives;
        while(n)
        {
            n >>=1;
            m_optimalDepth++;
        }
    }


    int moppSplitterMaxNodes;
    if ( m_compileParams->m_interleavedBuildingEnabled )
    {
        moppSplitterMaxNodes = HK_MOPP_ENABLED_INTERLEAVED_BUILDING_SPLITTER_MAX_NODES;
    }
    else
    {
        moppSplitterMaxNodes = params.m_numPrimitives + params.m_maxPrimitiveSplits;
    }

    {
        m_freeNodes.reset();
        hkpMoppTreeInternalNode* n = temp.m_nodes;
        for ( int i = 0; i < moppSplitterMaxNodes; i++)
        {
            m_freeNodes.addElem( n++);
        }
    }

    {
        m_freeTerminals.reset();
        hkpMoppTreeTerminal* n = temp.m_terminals;
        for ( int i = 0; i < moppSplitterMaxNodes; i++)
        {
            m_freeTerminals.addElem( n++);
        }
    }

    this->m_numPlaneDirections = assembler->getNumSplittingPlaneDirections();
    this->m_planeDirections = assembler->getSplittingPlaneDirections();

    hkpMoppMaxList maxList( m_compileParams->m_maxPrimitiveSplitsPerNode + 2 );
    m_maxList = &maxList;
    hkpMoppTreeNode* rootNode = split( HK_NULL, params , HK_MOPP_DS_LEFT, depth);


        // build the rest of the tree
    assembler->assemble( rootNode, this, moppSplitterMaxNodes*2 );


    m_maxList = HK_NULL;
    this->m_planeDirections = HK_NULL;
    this->m_numPlaneDirections = 0;

    return rootNode;
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
