// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

// Memory Optimised Partial Polytope Compiler Implementation
#include <Physics2012/Collide/hkpCollide.h>

#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/hkbuilder.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Code/hkpMoppCommands.h>


#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Splitter/hkpMoppDefaultSplitter.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Assembler/hkpMoppCodeGenerator.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Assembler/hkpMoppDefaultAssembler.h>

//
// standard constructor
//
hkpMoppCompiler::hkpMoppCompiler(hkpMoppMeshType meshType)
:
m_splitParams(meshType),
m_splitCostParams(meshType ),
m_assemblerParams()
{
    m_chunkInfo = HK_NULL;
}

//
// standard destructor
//
hkpMoppCompiler::~hkpMoppCompiler()
{
}

void hkpMoppCompiler::setSplitParams( const hkpMoppSplitter::hkpMoppSplitParams& sp )
{
    m_splitParams = sp;
}

void hkpMoppCompiler::setCostParams( const hkpMoppCostFunction::hkpMoppSplitCostParams& sp )
{
    m_splitCostParams = sp;
}


void hkpMoppCompiler::setAssemblerParams  ( const hkpMoppAssembler::hkpMoppAssemblerParams& ap )
{
    m_assemblerParams = ap;
}
// optionally set assembler params


void hkpMoppCompiler::enableInterleavedBuilding(bool enabledFlag)
{
    m_splitParams.m_interleavedBuildingEnabled = enabledFlag;
    m_assemblerParams.m_interleavedBuildingEnabled = enabledFlag;
}

int hkpMoppCompiler::calculateRequiredBufferSize( hkpMoppMediator* mediator )
{
    int maxTerms = this->m_splitParams.m_maxPrimitiveSplits + mediator->getNumPrimitives();

    const int sizePrim = sizeof(hkpMoppCompilerPrimitive);

    const int sizeNode = sizeof(hkpMoppTreeInternalNode) + sizeof(hkpMoppTreeTerminal);

    int maxNodes;
    if ( m_splitParams.m_interleavedBuildingEnabled )
    {
        maxNodes = HK_MOPP_ENABLED_INTERLEAVED_BUILDING_SPLITTER_MAX_NODES;
    }
    else
    {
        maxNodes = maxTerms;
    }

    return sizePrim * maxTerms + sizeNode * maxNodes;
}

static int readChunkIdFromCode(const hkUint8* PC)
{
    HK_ASSERT_NO_MSG(0x4b3d01b7, PC[0] == HK_MOPP_JUMP_CHUNK32);
    const int chunkId = (PC[1]<<24) + (PC[2]<<16) + (PC[3]<<8) + PC[4];
    return chunkId;
}

static void writeChunkOffsetIntoCode(hkUint8* PC, unsigned int offset)
{
    HK_ASSERT_NO_MSG(0x37fc1fb1, PC[0] == HK_MOPP_JUMP_CHUNK32);
    PC[1] = (hkUint8) ((offset & 0xff000000)>>24);  //HI
    PC[2] = (hkUint8) ((offset & 0x0ff0000)>>16);   //MID HI
    PC[3] = (hkUint8) ((offset & 0x0ff00)>>8);      //MID LO
    PC[4] = (hkUint8) ((offset & 0x0ff));           //LO
}

//
// compile
// compiles (generates) the tree
// returns MoppTree's root node on success, NULL on failure
//
hkpMoppCode* hkpMoppCompiler::compile(hkpMoppMediator* mediator, char* buffer, int bufferSize)
{
    int maxTerms = this->m_splitParams.m_maxPrimitiveSplits + mediator->getNumPrimitives();

    //: flexible call to compile anything into MOPP code
    const int estimatedSizePerPrim = 12;
    const int estimatedInitialSize = estimatedSizePerPrim * ( maxTerms );
    hkpMoppCodeGenerator codeGen( estimatedInitialSize );

    hkpMoppCostFunction     m_costFunction( m_splitCostParams );
    hkpMoppDefaultAssembler m_defaultAssembler ( m_assemblerParams, &codeGen, mediator );

    hkpMoppAssembler*    m_assembler =      &m_defaultAssembler;

    m_defaultAssembler.m_chunkInfo = m_chunkInfo;

    mediator->setSplittingPlaneDirections(m_assembler->getSplittingPlaneDirections(), m_assembler->getNumSplittingPlaneDirections());

    // build the tree
    int neededSpace = calculateRequiredBufferSize(mediator);

    char *finalBuffer;
    if (buffer && bufferSize >= neededSpace ){
        finalBuffer = buffer;
    }
    else
    {
        finalBuffer = hkAllocate<char>(neededSpace, HK_MEMORY_CLASS_MOPP);
    }

    hkpMoppTreeNode *rootNode;
    {
        hkpMoppSplitter::hkpMoppScratchArea sp;
        sp.m_primitives = reinterpret_cast<hkpMoppCompilerPrimitive *>( finalBuffer );
        sp.m_nodes      = reinterpret_cast<hkpMoppTreeInternalNode *> ( sp.m_primitives + maxTerms );

        int moppSplitterMaxNodes;
        if ( m_splitParams.m_interleavedBuildingEnabled )
        {
            moppSplitterMaxNodes = HK_MOPP_ENABLED_INTERLEAVED_BUILDING_SPLITTER_MAX_NODES;
        }
        else
        {
            moppSplitterMaxNodes = maxTerms;
        }
        sp.m_terminals  = reinterpret_cast<hkpMoppTreeTerminal*>  ( sp.m_nodes      + moppSplitterMaxNodes );

        // set up a new scratch area (block of memory) which will be used to group the compiler primitives
        mediator->getPrimitives(sp.m_primitives);

        hkpMoppDefaultSplitter splitter;
        rootNode = splitter.buildTree( mediator, &m_costFunction, m_assembler, m_splitParams, sp);
    }

    if ( finalBuffer != buffer)
    {
        hkDeallocate<char>(finalBuffer);
    }

    if( rootNode == HK_NULL )
    {
        return HK_NULL;
    }

    m_debugRootNode = rootNode;

    hkpMoppCode* moppCode = HK_NULL;

    if ( ! m_chunkInfo )
    {
        moppCode = codeGen.compileCode();
    }
    else
    {
        // Compute the offsets of each chunk
        // The offsets have to be multiples of 16 (so that the chunks can be DMAed) but don't need to be a multiple of the chunk size

        hkArray<int>::Temp chunkOffsets;
        chunkOffsets.setSize(m_chunkInfo->m_chunks.getSize() + 1);

        //  data:    |------------------|------------------|-- ...
        //  offsets: 0   size(chunk0)   1   size(chunk1)   2

        chunkOffsets[0] = 0;
        for (int chunkId=0 ; chunkId < m_chunkInfo->m_chunks.getSize(); chunkId++)
        {
            // Note: Chunk::m_codeSize is bigger than the final code returned by hkpMoppCodeGenerator::compileCode
            //       The latter uses hkpMoppCodeGenerator::m_pos to compute the size, so we should use that here too.
            const int codeSize = m_chunkInfo->m_chunks[chunkId].m_code->getPos();
            const int size = m_chunkInfo->m_compressor ? m_chunkInfo->m_maxChunkSize : HK_NEXT_MULTIPLE_OF(16, codeSize);
            chunkOffsets[chunkId+1] = chunkOffsets[chunkId] + size;
        }

        // Now that we know the final positions of each chunk, we can patch up the jump commands for each one.
        for (int chunkId=0 ; chunkId < m_chunkInfo->m_chunks.getSize(); chunkId++)
        {
            hkpMoppCompilerChunkInfo::Chunk& chunk = m_chunkInfo->m_chunks[chunkId];
            hkArray<hkpMoppCodeGenerator::JumpCommandInfo>& jumps = m_chunkInfo->m_chunks[chunkId].m_code->m_jumpCommands;
            for(int jumpId = 0; jumpId<jumps.getSize(); jumpId++)
            {
                hkUint8* pos = chunk.m_code->m_code + chunk.m_code->getSize() - jumps[jumpId].m_position;
                const int chunkIdToJump = readChunkIdFromCode(pos);
                HK_ASSERT(0x795fdb22, chunkIdToJump == jumps[jumpId].m_chunkId, "Mismatched chunk indices!");
                writeChunkOffsetIntoCode(pos, chunkOffsets[chunkIdToJump] );
            }
        }

        // Create a single buffer with all the chunks contiguously arranged
        const int prefetchInstructionPadding = 3;
        const int finalSize = chunkOffsets[ chunkOffsets.getSize() - 1 ] + prefetchInstructionPadding;
        moppCode = new hkpMoppCode();
        moppCode->m_data.reserveExactly( finalSize );
        moppCode->m_data.setSize( finalSize, 0xcd );

        // Copy the remaining blocks after this
        for (int chunkId=0 ; chunkId < m_chunkInfo->m_chunks.getSize(); chunkId++)
        {
            // Compute offset
            unsigned char* chunkBaseAddress = moppCode->m_data.begin() + chunkOffsets[ chunkId ];

            // Compile this chunk
            hkpMoppCode* compiledChunkCode =  m_chunkInfo->m_chunks[chunkId].m_code->compileCode();

            HK_ASSERT( 0x54e345e4, compiledChunkCode->getCodeSize() <= m_chunkInfo->m_maxChunkSize, "Chunk exceeds max chunk size" );

            //Copy code into the buffer
            hkString::memCpy( chunkBaseAddress, compiledChunkCode->m_data.begin(), compiledChunkCode->getCodeSize() );

            delete compiledChunkCode;

            m_chunkInfo->m_chunks[chunkId].m_code->removeReference();

        }
    }

    m_assembler->getScaleInfo( rootNode, &moppCode->m_info );
    return moppCode;
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
