// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

//
// Havok Partial Polytope Node Info Struct
//
// This struct will be passed to recursive calls with information about the previous level
// during the assembly process
//

#pragma once


#define HK_MOPP_RESCALE_BIT_RATIO 2
#define HK_MOPP_FIXED_POINT_BITS 16
#define HK_MOPP_BITS_PER_BYTE 8
#define HK_MAX_PRIMITIVES_PER_256_BYTES_BLOCK 22 // this check is used to avoid splitting off single terminals while building the tree in bands


#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Splitter/hkpMoppSplitTypes.h>



class hkpMoppAssemblerRescaleCommand
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MOPP, hkpMoppAssemblerRescaleCommand );

    hkBool      m_doRescale;
    int         m_scaleCommand;             //the scale command that will be added to the code
    int         m_moppOffset[3];                //the offset when converted to chars

    hkpMoppAssemblerRescaleCommand()
    {
        m_doRescale = false;
    }

};

class hkpMoppAssemblerCutCommand
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MOPP, hkpMoppAssemblerCutCommand );

    hkBool      m_doCut;                    //whether a cut is necessary or not
    int         m_cutCommand;               //the cut command that will be added to the code
    int         m_cutPlanes[2];             //the second cut plane (if there is one)

    hkpMoppAssemblerCutCommand()
    {
        m_doCut = false;
    }
};


//
// This struct will be passed to recursive assembler calls with information
// for top-down processing
//
class hkpMoppAssemblerNodeInfo
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MOPP, hkpMoppAssemblerNodeInfo );

        //level info
        int  m_level;                           //the depth so far (this will be modded by (m_groupLevels)
        hkBool m_ignoreBandControlCode;         // used internally to break recursion
        int m_maxExtentsSignificantBits;

        int         m_extents[3][2];            // the extents in 16.16 space

        int         m_currentBitsResolution;    // the number of bits after the . in a fixed point value
        int         m_accumOffset[3];           // the accumalted offset in mopp space

        //terminal info
        unsigned int m_terminalOffset;          //the offset used to reduce the size of terminal IDs
        unsigned int m_terminalSpread;          //the spread of the terminals beneath this node

        //terminal properties (copy from the hkpMoppTreeNode);
        int                 m_numProperties;
        hkpPrimitiveProperty    m_propertiesOffset[hkpMoppCode::MAX_PRIMITIVE_PROPERTIES];
        hkpPrimitiveProperty    m_propertiesSpread[hkpMoppCode::MAX_PRIMITIVE_PROPERTIES];

    protected:

        void updateOffsetAndSpread(const hkpMoppTreeNode* node)
        {
            m_terminalSpread = node->m_maxPrimitiveId - node->m_minPrimitiveId;
            m_numProperties = node->m_numProperties;
            for(int i = 0; i < node->m_numProperties; i++)
            {
                m_propertiesOffset[i] = node->m_minPropertyValue[i];
                m_propertiesSpread[i] = node->m_maxPropertyValue[i] - node->m_minPropertyValue[i];
            }
        }

        void updateExtents( const hkpMoppTreeNode* node, const hkpMoppCode::CodeInfo& codeInfo )
        {
            int extents = 0;
            for (int i=0; i < 3; i++)
            {
                m_extents[i][0] = hkMath::hkFloorToInt((node->m_extents.m_extent[i].m_min - codeInfo.m_offset(i)) * codeInfo.getScale());
                m_extents[i][1] = hkMath::hkFloorToInt((node->m_extents.m_extent[i].m_max - codeInfo.m_offset(i)) * codeInfo.getScale()) + 1;
                const int size = m_extents[i][1] - m_extents[i][0];
                extents = hkMath::max2( extents, size );
            }
        }

    protected:
        hkpMoppAssemblerNodeInfo()
        {
        }

    public:

        // return the maximum extends in fixed point resolution
        int getMaxExtentFixedPoint()
        {
            int mx = m_extents[0][1] - m_extents[0][0];
            mx = hkMath::max2(mx,m_extents[1][1] - m_extents[1][0]);
            mx = hkMath::max2(mx,m_extents[2][1] - m_extents[2][0]);
            return mx;
        }

                // returns the position of the highest bit set
        int calcSignificantBits( unsigned int a )
        {
            int bits = 0;
            while (a != 0)
            {
                a >>=1;
                bits++;
            }
            return bits;
        }


        void calcMaxExtentsSignificantBits(  )
        {
            const unsigned maxExtents           = (unsigned)getMaxExtentFixedPoint();
            const int maxExtentsSignificantBits = calcSignificantBits( maxExtents );

            int maxExtentsInclErrorSignificantBits = -1;
            for (int i=0; i < 3; i++)
            {
                const int error = 1 << (maxExtentsSignificantBits + (2 + HK_MOPP_RESCALE_BIT_RATIO - HK_MOPP_BITS_PER_BYTE) );
                const int h = calcSignificantBits( maxExtents + error);
                maxExtentsInclErrorSignificantBits = hkMath::max2( h, maxExtentsInclErrorSignificantBits );
            }
            m_maxExtentsSignificantBits =  hkMath::min2( (HK_MOPP_FIXED_POINT_BITS + HK_MOPP_BITS_PER_BYTE), maxExtentsInclErrorSignificantBits);
        }

        hkpMoppAssemblerNodeInfo(const hkpMoppAssemblerNodeInfo &parentInfo,const hkpMoppTreeNode* node, const hkpMoppCode::CodeInfo& codeInfo )
        {
            *this = parentInfo;
            updateOffsetAndSpread( node );
            updateExtents( node, codeInfo );

            m_level                 =   parentInfo.m_level + 1;
            m_ignoreBandControlCode =   false;
            calcMaxExtentsSignificantBits();
        }


        // this is the constructor for the root
        hkpMoppAssemblerNodeInfo(const hkpMoppTreeNode* node, const hkpMoppCode::CodeInfo& codeInfo )
        {

            updateExtents( node, codeInfo );
            m_currentBitsResolution = 16;
            m_accumOffset[0] = m_accumOffset[1] = m_accumOffset[2] = 0;
            m_level = -1;
            m_terminalOffset = 0;
            m_ignoreBandControlCode = false;

            updateOffsetAndSpread( node );
            calcMaxExtentsSignificantBits();
        }


};




class hkpMoppAssemblerCutInfo: public hkpMoppAssemblerNodeInfo
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_MOPP, hkpMoppAssemblerCutInfo );

    // catch spaces

    int m_lowCutPlanePosition[3];
    int m_highCutPlanePosition[3];

    hkpMoppAssemblerCutInfo(const hkpMoppAssemblerCutInfo &parentInfo,const hkpMoppTreeNode* node, const hkpMoppCode::CodeInfo& codeInfo ):
        hkpMoppAssemblerNodeInfo(parentInfo,node, codeInfo){
    }


    // this is the constructor for the root
    hkpMoppAssemblerCutInfo(const hkpMoppTreeNode* node, const hkpMoppCode::CodeInfo& codeInfo ) : hkpMoppAssemblerNodeInfo(node, codeInfo)
    {
    }

    hkpMoppAssemblerCutInfo()
    {
    }



    void calcPossibleCutPlanes(  )
    {
        for ( int direction = 0; direction < 3; direction ++ )
        {
            const int mn = m_extents[direction][0];
            const int mx = m_extents[direction][1];

            const int s = m_currentBitsResolution;

            m_lowCutPlanePosition[direction]  = (mn >> s) << s;
            m_highCutPlanePosition[direction] = ((mx >> s) + 1) << s;
        }
    }
};

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
