// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

//
// Memory Optimised Partial Polytope Assembler Implementation
//

// include all default MOPP headers
#include <Physics2012/Collide/hkpCollide.h>

#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/hkbuilder.h>

#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Code/hkpMoppCommands.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Assembler/hkpMoppDefaultAssembler.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Assembler/hkpMoppCodeGenerator.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Builder/Compiler/hkpPrimitiveCompressor.h>

#define DATA_OFFSET_COMMAND_SIZE 5
#define JUMP_CHUNK_COMMAND_SIZE 5
#define COMPRESSED_DATA_ALIGNMENT 4

void hkpMoppDefaultAssembler::updateParams(const hkpMoppAssembler::hkpMoppAssemblerParams &ap)
{
    m_params = ap;
    HK_ASSERT(0x165afc34,  ap.m_absoluteFitToleranceOfTriangles > 0.0f, "m_absoluteFitToleranceOfTriangles must be greater 0" );
    HK_ASSERT(0x1ece9117,  ap.m_absoluteFitToleranceOfInternalNodes > 0.0f, "m_absoluteFitToleranceOfInternalNodes must be > 0.0f" );
    HK_ASSERT(0x214fcd6b,  ap.m_relativeFitToleranceOfInternalNodes > 0.0f,  "m_relativeFitToleranceOfInternalNodes must be within ]0.0f .. 1.0f ]" );
    HK_ASSERT(0x125d9fcc,  ap.m_relativeFitToleranceOfInternalNodes <= 1.0f, "m_relativeFitToleranceOfInternalNodes must be within ]0.0f .. 1.0f ]" );
}


hkpMoppDefaultAssembler::hkpMoppDefaultAssembler(const hkpMoppAssemblerParams &ap, hkpMoppCodeGenerator* code, hkpMoppMediator* mediator)
:   m_directions(13)
{
    //x,y, and z axes
    m_directions[0].m_direction.set( 1.0f, 0.0f, 0.0f);
    m_directions[1].m_direction.set( 0.0f, 1.0f, 0.0f);
    m_directions[2].m_direction.set( 0.0f, 0.0f, 1.0f);

    m_directions[0].m_cost  = 0.0f;
    m_directions[1].m_cost  = 0.0f;
    m_directions[2].m_cost  = 0.0f;


    m_directions[3].m_direction.set( 0.0f, 1.0f, 1.0f);
    m_directions[4].m_direction.set( 1.0f, 0.0f, 1.0f);
    m_directions[5].m_direction.set( 1.0f, 1.0f, 0.0f);

    m_directions[3].m_cost  = 0.2f;
    m_directions[4].m_cost  = 0.2f;
    m_directions[5].m_cost  = 0.2f;

    m_directions[6].m_direction.set( 0.0f, 1.0f,-1.0f);
    m_directions[7].m_direction.set( 1.0f, 0.0f,-1.0f);
    m_directions[8].m_direction.set( 1.0f,-1.0f, 0.0f);

    m_directions[6].m_cost  = 0.25f;
    m_directions[7].m_cost  = 0.25f;
    m_directions[8].m_cost  = 0.25f;

    m_directions[9].m_direction.set( 1.0f, 1.0f, 1.0f);
    m_directions[10].m_direction.set( 1.0f, 1.0f,-1.0f);
    m_directions[11].m_direction.set( 1.0f,-1.0f, 1.0f);
    m_directions[12].m_direction.set( 1.0f,-1.0f,-1.0f);

    m_directions[9].m_cost  = 0.3f;
    m_directions[10].m_cost = 0.32f;
    m_directions[11].m_cost = 0.32f;
    m_directions[12].m_cost = 0.34f;


    // normalize m_directions
    for (int i = 0; i < 13; i++)
    {
        m_directions[i].m_direction.normalize<3>();
        m_directions[i].m_index = i;
    }

    updateParams(ap);

    m_code     = code;
    m_mediator = mediator;
    m_nodeMgr  = HK_NULL;
}

//
// Destructor
//
hkpMoppDefaultAssembler::~hkpMoppDefaultAssembler()
{
}

// release node when assembled
void hkpMoppDefaultAssembler::releaseNode( hkpMoppTreeNode* node )
{

    if ( node->m_isTerminal)
    {

    }
    else
    {
        if ( m_params.m_interleavedBuildingEnabled )
        {
            m_nodeMgr->releaseNode( node->toNode()->m_leftBranch );
            m_nodeMgr->releaseNode( node->toNode()->m_rightBranch );
            node->toNode()->m_leftBranch = HK_NULL;
            node->toNode()->m_rightBranch = HK_NULL;
        }
    }
}



//
//  ************************ SCALE command ****************************
//

hkBool hkpMoppDefaultAssembler::shouldAssemble( hkpMoppTreeNode* currentNode, const hkpMoppAssemblerNodeInfo& currentInfo )
{
    // this check is used to avoid splitting off single terminals.
    if (currentNode->m_numPrimitives < HK_MAX_PRIMITIVES_PER_256_BYTES_BLOCK )
    {
        return true;
    }

    if (currentInfo.m_level > m_minDepthToBuild)
    {
        return true;
    }
    return false;
}



int hkpMoppDefaultAssembler::calcRescaleBits( const hkpMoppAssemblerNodeInfo &parentInfo,   hkpMoppAssemblerNodeInfo& currentInfo )
{

    int maxExtentsSignificantBits = currentInfo.m_maxExtentsSignificantBits;
    maxExtentsSignificantBits -= HK_MOPP_BITS_PER_BYTE;
    maxExtentsSignificantBits = hkMath::max2( maxExtentsSignificantBits, 0 );

    int bits = parentInfo.m_currentBitsResolution - maxExtentsSignificantBits;

    // Make sure we do not get negative bits resolution afterwards
    if ( bits > parentInfo.m_currentBitsResolution)
    {
        bits = parentInfo.m_currentBitsResolution;
    }

    //HK_ASSERT_NO_MSG(0x3db2616f,  bits >= 0 );
    return bits;
}

hkBool hkpMoppDefaultAssembler::shouldRescale( hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo)
{
    const int bits = calcRescaleBits( parentInfo, currentInfo );
    if ( bits > HK_MOPP_RESCALE_BIT_RATIO)
    {
        return true;
    }
    return false;
}

void hkpMoppDefaultAssembler::recalcRescale( const hkpMoppAssemblerNodeInfo &parent, hkpMoppAssemblerNodeInfo& info, hkpMoppAssemblerRescaleCommand& cmd )
{
    int bits = calcRescaleBits(parent, info );
    cmd.m_doRescale     =   false;

    if ( bits > 0)
    {
        bits = hkMath::min2( bits, 4 );
L_calc_bits:

        info.m_currentBitsResolution = parent.m_currentBitsResolution - bits;
        HK_ASSERT_NO_MSG(0x6e6da5c0, info.m_currentBitsResolution >=0 );

        for (int i=0; i<3; i++)
        {
            const int shift = parent.m_currentBitsResolution;
            const int moppOffset = (info.m_extents[i][0] - parent.m_accumOffset[i]) >> shift;
            const int intOffset = moppOffset << shift;
            info.m_accumOffset[i] = parent.m_accumOffset[i] + intOffset;
            cmd.m_moppOffset[i] = moppOffset;

            HK_ASSERT_NO_MSG(0x523d8ec0,  moppOffset >= 0 && moppOffset < 0x100 );
            HK_ASSERT_NO_MSG(0x4b440cdc,  info.m_extents[i][0] >= info.m_accumOffset[i] );

            const int highMark = (info.m_extents[i][1] - info.m_accumOffset[i]) >> info.m_currentBitsResolution;
            //HK_ASSERT_NO_MSG(0x6a1572a1,  ( highMark  < 0xff );
            if ( highMark >= 0xff )
            {
                bits -= 1;
                //hkprintf("Warning, overflow detected\n");
                goto L_calc_bits;
            }
        }
        cmd.m_scaleCommand  = HK_MOPP_SCALE0 + bits;
        cmd.m_doRescale     = true;
    }
    else
    {
        info.m_currentBitsResolution = parent.m_currentBitsResolution;
        for (int i=0; i<3; i++)
        {
            info.m_accumOffset[i] = parent.m_accumOffset[i];
        }
    }
}


// assemble the subnode and return the size of the node

//
//  Put scale commands based on a bottom up strategy:
//      Each nodes calculates the maximum number of bits of the FixedPoint resolution it needs
//      Then the maximum allowed MOPP space size can be calculated using: res * HK_MOPP_RESOLUTION
//      This maximum size is returned. If the parent node requires a smaller number of accumulated scale bits
//      than rescale
//
int hkpMoppDefaultAssembler::preCalcScale( hkpMoppTreeNode* currentNode, const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo)
{
    //
    // some precalculations
    //
    const int maxExtentsSignificantBits = currentInfo.m_maxExtentsSignificantBits;
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;


    if ( currentNode->m_isTerminal )
    {
        hkReal tightness = m_params.m_absoluteFitToleranceOfTriangles;
        for (int i=0; i< 3; i++)
        {
            if ( !currentData->m_scaleIsValid )
            {
                currentData->m_cutCommandType[i] = hkpMoppAssemblerData::HK_MOPP_CUT8;
            }

            const hkReal nodeExtends =  currentNode->m_extents.m_extent[i].m_max - currentNode->m_extents.m_extent[i].m_min;

            // check for flat   objects and reduce tightness if flat
            const hkReal flatTightness = m_params.m_absoluteFitToleranceOfAxisAlignedTriangles(i);
            if ( nodeExtends < flatTightness )
            {
                if ( flatTightness < tightness )
                {
                    // check for 8/24 bit cuts
                    const int iflatTightness = hkMath::hkFloatToInt( flatTightness * m_codeInfo.getScale() );
                    const int nodeSizeForIflatTightness = iflatTightness << ( HK_MOPP_BITS_PER_BYTE-1);

                    // now, check whether there is a way a rescale command could give us the necessary resolution
                    if ( nodeSizeForIflatTightness >= (1<<maxExtentsSignificantBits)  )
                    {
                        // tell the rescale system
                        tightness = flatTightness;
                    }
                    else
                    {   // tell the cut system to use a high precission cut only for the first pass
                        if ( !currentData->m_scaleIsValid )
                        {
                            currentData->m_cutCommandType[i] = hkpMoppAssemblerData::HK_MOPP_CUT24;
                        }
                    }
                }
            }
        }
        // convert tightness into number of bits
        const unsigned tightnessFixedPoint = unsigned(hkMath::hkFloatToInt(tightness * ( 0.9f * m_codeInfo.getScale())));

        // count the number of significant bits
        const int tightnessBits = currentInfo.calcSignificantBits( tightnessFixedPoint );

        const int tightnessSignificantBits = tightnessBits + (HK_MOPP_BITS_PER_BYTE-2);

        const int significantBits = hkMath::max2( tightnessSignificantBits, maxExtentsSignificantBits );

        return significantBits;
    }

    //currentNode->m_doCommands |= HK_MOPP_DO_RESCALE;

    //
    // ****************************************
    // ******* top down precalculations *******
    // ****************************************
    //
    if ( !currentData->m_scaleIsValid )
    {
        if ( currentInfo.m_level < m_halfTreeDepth){
            if (shouldRescale( currentNode, parentInfo, currentInfo))
            {
                currentData->m_rescaleCommandType = hkpMoppAssemblerData::HK_MOPP_RESCALE;
            }
        }
    }

    if ( currentData->m_rescaleCommandType ==  hkpMoppAssemblerData::HK_MOPP_RESCALE )
    {
        hkpMoppAssemblerRescaleCommand rescaleCommandInfo;
        recalcRescale( parentInfo, currentInfo, rescaleCommandInfo);
    }


    //
    // ***********************
    // ******* Recurse *******
    // ***********************
    //
    int requiredBitsChildren[2] = { -1, -1 };
    int childMaxExtentsSignificantBits = 0x7fffffff;
    hkpMoppTreeNode *nodes[2] = { currentNode->toNode()->m_leftBranch, currentNode->toNode()->m_rightBranch };
    {
        for (int i = 1; i >= 0; i--)
        {
            hkpMoppTreeNode* node = nodes[i];
            if (node)
            {
                hkpMoppAssemblerNodeInfo childInfo(currentInfo, node, m_codeInfo);
                requiredBitsChildren[i] = preCalcScale(node,currentInfo, childInfo);
                childMaxExtentsSignificantBits = hkMath::min2( childMaxExtentsSignificantBits, childInfo.m_maxExtentsSignificantBits );
            }
        }
    }

    int requiredSignificantBits = static_cast<int>( hkMath::min2( maxExtentsSignificantBits + HK_MOPP_RESCALE_BIT_RATIO, childMaxExtentsSignificantBits + HK_MOPP_RESCALE_BIT_RATIO));

    //
    // *****************************************
    // ******* bottom up precalculations *******
    // *****************************************
    //

    //
    //  set the current size to the minimum of both children
    //
    //return 0;
    {
        for (int i=0; i<2; i++)
        {
            hkpMoppTreeNode* childNode  = nodes[i];
            int requiredBitsChild            =  requiredBitsChildren[i];

            if (!childNode)
            {
                continue;
            }
            hkpMoppAssemblerData* childData = &childNode->m_assemblerData;

            while(1)
            {
                // if my scale cannot be changed, we still have to check whether the children runs at proper scale
                if ( currentData->m_scaleIsValid )
                {
                    // see if our child requirement fits the current scale ratios
                    if ( !childData->m_scaleIsValid )
                    {
                        if( requiredBitsChild < requiredSignificantBits )
                        {
                            childData->m_rescaleCommandType = hkpMoppAssemblerData::HK_MOPP_RESCALE;
                        }
                    }
                    break;
                }

                // issue an extra rescale if needed
                if ( requiredBitsChild < maxExtentsSignificantBits )
                {
                    childData->m_rescaleCommandType = hkpMoppAssemblerData::HK_MOPP_RESCALE;
                }
                break;
            }

            // if the child has no rescale, we need to take care for the childs sizes
            if ( childData->m_rescaleCommandType == hkpMoppAssemblerData::HK_MOPP_NO_RESCALE )
            {
                requiredSignificantBits = static_cast<int>( hkMath::min2( requiredSignificantBits, requiredBitsChild ) );
            }
        }
    }

    return requiredSignificantBits;
}

void hkpMoppDefaultAssembler::fixScale( hkpMoppTreeNode* currentNode )
{   // fix all scale from this node to the root
    while ( currentNode )
    {
        hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

        if ( currentData->m_scaleIsValid )
        {
            return;
        }

        currentData->m_scaleIsValid = true;
        currentNode = currentNode->m_parent;
    }
}






//
//  ************************ CUT command ****************************
//  ************************ CUT command ****************************
//  ************************ CUT command ****************************
//



// make sure that our cuts actually fit the terminals very tightly
void hkpMoppDefaultAssembler::calcTermCut(hkpMoppTreeNode* currentNode, hkpMoppAssemblerCutInfo& currentInfo)
{

    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;


    for ( int i = 0; i < 3; i++)
    {
        const hkReal nodeExtends =  currentNode->m_extents.m_extent[i].m_max - currentNode->m_extents.m_extent[i].m_min;

        // check whether that the slice we will chop off will actually be large enough
        hkReal tightness = hkReal(1 << currentInfo.m_currentBitsResolution) / m_codeInfo.getScale();
        tightness = hkMath::max2( tightness, hkReal(m_params.m_absoluteFitToleranceOfTriangles));

        // check for flat   objects and reduce tightness if flat
        const hkReal flatTightness = m_params.m_absoluteFitToleranceOfAxisAlignedTriangles(i);
        if ( nodeExtends < flatTightness )
        {
            tightness = hkMath::min2( tightness, flatTightness);
        }

        const int tightnessFixed = hkMath::hkFloatToInt(tightness * m_codeInfo.getScale());

        const int mn = currentInfo.m_extents[i][0];
        const int mx = currentInfo.m_extents[i][1];

        currentData->m_minCutPlanePosition[i] = mn - tightnessFixed ;
        currentData->m_maxCutPlanePosition[i] = mx + tightnessFixed + 1;

        //hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;
    }
}

void hkpMoppDefaultAssembler::updateMaxCutPlanePosition(hkpMoppTreeNode* currentNode, HK_MOPP_SPLIT_DIRECTIONS directionCode, int planePosition)
{
    if ( directionCode > HK_MOPP_SD_Z)
    {
        return;
    }
    const int direction = directionCode;
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;
    // the split solves our cut problem
    currentData->m_maxCutPlanePosition[direction] = 0x7fffffff;
}

void hkpMoppDefaultAssembler::updateMinCutPlanePosition(hkpMoppTreeNode* currentNode, HK_MOPP_SPLIT_DIRECTIONS directionCode, int planePosition)
{
    if ( directionCode > HK_MOPP_SD_Z)
    {
        return;
    }
    const int direction = directionCode;
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;
    // the split acts like a cut
    currentData->m_minCutPlanePosition[direction] = 0;
}

void hkpMoppDefaultAssembler::preCalcCutPositions(hkpMoppTreeNode* currentNode,const hkpMoppAssemblerCutInfo &parentInfo, hkpMoppAssemblerCutInfo& currentInfo)
{

    //currentInfo.checkCatchSpace(currentNode);
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

    //
    // check, whether our node is already visited
    //

    if ( currentData->m_isAssembled )
    {
        return;
    }

    if ( currentNode->m_isTerminal )
    {
        currentInfo.calcPossibleCutPlanes();
        calcTermCut( currentNode, currentInfo );
        return;
    }

    //
    // ****************************************
    // ******* top down precalculations *******
    // ****************************************
    //

    hkpMoppAssemblerRescaleCommand rescaleCommandInfo;

    if ( currentData->m_rescaleCommandType == hkpMoppAssemblerData::HK_MOPP_RESCALE )
    {
        recalcRescale( parentInfo, currentInfo, rescaleCommandInfo );
    }
    currentInfo.calcPossibleCutPlanes();

    //calcCut( currentNode, currentInfo);

    const hkVector4 &norm           = currentNode->toNode()->getPlaneNormal();
    HK_MOPP_SPLIT_DIRECTIONS directionCode  = getSplitCode(norm);
    int pleft,pright;
    findPlanes(currentNode->toNode(),currentInfo,pleft,pright);


    //
    // *******************************
    // ******* Do Calculations *******
    // *******************************
    //

    //
    //  check, whether a cut command can be moved up
    //
    {
        // check node
        hkpMoppTreeNode* rightNode = currentNode->toNode()->m_rightBranch;
        hkpMoppTreeNode* leftNode  = currentNode->toNode()->m_leftBranch;

        hkpMoppTreeNode* nodes[2] = { rightNode, leftNode };
        for (int j=0;j<3;j++)
        {
            const int extents = currentInfo.m_extents[j][1] - currentInfo.m_extents[j][0];
            int delta = static_cast<int>(m_params.m_relativeFitToleranceOfInternalNodes * extents);
            const int minWidth = hkMath::hkFloatToInt(m_params.m_absoluteFitToleranceOfInternalNodes * m_codeInfo.getScale());
            delta = hkMath::max2( delta, minWidth );
            currentData->m_minCutPlanePosition[j] = 0; //currentInfo.m_extents[j][0] - delta;
            currentData->m_maxCutPlanePosition[j] = 0x7fffffff; //currentInfo.m_extents[j][1] + delta;
        }

        // check all three directions
        for (int c = 0; c < 2; c++)
        {
            hkpMoppTreeNode* childNode = nodes[c];
            hkpMoppAssemblerData* childData = &childNode->m_assemblerData;


            if ( !childNode)
            {
                continue;
            }
            //
            // recurse
            //
            hkpMoppAssemblerCutInfo childInfo(currentInfo, childNode, m_codeInfo);
            preCalcCutPositions(childNode,currentInfo, childInfo);

            if ( childNode == rightNode )
            {
                updateMinCutPlanePosition( childNode, directionCode, pright );
            }
            else
            {
                updateMaxCutPlanePosition( childNode, directionCode, pleft );
            }

            for (int i=0; i<3;i++)
            {
                while(1)
                {
                    int childCut = childData->m_cutCommandType[i];
                    if( childCut == hkpMoppAssemblerData::HK_MOPP_CUT8 )
                    {
                        if ( childData->m_minCutPlanePosition[i] <= currentInfo.m_lowCutPlanePosition[i] &&
                            childData->m_maxCutPlanePosition[i] >= currentInfo.m_highCutPlanePosition[i])
                        {
                            // now we can move the cut up
                            currentData->m_cutCommandType[i] = hkpMoppAssemblerData::hkpCutCommandType(hkMath::max2( childCut, int(currentData->m_cutCommandType[i])));
                            childData->m_cutCommandType[i]   = hkpMoppAssemblerData::HK_MOPP_NO_CUT;
                        }
                    }

                    if( childCut == hkpMoppAssemblerData::HK_MOPP_CUT24 )
                    {
                        if ( childData->m_minCutPlanePosition[i] <= currentInfo.m_extents[i][0] &&
                            childData->m_maxCutPlanePosition[i] >= currentInfo.m_extents[i][1] )
                        {
                            // now we can move the cut up
                            currentData->m_cutCommandType[i] = hkpMoppAssemblerData::hkpCutCommandType(childCut);
                            childData->m_cutCommandType[i]   = hkpMoppAssemblerData::HK_MOPP_NO_CUT;
                        }
                    }
                    break;
                }
                if ( childData->m_cutCommandType[i] == hkpMoppAssemblerData::HK_MOPP_NO_CUT)
                {
                    // update our min max cut planes using the child info
                    currentData->m_minCutPlanePosition[i] = hkMath::max2( currentData->m_minCutPlanePosition[i], childData->m_minCutPlanePosition[i]);
                    currentData->m_maxCutPlanePosition[i] = hkMath::min2( currentData->m_maxCutPlanePosition[i], childData->m_maxCutPlanePosition[i]);
                }
            }
        }

    }
}


void hkpMoppDefaultAssembler::initTopDown( hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo, hkpMoppAssemblerRescaleCommand& rescaleCommandInfo )
{
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

    if ( currentData->m_rescaleCommandType == hkpMoppAssemblerData::HK_MOPP_RESCALE )
    {
        recalcRescale( parentInfo, currentInfo, rescaleCommandInfo );
    }
    calcReoffset( currentNode, parentInfo, currentInfo );
}



//
//  ************************ assemble subtree ****************************
//


// returns the size of the assembled subtree
// returns -1 if one subtree could not been build
// algorithm:
//      The goal is to group all nodes with a depth between m_minDepthToBuild and m_minDepthToBuild+m_params.m_groupLevels.
//      We start setting m_minDepthToBuild to the estimated tree depth.
//      By default, all children which have a path deeper than m_minDepthToBuild are build.
//      Than we iteratively decrease m_minDepthToBuild by m_params.m_groupLevels and try to build the tree.
// only do special code if
//   - we our depth is a multiple of m_params.m_groupLevels
//   - we are not in a recursive call be the same code
int hkpMoppDefaultAssembler::assemblesubNode(hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo)
{
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

    if(currentData->m_isAssembled)
    {
        // if this node has already been assembled simply return the size of a jump16 to this node
        return 3;
    }

    // here we ensure that the tree gets compiled in depth "bands"
    if (currentInfo.m_level <= m_minDepthToBuild && (currentInfo.m_level % m_params.m_groupLevels) == 0 && currentInfo.m_ignoreBandControlCode == false )
    {
        currentInfo.m_ignoreBandControlCode = true; // disable recursive call to below code
        int size = -1;
        int oldMinDepth = m_minDepthToBuild;
        {
            // call this several times until my node is assembled.
            // the first time only the lowest level of the tree will be assembled, the next time all above levels,
            // and so on until our currentData is assembled.
            while ( !currentData->m_isAssembled )
            {
                int cPos = m_code->getPos();
                hkpMoppAssemblerNodeInfo childInfo = currentInfo;

                size = assemblesubNode( currentNode, parentInfo, childInfo);

                // already enough nodes assembled
                if ( m_numNodesAssembled > m_minAssembledElems )
                {
                    break;
                }

                // if nothing could be assembled, try to decrease the m_minDepthToBuild
                if( cPos == m_code->getPos() )
                {
                    // already negative depth, nothing to decrease
                    if (m_minDepthToBuild < 0)
                    {
                        break;
                    }
                    m_minDepthToBuild -= m_params.m_groupLevels;
                }
            }
        }
        m_minDepthToBuild = oldMinDepth;

        currentInfo.m_ignoreBandControlCode = false;
        return size;
    }


    //
    //  terminal
    //
    if ( currentNode->m_isTerminal )
    {
        fixScale( currentNode );
        if ( !shouldAssemble( currentNode, currentInfo ) )
        {
            return -1;
        }
        int size = assembleCutAndTerminalCommand( currentNode, parentInfo, currentInfo );
        return size;
    }

    //
    // ****************************************
    // ******* top down pre-calculations *******
    // ****************************************
    //
    hkpMoppAssemblerRescaleCommand rescaleCommandInfo;
    rescaleCommandInfo.m_scaleCommand = -1;
    initTopDown( currentNode, parentInfo, currentInfo, rescaleCommandInfo );


    //
    // ***********************
    // ******* Recurse *******
    // ***********************
    //
    int rsize = -1;     // the size of the right branch
    int lsize = -1;     // the size of the left branch

    if ( currentNode->toNode()->m_rightBranch )
    {
        hkpMoppTreeNode* childNode = currentNode->toNode()->m_rightBranch;
        hkpMoppAssemblerNodeInfo childInfo(currentInfo, childNode, m_codeInfo);

        rsize = assemblesubNode(childNode,currentInfo, childInfo);
        if ( rsize >= 0 )
        {
            releaseNode( childNode );
        }
    }

    if (currentNode->toNode()->m_leftBranch)
    {
        hkpMoppTreeNode* childNode = currentNode->toNode()->m_leftBranch;
        hkpMoppAssemblerNodeInfo childInfo(currentInfo, childNode, m_codeInfo);
        lsize = assemblesubNode(childNode,currentInfo, childInfo);

        if ( lsize >= 0)
        {
            releaseNode( childNode );
        }
    }


    //
    // *********************************************************************
    // ******* check whether we want to build this node now or later *******
    // *********************************************************************
    //

    // if any of our children are not build, do not build this node
    if( (lsize < 0) || (rsize < 0) )
    {
        return -1;
    }

    // here we ensure that the tree gets compiled in depth "bands"
    if ( !shouldAssemble( currentNode, currentInfo ) )
    {
        return -1;
    }


    //
    // **********************************************
    // ******* add all commands for this node *******
    // **********************************************
    //
    m_numNodesAssembled++;

    int size = assembleNonTerminalCommand( currentNode, parentInfo, currentInfo, rescaleCommandInfo );
    size += lsize + rsize;

    //now return the size that we have accumulated from adding our code
    return size;
}

    // this recursive function tries to find which interior nodes will become the root node of a new chunk
int hkpMoppDefaultAssembler::calcChunkPoints(hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo, hkArray< hkpMoppPrimitiveId >& terminalsOut)
{

    // ignore reoffsetting if embedding terminals
    if (m_chunkInfo->m_compressor)
    {
        currentInfo.m_terminalOffset = parentInfo.m_terminalOffset;
    }


    //
    //  terminal just assemble and return the size
    //
    if ( currentNode->m_isTerminal )
    {
        if (m_chunkInfo->m_compressor)
        {
            // force the terminal to be encoded into a single 8 bit command
            currentInfo.m_terminalOffset = currentNode->toTerminal()->m_primitive[0].m_primitiveID;
        }

        int terminalCodeSize = m_code->getPos();
        assembleCutAndTerminalCommand( currentNode, parentInfo, currentInfo );
        terminalCodeSize = m_code->getPos() - terminalCodeSize;

            // add our size to indicate we want to encode a terminal as well
        terminalsOut.pushBack( currentNode->toTerminal()->m_primitive[0].m_primitiveID );
        hkpMoppPrimitiveId primId= currentNode->toTerminal()->m_primitive[0].m_primitiveID;

        int terminalSize = terminalCodeSize;
        if (m_chunkInfo->m_compressor)
        {
            terminalSize += m_chunkInfo->m_compressor->calcPrimitivesSize( &primId, 1);
        }

        // Subtract space for data offset size
        HK_ASSERT(0x65434567, terminalSize <= m_chunkInfo->m_maxChunkSize - DATA_OFFSET_COMMAND_SIZE, "Chunk size too small for terminal");
        return terminalSize;
    }

    //
    // ****************************************
    // ******* top down pre-calculations *******
    // ****************************************
    //
    hkpMoppAssemblerRescaleCommand rescaleCommandInfo;
    rescaleCommandInfo.m_scaleCommand = -1;
    initTopDown( currentNode, parentInfo, currentInfo, rescaleCommandInfo );

    // override reoffset
    currentInfo.m_terminalOffset = parentInfo.m_terminalOffset;


    //
    // ***********************
    // ******* Recurse *******
    // ***********************
    //
    int rsize;      // the size of the right branch including terminals
    int rCodeSize;  // the size of the right branch MOPP code
    hkArray< hkpMoppPrimitiveId > rightTerminals;
    {
        hkpMoppTreeNode* childNode = currentNode->toNode()->m_rightBranch;
        hkpMoppAssemblerNodeInfo childInfo(currentInfo, childNode, m_codeInfo);
        rCodeSize = m_code->getPos();
        rsize = calcChunkPoints(childNode,currentInfo, childInfo, rightTerminals);
        rCodeSize = m_code->getPos() - rCodeSize;
        int computedSize = rCodeSize ;
        if ((m_chunkInfo->m_compressor) && (rightTerminals.getSize()!=0))
        {
            computedSize += m_chunkInfo->m_compressor->calcPrimitivesSize( rightTerminals.begin(), rightTerminals.getSize() );
        }
        HK_ASSERT_NO_MSG(0x2dcdba2, rsize == computedSize );
        rsize = computedSize;
    }

    int lsize;      // the size of the left branch including terminals
    int lCodeSize;  // the size of the left branch MOPP code
    hkArray< hkpMoppPrimitiveId > leftTerminals;
    {
        hkpMoppTreeNode* childNode = currentNode->toNode()->m_leftBranch;
        hkpMoppAssemblerNodeInfo childInfo(currentInfo, childNode, m_codeInfo);
        lCodeSize = m_code->getPos();
        lsize = calcChunkPoints(childNode,currentInfo, childInfo, leftTerminals);
        lCodeSize = m_code->getPos() - lCodeSize;
        int computedSize = lCodeSize ;
        if ((m_chunkInfo->m_compressor) && (leftTerminals.getSize()))
        {
            computedSize += m_chunkInfo->m_compressor->calcPrimitivesSize( leftTerminals.begin(), leftTerminals.getSize() );
        }
        HK_ASSERT_NO_MSG(0x4abb0ddf, lsize == computedSize );
        lsize = computedSize;
    }


    int nodeCodeSize = m_code->getPos();
    assembleNonTerminalCommand( currentNode, parentInfo, currentInfo, rescaleCommandInfo );
    nodeCodeSize = m_code->getPos() - nodeCodeSize;

    // We need to recompute the size required to store both branches
    // It is not necessarily rsize + lsize
    hkArray< hkpMoppPrimitiveId > allSubTerminals;
    allSubTerminals.insertAt(0, rightTerminals.begin(), rightTerminals.getSize());
    allSubTerminals.insertAt(allSubTerminals.getSize(), leftTerminals.begin(), leftTerminals.getSize() );

    // Size of both branches including terminals
    int sumCodeSize = lCodeSize + rCodeSize + nodeCodeSize;
    int sumSize = sumCodeSize;
    if ((m_chunkInfo->m_compressor) && (allSubTerminals.getSize()))
    {
        sumSize += m_chunkInfo->m_compressor->calcPrimitivesSize( allSubTerminals.begin(), allSubTerminals.getSize() );
    }

    HK_ASSERT( 0x54e54343, sumSize <= (nodeCodeSize + lsize + rsize), "Bad compression : It takes less space to store sub trees individually" );

    bool prunedLeft = false;
    bool prunedRight = false;

    while ( sumSize >= m_chunkInfo->m_maxChunkSize - m_chunkInfo->m_safetySize)
    {
        int chunkId = m_chunkInfo->m_chunks.getSize();
        hkpMoppCompilerChunkInfo::Chunk& chunk = m_chunkInfo->m_chunks.expandOne();
        hkpMoppTreeNode* childNode;
        int branchPrunedSize; // Size of the pruned branch including terminals
        int codePrunedSize;   // Size of the pruned code branch

        if ( lsize > rsize )
        {
            // Prune the left branch
            childNode = currentNode->toNode()->m_leftBranch;
            branchPrunedSize = lsize;
            lsize = JUMP_CHUNK_COMMAND_SIZE;
            codePrunedSize = lCodeSize;
            prunedLeft = true;
            sumSize = (prunedRight) ? nodeCodeSize + JUMP_CHUNK_COMMAND_SIZE * 2 : rsize + nodeCodeSize + JUMP_CHUNK_COMMAND_SIZE;
        }
        else
        {
            // Prune the right branch
            childNode = currentNode->toNode()->m_rightBranch;
            branchPrunedSize = rsize;
            rsize = JUMP_CHUNK_COMMAND_SIZE;
            codePrunedSize = rCodeSize;
            prunedRight = true;
            sumSize = (prunedLeft) ? nodeCodeSize + JUMP_CHUNK_COMMAND_SIZE * 2 : lsize + nodeCodeSize + JUMP_CHUNK_COMMAND_SIZE;
        }

        // Add 5 for data offset command - DG is the safety size needed?
        chunk.m_codeSize = branchPrunedSize + DATA_OFFSET_COMMAND_SIZE + m_chunkInfo->m_safetySize;

        hkpMoppAssemblerData* childData = &childNode->m_assemblerData;
        childData->m_chunkId = chunkId;

        m_code->undo(codePrunedSize - JUMP_CHUNK_COMMAND_SIZE); // the complete branch is replaced by a jump chunk command (5 bytes)

        //Adjust assembled addresses - otherwise the jumps are incorrect
        {
            hkpMoppTreeNode* rightChildNode = currentNode->toNode()->m_rightBranch;

            // Code is stored right branch first then left
            // If the left branch has been selected we do not need to adjust the right branch address
            if (prunedRight)
                rightChildNode->m_assemblerData.m_assembledAddress -= (codePrunedSize - JUMP_CHUNK_COMMAND_SIZE);

            hkpMoppTreeNode* leftChildNode = currentNode->toNode()->m_leftBranch;
            leftChildNode->m_assemblerData.m_assembledAddress -= (codePrunedSize - JUMP_CHUNK_COMMAND_SIZE);

            currentNode->m_assemblerData.m_assembledAddress -= (codePrunedSize - JUMP_CHUNK_COMMAND_SIZE);
        }
    }

    terminalsOut.clear();
    if (!prunedRight)
        terminalsOut.insertAt(terminalsOut.getSize(), rightTerminals.begin(), rightTerminals.getSize() );

    if (!prunedLeft)
        terminalsOut.insertAt(terminalsOut.getSize(), leftTerminals.begin(), leftTerminals.getSize() );

    return sumSize;
}

    // each chunk is getting a new set of terminal ids (==hkpShapeKey).
int hkpMoppDefaultAssembler::calcTerminalIdsForChunks(hkpMoppTreeNode* currentNode, int chunkId, int currentId )
{

    int origId = currentId;

    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;
    currentData->m_isAssembled = false;


    if( currentData->m_chunkId >=0  )
    {
        // reset the id if we start a new chunk
        currentId = 0;
        chunkId = currentData->m_chunkId;
    }

    if (m_chunkInfo->m_compressor)
    {
        // Reset min ID
        currentNode->m_minPrimitiveId   = currentId;
    }

    if ( currentNode->m_isTerminal )
    {
        if (  m_chunkInfo->m_compressor  )
        {
            // If embedding reindex the terminal
            currentNode->toTerminal()->m_primitive[0].m_origPrimitiveID = currentNode->toTerminal()->m_primitive[0].m_primitiveID;
            hkpMoppCodeReindexedTerminal& reindex = m_chunkInfo->m_reindexInfo.expandOne();
            reindex.m_reindexedShapeKey = (chunkId << 8) | currentId;
            reindex.m_origShapeKey = currentNode->toTerminal()->m_primitive[0].m_primitiveID;
            currentNode->toTerminal()->m_primitive[0].m_primitiveID = currentId++;
        }
    }
    else
    {
        {
            hkpMoppTreeNode* childNode = currentNode->toNode()->m_rightBranch;
            currentId = calcTerminalIdsForChunks(childNode, chunkId, currentId);
        }
        {
            hkpMoppTreeNode* childNode = currentNode->toNode()->m_leftBranch;
            currentId = calcTerminalIdsForChunks(childNode, chunkId, currentId);
        }
    }

    if (m_chunkInfo->m_compressor)
    {
        currentNode->m_minPrimitiveId  = 0;
        currentNode->m_maxPrimitiveId  = currentId;
    }

    if( currentData->m_chunkId  < 0 )
    {
        // if we do not start a new chunk, return the new id
        return currentId;
    }
    else
    {
        return origId;
    }
}

static void copyJumpCommandInfo(hkArray<hkpMoppCodeGenerator::JumpCommandInfo>& newCommands, hkArray<hkpMoppCodeGenerator::JumpCommandInfo>& originalCommands, int offset)
{
    for (int i=0; i<originalCommands.getSize(); i++)
    {
        hkpMoppCodeGenerator::JumpCommandInfo& newCommand = newCommands.expandOne();
        newCommand.m_chunkId = originalCommands[i].m_chunkId;
        newCommand.m_position = originalCommands[i].m_position + offset;
    }
}

static int copyCodeAndEmbedTerminals(hkpMoppCompilerChunkInfo* chunkInfo, hkpMoppCodeGenerator* codeWithTerminals, hkpMoppCodeGenerator* codeWithoutTerminals, hkArray< hkpMoppPrimitiveId >& terminals )
{
    // Alignment for
    const int padBytesToAlign = (COMPRESSED_DATA_ALIGNMENT - codeWithoutTerminals->getPos()) & (COMPRESSED_DATA_ALIGNMENT-1);

    // Don't embed any terminals if we have no compressor
    if (chunkInfo->m_compressor == HK_NULL)
    {
        terminals.clear();
    }

    // Embed the primitives in the MOPP code
    if (terminals.getSize())
    {
        // Compute storage space for terminals (ToDo: cache this value)
        int primsSize = chunkInfo->m_compressor->calcPrimitivesSize( terminals.begin(), terminals.getSize() );

        // Allocate space for the embedded primitives at the end of the code
        codeWithTerminals->pad( primsSize );

        // Store primitives in aligned local buffer
        hkLocalBuffer<unsigned char> alignedBuffer( primsSize );
        chunkInfo->m_compressor->storePrimitives( terminals.begin(), terminals.getSize(), alignedBuffer.begin());

        // Embed in MOPP code
        unsigned char* codeWithTerminalsPrimBase = codeWithTerminals->m_code + (codeWithTerminals->getSize() - codeWithTerminals->getPos());
        hkString::memCpy( codeWithTerminalsPrimBase, alignedBuffer.begin(), primsSize );

    }

    int oldCodeSize = codeWithTerminals->getPos();

    // Copy in the MOPP code
    {
        // Allocate space for the code
        codeWithTerminals->pad( codeWithoutTerminals->getPos() + padBytesToAlign );

        unsigned char* codeWithoutTerminalsBase = codeWithoutTerminals->m_code + (codeWithoutTerminals->getSize() - codeWithoutTerminals->getPos());
        unsigned char* codeWithTerminalsBase = codeWithTerminals->m_code + (codeWithTerminals->getSize() - codeWithTerminals->getPos());
        hkString::memCpy( codeWithTerminalsBase, codeWithoutTerminalsBase, codeWithoutTerminals->getPos() );
    }

    // Check the sizes match
    HK_ASSERT( 0x6e543e43, codeWithTerminals->getPos() <= chunkInfo->m_maxChunkSize, "Chunk Overflow" );

    // Copy the jump command positions, taking the difference in positions into account
    codeWithoutTerminals->validateJumpCommands();
    copyJumpCommandInfo(codeWithTerminals->m_jumpCommands, codeWithoutTerminals->m_jumpCommands, codeWithTerminals->getPos() - codeWithoutTerminals->getPos() );
    codeWithTerminals->validateJumpCommands();

    // Compute the offset to store the primitives at
    return codeWithTerminals->getPos() - oldCodeSize;
}


void hkpMoppDefaultAssembler::assembleSubNodeIntoChunks(hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo, int chunkId, hkArray<hkpMoppPrimitiveId>& terminalsOut)
{
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

    if( currentData->m_chunkId > 0 && !currentInfo.m_ignoreBandControlCode )
    {
        hkpMoppCodeGenerator* oldCode = m_code;

        //Recurse building a separate piece of code for this chunk
        {
            // if this node has been placed in a different chunk, just create a jump to that junk
            chunkId = currentData->m_chunkId;
            hkpMoppCompilerChunkInfo::Chunk& chunk = m_chunkInfo->m_chunks[chunkId];
            hkpMoppCodeGenerator chunkCode( chunk.m_codeSize );
            m_code = &chunkCode;

            // do not run into this chunk start control code, but proceed with the code below
            currentInfo.m_ignoreBandControlCode = true;

            // Recurse, writing the code and collecting the terminals
            hkArray< hkpMoppPrimitiveId > chunkTerminals;
            assembleSubNodeIntoChunks( currentNode, parentInfo, currentInfo, chunkId, chunkTerminals );

            // Store the offset to the embedded data at the start of the chunk
            if (m_chunkInfo->m_compressor)
            {
                const int codeSize = m_code->getPos() + DATA_OFFSET_COMMAND_SIZE;
                const int padBytesToAlign = (COMPRESSED_DATA_ALIGNMENT - codeSize) & (COMPRESSED_DATA_ALIGNMENT-1);
                int offset = codeSize + padBytesToAlign; // Add space for the addDataOffset command
                HK_ASSERT(0x43cc4568, (offset & (COMPRESSED_DATA_ALIGNMENT-1)) == 0, "Misaligned Data");
                addDataOffset( offset, chunkTerminals.getSize() );
            }

            // Create new code that will have the embedded primitives
            chunk.m_code = new hkpMoppCodeGenerator( chunk.m_codeSize );

            // Embed the primitives
            HK_ON_DEBUG(int actualOffset = ) copyCodeAndEmbedTerminals(m_chunkInfo, chunk.m_code, m_code, chunkTerminals);
            HK_ASSERT(0x43cc4568, (actualOffset & (COMPRESSED_DATA_ALIGNMENT-1)) == 0, "Misaligned Data");
            m_code->m_jumpCommands.clear();

        }
        m_code = oldCode;
        assembleJumpChunkCommand( currentNode );
        return;
    }

    //
    //  terminal just assemble and return the size
    //
    if ( currentNode->m_isTerminal )
    {
        HK_ASSERT(0x454e32e4, (m_chunkInfo->m_compressor == HK_NULL) || ((currentNode->toTerminal()->m_primitive[0].m_primitiveID - currentInfo.m_terminalOffset) < 256), "ID doesn't fit");
        assembleCutAndTerminalCommand( currentNode, parentInfo, currentInfo );
        terminalsOut.pushBack( currentNode->toTerminal()->m_primitive[0].m_origPrimitiveID );
        return;
    }

    //
    // ****************************************
    // ******* top down pre-calculations *******
    // ****************************************
    //
    hkpMoppAssemblerRescaleCommand rescaleCommandInfo;
    rescaleCommandInfo.m_scaleCommand = -1;
    initTopDown( currentNode, parentInfo, currentInfo, rescaleCommandInfo );


    //
    // ***********************
    // ******* Recurse *******
    // ***********************
    //
    {
        hkpMoppTreeNode* childNode = currentNode->toNode()->m_rightBranch;
        hkpMoppAssemblerNodeInfo childInfo(currentInfo, childNode, m_codeInfo);
        assembleSubNodeIntoChunks(childNode,currentInfo, childInfo, chunkId, terminalsOut);
    }

    {
        hkpMoppTreeNode* childNode = currentNode->toNode()->m_leftBranch;
        hkpMoppAssemblerNodeInfo childInfo(currentInfo, childNode, m_codeInfo);
        assembleSubNodeIntoChunks(childNode,currentInfo, childInfo, chunkId, terminalsOut);
    }
    assembleNonTerminalCommand( currentNode, parentInfo, currentInfo, rescaleCommandInfo );
}


void hkpMoppDefaultAssembler::calcReoffset( hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo )
{
    currentInfo.m_terminalOffset = parentInfo.m_terminalOffset;

    //
    //  check whether our term4 command already works
    //
    const unsigned int currentOffset = currentNode->m_minPrimitiveId - parentInfo.m_terminalOffset;
    const unsigned int maxTermId = currentOffset + currentInfo.m_terminalSpread;
    if ( maxTermId < HK_MOPP_MAX_TERM4 )
    {
        return;
    }

    // do not insert an term8, if we have only two children which can be converted to a term4
    if ( currentInfo.m_terminalSpread <= 2 && maxTermId < 0x100 )
    {
        return;
    }


    if(   (currentInfo.m_terminalSpread < HK_MOPP_MAX_TERM4      && parentInfo.m_terminalSpread >=  HK_MOPP_MAX_TERM4  ) ||
        (currentInfo.m_terminalSpread < 0x100        && parentInfo.m_terminalSpread >=  0x100) ||
        (currentInfo.m_terminalSpread < 0x10000    && parentInfo.m_terminalSpread >=    0x10000)   )
    {
        // reoffset
        currentInfo.m_terminalOffset = currentNode->m_minPrimitiveId;
    }
}


void hkpMoppDefaultAssembler::addProperty(int property,int value)
{
    //add the property
    if (value < 0x0)
    {
        m_code->addCommand32(property + HK_MOPP_PROPERTY32_0,value);
    }
    else if (value < 0x100)
    {
        m_code->addCommand8(property + HK_MOPP_PROPERTY8_0,value);
    }
    else if ( value < 0x10000 )
    {
        m_code->addCommand16(property + HK_MOPP_PROPERTY16_0,value);
    }
    else
    {
        m_code->addCommand32(property + HK_MOPP_PROPERTY32_0,value);
    }
}

int hkpMoppDefaultAssembler::addJumpChunk( int chunkId )
{
    m_code->addCommand32( HK_MOPP_JUMP_CHUNK32, chunkId );
    return JUMP_CHUNK_COMMAND_SIZE;
}

int hkpMoppDefaultAssembler::addDataOffset( int offset, int numTerminals )
{
    m_code->addCode((numTerminals&0x0ff));
    m_code->addCode((numTerminals&0x0ff00)>>8);
    m_code->addCommand16( HK_MOPP_DATA_OFFSET, offset );
    return DATA_OFFSET_COMMAND_SIZE;
}

void hkpMoppDefaultAssembler::addRescale(const hkpMoppAssemblerRescaleCommand &cmd)
{
    m_code->addCode(cmd.m_moppOffset[2]);
    m_code->addCode(cmd.m_moppOffset[1]);
    m_code->addCode(cmd.m_moppOffset[0]);
    m_code->addCode(cmd.m_scaleCommand );
}

void hkpMoppDefaultAssembler::addTermIdOffset(hkUint32 offset)
{
    if ( offset < 0x100 )
    {
        m_code->addCommand8(HK_MOPP_TERM_REOFFSET8, offset);
    }
    else if (offset < 0x10000)
    {
        m_code->addCommand16(HK_MOPP_TERM_REOFFSET16, offset);
    }
    else
    {
        m_code->addCommand32(HK_MOPP_TERM_REOFFSET32, offset);
    }
}



void hkpMoppDefaultAssembler::addCut(const hkpMoppTreeNode* currentNode, const hkpMoppAssemblerNodeInfo& currentInfo)
{
    const hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

    for (int i=0; i< 3; i++)
    {
        if (   currentData->m_cutCommandType[i] == hkpMoppAssemblerData::HK_MOPP_CUT8 )
        {
            int mn = currentInfo.m_extents[i][0] - currentInfo.m_accumOffset[i];
            int mx = currentInfo.m_extents[i][1] - currentInfo.m_accumOffset[i];
            mn >>= currentInfo.m_currentBitsResolution;
            mx >>= currentInfo.m_currentBitsResolution;
            mx += 1;

            m_code->addCode(mx);
            m_code->addCode(mn);
            //this is the cut
            m_code->addCode(HK_MOPP_DOUBLE_CUT_X + i);
        }
        if (   currentData->m_cutCommandType[i] == hkpMoppAssemblerData::HK_MOPP_CUT24 )
        {
            int mn = currentInfo.m_extents[i][0];
            int mx = currentInfo.m_extents[i][1];
            mx += 1;

            m_code->addParam24(mx);
            m_code->addParam24(mn);
            //this is the cut
            m_code->addCode(HK_MOPP_DOUBLE_CUT24_X + i);
        }
    }
}


void hkpMoppDefaultAssembler::addSplit(int pright,int pleft,HK_MOPP_SPLIT_DIRECTIONS directionCode,int lpos,int rpos)
{

    // add a jump to the right branch
    //int npos = m_code->getPos();
    //int nsize = m_code->getSize();
    //the offset to the right branch
    int rightOffset = m_code->getPos() - rpos;


    if( directionCode < 3 )     // check for major axis split
    {
        int leftOffset = m_code->getPos() - lpos;

        // check for split jump commands
        //here we have to check that both jumps can be put into 16 bits, and that the left jump
        //will have to jump PAST the right jump
        if( ((leftOffset > 0x00) || (rightOffset > 0x0f8)) && (leftOffset < 0x10000) && (rightOffset < 0x10000))
        {
            m_code->addCode(rightOffset & 0x0ff);
            m_code->addCode( (rightOffset >> 8) & 0x0ff);
            m_code->addCode(leftOffset & 0x0ff);
            m_code->addCode( (leftOffset >> 8) & 0x0ff);
            m_code->addCode(pright);
            m_code->addCode(pleft);
            //this makes the command a split_jump
            m_code->addCode(directionCode + HK_MOPP_SPLIT_JUMP_X);
            return;
        }

        // check for single split
        if ((pleft - pright) == 1)
        {
            if (rightOffset > 251)
            {
                //the jump to the right branch
                addJump(rpos);
                rpos = m_code->getPos();
            }
            addJump(lpos);
            rpos = m_code->getPos() - rpos;
            //add the offset to the right branch (could be a jump)
            m_code->addCode(rpos);
            m_code->addCode(pright);
            //should be a nicer look to this addition - makes the
            m_code->addCode(directionCode + HK_MOPP_SINGLE_SPLIT_X);
            return;
        }
    }

    // normal split with optional jumps
    {
        if (rightOffset > 251)
        {
            //the jump to the right branch
            addJump(rpos);
            rpos = m_code->getPos();
        }

        addJump(lpos);
        rpos = m_code->getPos() - rpos;
        //add the offset to the right branch (could be a jump)
        m_code->addCode(rpos);
        m_code->addCode(pright);            //the right splitting plane
        m_code->addCode(pleft);             //the left splitting plane#
        //this is the split - codes are added in reverse
        m_code->addCode(directionCode + HK_MOPP_SPLIT_X);
    }
}

void hkpMoppDefaultAssembler::checkAndAddProperties( hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo  )
{
    for(int pcount = 0; pcount < hkpMoppCode::MAX_PRIMITIVE_PROPERTIES; ++pcount)
    {
        if(currentNode->m_parent)
        {
            if(( currentInfo.m_propertiesSpread[pcount] == 0) && (parentInfo.m_propertiesSpread[pcount] != 0) )
            {
                addProperty(pcount,currentInfo.m_propertiesOffset[pcount]);
            }
        }
        //if we are at the root node, and the current spread is 0, then we must add the property (won't have been added before)
        else
        {
            if( (currentInfo.m_propertiesSpread[pcount] == 0) && (currentInfo.m_propertiesOffset[pcount] != 0 ) )
            {
                addProperty(pcount,currentInfo.m_propertiesOffset[pcount]);
            }
        }
    }
}

int hkpMoppDefaultAssembler::addTerminals(hkpMoppTreeTerminal* thisNode, const hkpMoppAssemblerNodeInfo& parentInfo, const hkpMoppAssemblerNodeInfo& currentInfo)
{
    const int oldPos = m_code->getPos();


    {
        //it is possible that this terminal node has a different mesh ID to the other
        //branch from the parent, if this is so, then the property must be added
        //before the primitive

        hkpMoppPrimitiveId ID = thisNode->m_primitive[0].m_primitiveID;

        //bring into normalized range
        ID -= currentInfo.m_terminalOffset;

        if(ID < HK_MOPP_MAX_TERM4)
        {
            //we use this only if there's one
            m_code->addCommand5(HK_MOPP_TERM4_0, ID);                       //MIDLO
        }
        else if(ID < 0x100)
        {
            m_code->addCommand8(HK_MOPP_TERM8, ID);                         //MIDLO
        }
        else if(ID < 0x10000)
        {
            m_code->addCommand16(HK_MOPP_TERM16, ID);                       //MIDLO
        }
        else if (ID < 0x1000000)
        {
            m_code->addCommand24(HK_MOPP_TERM24, ID);                       //MIDLO
        }
        else
        {
            m_code->addCommand32(HK_MOPP_TERM32, ID);                       //MIDLO
        }

        //the bottom up approach demands that this be done AFTER the above, to
        //appear before it in the code
        for(int p = 0; p < thisNode->m_numProperties; p++)
        {
            if (thisNode->m_minPropertyValue[p] != 0 && parentInfo.m_propertiesSpread[p] != 0 )
            {
                addProperty(p, thisNode->m_minPropertyValue[p]);
            }
        }
    }
    
    //this size is needed in case we are using multiple terminals

    int size = m_code->getPos() - oldPos;
    return size;
}

void hkpMoppDefaultAssembler::addJump(unsigned int jumpPos)
{
    //the address of the jump position
    int jump = m_code->getPos() - jumpPos;

    HK_ASSERTV(0x63e65d78,  jump >= 0 , "Negative jump calculated, should always be positive{}", jumpPos);

    //if you're only jumping ahead 1, then you don't need to jump, it's next :)
    if(jump > 0)
    {
        if(jump < 0xff)
        {
            m_code->addCommand8(HK_MOPP_JUMP8, jump);
        }
        else if (jump < 0xffff)
        {
            m_code->addCommand16(HK_MOPP_JUMP16, jump);
        }
        else if (jump < 0xffffff)
        {
            m_code->addCommand24(HK_MOPP_JUMP24, jump);
        }
        else
        {
            m_code->addCommand32(HK_MOPP_JUMP32, jump);
        }
    }
}


HK_MOPP_SPLIT_DIRECTIONS hkpMoppDefaultAssembler::getSplitCode(const hkVector4& norm)
{
    HK_MOPP_SPLIT_DIRECTIONS directionCode;

    if(norm(0) > 0.1f)
    {
        directionCode = HK_MOPP_SD_X;
        if(norm(2) > 0.1f)
        {
            directionCode = HK_MOPP_SD_XZ;
            if(norm(1) > 0.1f)
                directionCode = HK_MOPP_SD_XYZ;
            else if(norm(1) < -0.1f)
                directionCode = HK_MOPP_SD_XMYZ;
        }
        else if (norm(2) < -0.1f)
        {
            directionCode = HK_MOPP_SD_XMZ;
            if(norm(1) > 0.1f)
                directionCode = HK_MOPP_SD_XYMZ;
            else if(norm(1) < -0.1f)
                directionCode = HK_MOPP_SD_XMYMZ;
        }
        else if(norm(1) > 0.1f) // ie z = 0
        {
            directionCode = HK_MOPP_SD_XY;
        }
        else if(norm(1) < -0.1f)
        {
            directionCode = HK_MOPP_SD_XMY;
        }

    }
    else if(norm(1) > 0.1f)
    {
        directionCode = HK_MOPP_SD_Y;
        if(norm(2) > 0.1f)
        {
            directionCode = HK_MOPP_SD_YZ;
        }
        else if (norm(2) < -0.1f)
        {
            directionCode = HK_MOPP_SD_YMZ;
        }
    }
    // ie z = 0 is the only case left
    else
    {
        directionCode = HK_MOPP_SD_Z;
    }

    return directionCode;
}

#define ROOT2 1.4142135623730951f
#define ROOT3 1.7320508075688773f

void hkpMoppDefaultAssembler::findPlanes(const hkpMoppTreeInternalNode* node,const hkpMoppAssemblerNodeInfo& info,int &pleft,int &pright)
{

    int numAxes = 0;
    //the number of negative axes
    int numNeg = 0;
    int direction = 0;

    for (int i = 0; i< 3; i++)
    {
        const hkReal x = node->getPlaneNormal()(i);
        if ( x == 0.0f )
        {
            continue;
        }

        direction = i;
        numAxes++;

        if( x < 0)  //ie -1
        {
            numNeg++;
        }
    }

    const hkVector4& normalised = node->getPlaneNormal();

    // must be double, even if this means software emulation on PlayStation(R)2
    double rp0 = node->m_planeRightPosition;
    double rp1 = node->m_planeLeftPosition;

    //
    //  int version
    //
    // very accurate solution for the main axis
    if ( numAxes == 1)
    {
        // now in fixedPointSpace
        const int intRp0 = hkMath::hkFloorToInt( hkReal( (rp0 - m_codeInfo.m_offset(direction)) * m_codeInfo.getScale() ) );
        const int intRp1 = hkMath::hkFloorToInt( hkReal( (rp1 - m_codeInfo.m_offset(direction)) * m_codeInfo.getScale() ) );

        const int imoppRp0 = intRp0 - info.m_accumOffset[direction];
        const int imoppRp1 = intRp1 - info.m_accumOffset[direction];

        // now in moppSpace
        int moppRp0 = (imoppRp0 >> info.m_currentBitsResolution);
        int moppRp1 = (imoppRp1 >> info.m_currentBitsResolution)+1;

        moppRp0 = hkMath::max2( moppRp0, 0 );
        moppRp0 = hkMath::min2( moppRp0, 0xff );
        moppRp1 = hkMath::max2( moppRp1, 0 );
        moppRp1 = hkMath::min2( moppRp1, 0xff );


        pright = moppRp0;
        pleft  = moppRp1;

        // fix for numerical accuracy problems

        if (pright < 0)
        {
            pright = 0;
        }

        return;
    }

    //
    // floating point version
    //

    hkSimdReal sscale; sscale.setFromFloat(m_codeInfo.getScale());
    hkSimdReal sInv; sInv.setReciprocal<HK_ACC_FULL,HK_DIV_SET_MAX>(sscale);
    hkVector4 t0; t0.set( hkReal(info.m_accumOffset[0]), hkReal(info.m_accumOffset[1]), hkReal(info.m_accumOffset[2]) );
    hkVector4 offset = m_codeInfo.m_offset;
    offset.addMul( sInv, t0);
    double toOrigin = double(normalised.dot<3>(offset).getReal());

    // Now in float space - note: the offset is in the bigger space
    rp0 -= toOrigin;
    rp1 -= toOrigin;

    rp0 *= m_codeInfo.getScale();
    rp1 *= m_codeInfo.getScale();

    rp0 /= hkReal(1 << info.m_currentBitsResolution);
    rp1 /= hkReal(1 << info.m_currentBitsResolution);

    //rp0 = floor( rp0 );   // now rp are in moppSpace
    //rp1 = floor( rp1 ) + 1.0f;

    //the scale of the space is bigger when there is more than 1 axis
    double factor0 = 1.0;
    double factor1 = 1.0;
    if(numAxes == 2)
    {
        factor0 = ROOT2;
        factor1 = 0.5;
    }
    else if(numAxes == 3)
    {
        factor0 = ROOT3;
        factor1 = 1.0 / 3.0;
    }
    //after that step, the distance has been normalised
    rp0 *= factor0;
    rp1 *= factor0;

    if (numNeg)
    {
        rp0 += HK_MOPP_RESOLUTION*(numNeg);
        rp1 += HK_MOPP_RESOLUTION*(numNeg);
    }

    rp0 *= factor1;
    rp1 *= factor1;

    int moppRp0 = hkMath::hkFloorToInt( hkReal(rp0) );
    int moppRp1 = hkMath::hkFloorToInt( hkReal(rp1) ) + 1;

    moppRp0 = hkMath::max2( moppRp0, 0 );
    moppRp0 = hkMath::min2( moppRp0, 0xff );
    moppRp1 = hkMath::max2( moppRp1, 0 );
    moppRp1 = hkMath::min2( moppRp1, 0xff );

    pright  = moppRp0;
    pleft   = moppRp1;
}


void hkpMoppDefaultAssembler::getScaleInfo( hkpMoppTreeNode* rootNode, hkpMoppCode::CodeInfo* moppCodeInfo )
{
    //update the origin offset
    moppCodeInfo->m_offset.set(     rootNode->m_extents.m_extent[0].m_min,
        rootNode->m_extents.m_extent[1].m_min,
        rootNode->m_extents.m_extent[2].m_min);

    //update the scaling info
    hkReal worldSize = 0.0f;
    for (int k=0;k<3;k++)
    {
        worldSize = hkMath::max2( worldSize, rootNode->m_extents.m_extent[k].m_max - rootNode->m_extents.m_extent[k].m_min );
    }


    moppCodeInfo->m_offset(3) = 0.0f;
    //now that we have the scaling info, we put it into a proper conversion format
    //now is scaled down to 0..254 and binary left shifted by 16
    moppCodeInfo->setScale( (1<<HK_MOPP_FIXED_POINT_BITS) * ( hkReal(HK_MOPP_RESOLUTION-1) / worldSize));
}

int hkpMoppDefaultAssembler::assembleCutAndTerminalCommand( hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo )
{
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

    fixScale( currentNode );
    m_numNodesAssembled ++;
    hkpMoppTreeTerminal* term = currentNode->toTerminal();
    //the position in code before this sub-tree was built
    const int oldCodePos = m_code->getPos();

    addTerminals( term, parentInfo, currentInfo );

    addCut(currentNode, currentInfo);

    currentData->m_assembledAddress = m_code->getPos();
    int size = m_code->getPos() - oldCodePos;
    currentData->m_isAssembled = true;
    return size;
}

int hkpMoppDefaultAssembler::assembleNonTerminalCommand(hkpMoppTreeNode* currentNode,const hkpMoppAssemblerNodeInfo &parentInfo, hkpMoppAssemblerNodeInfo& currentInfo, hkpMoppAssemblerRescaleCommand& rescaleCommandInfo)
{

    // fix current scaling
    fixScale( currentNode );


    //the position in code before this sub-tree was built
    const int oldCodePos = m_code->getPos();

    //
    // add split command
    //
    {
        const hkVector4 &norm                   = currentNode->toNode()->getPlaneNormal();
        HK_MOPP_SPLIT_DIRECTIONS directionCode  = getSplitCode(norm);
        int pleft,pright;                       findPlanes(currentNode->toNode(),currentInfo,pleft,pright);
        int lpos, rpos;
        {
            hkpMoppTreeNode*      childNode = currentNode->toNode()->m_rightBranch;
            hkpMoppAssemblerData* childData = &childNode->m_assemblerData;
            rpos = childData->m_assembledAddress;
        }
        {
            hkpMoppTreeNode*      childNode = currentNode->toNode()->m_leftBranch;
            hkpMoppAssemblerData* childData = &childNode->m_assemblerData;
            lpos = childData->m_assembledAddress;
        }

        //add the bytes for the split command
        addSplit(pright,pleft,directionCode,lpos,rpos);
    }


    //
    // add term offset command
    //
    {
        unsigned int offset = currentInfo.m_terminalOffset - parentInfo.m_terminalOffset;
        if(offset)
        {
            addTermIdOffset(offset);
        }
    }

    //
    // add property offset commands if needed
    //
    {
        checkAndAddProperties( currentNode, parentInfo, currentInfo );
    }

    //
    // add cut command
    //
    {
        addCut(currentNode, currentInfo);
    }

    //
    // add rescale command
    //
    if(rescaleCommandInfo.m_doRescale)
    {
        addRescale(rescaleCommandInfo);
    }

    //we've gotten to the code addition for this particular node
    //we now note that it has been compiled - and store the address
    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;
    currentData->m_assembledAddress = m_code->getPos();
    currentData->m_isAssembled = true;


    int codePos = m_code->getPos();
    return codePos - oldCodePos;
}


int hkpMoppDefaultAssembler::assembleJumpChunkCommand( hkpMoppTreeNode* currentNode )
{
    //the position in code before this sub-tree was built
    const int oldCodePos = m_code->getPos();

    hkpMoppAssemblerData* currentData = &currentNode->m_assemblerData;

    // add jump chunk
    addJumpChunk( currentData->m_chunkId );

    // Set / reset assembled address for this node
    int codePos = m_code->getPos();
    currentData->m_assembledAddress = codePos;
    currentData->m_isAssembled = true;

    hkpMoppCodeGenerator::JumpCommandInfo& jumpInfo = m_code->m_jumpCommands.expandOne();
    jumpInfo.m_chunkId =  currentData->m_chunkId;
    jumpInfo.m_position = codePos;

    m_code->validateJumpCommands();

    return codePos - oldCodePos;
}

//
// assemble
// assemble the tree into BV machine code
// returns MoppCode on success, NULL on failure
//

void  hkpMoppDefaultAssembler::assemble(hkpMoppTreeNode* rootNode, class hkpMoppNodeMgr* mgr, int minAssembledElems )
{
    m_nodeMgr = mgr;
    m_numNodesAssembled = 0;
    m_minAssembledElems = minAssembledElems;

    if (rootNode == HK_NULL)
    {
        return;
    }

    getScaleInfo( rootNode, &this->m_codeInfo );

    // assemble code

    //
    //  calculate log2(treesize) as an initial guess for the tree depth
    //

    //set the level to be (it will be incremented to 0 in assembleSubNode)
    {
        //initial guess at the depth of the trees
        m_minDepthToBuild = 0;
        unsigned int s = rootNode->m_maxPrimitiveId;
        while ( s )
        {
            s = s>>1;
            m_minDepthToBuild++;
        }
    }

    m_halfTreeDepth = m_minDepthToBuild>>1;
    m_minDepthToBuild = (m_minDepthToBuild / m_params.m_groupLevels) * m_params.m_groupLevels;  // align for groupLevels
    m_minDepthToBuild += m_params.m_groupLevels + m_params.m_groupLevels -1;                    // some extra safety adds for slightly unbalanced trees

    hkpMoppAssemblerNodeInfo parentInfo( rootNode, m_codeInfo );

    //
    // calculate the scale positions
    //
    {
        hkpMoppAssemblerNodeInfo currentInfo( parentInfo, rootNode, m_codeInfo );
        preCalcScale(rootNode,parentInfo,currentInfo);
    }

    //
    //  calculate the cut positions
    //
    {
        hkpMoppAssemblerCutInfo  parentCutInfo( rootNode, m_codeInfo );
        hkpMoppAssemblerCutInfo currentInfo( parentCutInfo, rootNode, m_codeInfo );
        preCalcCutPositions(rootNode,parentCutInfo,currentInfo);
    }

    //
    //  assemble the tree
    //
    hkpMoppAssemblerNodeInfo currentInfo( parentInfo, rootNode, m_codeInfo );

    if ( ! m_chunkInfo)
    {
        //rootNode->m_doCommands &= ~(HK_MOPP_DO_CUT_X | HK_MOPP_DO_CUT_Y | HK_MOPP_DO_CUT_Z );     // no cut commands at the root node
        assemblesubNode(rootNode,parentInfo,currentInfo);
    }
    else
    {
        // assemble tree into chunks
        rootNode->m_assemblerData.m_chunkId = 0; // root gets first chunk id = 0
        m_chunkInfo->m_chunks.expandOne();

        // 1st pass - work out chunk sizes and parameters
        hkpMoppCodeGenerator* oldCode = m_code;
        {
            hkpMoppAssemblerNodeInfo dummyInfo( parentInfo, rootNode, m_codeInfo );
            hkArray< hkpMoppPrimitiveId > dummyPrims;
            hkpMoppCodeGenerator dummyCode;
            m_code = &dummyCode;

            calcChunkPoints( rootNode, parentInfo, dummyInfo, dummyPrims );
            calcTerminalIdsForChunks( rootNode, rootNode->m_assemblerData.m_chunkId, 0 );
        }

        // 2nd pass - assemble the data
        {
            hkArray< hkpMoppPrimitiveId > rootTerminals;
            hkpMoppCodeGenerator rootChunkCode;
            m_code = &rootChunkCode;

            assembleSubNodeIntoChunks(rootNode, parentInfo, currentInfo, rootNode->m_assemblerData.m_chunkId, rootTerminals );

            if (m_chunkInfo->m_compressor)
            {
                const int codeSize = m_code->getPos() + DATA_OFFSET_COMMAND_SIZE;
                const int padBytesToAlign = (COMPRESSED_DATA_ALIGNMENT - codeSize) & (COMPRESSED_DATA_ALIGNMENT-1);
                int offset =  codeSize + padBytesToAlign; // Add space for this addDataOffset Command
                addDataOffset( offset, rootTerminals.getSize() );
            }

            m_code = oldCode;
            copyCodeAndEmbedTerminals(m_chunkInfo, m_code, &rootChunkCode, rootTerminals);

            hkpMoppCompilerChunkInfo::Chunk& rootChunk = m_chunkInfo->m_chunks[ rootNode->m_assemblerData.m_chunkId ];
            rootChunk.m_code = m_code;
            rootChunk.m_code->addReference();
            rootChunk.m_codeSize = m_code->getPos();
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
