#pragma once		// Include this file only once

#ifndef __PHYSICS_MESH_SHAPE_H
#define __PHYSICS_MESH_SHAPE_H

#include <Physics2012/Collide/Shape/Compound/Collection/hkpShapeCollection.h>

#if defined(HK_PLATFORM_HAS_SPU)
	#include <Common/Base/Spu/Config/hkSpuConfig.h>
#endif // #if defined(HK_PLATFORM_HAS_SPU)

// --------------------
//
// Defines/Macros
//
// --------------------

// This is stored in the hkpShapeCollection's user data to identify it as a physics_mesh_shape.
#define PHYSICS_MESH_USER_DATA_ID 0xc0113c7

// --------------------
//
// Enumerated types
//
// --------------------

// Collision filters that can by set per triangle in the physics mesh.
enum physics_mesh_collision_filters {

	PMCF_CAMERA_NO_COLLIDE = 0,
	PMCF_BULLET_NO_COLLIDE,
	PMCF_AI_LOS_NO_COLLIDE,	
	PMCF_PLAYER_COLLIDE_ONLY,		
	PMCF_PLAYER_COLLIDE_ONLY_DEMO, 
	PMCF_PROJECTILE_NO_COLLIDE,	
	PMCF_HUMAN_COLLIDE_ONLY,
	PMCF_NO_COLLISION,
	PMCF_WORLD,

	PMCF_NUM_COLLISION_FILTERS
};

// Per triangle flags.
// NOTE: These are really just mutually exclusive collision filters and we should remove this and just
//			use physics_mesh_collision_filters instead.
enum physics_mesh_triangle_flags {

	PMTF_CAMERA_NO_COLLIDE			= 1 << PMCF_CAMERA_NO_COLLIDE,
	PMTF_BULLET_NO_COLLIDE			= 1 << PMCF_BULLET_NO_COLLIDE,
	PMTF_AI_LOS_NO_COLLIDE			= 1 << PMCF_AI_LOS_NO_COLLIDE,	
	PMTF_PLAYER_COLLIDE_ONLY		= 1 << PMCF_PLAYER_COLLIDE_ONLY,		
	PMTF_PLAYER_COLLIDE_ONLY_DEMO	= 1 << PMCF_PLAYER_COLLIDE_ONLY_DEMO, 
	PMTF_PROJECTILE_NO_COLLIDE		= 1 << PMCF_PROJECTILE_NO_COLLIDE,	
	PMTF_HUMAN_COLLIDE_ONLY			= 1 << PMCF_HUMAN_COLLIDE_ONLY,
	// Not a flag - PMCF_NO_COLLISION,
	// Not a flag - PMCF_WORLD,
};

// --------------------
//
// Structures/Classes
//
// --------------------

// Auxiliary data associated with a triangle.
struct physics_mesh_triangle_aux_data {

	// The material.
	hkUint8 m_collision_material;

	// Flags, see physics_mesh_collision_filter_flags.
	hkUint8 m_flags;
};

// Triangle mesh shape.
class physics_mesh_shape : public hkpShapeCollection
{
	public:

		HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_CDINFO);
		HK_DECLARE_REFLECTION();
		HKCD_DECLARE_SHAPE_TYPE(hkcdShapeType::V_PHYSICS_MESH);

	public:

		enum constants {

			// Number of bits to use for the first vertex index in a triangle.
			NUM_FIRST_VERTEX_INDEX_BITS = 18,

			// Number of bits to use for the offset of vertex indices.  The second and third
			// vertex indices of a triangle use offsets from the first vertex index.
			NUM_SECOND_VERTEX_INDEX_OFFSET_BITS = 7,
			NUM_THIRD_VERTEX_INDEX_OFFSET_BITS = 6,

			// The number of bits to use for flags (flip normal flag).
			NUM_SHAPE_KEY_FLAG_BITS = 1,

			// Mask used for extracting the vertex index of the first vertex.
			FIRST_VERTEX_INDEX_MASK = (1 << NUM_FIRST_VERTEX_INDEX_BITS) - 1,

			// Masks used for extracting the vertex index offsets for the second and third vertices.
			SECOND_VERTEX_INDEX_OFFSET_MASK = (1 << NUM_SECOND_VERTEX_INDEX_OFFSET_BITS) - 1,
			THIRD_VERTEX_INDEX_OFFSET_MASK = (1 << NUM_THIRD_VERTEX_INDEX_OFFSET_BITS) - 1,

			// Mask used to extract the flags (flip normal flag).
			SHAPE_KEY_FLAGS_MASK = (1 << NUM_SHAPE_KEY_FLAG_BITS) - 1,

			// Mask used for extracting the aux data index.
			AUX_DATA_INDEX_MASK = FIRST_VERTEX_INDEX_MASK,

			// Amount to shift the shape key to get the offset to the second vertex index.
			SECOND_VERTEX_INDEX_OFFSET_SHIFT	= NUM_FIRST_VERTEX_INDEX_BITS,

			// Amount to shift the shape key to get the offset to the third vertex index.
			THIRD_VERTEX_INDEX_OFFSET_SHIFT = NUM_FIRST_VERTEX_INDEX_BITS + NUM_SECOND_VERTEX_INDEX_OFFSET_BITS,		

			// Amount to shift the shape key to get the flags.
			SHAPE_KEY_FLAGS_SHIFT = NUM_FIRST_VERTEX_INDEX_BITS + NUM_SECOND_VERTEX_INDEX_OFFSET_BITS + NUM_THIRD_VERTEX_INDEX_OFFSET_BITS,		

			// Flags for the mesh.
			MESH_FLAG_DEMO_COLLISION_ENABLED = 0,
		};

	#if !defined(HK_PLATFORM_SPU)

		// Constructor.
		physics_mesh_shape(hkReal const* vert_buf, hkUint32 num_vertices, hkVector4 const& aabb_min, hkVector4 const& aabb_max,
						 hkUint32 num_triangle_aux_data, physics_mesh_triangle_aux_data const* p_triangle_aux_data);

	//#ifdef HK_PLATFORM_HAS_SPU
		// Default Constructor, only used to create Vtable on PS3
		physics_mesh_shape() {}
	//#endif

		//physics_mesh_shape( hkFinishLoadedObjectFlag flag ) : hkpShapeCollection(flag) {setType(HKCD_SHAPE_TYPE_FROM_CLASS(physics_mesh_shape));}
		// DSFL jfk

	#endif // #if !defined(HK_PLATFORM_SPU)

		//
		// hkpShape interface.
		//
		virtual void getAabb( const hkTransform& localToWorld, hkReal tolerance, hkAabb& out ) const;

	#if !defined(HK_PLATFORM_SPU)

		virtual int calcSizeForSpu(const CalcSizeForSpuInput& input, int spu_buffer_size_left) const;

		//
		// hkpShapeContainer interface.
		//
		/// The number of child shapes. The default implementation just iterates over all keys and is really slow
		virtual int getNumChildShapes() const;

		/// Get the first child shape key
		/// see getChildShape() for extra details
		virtual hkpShapeKey getFirstKey() const;

		/// Get the next child shape key
		/// If the "oldKey" parameter is the last key in the shape collection, this function
		/// returns HK_INVALID_SHAPE_KEY
		/// see getChildShape() for extra details
		virtual hkpShapeKey getNextKey(hkpShapeKey old_key) const;

	#endif // #if !defined(HK_PLATFORM_SPU)

		virtual const hkpShape* getChildShape( hkpShapeKey key, hkpShapeBuffer& buffer ) const;

		// hkpShapeContainer::getCollisionFilterInfo() interface implementation.
		virtual hkUint32 getCollisionFilterInfo( hkpShapeKey key ) const;

	#if !defined(HK_PLATFORM_SPU)

		/// Return whether welding should be enabled for this shape container
		virtual bool isWeldingEnabled() const { return false; }

	#endif // #if !defined(HK_PLATFORM_SPU)

		//
		// physics_mesh_shape interface.
		//

		// Get auxiliary data associated with a triangle.
		bool get_auxiliary_data(physics_mesh_triangle_aux_data& aux_data_out, hkpShapeKey shape_key) const;

	#if !defined(HK_PLATFORM_SPU)

		// Calculate the triangle normal.
		bool calculate_normal(hkVector4& hk_normal, hkpShapeKey shape_key) const;

		// Enable/disable demo collision.
		void enable_demo_collision(bool enable);

	#endif // #if !defined(HK_PLATFORM_SPU)

		// Shell radius around the triangles.
		inline hkReal get_triangle_radius() const { return m_triangle_radius; }

	private:

		// Cached AABB
		hkVector4 m_aabb_half_extents;
		hkVector4 m_aabb_center;

		// List of vertices.
		hkUint32 m_num_vertices;
		const hkReal* m_vertices;

		// Array of physics_mesh_triangle_aux_data structures.  This is the auxiliary data
		// for triangles.
		hkUint32 m_num_triangle_aux_data;
		const physics_mesh_triangle_aux_data* m_triangle_aux_data;

		// List of collision filters indexed by PMCF_*.
		hkUint32 m_collision_filter_table[ PMCF_NUM_COLLISION_FILTERS ];

		// Shell radius around the triangles.
		hkReal m_triangle_radius;

		// Flags for the mesh. See MESH_FLAG_* above.
		hkUint8 m_flags;
};

#if defined(HK_PLATFORM_HAS_SPU)
	HK_COMPILE_TIME_ASSERT(sizeof(physics_mesh_shape) <= HK_SPU_MAXIMUM_SHAPE_SIZE);
#endif

// --------------------
//
// Variables
//
// --------------------


// --------------------
//
// Prototypes
//
// --------------------

#if !defined(HK_PLATFORM_SPU)

// Initialize the physics mesh system.
void physics_mesh_initialize(const hkUint32 collision_filter_table[ PMCF_NUM_COLLISION_FILTERS ]);

#endif // #if !defined(HK_PLATFORM_SPU)

#endif // __PHYSICS_MESH_SHAPE_H
