// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0


#include <Physics2012/Collide/hkpCollide.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Compound/Tree/hkpBvTreeShape.h>
#include <Physics2012/Collide/Shape/Query/hkpRayShapeCollectionFilter.h>
#include <Physics2012/Collide/Shape/Query/hkpShapeRayCastInput.h>

#include <Common/Base/Container/BitField/hkBitField.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>

#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/Code/hkpMoppCode.h>

#include <Common/Base/Algorithm/Collide/1AxisSweep/hk1AxisSweep.h>

// HAVOK 2010.2 MODIFIED CODE JJZ 3.11.2010 ////////////////////////////////////////////////////////
//
// Don't transform the individual aabbs of the child shapes in getAabbImpl,
// just transform the aabb of the list shape.
#define __FAST_GET_AABB_IMPL
//
// HAVOK 2010.2 MODIFIED CODE JJZ 3.11.2010 ////////////////////////////////////////////////////////

hkpListShape::hkpListShape(const hkpShape*const* shapeArray, int numShapes, hkpShapeContainer::ReferencePolicy ref)
: hkpShapeCollection( HKCD_SHAPE_TYPE_FROM_CLASS(hkpListShape), COLLECTION_LIST )
{
    m_childInfo.reserve(4); // so that our array gets aligned16 to be downloaded to the spu

    setShapes( shapeArray, numShapes, HK_NULL, ref );
    // no need to call recalcAabbExtents explicitly since it is called in setShapes

    for (int i = 0; i < MAX_DISABLED_CHILDREN/32; i++)
    {
        m_enabledChildren[i] = unsigned(-1);
    }
    m_numDisabledChildren = 0;
    m_flags = ALL_FLAGS_CLEAR;
}

void hkpListShape::afterReflectNew()
{
    setType(HKCD_SHAPE_TYPE_FROM_CLASS(hkpListShape));
    m_collectionType = COLLECTION_LIST;
}



void hkpListShape::setShapes( const hkpShape*const* shapeArray, int numShapes, const hkUint32* filterInfo, hkpShapeContainer::ReferencePolicy ref )
{
    HK_ASSERT(0x282822c7,  m_childInfo.getSize()==0, "You can only call setShapes once during construction.");
    HK_ASSERT(0x221e5b17,  numShapes, "You cannot create a hkpListShape with no child shapes" );

    m_childInfo.setSize(numShapes);
    for (int i = 0; i < numShapes; i++)
    {
        if (shapeArray[i] != HK_NULL)
        {
            m_childInfo[i].m_shape = shapeArray[i];
            m_childInfo[i].m_collisionFilterInfo = filterInfo? filterInfo[i] : 0;
            m_childInfo[i].m_numChildShapes = numShapes;
            m_childInfo[i].m_shapeSize = 0;
            m_childInfo[i].m_shapeInfo = 0;
        }
    }

    if (ref == hkpShapeContainer::REFERENCE_POLICY_INCREMENT)
    {
        hkReferencedObject::addReferences(&m_childInfo[0].m_shape, m_childInfo.getSize(), sizeof(m_childInfo[0]));
    }

    recalcAabbExtents();
}

void hkpListShape::disableChild( hkpShapeKey index )
{
    HK_ASSERT( 0xf0f34fe5, index < MAX_DISABLED_CHILDREN && int(index) < m_childInfo.getSize(), "You can only disable the first 256 children" );
    int bitPattern = ~(1<<(index&0x1f));
    int i = index>>5;
    int value = m_enabledChildren[ i ];
    int newVal = value & bitPattern;
    if ( value != newVal )
    {
        m_enabledChildren[i] = newVal;
        m_numDisabledChildren++;
    }
}

/// Allows for quickly enabling a child shape.
void hkpListShape::enableChild( hkpShapeKey index )
{
    HK_ASSERT( 0xf0f34fe6, index < MAX_DISABLED_CHILDREN && int(index) < m_childInfo.getSize(), "You can only disable the first 256 children" );
    int bitPattern = (1<<(index&0x1f));
    int i = index>>5;
    int value = m_enabledChildren[ i ];
    int newVal = value | bitPattern;
    if ( value != newVal )
    {
        m_enabledChildren[i] = newVal;
        m_numDisabledChildren--;
    }
}

void hkpListShape::setEnabledChildren( const hkBitField& enabledChildren )
{
    HK_ASSERT( 0xf03465fe, enabledChildren.getSize() == m_childInfo.getSize(), "Your bitfield does not match the list shape" );
    HK_ASSERT( 0xf03465fe, enabledChildren.getSize() <= 256, "Your bitfield is too large, you can only disable 256 children" );

    const hkUint32* HK_RESTRICT enabledChildrenWords = enabledChildren.getWords();
    for (int i =0; i < enabledChildren.getNumWords(); i++)
    {
        m_enabledChildren[i] = enabledChildrenWords[i];
    }
    m_numDisabledChildren = hkUint16(m_childInfo.getSize() - enabledChildren.bitCount());
}


hkBool hkpListShape::castRay(const hkpShapeRayCastInput& input, hkpShapeRayCastOutput& results) const
{
    HK_TIMER_BEGIN("rcList",HK_NULL);

    const hkpShapeRayCastOutput originalResults = results;  

    hkpShapeKey bestKey = HK_INVALID_SHAPE_KEY;

    if ( !input.m_rayShapeCollectionFilter )
    {
        for (int i = 0; i < m_childInfo.getSize(); i++)
        {
            if (isChildEnabled(i))
            {
                const hkpShape* childShape = m_childInfo[i].m_shape;
                results.setKey(i);
                results.changeLevel(1);
                if ( childShape->castRay( input, results ) )
                {
                    bestKey = i;
                }
                results.changeLevel(-1);
            }
        }
    }
    else
    {
        for (int i = 0; i < m_childInfo.getSize(); i++)
        {
            if ( isChildEnabled(i) && (false!=input.m_rayShapeCollectionFilter->isCollisionEnabled( input, *this, i )) )
            {
                const hkpShape* childShape = m_childInfo[i].m_shape;
                results.setKey(i);
                results.changeLevel(1);
                if ( childShape->castRay( input, results ) )
                {
                    bestKey = i;
                }
                results.changeLevel(-1);
            }
        }
    }

    results.setKey(bestKey);

    if (bestKey == HK_INVALID_SHAPE_KEY)
    {
        results = originalResults;
    }

    HK_TIMER_END();
    return bestKey != HK_INVALID_SHAPE_KEY;
}

void hkpListShape::castRayWithCollector(const hkpShapeRayCastInput& input, const hkpCdBody& cdBody, hkpRayHitCollector& collector) const
{
// copy of castRayImpl() with modifications

    HK_TIMER_BEGIN("rcList",HK_NULL);

    if ( !input.m_rayShapeCollectionFilter )
    {
        for (int i = 0; i < m_childInfo.getSize(); i++)
        {
            if (isChildEnabled(i))
            {
                const hkpShape* childShape = m_childInfo[i].m_shape;
                hkpCdBody childBody(&cdBody);
                childBody.setShape(childShape, i);
                childShape->castRayWithCollector( input, childBody, collector );
            }
        }
    }
    else
    {
        for (int i = 0; i < m_childInfo.getSize(); i++)
        {
            if ( isChildEnabled(i) && (false!=input.m_rayShapeCollectionFilter->isCollisionEnabled( input, *this, i )) )
            {
                const hkpShape* childShape = m_childInfo[i].m_shape;
                hkpCdBody childBody(&cdBody);
                childBody.setShape(childShape, i);
                childShape->castRayWithCollector( input, childBody, collector );
            }
        }
    }

    HK_TIMER_END();
}

void hkpListShape::getAabb(const hkTransform& localToWorld, hkReal tolerance, hkAabb& out ) const
{
    // HAVOK 2010.2 MODIFIED CODE JJZ 3.11.2010 ////////////////////////////////////////////////////////
    //
    // Don't transform the individual aabbs of the child shapes in getAabbImpl,
    // just transform the aabb of the list shape.
#if defined(__FAST_GET_AABB_IMPL)

    // Don't transform the individual aabbs of the child shapes, just transform the aabb of the list shape.
    hkpListShape const* thisObj = static_cast<hkpListShape const*>(HK_GET_THIS_PTR);
    hkAabbUtil::calcAabb(localToWorld, thisObj->m_aabbHalfExtents, thisObj->m_aabbCenter, hkSimdFloat32::fromFloat(tolerance), out);

#else
    // On cpu/ppu we will recalculate the hkpListShape's AABB on the fly by calculating the AABBs of its
    // children. The reason for this is that the costs for this function are much smaller than dealing
    // with an too large AABB.

    // Set up for SPU. <<sk.todo.aa allow serialization to do this calc on load.
    if ( m_aabbHalfExtents.lessEqualZero().allAreSet() )
    {
        (const_cast<hkpListShape*>(this))->recalcAabbExtents();
    }

    out.m_min.setAll( hkSimdReal_Max*hkSimdReal_Half ); 
    out.m_max.setNeg<4>( out.m_min);

    hkAabb t;
    for (int i = 0; i < m_childInfo.getSize(); i++)
    {
        const hkpShape* childShape = m_childInfo[i].m_shape;
        childShape->getAabb( localToWorld, tolerance, t );
        out.m_min.setMin( out.m_min, t.m_min );
        out.m_max.setMax( out.m_max, t.m_max );
    }
#endif // #if defined(__FAST_GET_AABB_IMPL)
    //
    // HAVOK 2010.2 MODIFIED CODE JJZ 3.11.2010 ////////////////////////////////////////////////////////

}

HK_COMPILE_TIME_ASSERT( sizeof(hkAabbUint32) == sizeof(hk1AxisSweep::AabbInt) );

void hkpListShape::recalcAabbExtents( )
{
    hkAabb aabb;
    recalcAabbExtents( aabb );
}

void hkpListShape::recalcAabbExtents( hkAabb& aabb )
{
    m_childInfo[0].m_shape->getAabb( hkTransform::getIdentity(), 0.0f, aabb );

    hkAabb t;
    {
        for (int i = 1; i < m_childInfo.getSize(); i++)
        {
            m_childInfo[i].m_shape->getAabb( hkTransform::getIdentity(), 0.0f, t );
            aabb.m_min.setMin( aabb.m_min, t.m_min );
            aabb.m_max.setMax( aabb.m_max, t.m_max );
        }
    }
    aabb.getCenter( m_aabbCenter );
    aabb.getHalfExtents( m_aabbHalfExtents );
}


const hkpShape* hkpListShape::getChildShape(hkpShapeKey key, hkpShapeBuffer& buffer) const
{
    return m_childInfo[key].m_shape;
}

hkUint32 hkpListShape::getCollisionFilterInfo(hkpShapeKey key) const
{
    return m_childInfo[ key ].m_collisionFilterInfo;
}

void hkpListShape::setCollisionFilterInfo( hkpShapeKey index, hkUint32 filterInfo )
{
    m_childInfo[ index ].m_collisionFilterInfo = filterInfo;
}

int hkpListShape::calcSizeForSpu(const CalcSizeForSpuInput& input, int spuBufferSizeLeft) const
{
    return -1;
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
