// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

template <typename CONFIG>
inline int  hkpGroupFilterBase<CONFIG>::getNewSystemGroup()
{
    return ++m_nextFreeSystemGroup;
}

template <typename CONFIG>
/*static*/ HK_INLINE hkUint32 HK_CALL hkpGroupFilterBase<CONFIG>::calcFilterInfo( int layer, int systemGroup, int subSystemId, int subSystemDontCollideWith )
{
    HK_ASSERT_NO_MSG(0x1902b596,  (layer >= 0)                      && (layer < (1 << NUM_LAYER_BITS)) );
    HK_ASSERT_NO_MSG(0x1902b597,  (subSystemId >= 0)                && (subSystemId < (1 << NUM_SUB_SYSTEM_BITS)) );
    HK_ASSERT_NO_MSG(0x1902b598,  (subSystemDontCollideWith >= 0)   && (subSystemDontCollideWith < (1 << NUM_DONT_COLLIDE_BITS)) );
    HK_ASSERT_NO_MSG(0x5ae6770c,  (systemGroup >= 0)                && (systemGroup < (1 << NUM_SYSTEM_GROUP_BITS)));

    return hkUint32(    (subSystemId << NUM_SUB_SYSTEM_SHIFT)                   |
                        (subSystemDontCollideWith << NUM_DONT_COLLIDE_SHIFT)    |
                        (systemGroup << NUM_SYSTEM_GROUP_SHIFT)                 |
                        (layer << NUM_LAYER_SHIFT));
}

template <typename CONFIG>
/*static*/ HK_INLINE int HK_CALL hkpGroupFilterBase<CONFIG>::getLayerFromFilterInfo( hkUint32 filterInfo )
{
    return (filterInfo >> NUM_LAYER_SHIFT) & ((1 << NUM_LAYER_BITS) - 1);
}

template <typename CONFIG>
int hkpGroupFilterBase<CONFIG>::setLayer( hkUint32 filterInfo, int newLayer )
{
    hkUint32 collisionLayerMask = 0xffffffff - ((1 << NUM_LAYER_BITS) - 1);
    return newLayer + ( collisionLayerMask & filterInfo);
}

template <typename CONFIG>
/*static*/ HK_INLINE int HK_CALL hkpGroupFilterBase<CONFIG>::getSystemGroupFromFilterInfo( hkUint32 filterInfo )
{
    return (filterInfo >> NUM_SYSTEM_GROUP_SHIFT) & ((1 << NUM_SYSTEM_GROUP_BITS) - 1);
}

template <typename CONFIG>
/*static*/ HK_INLINE int HK_CALL hkpGroupFilterBase<CONFIG>::getSubSystemIdFromFilterInfo( hkUint32 filterInfo )
{
    return (filterInfo >> NUM_SUB_SYSTEM_SHIFT) & ((1 << NUM_SUB_SYSTEM_BITS) - 1);
}

template <typename CONFIG>
/*static*/ HK_INLINE int HK_CALL hkpGroupFilterBase<CONFIG>::getSubSystemDontCollideWithFromFilterInfo( hkUint32 filterInfo )
{
    return (filterInfo >> NUM_DONT_COLLIDE_SHIFT) & ((1 << NUM_DONT_COLLIDE_BITS) - 1);
}

#if !defined(HK_PLATFORM_SPU)

template <typename CONFIG>
hkpGroupFilterBase<CONFIG>::hkpGroupFilterBase()
{
    m_type = HK_FILTER_GROUP;
    // Initially enable all collision groups
    for (int i=0; i<COLLISION_LUT_SIZE; i++)
    {
        m_collisionLookupTable[i] = Mask(-1);
    }
    m_nextFreeSystemGroup = 0;
}
#else
HK_COMPILE_TIME_ASSERT(sizeof(hkpGroupFilter)==256);
#endif

template <typename CONFIG>
hkBool hkpGroupFilterBase<CONFIG>::isCollisionEnabled(hkUint32 infoA, hkUint32 infoB) const
{
    // If the objects are in the same system group, but not system group 0,
    // then the decision of whether to collide is based exclusively on the
    // objects' SubSystemId and SubSystemDontCollideWith.
    // Otherwise, the decision is based exclusively on the objects' layers.

    const hkUint32 zeroIfSameSystemGroup = getSystemGroupFromFilterInfo(infoA^infoB);

    // check for identical system groups
    if( zeroIfSameSystemGroup == 0)
    {
        // check whether system group was set (nonzero)
        const hkUint32 sysGroupA = getSystemGroupFromFilterInfo(infoA);
        if( sysGroupA != 0 )
        {
            // check whether we allow collisions
            const int idA = getSubSystemIdFromFilterInfo(infoA);
            const int dontCollideB = getSubSystemDontCollideWithFromFilterInfo(infoB);
            if( idA == dontCollideB )
            {
                return false;
            }

            const int idB = getSubSystemIdFromFilterInfo(infoB);
            const int dontCollideA = getSubSystemDontCollideWithFromFilterInfo(infoA);
            return ( idB - dontCollideA );
        }
    }

    // use the layers to decide
    const Mask layerBitsA = m_collisionLookupTable[getLayerFromFilterInfo(infoA)];
    const Mask layerBitsB = Mask(1) << Mask(getLayerFromFilterInfo(infoB));

    return layerBitsA & layerBitsB;
}

#if !defined(HK_PLATFORM_SPU)
template <typename CONFIG>
hkBool hkpGroupFilterBase<CONFIG>::isCollisionEnabled( const hkpCollidable& a, const hkpCollidable& b ) const
{
    return isCollisionEnabled( a.getCollisionFilterInfo(), b.getCollisionFilterInfo() );
}
#endif
template <typename CONFIG>
hkBool hkpGroupFilterBase<CONFIG>::isCollisionEnabled( const hkpCollisionInput& input, const hkpCdBody& collectionBodyA, const hkpCdBody& collectionBodyB, const HK_SHAPE_CONTAINER& containerShapeA, const HK_SHAPE_CONTAINER& containerShapeB, hkpShapeKey keyA, hkpShapeKey keyB ) const
{
    hkUint32 infoA = containerShapeA.getCollisionFilterInfo( keyA );

    if ( infoA == USE_COLLIDABLE_FILTER_INFO)
    {
        infoA = collectionBodyA.getRootCollidable()->getCollisionFilterInfo();
    }

    hkUint32 infoB = containerShapeB.getCollisionFilterInfo( keyB );
    if ( infoB == USE_COLLIDABLE_FILTER_INFO)
    {
        infoB = collectionBodyB.getRootCollidable()->getCollisionFilterInfo();
    }

    return isCollisionEnabled( infoA, infoB );
}

template <typename CONFIG>
hkBool hkpGroupFilterBase<CONFIG>::isCollisionEnabled( const hkpCollisionInput& input, const hkpCdBody& a, const hkpCdBody& b, const HK_SHAPE_CONTAINER& bContainer, hkpShapeKey bKey  ) const
{
    hkUint32 infoB = bContainer.getCollisionFilterInfo( bKey );
    if ( infoB == USE_COLLIDABLE_FILTER_INFO)
    {
        infoB = b.getRootCollidable()->getCollisionFilterInfo();
    }

    hkUint32 infoA;
    if ( !a.getParent() )
    {
        infoA = a.getRootCollidable()->getCollisionFilterInfo();
    }
    else
    {
        // If a has a parent, then two shape collections are colliding with each other,
        // in this case we have to try to reverse engineer the parent of a to get
        // the proper collision filter
        const hkpCdBody* p = a.getParent();
        const hkpCdBody* lastp = &a;
        while(1)
        {
            hkpShapeType t = p->getShape()->getType();

            if ( input.m_dispatcher->hasAlternateType( t, hkcdShapeType::COLLECTION ) )
            {
                const hkpShapeCollection* aCollection = static_cast<const hkpShapeCollection*>( p->getShape() );
                infoA = aCollection->getCollisionFilterInfo( lastp->getShapeKey() );
                break;
            }

            else if ( input.m_dispatcher->hasAlternateType( t, hkcdShapeType::BV_TREE ) )
            {
#ifndef HK_PLATFORM_SPU // <nat> need this to compile on SPU. fixme.
                hkpShapeBuffer buffer;
                const HK_SHAPE_CONTAINER* shapeContainer = hkBvTreeAgent3::getShapeContainerFrom(p, buffer);

                infoA = shapeContainer->getCollisionFilterInfo( lastp->getShapeKey() );
                break;
#else
                return true;
#endif

            }
            else if ( input.m_dispatcher->hasAlternateType( t, hkcdShapeType::MULTI_SPHERE ) )
            {
                infoA = a.getRootCollidable()->getCollisionFilterInfo();
                break;
            }
            else
            {
                // We disable filtering for convex list shapes, because we do not filter
                // the collisions in the get supporting vertex call, so the filtering will be inconsistent
                if (input.m_dispatcher->hasAlternateType( t, hkcdShapeType::CONVEX_LIST ) )
                {
                    return true;
                }
            }

            lastp = p;
            p = p->getParent();
            if ( p )
            {
                continue;
            }
            // parent of lastp is zero, therefor lastp is the root collidable
            infoA = reinterpret_cast<const hkpCollidable*>(lastp)->getCollisionFilterInfo();
            break;
        }
    }
    return isCollisionEnabled( infoA, infoB );
}

template <typename CONFIG>
hkBool hkpGroupFilterBase<CONFIG>::isCollisionEnabled( const hkpShapeRayCastInput& aInput, const HK_SHAPE_CONTAINER& bContainer, hkpShapeKey bKey ) const
{
    hkUint32 infoB = bContainer.getCollisionFilterInfo( bKey );
    return isCollisionEnabled( aInput.m_filterInfo, infoB );
}

template <typename CONFIG>
hkBool hkpGroupFilterBase<CONFIG>::isCollisionEnabled( const hkpWorldRayCastInput& aInput, const hkpCollidable& collidableB ) const
{
    return isCollisionEnabled( aInput.m_filterInfo, collidableB.getCollisionFilterInfo() );
}

#if !defined(HK_PLATFORM_SPU)

template <typename CONFIG>
void hkpGroupFilterBase<CONFIG>::enableCollisionsUsingBitfield( Mask layerBitsA, Mask layerBitsB )
{
    HK_ASSERT(0x3c3a0084,  (layerBitsA|layerBitsB) != 0, "layer bits not set");
    for( int i=0; i< COLLISION_LUT_SIZE; i++ )
    {
        const Mask b = Mask(1) << Mask(i);
        if( b & layerBitsA )
        {
            m_collisionLookupTable[i] |= layerBitsB;
        }
        if( b & layerBitsB )
        {
            m_collisionLookupTable[i] |= layerBitsA;
        }
    }
}


template <typename CONFIG>
void hkpGroupFilterBase<CONFIG>::enableCollisionsBetween( int layerA, int layerB )
{
    HK_ASSERT_NO_MSG(0x66c2b6fd,  0 <= layerA && layerA < COLLISION_LUT_SIZE );
    HK_ASSERT_NO_MSG(0x5a285631,  0 <= layerB && layerB < COLLISION_LUT_SIZE );

    m_collisionLookupTable[layerA] |= Mask(1) << Mask(layerB);
    m_collisionLookupTable[layerB] |= Mask(1) << Mask(layerA);
}


template <typename CONFIG>
void hkpGroupFilterBase<CONFIG>::disableCollisionsBetween( int layerA, int layerB )
{
    HK_ASSERT_NO_MSG(0x2a168aec,  0 <= layerA && layerA < COLLISION_LUT_SIZE );
    HK_ASSERT_NO_MSG(0x234fb60b,  0 <= layerB && layerB < COLLISION_LUT_SIZE );
    HK_ASSERT(0x4ab45935,  layerA > 0, "You are not allowed to disable collision of layer 0");
    HK_ASSERT(0x358c7ccd,  layerB > 0, "You are not allowed to disable collision of layer 0");

    m_collisionLookupTable[layerA] &= ~(Mask(1) << Mask(layerB));
    m_collisionLookupTable[layerB] &= ~(Mask(1) << Mask(layerA));
}


template <typename CONFIG>
void hkpGroupFilterBase<CONFIG>::disableCollisionsUsingBitfield( Mask layerBitsA, Mask layerBitsB )
{
    HK_ASSERT(0x41c4fad2,  (layerBitsA|layerBitsB) != 0, "layer bits not set");
    HK_ASSERT(0x49059b77,  (layerBitsA&1) == 0, "You are not allowed to disable collision of layer 0");
    HK_ASSERT(0x371ca278,  (layerBitsB&1) == 0, "You are not allowed to disable collision of layer 0");
    for( int i=0; i< COLLISION_LUT_SIZE; i++ )
    {
        const Mask b = Mask(1) << Mask(i);
        if( b & layerBitsA )
        {
            m_collisionLookupTable[i] &= ~layerBitsB;
        }
        if( b & layerBitsB )
        {
            m_collisionLookupTable[i] &= ~layerBitsA;
        }
    }
}

#endif

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
