// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics2012/Collide/Filter/hkpCollisionFilter.h>
#include <Physics2012/Collide/Shape/Query/hkpShapeRayCastInput.h>
#include <Physics2012/Collide/Shape/Compound/Tree/hkpBvTreeShape.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppBvTreeShape.h>
#include <Physics2012/Collide/Agent3/BvTree3/hkpBvTreeAgent3.h>

namespace hkpGroupFilterTypes
{
    /// Group filter configuration template <LAYER_BITS, SUB_SYSTEM_BITS, DONT_COLLIDE_BITS, SYSTEM_GROUP_BITS>
    template <int LAYER_BITS, int SUB_SYSTEM_BITS, int DONT_COLLIDE_BITS, int SYSTEM_GROUP_BITS>
    struct Config
    {
        HK_DECLARE_CLASS(Config, New, Reflect);
        enum
        {
            NUM_LAYER_BITS          = LAYER_BITS,
            NUM_SUB_SYSTEM_BITS     = SUB_SYSTEM_BITS,
            NUM_DONT_COLLIDE_BITS   = DONT_COLLIDE_BITS,
            NUM_SYSTEM_GROUP_BITS   = SYSTEM_GROUP_BITS,
        };
    };

    /// Layer mask storage
    template <int LAYER_BITS> struct LayerMask {};
    template <> struct LayerMask<5> {   typedef hkUint32 Mask;  };
    template <> struct LayerMask<6> {   typedef hkUint64 Mask;  };
}

/// This is an example collision filter.
/// The collision filter implementation can be a significant factor in performance of your physics.
/// Using this filter (or one very like it) for all your filtering will give optimal cache performance
/// and we strongly recommend using this if possible.
///
/// This filter is designed to allow custom filtering between bones in different
/// ragdolls or other constrained systems in your world, by doing simple operations on two 32 bit numbers.
/// The collision detection pipeline is setup to produce these 32 bit numbers, called collision filter infos.
///
/// It inherits from hkpCollisionFilter, and so implements each of the 4 hkpCollisionFilter::isCollisionEnabled
/// methods. For each method, it extracts two 32 bit values from the inputs (one representing each object to be
/// tested), and passes these values to a common filtering mechanism.
///
/// We split the 32 bit value into a system group (16 bits), a layer (5 or 6 bits), and a subsystem id (5 bits),
/// and the id of another subsystem that we do not want to collide with (5 bits), as follows:
///     5 bit layer setup: [ (systemGroup<<16) | (subSystemDontCollideWith<<10) | (subSystemId<<5) | layer ].
///     6 bit layer setup: [ (systemGroup<<17) | (subSystemDontCollideWith<<11) | (subSystemId<<6) | layer ].
///
/// The filter allows the user to specify whether collisions are enabled or disabled between every pair of layers
/// and between members of the same system.
///
/// The exact logic of the filter is as follows:
/// If both bodies are in the same system group, use the subsystem check.
/// If bodies are in different systems, use the layer check.
/// If both bodies are in same system AND that system is 0, use the layer check.
///
/// Layer-Layer collision disabling is set in the filter itself.
/// Subsystem collision disabling is set in the filterinfo.
///
/// Some things to keep in mind:
///
///     - Collisions between layer 0 and any other layer should not be disabled (by default all layers
///       collide with all other layers, including themselves). This is a policy decision for shape collection filtering.
///       All shape collections are required to implement hkpShapeCollection::getCollisionFilterInfo for each child shape.
///       By default this returns 0 (i.e., if no special child shape filter information is set).
///       So if you disable layer 0 in the group filter and use this filter as a shape collection filter,
///       it will disable collisions with children of all default shape collections.
///     - System group 0 is special in that if two objects of system group 0 are under consideration, their subsystem id
///       will not be considered. Instead, only the layer information will be used.
///
/// The class provides helper methods for getting and setting these values.
///
/// Examples:
///
///     - Keyframed objects should not collide with static geometry. Therefore
///       we can put all static geometry into one layer and all the keyframed object into
///       another layer. Then we disable the collision between these layers.
///     - All elements of a ragdoll should collide with each other, except those that are connected by
///       a constraint. In this case we assign all bodies of the ragdoll to the same system group. Also we assign
///       a unique subSystemId for each body in the ragdoll. Then we set subSystemDontCollideWith
///       to the id of the parent rigidbody (assuming you see the ragdoll as a hierarchy of bodies
///       and constraints, see the hkaRagdollUtils for details).
///
/// Note:
///
/// If two shape collections A and B collide, the following will happen
///     - The collidable of A and B will be checked.
///     - Then, one of the collections will be expanded, say in our case A will be replaced by
///       its children A# = { A0, A1, ... An }.
///     - Each collisionInfo of child A# (see hkpShapeCollection::getCollisionFilterInfo()) will
///       be tested against hkpCollidable::getCollisionFilterInfo() of B.
///     - Then for each A# B will be expanded into its children B# = {B0, B1, ...Bn}.
///     - Each collisionInfo of child A# (see hkpShapeCollection::getCollisionFilterInfo()) will be tested
///       against collisionInfo of child B#.
///
/// Interestingly, B was never tested against A#. That means if you have a complex shape collection hierarchy,
/// you think about your filters very carefully.
template <typename CONFIG>
class hkpGroupFilterBase : public hkpCollisionFilter
{
    public:

        enum
        {
            NUM_LAYER_BITS          = CONFIG::NUM_LAYER_BITS,
            NUM_SUB_SYSTEM_BITS     = CONFIG::NUM_SUB_SYSTEM_BITS,
            NUM_DONT_COLLIDE_BITS   = CONFIG::NUM_DONT_COLLIDE_BITS,
            NUM_SYSTEM_GROUP_BITS   = CONFIG::NUM_SYSTEM_GROUP_BITS,

            COLLISION_LUT_SIZE      = (1 << NUM_LAYER_BITS),

            NUM_LAYER_SHIFT         = 0,
            NUM_SUB_SYSTEM_SHIFT    = (NUM_LAYER_SHIFT + NUM_LAYER_BITS),
            NUM_DONT_COLLIDE_SHIFT  = (NUM_SUB_SYSTEM_SHIFT + NUM_SUB_SYSTEM_BITS),
            NUM_SYSTEM_GROUP_SHIFT  = (NUM_DONT_COLLIDE_SHIFT + NUM_DONT_COLLIDE_BITS),
        };

        // Filter info is stored in a hkUint32 so we must make sure that the config is specified within 32 bits.
        HK_COMPILE_TIME_ASSERT(NUM_LAYER_BITS + NUM_SUB_SYSTEM_BITS + NUM_DONT_COLLIDE_BITS + NUM_SYSTEM_GROUP_BITS <= 32);

        typedef typename hkpGroupFilterTypes::LayerMask<CONFIG::NUM_LAYER_BITS>::Mask Mask;

    // we make the size of this class 256 bytes, so that the spu simulator always finds a valid allocated memory piece
    public:
        HK_DECLARE_CLASS(hkpGroupFilterBase, New, Reflect);

            /// A collision filter info value which tells the group filter to take the collision filter info from the root collidable.
            /// Usually this will be used for the hkpExtendedMeshShape::m_defaultCollisionFilterInfo.
        enum { USE_COLLIDABLE_FILTER_INFO = 0xffffffff };

    public:

            /// Constructor, by default enables all collisions between all layers.
        hkpGroupFilterBase();

            /// Enable the collision between layer layerA and layerB.
        void enableCollisionsBetween(int layerA, int layerB);

            /// Disables collisions between the layers A and B.
        void disableCollisionsBetween(int layerA, int layerB);

            /// Enables collisions between the specified layers.
            /// layerBitsA and layerBitsB are bitfields, one bit for every layer.
            /// e.g., to enable collisions between one layer and all other layers for a 5 bit layer setup,
            /// call enableCollisionsUsingBitfield( 1<< myLayer, 0xfffffffe)
        void enableCollisionsUsingBitfield( Mask layerBitsA, Mask layerBitsB );

            /// Disables collisions between the specified collision layers.
            /// See enableCollisionsUsingBitfield for how to use bitfields.
        void disableCollisionsUsingBitfield( Mask layerBitsA, Mask layerBitsB );

            /// Combine the layer and systemGroup information into one 32 bit integer.
            /// This resulting collision filter info can be used in entities and hkEntityCinfo.
        static inline hkUint32 HK_CALL calcFilterInfo( int layer, int systemGroup = 0, int subSystemId = 0, int subSystemDontCollideWith = 0);

            /// Extract the layer from a given filterInfo.
        static int HK_CALL getLayerFromFilterInfo( hkUint32 filterInfo );

            /// Returns the filter info provided with the layer replaced by newLayer.
        static int HK_CALL setLayer( hkUint32 filterInfo, int newLayer );

            /// Extract the system group from a given filterInfo.
        static int HK_CALL getSystemGroupFromFilterInfo( hkUint32 filterInfo );

            /// Extract the subsystem id from a given filterInfo.
        static int HK_CALL getSubSystemIdFromFilterInfo( hkUint32 filterInfo );

            /// Extract the subSystemDontCollideWith from a given filterInfo.
        static int HK_CALL getSubSystemDontCollideWithFromFilterInfo( hkUint32 filterInfo );


            /// Creates a new unique identifier for system groups (maximum 65k).
        inline int getNewSystemGroup();

            /// Checks two collidables. This is the implementation of the hkpCollidableCollidableFilter::isCollisionEnabled method.
            /// This extracts a 32 bit value from each hkpCollidable, using the getCollisionFilterInfo() method.
        virtual hkBool isCollisionEnabled( const hkpCollidable& a, const hkpCollidable& b ) const;

            /// This is an implementation of hkpShapeCollectionFilter::isCollisionEnabled().
            /// This gets the 32 bit collision info for hkpCdBody "b" by calling hkpShapeContainer::getCollisionFilterInfo( bKey ).
            /// For body "a" we do one of two things, depending on whether a has a parent or not.
            ///     -# If "a" does not have a parent, we know that it is not a child shape of a shape collection, and so we simply call
            ///        hkpCollidable::getCollisionFilterInfo() on its root collidable (hkpCdBody::getRootCollidable).
            ///     -# If "a" does have a parent, we know that it _may_ have a hkpShapeCollection in an ancestor hkpCdBody. We then travel up the hkpCdBody
            ///        hierarchy until we reach the first hkpCdBody being a child of a hkpShapeCollection or hkpBvTreeShape (or we reach the root collidable and act as in point 1.). We then call
            ///        hkpShapeCollection::getCollisionFilterInfo() using the found shape collection and the found child hkpCdBody.
        virtual hkBool isCollisionEnabled( const hkpCollisionInput& input, const hkpCdBody& a, const hkpCdBody& b, const HK_SHAPE_CONTAINER& bContainer, hkpShapeKey bKey  ) const;

            // hkpShapeCollectionFilter interface forwarding
        virtual hkBool isCollisionEnabled( const hkpCollisionInput& input, const hkpCdBody& collectionBodyA, const hkpCdBody& collectionBodyB, const HK_SHAPE_CONTAINER& containerShapeA, const HK_SHAPE_CONTAINER& containerShapeB, hkpShapeKey keyA, hkpShapeKey keyB ) const;

            /// This is an implementation of hkpRayShapeCollectionFilter::isCollisionEnabled().
            /// This gets the 32 bit collision info for the ray by getting hkpShapeRayCastInput::m_filterInfo (for aInput).
            /// To get the 32 bit collision info for the shape, we call hkpShapeCollection::getCollisionFilterInfo() for bContainer, using bKey.
        virtual hkBool isCollisionEnabled( const hkpShapeRayCastInput& aInput, const HK_SHAPE_CONTAINER& bContainer, hkpShapeKey bKey ) const;

            /// This is an implementation of hkpRayCollidableFilter::isCollisionEnabled().
            /// This gets the 32 bit collision info for the ray by getting hkpWorldRayCastInput::m_filterInfo (for inputA).
            /// To get the 32 bit collision info for the collidableB, we simply call hkpCollidable::getCollisionFilterInfo() collidableB
        virtual hkBool isCollisionEnabled( const hkpWorldRayCastInput& inputA, const hkpCollidable& collidableB ) const;

            /// The actual filter implementation between two hkUint32 values. All the other isCollisionEnabled functions call this method.
            /// Returns true if the objects are enabled to collide, based on their collision groups.
        hkBool isCollisionEnabled(hkUint32 infoA, hkUint32 infoB) const;

    public:

        int m_nextFreeSystemGroup;
        HK_ALIGN8( Mask m_collisionLookupTable[COLLISION_LUT_SIZE] );
};

#include <Physics2012/Collide/Filter/Group/hkpGroupFilter.inl>
#include <Physics2012/Collide/_Auto/TemplateTypes/hkpGroupFilter_Types.inl>

/// Default group filter specialization using 32 layers, 32 subsystems.
class HK_EXPORT_PHYSICS_2012 hkpGroupFilter : public hkpGroupFilterBase< hkpGroupFilterTypes::Config<5,5,5,16> >
{
    //+version(1)
public:

    HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE );
    HK_DECLARE_REFLECTION();

    typedef hkpGroupFilterBase< hkpGroupFilterTypes::Config<5,5,5,16> > Super;

    static hkUint32 HK_CALL calcFilterInfo( int layer, int systemGroup=0, int subSystemId=0, int subSystemDontCollideWith=0 );
};

/// 64 layer group filter specialization with 32 subsystems.
class HK_EXPORT_PHYSICS_2012 hkpGroupFilter64Layers : public hkpGroupFilterBase< hkpGroupFilterTypes::Config<6,5,5,16> >
{
public:

    HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE );
    HK_DECLARE_REFLECTION();

    typedef hkpGroupFilterBase< hkpGroupFilterTypes::Config<6,5,5,16> > Super;

    static hkUint32 HK_CALL calcFilterInfo( int layer, int systemGroup=0, int subSystemId=0, int subSystemDontCollideWith=0 );
};

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
