// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

/// A container consisting of a list of same-sized blocks holding Ts.
/// The list is written to using a BatchWriter, and then read from
/// using a BatchReader. No other use-case is supported.
/// This is a simplified version of the hkBlockStream.
template<typename T, int BLOCK_SIZE>
class hkpBlockList
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_COLLIDE, hkpBlockList );

    public:
        struct Block
        {
            enum
            {
                NUM_ELEMENTS_IN_BLOCK = ( BLOCK_SIZE - hkSizeOf( Block* ) ) / hkSizeOf( T )
            };

            HK_INLINE void initialize()
            {
                this->m_next = HK_NULL;
                HK_ON_DEBUG( hkString::memSet( m_data, 0xcf, NUM_ELEMENTS_IN_BLOCK * hkSizeOf( T ) ) );
            }

                /// The link to the next block.
            HK_ALIGN16( Block* m_next );

                /// The data contained in the block.
            T m_data[NUM_ELEMENTS_IN_BLOCK];
        };

    public:
            /// Create a list with no blocks.
        hkpBlockList();

#if defined(HK_DEBUG)
            /// Return the number of blocks in the list.
        int getTotalNumElems() { return m_numTotalElements; }

            /// Assert that the list has no blocks.
        void checkEmpty();
#endif

    public:

        class BatchWriter
        {
            public:
                HK_INLINE BatchWriter();

                    /// Sets the writer to the start of an empty list
                HK_INLINE void setToStartOfList( hkpBlockList* blockList );

                    /// Write a batch of same-sized elements into the list.
                HK_INLINE void writeBatch( const T* data, int numElements );

                    /// Finish up.
                HK_INLINE void finalize();

            protected:
                    /// A new block is return which is added after lastBlock.
                HK_INLINE Block* addBlockAtEnd( Block* lastBlock );

            public:
                hkpBlockList<T, BLOCK_SIZE>* m_blockList;

                    /// The current block (a PPU pointer when on SPU)
                Block* m_currentBlock;

                    /// The index of the next element at which writing will be done.
                int m_currentElement;

                    /// The last block written to (a PPU pointer when on SPU).
                Block* m_prevBlock;
        };

            /// The batch consumer reads a hkpBlockList. It also frees the memory once it is read.
        class BatchConsumer
        {
            public:
                HK_INLINE BatchConsumer();

                HK_INLINE ~BatchConsumer();

                    /// Sets the reader to the start of a list
                HK_INLINE void setToStartOfList( hkpBlockList* list );

                    /// Deallocate the last block in the list if the consumer hasn't
                    /// done so already.
                HK_INLINE void finalize();

                    /// Set the maximum number of elements to obtain.
                HK_INLINE void setNumElements( int numElements );

                    /// Get the pointer to and return the count of the next batch.
                    /// \param numElementsInBatch is zero when it has reached the end.
                HK_INLINE   int accessBatch( T*& dataOut );

                    /// Get a pointer to the remaining elements in the current block (use 0xcd as terminator)
                HK_INLINE   int getElements( T*& dataOut );

            protected:
                /// Remove the first block.
                HK_INLINE void removeBlockFromStart( Block* firstBlock, Block* secondBlock );

            protected:
                hkpBlockList<T, BLOCK_SIZE>* m_blockList;

                /// The current block. This is a PPU pointer on SPU.
                Block* m_currentBlock;

                    /// The index to the next element to read within the current block.
                int m_currentElement;

                    /// Cause accessBatch to return only this many elements.
                int m_numElementsToRead;
        };

    protected:
            /// The first block in the list.
            /// This is invalidated by the consumer (in release builds).
        HK_ALIGN16( Block* m_firstBlock );

            /// The number of blocks in the list (debug only)
        int m_numBlocks;

            /// The number of Ts in the list (debug only)
        int m_numTotalElements;
};

#include <Physics2012/Collide/Agent3/Machine/Midphase/hkpBlockList.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
