// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/Physics/hkStepInfo.h>
#include <Physics/ConstraintSolver/Solve/hkpSolverInfo.h>

class hkpVelocityAccumulator;
class hkpSolverResults;
class hkpConstraintQueryOut;

    /// The time information, the constraints get access to
class HK_EXPORT_PHYSICS hkpConstraintQueryStepInfo
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT_SOLVER, hkpConstraintQueryStepInfo );

            /// The delta time of each solvers substep
        HK_ALIGN16( hkReal  m_subStepDeltaTime );

        hkReal  m_microStepDeltaTime;

            /// The 1.0f/m_substepDeltaTime
        hkReal  m_subStepInvDeltaTime;


        hkReal  m_frameDeltaTime;
        hkReal  m_frameInvDeltaTime;
        hkReal  m_invNumSteps;
        hkReal  m_invNumStepsTimesMicroSteps;   // = 1.0f/(numSubsteps*numMicroSteps)

        hkReal  m_maxConstraintViolationSqrd;

            // a factor all rhs should be multiplied
        hkReal  m_rhsFactor;

            // a factor all invMasses should be multiplied
        hkReal  m_virtMassFactor;

            // a factor for all rhs used for friction calculations
        hkReal  m_frictionRhsFactor;
};

#define HK_ACCUMULATOR_OFFSET_TO_INDEX(offset) hkUint16(unsigned(offset) / sizeof(hkpVelocityAccumulator))


    /// The input structure to hkpConstraintData::buildJacobian()
class HK_EXPORT_PHYSICS hkpConstraintQueryIn : public hkpConstraintQueryStepInfo
{
    public:

        typedef void (HK_CALL *FnBeginConstraints)( const hkpConstraintQueryIn &in, hkpConstraintQueryOut &out, hkpSolverResults* sr, int solverResultStriding );

    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT_SOLVER, hkpConstraintQueryIn );

        hkpConstraintQueryIn( _In_ FnBeginConstraints beginConstraintsFunc ) : m_beginConstraints(beginConstraintsFunc) {}

        void HK_INLINE beginConstraints( hkpConstraintQueryOut &out, _Inout_ hkpSolverResults* sr, int solverResultStriding ) const
        {
            (*m_beginConstraints)( *this, out, sr, solverResultStriding );
        }

        inline void set(const hkpSolverInfo& si, const hkStepInfo& stepInfo, hkpViolatedConstraintArray* violatedConstraintsBuffer)
        {
            m_subStepDeltaTime    = si.m_deltaTime;
            m_microStepDeltaTime  = si.m_deltaTime * si.m_invNumMicroSteps;
            m_subStepInvDeltaTime = si.m_invDeltaTime;
            m_invNumSteps         = si.m_invNumSteps;
            m_invNumStepsTimesMicroSteps      = si.m_invNumSteps * si.m_invNumMicroSteps;
            m_tau                 = si.m_tau;
            m_damping             = si.m_damping;

            m_violatedConstraints = violatedConstraintsBuffer;

            m_rhsFactor           = si.m_tauDivDamp * si.m_invDeltaTime;
            m_virtMassFactor      = si.m_damping;
            m_frictionRhsFactor   = si.m_frictionTauDivDamp * si.m_invDeltaTime;

            m_frameDeltaTime      = stepInfo.m_deltaTime;
            m_frameInvDeltaTime   = stepInfo.m_invDeltaTime;
            m_maxConstraintViolationSqrd.setFromFloat(si.m_maxConstraintViolationSqrd);
        }

#ifdef HK_PLATFORM_RVL
        /// Assignment operator (for Wii)
        hkpConstraintQueryIn & operator = (const hkpConstraintQueryIn & other)
        {
            // Base class members
            {
                m_subStepDeltaTime              = other.m_subStepDeltaTime;
                m_microStepDeltaTime            = other.m_microStepDeltaTime;
                m_subStepInvDeltaTime           = other.m_subStepInvDeltaTime;
                m_frameDeltaTime                = other.m_frameDeltaTime;
                m_frameInvDeltaTime             = other.m_frameInvDeltaTime;
                m_invNumSteps                   = other.m_invNumSteps;
                m_invNumStepsTimesMicroSteps    = other.m_invNumStepsTimesMicroSteps;
                m_maxConstraintViolationSqrd    = other.m_maxConstraintViolationSqrd;
                m_rhsFactor                     = other.m_rhsFactor;
                m_virtMassFactor                = other.m_virtMassFactor;
                m_frictionRhsFactor             = other.m_frictionRhsFactor;
            }

            // My members
            {
                m_bodyA                         = other.m_bodyA;
                m_bodyB                         = other.m_bodyB;
                m_transformA                    = other.m_transformA;
                m_transformB                    = other.m_transformB;
                m_tau                           = other.m_tau;
                m_damping                       = other.m_damping;
                m_maxConstraintViolationSqrd    = other.m_maxConstraintViolationSqrd;
                m_constraintInstance            = other.m_constraintInstance;
                m_violatedConstraints           = other.m_violatedConstraints;
                m_accumulatorAIndex             = other.m_accumulatorAIndex;
                m_accumulatorBIndex             = other.m_accumulatorBIndex;
            }

            return *this;
        }
#endif

    public:

            /// The velocity accumulator of bodyA. Typically this is a hkpVelocityAccumulator
        HK_ALIGN16( const hkpVelocityAccumulator* m_bodyA );

            /// The velocity accumulator of bodyB. Typically this is a hkpVelocity
        const hkpVelocityAccumulator* m_bodyB;

            /// the transform of rigid body A, identity transform if not available
        const hkTransform* m_transformA;

            /// the transform of rigid body B, identity transform if not available
        const hkTransform* m_transformB;

            /// the current global solver settings
        hkReal m_tau;

            /// the current global solver settings
        hkReal m_damping;

            /// the current constraint violation settings
        hkSimdReal m_maxConstraintViolationSqrd;

            /// if this class is used with the hkDynamics library, this points to an hkpConstraintInstance
        class hkpConstraintInstance* m_constraintInstance;

            /// listeners
        hkpViolatedConstraintArray* m_violatedConstraints;

            /// Data to be written to the header schema
        hkUint32 m_accumulatorAIndex;
        hkUint32 m_accumulatorBIndex;

            /// Implementation of beginConstraints
        FnBeginConstraints m_beginConstraints;

            // A debug identifier used by asserts
        HK_DEBUG_ONLY_MEMBER(hkUint64, m_constraintIdentifier);
};

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
