// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics/Constraint/hkpConstraint.h>
#include <Physics/Constraint/VehicleFriction/hkpVehicleFrictionSolver.h>

#include <Physics/ConstraintSolver/Jacobian/hkpJacobianElement.h>
#include <Physics/ConstraintSolver/Solve/hkpSolverElemTemp.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>


// Temporary axis variables used during solving
struct hkpVehicleFrictionAxisTmp
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_CONSTRAINT, hkpVehicleFrictionAxisTmp );

    // Some Jacobians
    hkp1Lin2AngJacobian m_forwardJac;   // the forward backward jacobian
    hkp1Lin2AngJacobian m_sideJac;      // the side angular jacobian

    // Input parameters to the friction solver
    hkReal m_forwardVel;                // the velocity of the car at the contact point
    hkReal m_forwardSlipVel;            // the slip velocity of the wheel in the forward direction
    hkReal m_sideSlipVel;               // the slip velocity of the wheel in the side direction
    hkReal m_wheelDownForce;            // the down force after equalize
    hkReal m_sideVirtMass;
    hkReal m_invSideVirtMass;           // the inverse virtual mass of the wheel into the side direction (the other is m_rotation inertia)
    hkReal m_wheelForwardVirtMass;
    hkReal m_invWheelForwardVirtMass;

    // The forces needed to stop the wheel from sliding completely
    hkReal m_forwardImpulse;
    hkReal m_sideImpulse;

    // Parameters used to clip the force
    hkReal m_maxForwardImpulse;
    hkReal m_invMaxForwardImpulse;
    hkReal m_maxSideImpulse;
    hkReal m_invMaxSideImpulse;

    // The impulse which is needed to stop the wheel from sliding after the friction solver has done it's work.
    // It is pretty much the rest which has been clipped away.
    hkReal m_restForwardSlipImpulse;
};


// Setup internal cache information hkpVehicleFrictionDescription for fast solving of the vehicle
void hkVehicleFrictionDescriptionInitValues(
    const hkpVehicleFrictionDescription::Cinfo& ci, _Inout_ hkpVehicleFrictionDescription& out )
{
    out.m_chassisMassInv = ci.m_chassisMassInv;

    // Some basic wheel geometry
    {
        out.m_axleDescr[0].m_wheelRadius = ci.m_wheelRadius[0];
        out.m_axleDescr[1].m_wheelRadius = ci.m_wheelRadius[1];
        out.m_axleDescr[0].m_wheelRadiusInv = 1.0f / ci.m_wheelRadius[0];
        out.m_axleDescr[1].m_wheelRadiusInv = 1.0f / ci.m_wheelRadius[1];

        // calculate the virtual mass on the surface of the wheel
        out.m_axleDescr[0].m_wheelSurfaceInertia = ci.m_wheelAxleAngularInertia[0] * out.m_axleDescr[0].m_wheelRadiusInv;
        out.m_axleDescr[1].m_wheelSurfaceInertia = ci.m_wheelAxleAngularInertia[1] * out.m_axleDescr[1].m_wheelRadiusInv;
        out.m_axleDescr[0].m_wheelSurfaceInertiaInv = 1.0f / out.m_axleDescr[0].m_wheelSurfaceInertia;
        out.m_axleDescr[1].m_wheelSurfaceInertiaInv = 1.0f / out.m_axleDescr[1].m_wheelSurfaceInertia;
    }

    // Calculate wheel distances
    hkReal dist0,dist1;
    {
        dist0 = ci.m_wheelPosition[0].dot<3>( ci.m_directionFront ).getReal();
        dist1 = ci.m_wheelPosition[1].dot<3>( ci.m_directionFront ).getReal();
        out.m_wheelDistance = hkMath::fabs(dist0 - dist1);
    }

    // Calculate the values for the friction equalizer
    {
        HK_ASSERT( 0x6e5c0a34, hkMath::fabs( out.m_wheelDistance ) > 0.0f, "The distance between the axles must be non zero" );
        const hkReal q = 1.0f / out.m_wheelDistance;
        const hkReal ideal_a = hkMath::fabs( -dist1 * q );      // ideal front wheel pressure
        const hkReal ideal_b = hkMath::fabs(  dist0 * q );

        out.m_axleDescr[0].m_wheelDownForceFactor    = 1.0f - ci.m_frictionEqualizer;
        out.m_axleDescr[0].m_wheelDownForceSumFactor = ideal_a * ci.m_frictionEqualizer;

        out.m_axleDescr[1].m_wheelDownForceFactor    = 1.0f - ci.m_frictionEqualizer;
        out.m_axleDescr[1].m_wheelDownForceSumFactor = ideal_b * ci.m_frictionEqualizer;
    }

    // Build the jacobians to find the virtual masses for the wheels in the forward direction
    hkReal virtInvMass[2];
    {
        for ( int i = 0; i < 2; i++ )
        {
            hkVector4 mr; mr.setSub( ci.m_wheelPosition[i], ci.m_chassisCenterOfMass );
            mr.addMul( hkSimdReal::fromFloat(-ci.m_wheelRadius[i]), ci.m_directionUp );
            hkVector4 jac; jac.setCross( mr, ci.m_directionRight );
            hkVector4 mj;  mj._setRotatedDir( ci.m_chassisFrictionInertiaInv, jac );
            virtInvMass[i] = out.m_chassisMassInv + mj.dot<3>( jac ).getReal();
        }
    }

    // Setup the friction clip table
    {
        for ( int a = 0; a < 2; a++ )
        {
            const int sizeOfYTable = hkpVehicleFrictionDescription::hkFricionEllipseLineSegments;
            const hkReal yToXmassRatio = virtInvMass[a] * out.m_axleDescr[a].m_wheelSurfaceInertia;
            out.m_axleDescr[a].m_wheelChassisMassRatio = yToXmassRatio;
            hkReal x = yToXmassRatio / sizeOfYTable;
            out.m_axleDescr[a].m_xStart = x;
            const hkReal xStep = hkMath::pow( hkReal(1.0f / x), hkReal(1.0f / (sizeOfYTable-1)) );
            out.m_axleDescr[a].m_xStep = xStep;

            int i;
            for ( i = 0; i < sizeOfYTable-1; i++ )
            {
                out.m_axleDescr[a].m_frictionCircleYtab[i] = 1.0f - hkMath::pow( 1.0f - x, 1.0f / yToXmassRatio );
                x *= xStep;
            }
            out.m_axleDescr[a].m_frictionCircleYtab[i] = 1.0f;
        }
    }
}


// Update the friction side force / forward force
static inline void hkVehicleFrictionSolverCalcRelativeForces(
    const hkpVehicleFrictionAxisTmp &axis, hkpVehicleFrictionStatus::AxisStatus &status )
{
    //  Scale the friction circle, so we get our ellipse
    status.m_relativeSideForce      = axis.m_sideImpulse    * axis.m_invMaxSideImpulse;
    status.m_relativeForwardForce   = axis.m_forwardImpulse * axis.m_invMaxForwardImpulse;
}


static inline void hkCalculateFrictionalEllipsoidScale(
    const hkpVehicleFrictionDescription::AxisDescription& adesc, hkReal xs, hkReal ys,
    hkReal& scaleXOut, hkReal& scaleYOut )
{
    hkReal x = adesc.m_xStart * xs;
    hkReal y;
    hkReal w0;
    hkReal x1 = 0.0f;
    hkReal y1 = 0.0f;
    hkReal w1 = 0.0f;

    int i = 0;
    while( true )
    {
        y = adesc.m_frictionCircleYtab[i] * ys;
        w0 = x*x + y*y;
        HK_ASSERT_NO_MSG( 0x15bcbb84, hkMath::isFinite(w0) ); 
        if ( w0 > 1.0f || ++i >= hkpVehicleFrictionDescription::hkFricionEllipseLineSegments )
        {
            break;
        }

        x1 = x;
        y1 = y;
        w1 = w0;
        x *= adesc.m_xStep;
    }

    // Get the two interpolation weights, a0 and a1
    w0 = hkMath::sqrt( w0 ) - 1.0f;
    w1 = hkMath::sqrt( w1 ) - 1.0f;
    hkReal diff = (w0 - w1);
    if (hkMath::equal(diff, hkReal(0), HK_REAL_EPSILON)) 
    {
        diff = HK_REAL_EPSILON;
    }
    const hkReal invDiff = 1.0f / diff;
    const hkReal a0 = - w1 * invDiff;
    const hkReal a1 =   w0 * invDiff;

    // Interpolate our new impulses
    scaleXOut = hkMath::clamp( ( x * a0 + x1 * a1 ), hkReal(-1.1f), hkReal(1.1f) ); 
    scaleYOut = hkMath::clamp( ( y * a0 + y1 * a1 ), hkReal(-1.1f), hkReal(1.1f) ); 
}


// Clip the forces between the wheel and the ground
// Output:
//  - atmp.m_real_forward_impulse: The impulse in the forward direction
//  - atmp.m_real_constraint_impulse: The impulse in the side direction
//  - status.m_real_skid_energy_density
//  - status.m_sideForce
static inline void hkVehicleFrictionSolverClipImpulse(
    const hkpVehicleFrictionSolverAxleParams &apar,
    hkpVehicleFrictionAxisTmp &atmp,
    const hkpVehicleFrictionDescription::AxisDescription& adesc,
    hkpVehicleFrictionStatus::AxisStatus &astatus )
{
    // Scale the friction circle, so we get our ellipse     
    const hkReal xs = hkMath::clamp(atmp.m_sideImpulse    * atmp.m_invMaxSideImpulse, -HK_REAL_HIGH, HK_REAL_HIGH);
    const hkReal ys = hkMath::clamp(atmp.m_forwardImpulse * atmp.m_invMaxForwardImpulse, -HK_REAL_HIGH, HK_REAL_HIGH);

    astatus.m_relativeSideForce = xs;
    astatus.m_relativeForwardForce = ys;

    // Return if our start point is within our ellipse
    if ( xs*xs + ys*ys < 1.0f )
    {
        atmp.m_restForwardSlipImpulse = 0.0f;
        return;
    }

    // We cannot use a simple line clip, as the inertia tensor in the x and y direction differ greatly.
    // Instead our point xs,ys is connected to the origin via an exponential curve, which is encoded in our friction table.
    // So we need to find a point in our table, where the segment crosses the friction circle.
    hkReal restSideImpulse;
    {
        hkReal scaleX, scaleY;
        hkCalculateFrictionalEllipsoidScale( adesc, xs, ys, scaleX, scaleY );

        const hkReal impX = scaleX * atmp.m_maxSideImpulse;
        const hkReal impY = scaleY * atmp.m_maxForwardImpulse;

        restSideImpulse = atmp.m_sideImpulse - impX;
        atmp.m_sideImpulse = impX;
        atmp.m_restForwardSlipImpulse = atmp.m_forwardImpulse - impY;
        atmp.m_forwardImpulse = impY;
    }

    // Calculate skid mark information
    {
        // Calculate vehicle over ground velocity.
        hkReal velOverGroundSqr = atmp.m_forwardVel * atmp.m_forwardVel + atmp.m_sideSlipVel * atmp.m_sideSlipVel;
        if ( hkMath::equal(velOverGroundSqr, hkReal(0), HK_REAL_EPSILON) )
        {
            velOverGroundSqr = HK_REAL_EPSILON; // avoid dividing by zero
        }
        const hkReal velOverGroundSqrInv = 1.0f / velOverGroundSqr;

        // calc the energy eliminated
        const hkReal energyForward = atmp.m_restForwardSlipImpulse * atmp.m_invWheelForwardVirtMass * atmp.m_maxForwardImpulse;
        const hkReal energySide    = restSideImpulse               * atmp.m_invSideVirtMass         * atmp.m_maxSideImpulse;

        const hkReal energySquare = energyForward * energyForward + energySide * energySide;

        astatus.m_skid_energy_density = hkMath::sqrt( energySquare * velOverGroundSqrInv );
    }
}


// Internal functions
extern void hkVehicleFrictionSolver_buildJacobian(
    const hkpVelocityAccumulator* opaA, const hkpVelocityAccumulator* opaB,
    hkVector4Parameter position, hkVector4Parameter direction,
    hkp1Lin2AngJacobian* jac );

extern hkReal hkVehicleFrictionSolver_getVelocity(
    const hkp1Lin2AngJacobian& jac,
    const hkpVelocityAccumulator* opaA, const hkpVelocityAccumulator* opaB );

extern void hkVehicleFrictionSolver_applyImpulse(
    hkSimdRealParameter imp, const hkp1Lin2AngJacobian& jac,
    hkpVelocityAccumulator* opaA, hkpVelocityAccumulator* opaB );


void hkVehicleFrictionApplyVehicleFriction(
    const hkpVehicleStepInfo& stepInfo,
    const hkpVehicleFrictionDescription& descr,
    _Inout_ hkpVehicleFrictionSolverParams &params,
    _Inout_ hkpVehicleFrictionStatus &status )
{
    // Just a simple 2 equation, 2 unknown system.
    // Error will be velocities violating the constraints.
    // Solving for friction impulses such that projected velocities perpendicular to the tires = 0.
    // invMassMatrix dp = -invMassMatrix p, where p is the current momentum.  Solve this.
    // invMassMatrix = J M^-1, where J projects velocities to constraint space
    // dp = J^t _dp
    // J M^-1 J^t _dp = -J M^-1 p
    // J M^-1 J^t _dp = -J v, where v is the current velocities.  Solve for _dp


    // Build jacobians and mass matrix, also update positional friction
    hkpVehicleFrictionAxisTmp m_axis[HK_VEHICLE_FRICTION_N_AXIS];
    HK_ALIGN_REAL(hkReal invMassMatrix[4]);
    {
        for( int a = 0; a < HK_VEHICLE_FRICTION_N_AXIS; a++ )
        {
            const hkpVehicleFrictionSolverAxleParams& apar = params.m_axleParams[a];

            // build jacobians
            hkpVehicleFrictionAxisTmp& atmp = m_axis[a];
            {
                const hkpVelocityAccumulator* opaA = &params.m_chassis;
                const hkpVelocityAccumulator* opaB = apar.m_groundObject;

                hkVehicleFrictionSolver_buildJacobian(
                    opaA, opaB, apar.m_contactPointWs, apar.m_forwardDirWs, &atmp.m_forwardJac );

                atmp.m_invWheelForwardVirtMass = atmp.m_forwardJac.getUserValue() + descr.m_axleDescr[a].m_wheelSurfaceInertiaInv;
                atmp.m_wheelForwardVirtMass = 1.0f / atmp.m_invWheelForwardVirtMass;

                hkVehicleFrictionSolver_buildJacobian(
                    opaA, opaB, apar.m_contactPointWs, apar.m_constraintNormalWs, &atmp.m_sideJac );

                atmp.m_sideVirtMass     = atmp.m_sideJac.getEffMass();
                atmp.m_invSideVirtMass  = atmp.m_sideJac.getUserValue();
            }

            // get velocity for positional friction
            hkpVehicleFrictionStatus::AxisStatus &astatus = status.m_axis[a];
            {
                const hkpVelocityAccumulator* opaA = &params.m_chassisAtLastIntegration;
                const hkpVelocityAccumulator* opaB = apar.m_groundObjectAtLastIntegration;

                const hkReal forwardVel = hkVehicleFrictionSolver_getVelocity( atmp.m_forwardJac, opaA, opaB );
                const hkReal sideVel    = hkVehicleFrictionSolver_getVelocity( atmp.m_sideJac, opaA, opaB );

                astatus.m_sideRhs    += sideVel;
                astatus.m_forwardRhs += forwardVel;
            }
        }

        hkReal nonDiag;
        {
            const hkpVelocityAccumulator* opaA = &params.m_chassis;
            const hkpVelocityAccumulator* opaB = params.m_axleParams[0].m_groundObject;
            nonDiag = ( opaB == params.m_axleParams[1].m_groundObject) ?
                m_axis[0].m_sideJac.getNonDiagSameObjects( *opaA, *opaB, m_axis[1].m_sideJac ).getReal() :
                m_axis[0].m_sideJac.getNonDiagDifferentObjects( *opaA, m_axis[1].m_sideJac ).getReal();
        }

        invMassMatrix[0] = m_axis[0].m_invSideVirtMass;
        invMassMatrix[3] = m_axis[1].m_invSideVirtMass;
        invMassMatrix[1] = nonDiag;
        invMassMatrix[2] = nonDiag;
    }

    // Apply the remembered forward/backward impulses
    {
        for( int a = 0; a < HK_VEHICLE_FRICTION_N_AXIS; a++ )
        {
            const hkSimdReal impulse = hkSimdReal::fromFloat( status.m_axis[a].m_delayed_forward_impulse );
            if ( impulse.isEqualZero() )
            {
                continue;
            }

            const hkp1Lin2AngJacobian& jac = m_axis[a].m_forwardJac;
            hkpVelocityAccumulator* opaA = &params.m_chassis;
            hkpVelocityAccumulator* opaB = params.m_axleParams[a].m_groundObject;

            hkVehicleFrictionSolver_applyImpulse( impulse, jac, opaA, opaB );
        }
    }

    // Equalize the friction force
    {
        const hkReal w0 = params.m_axleParams[0].m_wheelDownForce;
        const hkReal w1 = params.m_axleParams[1].m_wheelDownForce;
        const hkReal sum = w0 + w1;

        m_axis[0].m_wheelDownForce = descr.m_axleDescr[0].m_wheelDownForceFactor * w0 + descr.m_axleDescr[0].m_wheelDownForceSumFactor * sum;
        m_axis[1].m_wheelDownForce = descr.m_axleDescr[1].m_wheelDownForceFactor * w1 + descr.m_axleDescr[1].m_wheelDownForceSumFactor * sum;
    }

    status.m_axis[0].m_skid_energy_density = 0.0f;
    status.m_axis[1].m_skid_energy_density = 0.0f;

    // Calculate some per wheel behavior
    {
        for ( int i = 0; i < 2; i++ )
        {
            hkpVehicleFrictionAxisTmp& atmp = m_axis[i];
            const hkpVehicleFrictionSolverAxleParams& apar = params.m_axleParams[i];
            hkpVehicleFrictionStatus::AxisStatus& astatus = status.m_axis[i];

            // get the current velocities and slip velocities
            {
                hkpVelocityAccumulator* opaA = &params.m_chassis;
                hkpVelocityAccumulator* opaB = apar.m_groundObject;

                {
                    hkp1Lin2AngJacobian& jac = atmp.m_forwardJac;
                    atmp.m_forwardVel = hkVehicleFrictionSolver_getVelocity( jac, opaA, opaB );
                }
                {
                    hkp1Lin2AngJacobian& jac = atmp.m_sideJac;
                    atmp.m_sideSlipVel = hkVehicleFrictionSolver_getVelocity( jac, opaA, opaB );
                    atmp.m_sideSlipVel += astatus.m_sideRhs * 0.25f;
                }
            }

            const hkReal downImpulse = hkMath::fabs(atmp.m_wheelDownForce) * stepInfo.m_deltaTime;
            hkReal forwardForce = apar.m_forwardForce;

            // slip velocity used for viscosity friction
            hkReal forwardSlipForFriction;
            if ( apar.m_wheelFixed ) //|| params.m_axleParams[1-i].m_wheelFixed)
            {
                const hkReal wheelChassisMassRatio = atmp.m_sideVirtMass * atmp.m_invWheelForwardVirtMass;

                atmp.m_forwardSlipVel = -atmp.m_forwardVel;
                forwardSlipForFriction = atmp.m_forwardSlipVel;

                // extra slip to also stop the chassis (clipped)
                hkReal extra = (astatus.m_forwardRhs * 0.5f + atmp.m_forwardVel) * wheelChassisMassRatio * 0.5f;
                const hkReal maxExtra = downImpulse * apar.m_frictionCoefficient * atmp.m_invWheelForwardVirtMass;
                if ( hkMath::fabs( extra ) > maxExtra )
                {
                    extra = ( extra > 0 ) ? maxExtra : -maxExtra;
                }
                atmp.m_forwardSlipVel -= extra;
                forwardForce = 0.0f;
            }
            else
            {
                astatus.m_forwardRhs = 0.0f;
                atmp.m_forwardSlipVel = status.m_axis[i].m_forward_slip_velocity;
                forwardSlipForFriction = atmp.m_forwardSlipVel;
            }

            // calculate the maximum impulses and forward impulses
            {
                
                // maxFriction is only used when viscosityFriction != 0; limit friction to maxFriction in that case. Friction is a constant otherwise
                hkReal friction = apar.m_frictionCoefficient;
                if ( apar.m_viscosityFrictionCoefficient )
                {
                    const hkReal slipVelocitySquared = forwardSlipForFriction * forwardSlipForFriction + atmp.m_sideSlipVel * atmp.m_sideSlipVel;
                    const hkReal slipVelocity = hkMath::sqrt(slipVelocitySquared);
                    friction += slipVelocity * apar.m_viscosityFrictionCoefficient;
                    friction = hkMath::min2( friction, apar.m_maxFrictionCoefficient);
                    friction = hkMath::max2( hkReal(0.0f), friction );
                }
                atmp.m_maxForwardImpulse = downImpulse * friction;
                atmp.m_maxSideImpulse    = atmp.m_maxForwardImpulse;
                atmp.m_invMaxSideImpulse = atmp.m_invMaxForwardImpulse = 1.0f / (atmp.m_maxSideImpulse + HK_REAL_EPSILON * HK_REAL_EPSILON);

                const hkReal stopTheWheelFromSlidingImpulse = atmp.m_forwardSlipVel * atmp.m_wheelForwardVirtMass + stepInfo.m_deltaTime * forwardForce;
                atmp.m_forwardImpulse = stopTheWheelFromSlidingImpulse;
            }
        }
    }

    // Tyre slip:
    // Simply reduce the virtual mass for each tyre and our iterative solver can't solve our wheel perfectly,
    // so that our wheels starts to slide
    if ( params.m_axleParams[0].m_slipVelocityFactor + params.m_axleParams[1].m_slipVelocityFactor > 0.0f )
    {
        hkReal invTaus[HK_VEHICLE_FRICTION_N_AXIS];
        {
            const hkReal totalDownForce = hkMath::fabs( params.m_axleParams[0].m_wheelDownForce ) + hkMath::fabs( params.m_axleParams[1].m_wheelDownForce );
            const hkReal sideAccelBase = totalDownForce * descr.m_chassisMassInv;
            const hkReal MAX_INV_TAU = hkReal( 10000.0f );
            for ( int a = 0; a < HK_VEHICLE_FRICTION_N_AXIS; a++ )
            {
                hkReal Z = sideAccelBase * stepInfo.m_deltaTime;
                if (Z == 0)
                {
                    invTaus[a] = MAX_INV_TAU;
                }
                else
                {
                    hkReal N = hkMath::max2( params.m_axleParams[a].m_slipVelocityFactor, hkReal(0.0f) );
                    invTaus[a] = (Z>=N)? 1.0f : N/Z;
                    invTaus[a] = hkMath::min2( MAX_INV_TAU, invTaus[a] );
                }
            }
        }

        // update mass information
        {
            invMassMatrix[0] *= invTaus[0];
            invMassMatrix[3] *= invTaus[1];
            const hkReal f = hkMath::sqrt( invTaus[0] * invTaus[1] );
            invMassMatrix[1] *= f;
            invMassMatrix[2] *= f;
            for ( int a = 0; a < HK_VEHICLE_FRICTION_N_AXIS; a++ )
            {
                hkpVehicleFrictionAxisTmp& atmp = m_axis[a];
                atmp.m_sideVirtMass    /= invTaus[a];
                atmp.m_invSideVirtMass *= invTaus[a];
            }
        }
    }

    // Calculate real impulses and do a first quick check for friction
    hkReal slidingMagnitude[HK_VEHICLE_FRICTION_N_AXIS];
    {
        // friction
        // invert the inverse mass matrix
        HK_ALIGN_REAL(hkReal effMassMatrix[4]);
        {
            hkVector4 mm; mm.load<4>(&invMassMatrix[0]);
            hkVector4 imm; imm.set(1,0,0,1);
            HK_ON_DEBUG(hkResult success =) hkVector4Util::invert2x2Matrix( mm, hkSimdReal_0, imm );
            HK_ASSERT(0x4e7da39b, success.isSuccess(), "determinant of vehicle friction matrix = 0, maybe your position of the wheels are not set correctly" );
            imm.store<4>(&effMassMatrix[0]);
        }

        m_axis[0].m_sideImpulse = - (effMassMatrix[0] * m_axis[0].m_sideSlipVel + effMassMatrix[1] * m_axis[1].m_sideSlipVel);
        m_axis[1].m_sideImpulse = - (effMassMatrix[2] * m_axis[0].m_sideSlipVel + effMassMatrix[3] * m_axis[1].m_sideSlipVel);

        for ( int a = 0; a < HK_VEHICLE_FRICTION_N_AXIS; a++ )
        {
            // calculate the magnitude of sliding. if <0; than no sliding occurs
            hkpVehicleFrictionAxisTmp& atmp = m_axis[a];
            const hkReal x = atmp.m_forwardImpulse * atmp.m_invMaxForwardImpulse;
            const hkReal y = atmp.m_sideImpulse    * atmp.m_invMaxSideImpulse;
            slidingMagnitude[a] = x*x + y*y;
        }
    }

    if ( hkMath::fabs( m_axis[0].m_forwardVel) >= params.m_maxVelocityForPositionalFriction )
    {
        status.m_axis[0].m_sideRhs    = 0.0f;
        status.m_axis[0].m_forwardRhs = 0.0f;
        status.m_axis[1].m_sideRhs    = 0.0f;
        status.m_axis[1].m_forwardRhs = 0.0f;
    }

    // If no wheels slide then the first guess is correct
    // Otherwise try the local constraint approach, where the sliding wheel will be calculated first
    if ( slidingMagnitude[0] > 1.0f || slidingMagnitude[1] > 1.0f )
    {
        if (slidingMagnitude[0] > 1.0f)
        {
            hkReal inv = 1.f / slidingMagnitude[0];
            status.m_axis[0].m_sideRhs    *= inv;
            status.m_axis[0].m_forwardRhs *= inv;
        }

        if (slidingMagnitude[1] > 1.0f)
        {
            hkReal inv = 1.f / slidingMagnitude[1];
            status.m_axis[1].m_sideRhs    *= inv;
            status.m_axis[1].m_forwardRhs *= inv;
        }

        // order the axes
        int firstSlidingAxis, other;
        if (slidingMagnitude[0] > slidingMagnitude[1])
        {
            firstSlidingAxis = 0; other = 1;
        }
        else
        {
            firstSlidingAxis = 1; other = 0;
        }

        // loop over both axes
        for ( int i = 0; i < 2; i++ )
        {
            hkpVehicleFrictionAxisTmp &atmp = m_axis[firstSlidingAxis];
            atmp.m_sideImpulse = -atmp.m_sideSlipVel * atmp.m_sideVirtMass;
            hkVehicleFrictionSolverClipImpulse( params.m_axleParams[firstSlidingAxis], atmp, descr.m_axleDescr[firstSlidingAxis], status.m_axis[firstSlidingAxis] );

            // update needed second velocity (unnecessary for second axis)
            m_axis[other].m_sideSlipVel += atmp.m_sideImpulse * invMassMatrix[1];   // invMassMatrix is symmetric, just take any outer diagonal element
            firstSlidingAxis = other;
        }

        // update slip velocities
        status.m_axis[0].m_forward_slip_velocity = m_axis[0].m_restForwardSlipImpulse * m_axis[0].m_invWheelForwardVirtMass;
        status.m_axis[1].m_forward_slip_velocity = m_axis[1].m_restForwardSlipImpulse * m_axis[1].m_invWheelForwardVirtMass;
        status.m_axis[0].m_side_slip_velocity = m_axis[0].m_sideSlipVel;
        status.m_axis[1].m_side_slip_velocity = m_axis[1].m_sideSlipVel;
    }
    else
    {
        for ( int i = 0; i < 2; i++ )
        {
            hkpVehicleFrictionAxisTmp &atmp = m_axis[i];

            const hkReal xs = atmp.m_sideImpulse * atmp.m_invMaxSideImpulse;
            const hkReal ys = atmp.m_forwardImpulse * atmp.m_invMaxForwardImpulse;

            if ( xs*xs + ys*ys >= 1.0f )
            {
                hkReal scaleX, scaleY;
                hkCalculateFrictionalEllipsoidScale( descr.m_axleDescr[i], xs, ys, scaleX, scaleY );

                const hkReal impX = scaleX * atmp.m_maxSideImpulse;
                const hkReal absVal = hkMath::fabs( impX );

                atmp.m_sideImpulse = hkMath::clamp( atmp.m_sideImpulse, -absVal, absVal );
            }
            else
            {
                atmp.m_sideImpulse = -atmp.m_sideSlipVel * atmp.m_sideVirtMass;
            }

            // This only matters during the first iteration to accumulate velocity from the impulse when calculating the second axis.  Same
            // as the code that updates m_sideSlipVel in the sliding path, except in this path we always solve the axles in the same order.
            m_axis[1].m_sideSlipVel += atmp.m_sideImpulse * invMassMatrix[1];   // invMassMatrix is symmetric, just take any outer diagonal element
        }

        hkVehicleFrictionSolverCalcRelativeForces( m_axis[0], status.m_axis[0] );
        hkVehicleFrictionSolverCalcRelativeForces( m_axis[1], status.m_axis[1] );

        status.m_axis[0].m_forward_slip_velocity = 0.0f;
        status.m_axis[1].m_forward_slip_velocity = 0.0f;
        status.m_axis[0].m_side_slip_velocity = 0.0f;
        status.m_axis[1].m_side_slip_velocity = 0.0f;
    }

    // Remember forward/backward impulses to be applied at the next frame
    status.m_axis[0].m_delayed_forward_impulse = m_axis[0].m_forwardImpulse;
    status.m_axis[1].m_delayed_forward_impulse = m_axis[1].m_forwardImpulse;

    // Export side forces ( for force feedback devices )
    status.m_axis[0].m_side_force = m_axis[0].m_sideImpulse * stepInfo.m_invDeltaTime;
    status.m_axis[1].m_side_force = m_axis[1].m_sideImpulse * stepInfo.m_invDeltaTime;

    // Apply left and right impulses
    {
        for( int a = 0; a < 2; a++ )
        {
            const hkpVehicleFrictionAxisTmp& atmp = m_axis[a];
            const hkp1Lin2AngJacobian& jac = atmp.m_sideJac;
            hkpVelocityAccumulator* opaA = &params.m_chassis;
            hkpVelocityAccumulator* opaB = params.m_axleParams[a].m_groundObject;

            hkVehicleFrictionSolver_applyImpulse( hkSimdReal::fromFloat(atmp.m_sideImpulse), jac, opaA, opaB );
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
