// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : PHYSICS PHYSICS_2012
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics/Constraint/hkpConstraint.h>
#include <Physics/Constraint/Data/hkpConstraintData.h>

class hkpConstraintMotor;
struct hkpConstraintRuntime;

/// Utility functions to operate on a constraint data.
namespace hkpConstraintDataUtils
{
    /// Converts a constraint data to its powered version. Works only with Ragdoll and Hinge constraints.
    HK_EXPORT_PHYSICS void HK_CALL convertToPowered( _In_ hkpConstraintData* data, _In_opt_ hkpConstraintMotor* motor, bool enableMotors );

    /// Creates a copy of the constraint data in its limited version (dynamically allocates).
    HK_EXPORT_PHYSICS hkpConstraintData* HK_CALL createLimited( _In_ const hkpConstraintData* data );

    /// Extracts pivot vectors from the constraint.
    HK_EXPORT_PHYSICS hkResult HK_CALL getConstraintPivots ( _In_ const hkpConstraintData* data, _Out_ hkVector4& pivotInAOut, _Out_ hkVector4& pivotInBOut );

    /// Extracts motors from the constraints.
    HK_EXPORT_PHYSICS hkResult HK_CALL getConstraintMotors( _In_ const hkpConstraintData* data, _Out_ hkpConstraintMotor*& motor0, _Out_ hkpConstraintMotor*& motor1, _Out_ hkpConstraintMotor*& motor2 );

    /// Loosen the constraint limits so that its constrained bodies' transforms are no longer violating them.
    ///
    /// NOTE: This class only operates on the limits of CONSTRAINT_TYPE_RAGDOLL and CONSTRAINT_TYPE_LIMITEDHINGE constraints.
    ///       Other constraint types are currently not supported. Additionally, this cannot loosen constrained axes which
    ///       are not controlled by a limit (i.e., the non-principle axes of the limited hinge constraint, positional constraint, etc.).
    HK_EXPORT_PHYSICS hkResult HK_CALL loosenConstraintLimits( _In_ hkpConstraintData* data, const hkTransform& bodyATransform, const hkTransform& bodyBTransform );

    /// Returns the maximum size of the runtime for the given constraint data
    HK_EXPORT_PHYSICS int HK_CALL getSizeOfRuntime( _In_ const hkpConstraintData* dataIn );

    /// Checks if the constraint supports getting and setting of its pivot points.
    HK_EXPORT_PHYSICS bool HK_CALL constraintSupportsPivotGetSet( _In_ const hkpConstraintData* data );

    /// Get the constraint's pivot for body A.
    HK_EXPORT_PHYSICS const hkVector4& HK_CALL getPivotA(_In_ const hkpConstraintData* data );

    /// Get the constraint's pivot for body B.
    HK_EXPORT_PHYSICS const hkVector4& HK_CALL getPivotB(_In_ const hkpConstraintData* data );

    /// Get the constraint's pivot for body i = 0 / 1.
    HK_EXPORT_PHYSICS hkVector4Parameter HK_CALL getPivot( _In_ const hkpConstraintData* data, int pivotIndex );

    /// Set the constraint's pivot for body A.
    HK_EXPORT_PHYSICS void HK_CALL setPivot( _In_ hkpConstraintData* data, const hkVector4& pivot, int pivotIndex );

    /// Set the constraint's pivot for body A.
    HK_EXPORT_PHYSICS void HK_CALL setPivotTransform( _In_ hkpConstraintData* data, const hkTransform& pivot, int pivotIndex );

    /// Get the pivot transform.
    HK_EXPORT_PHYSICS void HK_CALL getPivotTransform( _In_ const hkpConstraintData* data, int index, _Out_ hkTransform& pivot );

    /// This creates a clone of the three constraints that can have motors,
    /// namely hkpLimitedHingeConstraintData, hkpPrismaticConstraintData and hkpRagdollConstraintData.
    /// This does not support the powered chains.
    /// The function returns HK_NULL for unsupported types of constraints.
    HK_EXPORT_PHYSICS hkpConstraintData* HK_CALL cloneIfCanHaveMotors( _In_ const hkpConstraintData* data );

    enum Pivot
    {
        PIVOT_A,
        PIVOT_B
    };

    /// Transform a constraint pivot; use when constraining a body to the world
    HK_EXPORT_PHYSICS void transformPivot( _In_ hkpConstraintData* data, const hkTransform& transformation, Pivot pivotToTransform );

    /// Returns a deep copy of the constraint data.
    /// If the constraint wraps another, for example, a breakable or
    /// malleable constraint, it clones the child constraint also.
    HK_EXPORT_PHYSICS hkpConstraintData* HK_CALL deepClone(_In_ const hkpConstraintData* data );

    // Returns if the provided motor is active.
    // An active motor is a motor which will impart a force on the constrained bodies.
    // This *can not* know if the motor is enabled/disabled on it's owning constraint (see hasActiveMotors).
    HK_EXPORT_PHYSICS bool HK_CALL isMotorActive( _In_opt_ const hkpConstraintMotor* motor );

    // Returns if the constraint data has any active motors.
    // An active motor is a motor which will impart a force on the constrained bodies.
    HK_EXPORT_PHYSICS bool HK_CALL hasActiveMotor( _In_opt_ const hkpConstraintData* data );
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
