// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : PHYSICS PHYSICS_2012
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Physics/Constraint/hkpConstraint.h>

#include <Physics/Constraint/Data/hkpConstraintDataUtils.h>

#include <Physics/Constraint/Data/hkpConstraintData.h>
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>
#include <Physics/Constraint/Data/Fixed/hkpFixedConstraintData.h>
#include <Physics/Constraint/Data/DeformableFixed/hkpDeformableFixedConstraintData.h>
#include <Physics/Constraint/Data/RagdollLimits/hkpRagdollLimitsData.h>
#include <Physics/Constraint/Data/HingeLimits/hkpHingeLimitsData.h>

// Constraints
#include <Physics/ConstraintSolver/Solve/hkpSolverResults.h>
#include <Physics/Constraint/Data/StiffSpring/hkpStiffSpringConstraintData.h>
#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics/Constraint/Data/Wheel/hkpWheelConstraintData.h>
#include <Physics/Constraint/Data/Hinge/hkpHingeConstraintData.h>
#include <Physics/Constraint/Data/Prismatic/hkpPrismaticConstraintData.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics/Constraint/Data/PointToPlane/hkpPointToPlaneConstraintData.h>
#include <Physics/Constraint/Data/Pulley/hkpPulleyConstraintData.h>
#include <Physics/Constraint/Data/PointToPath/hkpPointToPathConstraintData.h>
#include <Physics/Constraint/Data/Wrapper/hkpWrappedConstraintData.h>
#include <Physics/Constraint/Data/Clearance/hkpLinearClearanceConstraintData.h>
#include <Physics/Constraint/Data/Rotational/hkpRotationalConstraintData.h>
#include <Physics/Constraint/Data/AngularFriction/hkpAngularFrictionConstraintData.h>
#include <Physics/Constraint/Data/CogWheel/hkpCogWheelConstraintData.h>
#include <Physics/Constraint/Data/RackAndPinion/hkpRackAndPinionConstraintData.h>
#include <Physics/Constraint/Motor/hkpConstraintMotor.h>
#include <Physics/Constraint/Data/6Dof/hkp6DofConstraintData.h>

#include <Physics/ConstraintSolver/Constraint/Bilateral/hkpInternalConstraintUtils.h>


namespace hkpConstraintDataUtils
{
    static const hkReal smallTolerance = 0.01f;

    // Sets the constraint to a powered version. Only works with hinges and ragdolls (and will warn if it's not the case)
    void HK_CALL convertToPowered( _In_ hkpConstraintData* data, _In_opt_ hkpConstraintMotor* motor, bool enableMotors )
    {
        switch (data->getType())
        {
            case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
            {
                hkpLimitedHingeConstraintData* hingeData = static_cast<hkpLimitedHingeConstraintData*>(data);
                hingeData->setMotor( motor );
                hingeData->setMotorEnabled(HK_NULL, enableMotors);
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
            {
                hkpRagdollConstraintData* ragdollData = static_cast<hkpRagdollConstraintData*>(data);
                ragdollData->setTwistMotor(motor);
                ragdollData->setPlaneMotor(motor);
                ragdollData->setConeMotor(motor);
                ragdollData->setMotorsEnabled(HK_NULL, enableMotors);
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
            {
                hkp6DofConstraintData* ragdollData = static_cast<hkp6DofConstraintData*>(data);
                ragdollData->setAngularMotor(hkp6DofConstraintData::MOTOR_ALL, motor);
                ragdollData->setAngularMotorsEnabled( enableMotors );
                ragdollData->updateDirtyAtoms();
                break;
            }

            default:
            {
                HK_WARN_ALWAYS (0xabba1b34, "Cannot convert constraint data to a powered constraint.");
            }
        }
    }


    hkpConstraintData* HK_CALL createLimited( _In_ const hkpConstraintData* data )
    {
        // We have to create a new data.
        hkpConstraintData * result = HK_NULL;

        switch(data->getType())
        {
            case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
            {
                hkpRagdollLimitsData* n = new hkpRagdollLimitsData();
                const hkpRagdollConstraintData* o = static_cast<const hkpRagdollConstraintData*>(data);
                n->m_atoms.m_rotations.m_rotationA = o->m_atoms.m_transforms.m_transformA.getRotation();
                n->m_atoms.m_rotations.m_rotationB = o->m_atoms.m_transforms.m_transformB.getRotation();
                n->m_atoms.m_twistLimit  = o->m_atoms.m_twistLimit;
                n->m_atoms.m_planesLimit = o->m_atoms.m_planesLimit;
                n->m_atoms.m_coneLimit   = o->m_atoms.m_coneLimit;
                const bool stabilizationEnabled = o->getConeLimitStabilization();
                n->setConeLimitStabilization(stabilizationEnabled);
                result = n;


                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
            {
                hkpHingeLimitsData* n = new hkpHingeLimitsData();
                const hkpLimitedHingeConstraintData* o = static_cast<const hkpLimitedHingeConstraintData*>(data);

                n->m_atoms.m_rotations.m_rotationA = o->m_atoms.m_transforms.m_transformA.getRotation();
                n->m_atoms.m_rotations.m_rotationB = o->m_atoms.m_transforms.m_transformB.getRotation();
                n->m_atoms.m_angLimit = o->m_atoms.m_angLimit;
                n->m_atoms.m_2dAng     = o->m_atoms.m_2dAng;
                result = n;

                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
            case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL_LIMITS:
            case hkpConstraintData::CONSTRAINT_TYPE_HINGE_LIMITS:
            {
                // Just return the original constraint.
                // Remember to add a reference.
                HK_WARN(0xad67d7db, "The original constraint already is of the 'limits' type.");
                data->addReference();
                return const_cast<hkpConstraintData*>(data);
            }

            default:
            {
                HK_WARN_ALWAYS(0xabbad7da, "Unsupported constraint type. Cannot generate limits constraints.");
                return HK_NULL;
            }
        }
        return result;
    }


    hkResult HK_CALL getConstraintPivots( _In_ const hkpConstraintData* data, _Out_ hkVector4& pivotInAOut, _Out_ hkVector4& pivotInBOut )
    {
        // Unwrap wrapped constraints
        switch (data->getType())
        {
            case hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE:
            case hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE:
            {
                const hkpWrappedConstraintData* b = static_cast<const hkpWrappedConstraintData*>(data);
                data = b->getWrappedConstraintData();
                break;
            }
            default:
                break;
        }

        switch (data->getType())
        {
            case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:
            {
                const hkpBallAndSocketConstraintData* bsConstraint = static_cast<const hkpBallAndSocketConstraintData*> (data);
                pivotInAOut = bsConstraint->m_atoms.m_pivots.m_translationA;
                pivotInBOut = bsConstraint->m_atoms.m_pivots.m_translationB;
                break;
            }

                case hkpConstraintData::CONSTRAINT_TYPE_FIXED:
            {
                const hkpFixedConstraintData* fixedConstraint = static_cast<const hkpFixedConstraintData*> (data);
                pivotInAOut = fixedConstraint->m_atoms.m_transforms.m_transformA.getTranslation();
                pivotInBOut = fixedConstraint->m_atoms.m_transforms.m_transformB.getTranslation();
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:
            {
                const hkpDeformableFixedConstraintData* dcd = static_cast<const hkpDeformableFixedConstraintData*> (data);
                pivotInAOut = dcd->m_atoms.m_transforms.m_transformA.getTranslation();
                pivotInBOut = dcd->m_atoms.m_transforms.m_transformB.getTranslation();
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_HINGE:
            {
                const hkpHingeConstraintData* hConstraint = static_cast<const hkpHingeConstraintData*> (data);
                pivotInAOut = hConstraint->m_atoms.m_transforms.m_transformA.getTranslation();
                pivotInBOut = hConstraint->m_atoms.m_transforms.m_transformB.getTranslation();
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
            {
                const hkpLimitedHingeConstraintData* hConstraint = static_cast<const hkpLimitedHingeConstraintData*> (data);
                pivotInAOut = hConstraint->m_atoms.m_transforms.m_transformA.getTranslation();
                pivotInBOut = hConstraint->m_atoms.m_transforms.m_transformB.getTranslation();
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
            {
                const hkpRagdollConstraintData* rConstraint = static_cast<const hkpRagdollConstraintData*> (data);
                pivotInAOut = rConstraint->m_atoms.m_transforms.m_transformA.getTranslation();
                pivotInBOut = rConstraint->m_atoms.m_transforms.m_transformB.getTranslation();
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
            {
                const hkp6DofConstraintData* rConstraint = static_cast<const hkp6DofConstraintData*> (data);
                pivotInAOut = rConstraint->m_blueprints.m_transforms.m_transformA.getTranslation();
                pivotInBOut = rConstraint->m_blueprints.m_transforms.m_transformB.getTranslation();
                break;
            }

            default:
            {
                HK_WARN_ALWAYS (0xabbabf3b, "Unsupported type of constraint in getConstraintPivots");
                pivotInAOut.setZero();
                pivotInBOut.setZero();
                return HK_FAILURE;
            }
        }

        return HK_SUCCESS;
    }


    hkResult HK_CALL getConstraintMotors( _In_ const hkpConstraintData* data, _Out_ hkpConstraintMotor*& motor0, _Out_ hkpConstraintMotor*& motor1, _Out_ hkpConstraintMotor*& motor2 )
    {
        switch (data->getType())
        {
            case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
            {
                const hkpLimitedHingeConstraintData* hConstraint = static_cast<const hkpLimitedHingeConstraintData*> (data);
                motor0 = hConstraint->getMotor();
                motor1 = HK_NULL;
                motor2 = HK_NULL;
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
            {
                const hkpRagdollConstraintData* rConstraint = static_cast<const hkpRagdollConstraintData*> (data);
                // Possibly, those motors should be extracted in a different order.
                motor0 = rConstraint->getTwistMotor();
                motor1 = rConstraint->getConeMotor();
                motor2 = rConstraint->getPlaneMotor();
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
            {
                const hkp6DofConstraintData* rConstraint = static_cast<const hkp6DofConstraintData*> (data);
                // Possibly, those motors should be extracted in a different order.
                motor0 = rConstraint->getAngularMotor(hkp6DofConstraintData::MOTOR_TWIST);
                motor1 = rConstraint->getAngularMotor(hkp6DofConstraintData::MOTOR_SWING0);
                motor2 = rConstraint->getAngularMotor(hkp6DofConstraintData::MOTOR_SWING1);
                break;
            }

            default:
            {
                motor0 = motor1 = motor2 = HK_NULL;
                HK_WARN_ALWAYS (0xabbae233, "This type of constraint does not have motors");
                return HK_FAILURE;
            }
        }

        return HK_SUCCESS;
    }


    hkResult HK_CALL loosenConstraintLimits(_In_ hkpConstraintData* data, const hkTransform& bodyATransform, const hkTransform& bodyBTransform)
    {
        switch(data->getType())
        {
            case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
            {
                // NOTE: Body B is the reference object, A is the constrained object

                hkpRagdollConstraintData* ragdollData = static_cast<hkpRagdollConstraintData*>(data);
                hkpRagdollConstraintData::Atoms& atoms = ragdollData->m_atoms;

                // Base transforms
                const hkTransform &pivotTransformA = atoms.m_transforms.m_transformA; // parent
                const hkTransform &pivotTransformB = atoms.m_transforms.m_transformB; // child

                // Get axes information for cone limit
                hkVector4 twistAxisInA_WS;  twistAxisInA_WS._setRotatedDir(bodyATransform.getRotation(), pivotTransformA.getColumn(atoms.m_coneLimit.m_twistAxisInA));
                hkVector4 refAxisInB_WS;    refAxisInB_WS._setRotatedDir(bodyBTransform.getRotation(), pivotTransformB.getColumn(atoms.m_coneLimit.m_refAxisInB));

                // Get axes information for plane limit
                hkVector4 twistAxisInA_WS2; twistAxisInA_WS2._setRotatedDir(bodyATransform.getRotation(), pivotTransformA.getColumn(atoms.m_planesLimit.m_twistAxisInA));
                hkVector4 refAxisInB_WS2;   refAxisInB_WS2._setRotatedDir(bodyBTransform.getRotation(), pivotTransformB.getColumn(atoms.m_planesLimit.m_refAxisInB));

#if (defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_WIN32)) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED) && !defined(HK_ARCH_ARM)
                hkVector4 acosDots; acosDots.set(twistAxisInA_WS.dot<3>(refAxisInB_WS), twistAxisInA_WS2.dot<3>(refAxisInB_WS2), hkSimdReal_0, hkSimdReal_0);
                acosDots.setClamped(acosDots, hkVector4::getConstant<HK_QUADREAL_MINUS1>(), hkVector4::getConstant<HK_QUADREAL_1>());
                hkVector4 acs; hkVector4Util::aCos(acosDots,acs);
                hkReal currentConeAngle = acs(0) + hkpConstraintDataUtils::smallTolerance;
                hkReal currentPlaneAngle = acs(1);
#else
                hkReal currentConeAngle = hkMath::acos(twistAxisInA_WS.dot<3>(refAxisInB_WS).getReal()) + hkpConstraintDataUtils::smallTolerance;
                hkReal currentPlaneAngle = hkMath::acos(twistAxisInA_WS2.dot<3>(refAxisInB_WS2).getReal());
#endif

                // Loosen cone constraint to accommodate if needed
                {
                    hkReal maxConeAngle = atoms.m_coneLimit.m_maxAngle;
                    atoms.m_coneLimit.m_maxAngle = hkMath::max2(maxConeAngle, currentConeAngle);
                }

                // Loosen plane constraint to accommodate if needed
                {
                    hkReal maxPlaneAngle = atoms.m_planesLimit.m_maxAngle;
                    hkReal minPlaneAngle = atoms.m_planesLimit.m_minAngle;
                    const hkReal piDivBy2 = (HK_REAL_PI * hkReal(0.5f));
                    if (currentPlaneAngle <= piDivBy2)
                    {
                        atoms.m_planesLimit.m_maxAngle = hkMath::max2(maxPlaneAngle, piDivBy2 - currentPlaneAngle + hkpConstraintDataUtils::smallTolerance);
                    }
                    else
                    {
                        atoms.m_planesLimit.m_minAngle = hkMath::min2(minPlaneAngle, -(currentPlaneAngle - piDivBy2) - hkpConstraintDataUtils::smallTolerance);
                    }
                }

                // Get axes information for cone limit
                twistAxisInA_WS._setRotatedDir(bodyATransform.getRotation(), pivotTransformA.getColumn(atoms.m_twistLimit.m_twistAxis));
                refAxisInB_WS._setRotatedDir(bodyBTransform.getRotation(), pivotTransformB.getColumn(atoms.m_twistLimit.m_twistAxis)); // reused for twist axis
                hkVector4 planeAxisA_WS;    planeAxisA_WS._setRotatedDir(bodyATransform.getRotation(), pivotTransformA.getColumn(atoms.m_twistLimit.m_refAxis));
                hkVector4 planeAxisB_WS;    planeAxisB_WS._setRotatedDir(bodyBTransform.getRotation(), pivotTransformB.getColumn(atoms.m_twistLimit.m_refAxis));
                hkVector4 currentTwistAxis_WS;
                hkReal currentTwistAngle;
                hkInternalConstraintUtils_calcRelativeAngle( twistAxisInA_WS, refAxisInB_WS, planeAxisA_WS, planeAxisB_WS, currentTwistAxis_WS, currentTwistAngle );

                // Loosen twist constraint to accommodate if needed
                {
                    hkReal minTwistAngle = atoms.m_twistLimit.m_minAngle;
                    hkReal maxTwistAngle = atoms.m_twistLimit.m_maxAngle;
                    if (currentTwistAngle >= hkReal(0))
                    {
                        atoms.m_twistLimit.m_maxAngle = hkMath::max2(maxTwistAngle, currentTwistAngle + hkpConstraintDataUtils::smallTolerance);
                    }
                    else
                    {
                        atoms.m_twistLimit.m_minAngle = hkMath::min2(minTwistAngle, currentTwistAngle - hkpConstraintDataUtils::smallTolerance);
                    }
                }
                break;
            }
            case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
            {
                hkpLimitedHingeConstraintData* hingeData = static_cast<hkpLimitedHingeConstraintData*>(data);
                hkpLimitedHingeConstraintData::Atoms& atoms = hingeData->m_atoms;

                // Base transforms
                const hkTransform &pivotTransformA = atoms.m_transforms.m_transformA; // parent
                const hkTransform &pivotTransformB = atoms.m_transforms.m_transformB; // child

                // Get limited axis in world space
                // NOTE: looking in hkpLimitedHingeConstraintData, found that column(1) contains perpAxis, column(2) contains a second perpAxis
                hkVector4 perpAxisA_WS;  perpAxisA_WS._setRotatedDir(bodyATransform.getRotation(), pivotTransformA.getColumn<1>());
                hkVector4 perpAxisB_WS;  perpAxisB_WS._setRotatedDir(bodyBTransform.getRotation(), pivotTransformB.getColumn<1>());
                hkVector4 perpAxisBPerp_WS;   perpAxisBPerp_WS._setRotatedDir(bodyBTransform.getRotation(), pivotTransformB.getColumn<2>());

                // Loosen bend constraint to accommodate if needed
                {
                    hkReal minBendAngle = atoms.m_angLimit.m_minAngle;
                    hkReal maxBendAngle = atoms.m_angLimit.m_maxAngle;
#if (defined(HK_ARCH_IA32) || defined(HK_ARCH_X64)) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
                    hkSimdReal acosDot; acosDot.setClamped( perpAxisA_WS.dot<3>(perpAxisB_WS), hkSimdReal_Minus1, hkSimdReal_1 );
                    acosDot = hkVector4Util::aCos(acosDot);
                    acosDot.setFlipSign(acosDot, perpAxisA_WS.dot<3>(perpAxisBPerp_WS).lessZero());
                    hkReal currentBendAngle = acosDot.getReal();
#else
                    hkReal currentBendAngleSign = perpAxisA_WS.dot<3>(perpAxisBPerp_WS).isGreaterEqualZero() ? hkReal(1) : hkReal(-1);
                    hkReal currentBendAngle = currentBendAngleSign * hkMath::acos(perpAxisA_WS.dot<3>(perpAxisB_WS).getReal());
#endif
                    if (currentBendAngle >= hkReal(0))
                    {
                        atoms.m_angLimit.m_maxAngle = hkMath::max2(maxBendAngle, currentBendAngle + hkpConstraintDataUtils::smallTolerance);
                    }
                    else
                    {
                        atoms.m_angLimit.m_minAngle = hkMath::min2(minBendAngle, currentBendAngle - hkpConstraintDataUtils::smallTolerance);
                    }
                }
            }
            break;

        default:
            HK_WARN_ALWAYS (0x7a501290, "This type of constraint is not supported for loosening.");
            return HK_FAILURE;
        }

        return HK_SUCCESS;
    }


    /// Returns the maximum size of the runtime for the given constraint data
    int HK_CALL getSizeOfRuntime(_In_ const hkpConstraintData* dataIn)
    {
        switch ( dataIn->getType() )
        {
        case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:      return sizeof(hkpBallAndSocketConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:   return sizeof(hkpDeformableFixedConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_FIXED:              return sizeof(hkpFixedConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_HINGE:              return sizeof(hkpHingeConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:       return sizeof(hkpLimitedHingeConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:            return sizeof(hkpRagdollConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL_LIMITS:     return sizeof(hkpRagdollLimitsData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_6DOF:               return sizeof(hkp6DofConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:          return sizeof(hkpPrismaticConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:        return sizeof(hkpStiffSpringConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:              return sizeof(hkpWheelConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_PULLEY:             return sizeof(hkpPulleyConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_HINGE_LIMITS:       return sizeof(hkpHingeLimitsData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPATH:        return sizeof(hkpPointToPathConstraintData::Runtime);
        case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:       return sizeof(hkpPointToPlaneConstraintData::Runtime);

//      case hkpConstraintData::CONSTRAINT_TYPE_COG_WHEEL:          return sizeof(hkpCogWheelConstraintData::Runtime);
//      case hkpConstraintData::CONSTRAINT_TYPE_ROTATIONAL:         return sizeof(hkpRotationalConstraintData::Runtime);
//      case hkpConstraintData::CONSTRAINT_TYPE_RACK_AND_PINION:    return sizeof(hkpRackAndPinionConstraintData::Runtime);
        default:
            HK_ASSERT(0x40a64513, false, "Unsupported constraint type" );
            break;
        }

        return 0;
    }
}


bool HK_CALL hkpConstraintDataUtils::constraintSupportsPivotGetSet(_In_ const hkpConstraintData* data)
{
    switch(data->getType())
    {
    case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:
    case hkpConstraintData::CONSTRAINT_TYPE_FIXED:
    case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:
    case hkpConstraintData::CONSTRAINT_TYPE_HINGE:
    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:
    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
    case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:
    case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:
    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:
    case hkpConstraintData::CONSTRAINT_TYPE_PULLEY:
    case hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE:
    case hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE:
        return true;

    default:
        return false;
    }
}

//
//  Get the constraint's pivot for body i = 0 / 1.

hkVector4Parameter HK_CALL hkpConstraintDataUtils::getPivot(_In_ const hkpConstraintData* data, int pivotIndex)
{
    switch ( data->getType() )
    {
    case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:      { const hkpBallAndSocketConstraintData*     d = static_cast<const hkpBallAndSocketConstraintData*>(data);       return (&d->m_atoms.m_pivots.m_translationA)[pivotIndex]; }
    case hkpConstraintData::CONSTRAINT_TYPE_FIXED:              { const hkpFixedConstraintData*             d = static_cast<const hkpFixedConstraintData*>(data);               return (&d->m_atoms.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:   { const hkpDeformableFixedConstraintData*   d = static_cast<const hkpDeformableFixedConstraintData*>(data);     return (&d->m_atoms.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_HINGE:              { const hkpHingeConstraintData*             d = static_cast<const hkpHingeConstraintData*>(data);               return (&d->m_atoms.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:       { const hkpLimitedHingeConstraintData*      d = static_cast<const hkpLimitedHingeConstraintData*>(data);        return (&d->m_atoms.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:          { const hkpPrismaticConstraintData*         d = static_cast<const hkpPrismaticConstraintData*>(data);           return (&d->m_atoms.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:            { const hkpRagdollConstraintData*           d = static_cast<const hkpRagdollConstraintData*>(data);             return (&d->m_atoms.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:               { const hkp6DofConstraintData*              d = static_cast<const hkp6DofConstraintData*>(data);                return (&d->m_blueprints.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:        { const hkpStiffSpringConstraintData*       d = static_cast<const hkpStiffSpringConstraintData*>(data);         return (&d->m_atoms.m_pivots.m_translationA)[pivotIndex]; }
    case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:              { const hkpWheelConstraintData*             d = static_cast<const hkpWheelConstraintData*>(data);               return (&d->m_atoms.m_suspensionBase.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:       { const hkpPointToPlaneConstraintData*      d = static_cast<const hkpPointToPlaneConstraintData*>(data);        return (&d->m_atoms.m_transforms.m_transformA)[pivotIndex].getTranslation(); }
    case hkpConstraintData::CONSTRAINT_TYPE_PULLEY:             { const hkpPulleyConstraintData*            d = static_cast<const hkpPulleyConstraintData*>(data);              return (&d->m_atoms.m_translations.m_translationA)[pivotIndex]; }

    case hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE:
    case hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE:
                                                                return getPivot(static_cast<const hkpWrappedConstraintData*>(data)->getWrappedConstraintData(), pivotIndex);

    default:
        {
            HK_WARN(0xad873344, "This constraint doesn't support getPivot() functionality.");
            return hkVector4::getZero();
        }
    }
}

const hkVector4& HK_CALL hkpConstraintDataUtils::getPivotA(_In_ const hkpConstraintData* data)
{
    return getPivot( data, 0 );
}

const hkVector4& HK_CALL hkpConstraintDataUtils::getPivotB(_In_ const hkpConstraintData* data)
{
    return getPivot( data, 1 );
}


void HK_CALL hkpConstraintDataUtils::setPivot(_In_ hkpConstraintData* data, const hkVector4& pivot, int index)
{
    switch(data->getType())
    {
    case hkpConstraintData::CONSTRAINT_TYPE_HINGE:              { hkpHingeConstraintData* d             = static_cast<hkpHingeConstraintData*>(             data); (&d->m_atoms.m_transforms.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:       { hkpLimitedHingeConstraintData* d      = static_cast<hkpLimitedHingeConstraintData*>(      data); (&d->m_atoms.m_transforms.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:          { hkpPrismaticConstraintData* d         = static_cast<hkpPrismaticConstraintData*>(         data); (&d->m_atoms.m_transforms.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:            { hkpRagdollConstraintData* d           = static_cast<hkpRagdollConstraintData*>(           data); (&d->m_atoms.m_transforms.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:              { hkpWheelConstraintData* d             = static_cast<hkpWheelConstraintData*>(             data); (&d->m_atoms.m_suspensionBase.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:       { hkpPointToPlaneConstraintData* d      = static_cast<hkpPointToPlaneConstraintData*>(      data); (&d->m_atoms.m_transforms.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_PULLEY:             { hkpPulleyConstraintData* d            = static_cast<hkpPulleyConstraintData*>(            data); (&d->m_atoms.m_translations.m_translationA)[index] = pivot; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:      { hkpBallAndSocketConstraintData* d     = static_cast<hkpBallAndSocketConstraintData*>(     data); (&d->m_atoms.m_pivots.m_translationA)[index] = pivot; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_FIXED:              { hkpFixedConstraintData* d             = static_cast<hkpFixedConstraintData*>(             data); (&d->m_atoms.m_transforms.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:   { hkpDeformableFixedConstraintData* d   = static_cast<hkpDeformableFixedConstraintData*>(   data); (&d->m_atoms.m_transforms.m_transformA)[index].setTranslation(pivot); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:        { hkpStiffSpringConstraintData* d       = static_cast<hkpStiffSpringConstraintData*>(       data); (&d->m_atoms.m_pivots.m_translationA)[index] = pivot; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:               { hkp6DofConstraintData* d              = static_cast<hkp6DofConstraintData*>(              data); (&d->m_blueprints.m_transforms.m_transformA)[index].setTranslation(pivot); d->m_isDirty = true; d->updateDirtyAtoms(); break; }

    case hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE:
    case hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE:
                                                                { setPivot( static_cast<hkpWrappedConstraintData*>(data)->accessWrappedConstraintData(), pivot, index ); break; }

                                                            //
                                                            // Those constraints are a contact, or don't have pivots (rotational only), or have multiple pivots (chains), are just left out for the moment
                                                            //

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_CONTACT:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_ROTATIONAL:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_HINGE_LIMITS:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL_LIMITS:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_STIFF_SPRING_CHAIN:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_BALL_SOCKET_CHAIN:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_POWERED_CHAIN:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPATH:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_CUSTOM:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_GENERIC:

    default:
        {
            HK_WARN(0xad873344, "This constraint doesn't support setPivot() functionality.");
        }
    }
}

void HK_CALL hkpConstraintDataUtils::setPivotTransform(_In_ hkpConstraintData* data, const hkTransform& pivot, int index)
{
    switch(data->getType())
    {
    case hkpConstraintData::CONSTRAINT_TYPE_HINGE:              { hkpHingeConstraintData* d             = static_cast<hkpHingeConstraintData*>(        data); (&d->m_atoms.m_transforms.m_transformA)[index] = pivot; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:       { hkpLimitedHingeConstraintData* d      = static_cast<hkpLimitedHingeConstraintData*>( data); (&d->m_atoms.m_transforms.m_transformA)[index] = pivot; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:          { hkpPrismaticConstraintData* d         = static_cast<hkpPrismaticConstraintData*>(    data); (&d->m_atoms.m_transforms.m_transformA)[index] = pivot; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:            { hkpRagdollConstraintData* d           = static_cast<hkpRagdollConstraintData*>(      data); (&d->m_atoms.m_transforms.m_transformA)[index] = pivot; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:               { hkp6DofConstraintData* d              = static_cast<hkp6DofConstraintData*>(         data); (&d->m_blueprints.m_transforms.m_transformA)[index] = pivot; d->m_isDirty = true; d->updateDirtyAtoms(); break; }

    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:       { hkpPointToPlaneConstraintData* d      = static_cast<hkpPointToPlaneConstraintData*>(      data); (&d->m_atoms.m_transforms.   m_transformA)[index]    = pivot;                    break; }
    case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:              { hkpWheelConstraintData* d             = static_cast<hkpWheelConstraintData*>(             data); (&d->m_atoms.m_suspensionBase.m_transformA)[index]   = pivot;                    break; }
    case hkpConstraintData::CONSTRAINT_TYPE_FIXED:              { hkpFixedConstraintData* d             = static_cast<hkpFixedConstraintData*>(             data); (&d->m_atoms.m_transforms.   m_transformA)[index]    = pivot;                    break; }
    case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:   { hkpDeformableFixedConstraintData* d   = static_cast<hkpDeformableFixedConstraintData*>(   data); (&d->m_atoms.m_transforms.   m_transformA)[index]    = pivot;                    break; }
    case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:      { hkpBallAndSocketConstraintData* d     = static_cast<hkpBallAndSocketConstraintData*>(     data); (&d->m_atoms.m_pivots.       m_translationA)[index]  = pivot.getTranslation();   break; }
    case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:        { hkpStiffSpringConstraintData* d       = static_cast<hkpStiffSpringConstraintData*>(       data); (&d->m_atoms.m_pivots.       m_translationA)[index]  = pivot.getTranslation();   break; }
    case hkpConstraintData::CONSTRAINT_TYPE_PULLEY:             { hkpPulleyConstraintData* d            = static_cast<hkpPulleyConstraintData*>(            data); (&d->m_atoms.m_translations. m_translationA)[index]  = pivot.getTranslation();   break; }

    case hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE:
    case hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE:
                                                                { setPivotTransform( static_cast<hkpWrappedConstraintData*>(data)->accessWrappedConstraintData(), pivot, index ); break; }

                                                            //
                                                            // Those constraints are a contact, or don't have pivots (rotational only), or have multiple pivots (chains), are just left out for the moment
                                                            //

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_CONTACT:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_ROTATIONAL:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_HINGE_LIMITS:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL_LIMITS:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_STIFF_SPRING_CHAIN:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_BALL_SOCKET_CHAIN:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_POWERED_CHAIN:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPATH:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_CUSTOM:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_GENERIC:

    default:
        {
            HK_WARN(0xad873344, "This constraint doesn't support setPivot() functionality.");
        }
    }
}

void HK_CALL hkpConstraintDataUtils::getPivotTransform(_In_ const hkpConstraintData* data, int index, _Out_ hkTransform& pivot)
{
    pivot.setIdentity();

    switch(data->getType())
    {
    case hkpConstraintData::CONSTRAINT_TYPE_HINGE:              { const hkpHingeConstraintData* d           = static_cast<const hkpHingeConstraintData*>(           data); pivot = (&d->m_atoms.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:       { const hkpLimitedHingeConstraintData* d    = static_cast<const hkpLimitedHingeConstraintData*>(    data); pivot = (&d->m_atoms.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:          { const hkpPrismaticConstraintData* d       = static_cast<const hkpPrismaticConstraintData*>(       data); pivot = (&d->m_atoms.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:            { const hkpRagdollConstraintData* d         = static_cast<const hkpRagdollConstraintData*>(         data); pivot = (&d->m_atoms.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:               { const hkp6DofConstraintData* d            = static_cast<const hkp6DofConstraintData*>(            data); pivot = (&d->m_blueprints.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:       { const hkpPointToPlaneConstraintData* d    = static_cast<const hkpPointToPlaneConstraintData*>(    data); pivot = (&d->m_atoms.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_FIXED:              { const hkpFixedConstraintData* d           = static_cast<const hkpFixedConstraintData*>(           data); pivot = (&d->m_atoms.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:   { const hkpDeformableFixedConstraintData* d = static_cast<const hkpDeformableFixedConstraintData*>( data); pivot = (&d->m_atoms.m_transforms.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:              { const hkpWheelConstraintData* d           = static_cast<const hkpWheelConstraintData*>(           data); pivot = (&d->m_atoms.m_suspensionBase.m_transformA)[index]; break; }
    case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:      { const hkpBallAndSocketConstraintData* d   = static_cast<const hkpBallAndSocketConstraintData*>(   data); pivot.setTranslation((&d->m_atoms.m_pivots.m_translationA)[index] ); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:        { const hkpStiffSpringConstraintData* d     = static_cast<const hkpStiffSpringConstraintData*>(     data); pivot.setTranslation((&d->m_atoms.m_pivots.m_translationA)[index] ); break; }
    case hkpConstraintData::CONSTRAINT_TYPE_PULLEY:             { const hkpPulleyConstraintData* d          = static_cast<const hkpPulleyConstraintData*>(          data); pivot.setTranslation((&d->m_atoms.m_translations.m_translationA)[index] ); break; }

    case hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE:
    case hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE:
                                                                { getPivotTransform( static_cast<const hkpWrappedConstraintData*>(data)->getWrappedConstraintData(), index, pivot ); break; }

                                                            //
                                                            // Those constraints are a contact, or don't have pivots (rotational only), or have multiple pivots (chains), are just left out for the moment
                                                            //

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_CONTACT:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_ROTATIONAL:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_HINGE_LIMITS:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL_LIMITS:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_STIFF_SPRING_CHAIN:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_BALL_SOCKET_CHAIN:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_POWERED_CHAIN:

                                                            //case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPATH:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_CUSTOM:
                                                            //case hkpConstraintData::CONSTRAINT_TYPE_GENERIC:

    default:
        {
            HK_WARN(0xad873344, "This constraint doesn't support setPivot() functionality.");
        }
    }
}


hkpConstraintData* HK_CALL hkpConstraintDataUtils::cloneIfCanHaveMotors(_In_ const hkpConstraintData* data)
{
    hkpConstraintData::ConstraintType type = (hkpConstraintData::ConstraintType)data->getType();

    switch(type)
    {
    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
        {
            const hkpLimitedHingeConstraintData* oldData = static_cast<const hkpLimitedHingeConstraintData*>(data);

            // Create new instance
            hkpLimitedHingeConstraintData* newData = new hkpLimitedHingeConstraintData();

            // Copy all data
            newData->m_atoms = oldData->m_atoms;
            newData->m_userData = oldData->m_userData;

            // Clone motors
            class hkpConstraintMotor*& motor = newData->m_atoms.m_angMotor.m_motor;
            if (motor)
            {
                motor = motor->clone();
            }

            // All done
            return newData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:
        {
            const hkpPrismaticConstraintData* oldData = static_cast<const hkpPrismaticConstraintData*>(data);

            // Create new instance
            hkpPrismaticConstraintData* newData = new hkpPrismaticConstraintData();

            // Copy all data
            newData->m_atoms = oldData->m_atoms;
            newData->m_userData = oldData->m_userData;

            // Clone motors
            class hkpConstraintMotor*& motor = newData->m_atoms.m_motor.m_motor;
            if (motor)
            {
                motor = motor->clone();
            }

            // All done
            return newData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
        {
            const hkpRagdollConstraintData* oldData = static_cast<const hkpRagdollConstraintData*>(data);

            // Create new instance
            hkpRagdollConstraintData* newData = new hkpRagdollConstraintData();

            // Copy all data
            newData->m_atoms = oldData->m_atoms;
            newData->m_userData = oldData->m_userData;

            // Clone motors
            class hkpConstraintMotor** motors = newData->m_atoms.m_ragdollMotors.m_motors;
            for (int i = 0; i < 3; i++)
            {
                if (motors[i])
                {
                    motors[i] = motors[i]->clone();
                }
            }

            // All done
            return newData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
        {
            const hkp6DofConstraintData* oldData = static_cast<const hkp6DofConstraintData*>(data);

            // Create new instance
            hkp6DofConstraintData* newData = new hkp6DofConstraintData();

            // Copy all data
            newData->m_blueprints = oldData->m_blueprints;
            newData->m_userData = oldData->m_userData;

            // Clone motors
            for (int i = 0; i < 3; i++)
            {
                if ( oldData->m_blueprints.linearMotor(i).m_motor )
                {
                    newData->m_blueprints.linearMotor(i).m_motor = oldData->m_blueprints.linearMotor(i).m_motor;
                }
                if ( oldData->m_blueprints.m_ragdollMotors.m_motors[i] )
                {
                    newData->m_blueprints.m_ragdollMotors.m_motors[i] = oldData->m_blueprints.m_ragdollMotors.m_motors[i];
                }
            }

            // All done
            return newData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_LINEAR_SLACK:
        {
            const hkpLinearClearanceConstraintData* oldData = static_cast<const hkpLinearClearanceConstraintData*>(data);

            hkpLinearClearanceConstraintData* newData = new hkpLinearClearanceConstraintData();
            newData->m_atoms = oldData->m_atoms;
            newData->m_userData = oldData->m_userData;
            hkpConstraintMotor* motor = oldData->getMotor();
            if(motor)
            {
                newData->m_atoms.m_motor.m_motor = motor->clone();
            }

            return newData;
        }

    default:
        {
            // Constraint doesn't have motors
            return HK_NULL;
        }
    }
}

template <class ConstraintDataType>
static ConstraintDataType* HK_CALL hkpConstraintDataUtils_copyAtoms(const hkpConstraintData* constraintIn)
{
    ConstraintDataType* constraintOut = new ConstraintDataType();

    const int size  = static_cast<const ConstraintDataType*>(constraintIn)->m_atoms.getSizeOfAllAtoms();
    const hkpConstraintAtom* src_atom = static_cast<const ConstraintDataType*>(constraintIn)->m_atoms.getAtoms();
    const hkpConstraintAtom* dst_atom = static_cast<ConstraintDataType*>(constraintOut)->m_atoms.getAtoms();
    void* dst = (void*)dst_atom;

    hkString::memCpy(dst, src_atom, size);

    return constraintOut;
}

static hkp6DofConstraintData* HK_CALL hkpConstraintDataUtils_copyAtoms2(const hkp6DofConstraintData* constraintIn)
{
    hkp6DofConstraintData* constraintOut = new hkp6DofConstraintData();

    const int size  = sizeof( constraintIn->m_blueprints );
    const void* src_atom = &static_cast<const hkp6DofConstraintData*>(constraintIn)->m_blueprints;
    const void* dst_atom = &static_cast<const hkp6DofConstraintData*>(constraintOut)->m_blueprints;
    void* dst = (void*)dst_atom;

    hkString::memCpy(dst, src_atom, size);

    return constraintOut;
}

namespace
{
    // Helper used by hkpConstraintDataUtils::deepClone()
    static HK_INLINE void _addRefChecked( const hkReferencedObject* o)
    {
        if(o)
        {
            o->addReference();
        }
    }
}

hkpConstraintData* HK_CALL hkpConstraintDataUtils::deepClone(_In_ const hkpConstraintData* data)
{
    HK_ASSERT(0x38dbcef2, data, "Constraint data is null");

    switch(data->getType())
    {
    case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:      return hkpConstraintDataUtils_copyAtoms<hkpBallAndSocketConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_FIXED:              return hkpConstraintDataUtils_copyAtoms<hkpFixedConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:   return hkpConstraintDataUtils_copyAtoms<hkpDeformableFixedConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_HINGE:              return hkpConstraintDataUtils_copyAtoms<hkpHingeConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:        return hkpConstraintDataUtils_copyAtoms<hkpStiffSpringConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:              return hkpConstraintDataUtils_copyAtoms<hkpWheelConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:       return hkpConstraintDataUtils_copyAtoms<hkpPointToPlaneConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_PULLEY:             return hkpConstraintDataUtils_copyAtoms<hkpPulleyConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_HINGE_LIMITS:       return hkpConstraintDataUtils_copyAtoms<hkpHingeLimitsData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL_LIMITS:     return hkpConstraintDataUtils_copyAtoms<hkpRagdollLimitsData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_ROTATIONAL:         return hkpConstraintDataUtils_copyAtoms<hkpRotationalConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_ANGULAR_FRICTION:   return hkpConstraintDataUtils_copyAtoms<hkpAngularFrictionConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_COG_WHEEL:          return hkpConstraintDataUtils_copyAtoms<hkpCogWheelConstraintData>(data);
    case hkpConstraintData::CONSTRAINT_TYPE_RACK_AND_PINION:    return hkpConstraintDataUtils_copyAtoms<hkpRackAndPinionConstraintData>(data);

    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
        {
            hkpLimitedHingeConstraintData* limitedHingeData = hkpConstraintDataUtils_copyAtoms<hkpLimitedHingeConstraintData>(data);

            // Add a reference to the motor, if it exists. If you wish to deepClone this motor, use cloneIfCanHaveMotors()
            
            if( limitedHingeData->getMotor() )
            {
                limitedHingeData->getMotor()->addReference();
            }

            return limitedHingeData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPATH:
        {
            hkpPointToPathConstraintData* copy = hkpConstraintDataUtils_copyAtoms<hkpPointToPathConstraintData>(data);
            hkpParametricCurve* path = static_cast<const hkpPointToPathConstraintData*>(data)->getPath();

            hkpParametricCurve* clone = path->clone();

            copy->setPath(path);
            clone->removeReference();

            return copy;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:
        {
            hkpPrismaticConstraintData* prismaticData = hkpConstraintDataUtils_copyAtoms<hkpPrismaticConstraintData>(data);

            // Add a reference to the motor, if it exists. If you wish to deepClone this motor, use cloneIfCanHaveMotors()
            
            _addRefChecked( prismaticData->getMotor() );

            return prismaticData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
        {
            hkpRagdollConstraintData* ragdollData = hkpConstraintDataUtils_copyAtoms<hkpRagdollConstraintData>(data);

            // Add a reference to the motors, if they exist. If you wish to deepClone this motor, use cloneIfCanHaveMotors()
            
            _addRefChecked( ragdollData->getTwistMotor() );
            _addRefChecked( ragdollData->getConeMotor() );
            _addRefChecked( ragdollData->getPlaneMotor() );

            return ragdollData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
        {
            hkp6DofConstraintData* ragdollData = hkpConstraintDataUtils_copyAtoms2((const hkp6DofConstraintData*)data);

            // Add a reference to the motors, if they exist. If you wish to deepClone this motor, use cloneIfCanHaveMotors()
            
            _addRefChecked( ragdollData->getAngularMotor( hkp6DofConstraintData::MOTOR_SWING0 ) );
            _addRefChecked( ragdollData->getAngularMotor( hkp6DofConstraintData::MOTOR_SWING1 ) );
            _addRefChecked( ragdollData->getAngularMotor( hkp6DofConstraintData::MOTOR_TWIST ) );

            _addRefChecked( ragdollData->getLinearMotor( 0 ) );
            _addRefChecked( ragdollData->getLinearMotor( 1 ) );
            _addRefChecked( ragdollData->getLinearMotor( 2 ) );


            return ragdollData;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_LINEAR_SLACK:
        {
            hkpLinearClearanceConstraintData* linearClearanceData = hkpConstraintDataUtils_copyAtoms<hkpLinearClearanceConstraintData>(data);
            if(linearClearanceData->getMotor())
            {
                linearClearanceData->getMotor()->addReference();
            }

            return linearClearanceData;
        }

        // Breakable constraint
    case hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE:
    case hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE:
        {
            const hkpWrappedConstraintData* srcData = reinterpret_cast<const hkpWrappedConstraintData*>(data);
            hkpWrappedConstraintData* dstData = srcData->deepClone();
            return dstData;
        }

        // Generic & contact constrainsts
    case hkpConstraintData::CONSTRAINT_TYPE_GENERIC:
    case hkpConstraintData::CONSTRAINT_TYPE_CONTACT:
    case hkpConstraintData::CONSTRAINT_TYPE_STIFF_SPRING_CHAIN:
    case hkpConstraintData::CONSTRAINT_TYPE_BALL_SOCKET_CHAIN:
    case hkpConstraintData::CONSTRAINT_TYPE_POWERED_CHAIN:
    default:
        {
            HK_ASSERT(0xad8754dd, false, "Cloning of chain/generic/simplecontact costraints not supported.");
            return HK_NULL;
        }
    }
}

void hkpConstraintDataUtils::transformPivot( _In_ hkpConstraintData* data, const hkTransform& extraTransform, hkpConstraintDataUtils::Pivot pivotId )
{
    while (data->getType() == hkpConstraintData::CONSTRAINT_TYPE_MALLEABLE
        || data->getType() == hkpConstraintData::CONSTRAINT_TYPE_BREAKABLE)
    {
        data = static_cast<hkpWrappedConstraintData*>(data)->accessWrappedConstraintData();
    }

    hkTransform transformationA;
    hkTransform transformationB;
    if(pivotId == PIVOT_A)
    {
        transformationA = extraTransform;
        transformationB = hkTransform::getIdentity();
    }
    else
    {
        transformationA = hkTransform::getIdentity();
        transformationB = extraTransform;
    }

    switch(data->getType())
    {
    case hkpConstraintData::CONSTRAINT_TYPE_BALLANDSOCKET:
        {
            hkpBallAndSocketConstraintData* ballSocketData = static_cast<hkpBallAndSocketConstraintData*>(data);

            hkVector4& pivotA = ballSocketData->m_atoms.m_pivots.m_translationA;
            hkVector4& pivotB = ballSocketData->m_atoms.m_pivots.m_translationB;
            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);
            ballSocketData->setInBodySpace(pivotA,pivotB);
            break;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_FIXED:
        {
            hkpFixedConstraintData* fixedConstraintData = static_cast<hkpFixedConstraintData*>(data);

            hkTransform pivotA; pivotA.setMul(transformationA, fixedConstraintData->m_atoms.m_transforms.m_transformA);
            hkTransform pivotB; pivotB.setMul(transformationB, fixedConstraintData->m_atoms.m_transforms.m_transformB);
            fixedConstraintData->setInBodySpace(pivotA, pivotB);
            break;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_DEFORMABLE_FIXED:
        {
            hkpDeformableFixedConstraintData* constraintData = static_cast<hkpDeformableFixedConstraintData*>(data);

            hkTransform pivotA; pivotA.setMul(transformationA, constraintData->m_atoms.m_transforms.m_transformA);
            hkTransform pivotB; pivotB.setMul(transformationB, constraintData->m_atoms.m_transforms.m_transformB);
            constraintData->setInBodySpace(pivotA, pivotB);
            break;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_HINGE:
        {
            hkpHingeConstraintData* hingeData = static_cast<hkpHingeConstraintData*>(data);

            hkTransform aTransform = hingeData->m_atoms.m_transforms.m_transformA;
            hkTransform bTransform = hingeData->m_atoms.m_transforms.m_transformB;
            hkVector4& pivotA =  aTransform.getTranslation();
            hkVector4& pivotB =  bTransform.getTranslation();
            hkVector4 axisA = aTransform.getColumn<0>();
            hkVector4 axisB = bTransform.getColumn<0>();
            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);
            axisA.setRotatedDir(transformationA.getRotation(), axisA);
            axisB.setRotatedDir(transformationB.getRotation(), axisB);
            hingeData->setInBodySpace(pivotA,pivotB,axisA,axisB);
            break;
        }
    case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
        {
            hkpLimitedHingeConstraintData* hingeData = static_cast<hkpLimitedHingeConstraintData*>(data);

            hkTransform aTransform = hingeData->m_atoms.m_transforms.m_transformA;
            hkTransform bTransform = hingeData->m_atoms.m_transforms.m_transformB;
            hkVector4 axisA = aTransform.getColumn<0>();
            hkVector4 axisB = bTransform.getColumn<0>();
            hkVector4 axisAPerp = aTransform.getColumn<1>();
            hkVector4 axisBPerp = bTransform.getColumn<1>();
            hkVector4& pivotA =  aTransform.getColumn(3);
            hkVector4& pivotB =  bTransform.getColumn(3);
            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);
            axisA.setRotatedDir(transformationA.getRotation(), axisA);
            axisB.setRotatedDir(transformationB.getRotation(), axisB);
            axisAPerp.setRotatedDir(transformationA.getRotation(), axisAPerp);
            axisBPerp.setRotatedDir(transformationB.getRotation(), axisBPerp);
            hingeData->setInBodySpace(pivotA,pivotB,axisA,axisB,axisAPerp,axisBPerp);
            break;
        }

    case hkpConstraintData::CONSTRAINT_TYPE_POINTTOPLANE:
        {
            hkpPointToPlaneConstraintData* planeData = static_cast<hkpPointToPlaneConstraintData*>(data);

            hkVector4 pivotA = planeData->m_atoms.m_transforms.m_transformA.getTranslation();
            hkVector4 pivotB = planeData->m_atoms.m_transforms.m_transformB.getTranslation();
            int planeNormalIndex = planeData->m_atoms.m_lin.m_axisIndex;
            hkVector4 normalB = planeData->m_atoms.m_transforms.m_transformB.getColumn(planeNormalIndex);
            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);
            normalB.setRotatedDir(transformationB.getRotation(),normalB);
            planeData->setInBodySpace(pivotA,pivotB,normalB);
            break;
        }
    case hkpConstraintData::CONSTRAINT_TYPE_PRISMATIC:
        {
            hkpPrismaticConstraintData* prisData = static_cast<hkpPrismaticConstraintData*>(data);

            hkTransform aTransform = prisData->m_atoms.m_transforms.m_transformA;
            hkTransform bTransform = prisData->m_atoms.m_transforms.m_transformB;
            hkVector4 axisA = aTransform.getColumn<0>();
            hkVector4 axisPerpA = aTransform.getColumn<1>();
            hkVector4 pivotA = aTransform.getTranslation();
            hkVector4 axisB = bTransform.getColumn<0>();
            hkVector4 axisPerpB = bTransform.getColumn<1>();
            hkVector4 pivotB = bTransform.getTranslation();
            pivotA.setTransformedPos(transformationA, pivotA);
            pivotB.setTransformedPos(transformationB, pivotB);
            axisA.setRotatedDir(transformationA.getRotation(), axisA);
            axisB.setRotatedDir(transformationB.getRotation(), axisB);
            axisPerpA.setRotatedDir(transformationA.getRotation(), axisPerpA);
            axisPerpB.setRotatedDir(transformationB.getRotation(), axisPerpB);
            prisData->setInBodySpace(pivotA,pivotB,axisA,axisB,axisPerpA,axisPerpB);
            break;
        }
    case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
        {
            hkpRagdollConstraintData* ragDollData = static_cast<hkpRagdollConstraintData*>(data);

            hkTransform aTransform = ragDollData->m_atoms.m_transforms.m_transformA;
            hkTransform bTransform = ragDollData->m_atoms.m_transforms.m_transformB;

            hkVector4 pivotA = aTransform.getTranslation();
            hkVector4 twistAxisA = aTransform.getColumn(ragDollData->m_atoms.AXIS_TWIST);
            hkVector4 planeAxisA = aTransform.getColumn(ragDollData->m_atoms.AXIS_PLANES);

            hkVector4 pivotB = bTransform.getTranslation();
            hkVector4 twistAxisB = bTransform.getColumn(ragDollData->m_atoms.AXIS_TWIST);
            hkVector4 planeAxisB = bTransform.getColumn(ragDollData->m_atoms.AXIS_PLANES);

            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);

            planeAxisA.setRotatedDir(transformationA.getRotation(),planeAxisA);
            planeAxisB.setRotatedDir(transformationB.getRotation(),planeAxisB);
            twistAxisA.setRotatedDir(transformationA.getRotation(),twistAxisA);
            twistAxisB.setRotatedDir(transformationB.getRotation(),twistAxisB);
            ragDollData->setInBodySpace(pivotA, pivotB, planeAxisA, planeAxisB, twistAxisA, twistAxisB);
            break;
        }
    case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
        {
            hkp6DofConstraintData* ragDollData = static_cast<hkp6DofConstraintData*>(data);

            hkTransform aTransform = ragDollData->m_blueprints.m_transforms.m_transformA;
            hkTransform bTransform = ragDollData->m_blueprints.m_transforms.m_transformB;

            hkVector4 pivotA =   aTransform.getTranslation();
            hkVector4 twistAxisA = aTransform.getColumn(ragDollData->AXIS_TWIST);
            hkVector4 planeAxisA = aTransform.getColumn(ragDollData->AXIS_SWING0);

            hkVector4 pivotB =   bTransform.getTranslation();
            hkVector4 twistAxisB = bTransform.getColumn(ragDollData->AXIS_TWIST);
            hkVector4 planeAxisB = bTransform.getColumn(ragDollData->AXIS_SWING0);

            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);

            planeAxisA.setRotatedDir(transformationA.getRotation(),planeAxisA);
            planeAxisB.setRotatedDir(transformationB.getRotation(),planeAxisB);
            twistAxisA.setRotatedDir(transformationA.getRotation(),twistAxisA);
            twistAxisB.setRotatedDir(transformationB.getRotation(),twistAxisB);
            ragDollData->setInBodySpace(pivotA, pivotB, twistAxisA, twistAxisB, planeAxisA, planeAxisB );
            ragDollData->updateDirtyAtoms();
            break;
        }
    case hkpConstraintData::CONSTRAINT_TYPE_STIFFSPRING:
        {
            hkpStiffSpringConstraintData* springData = static_cast<hkpStiffSpringConstraintData*>(data);

            hkVector4 pivotA = springData->m_atoms.m_pivots.m_translationA;
            hkVector4 pivotB = springData->m_atoms.m_pivots.m_translationB;
            hkReal minLength = springData->m_atoms.m_spring.m_length;
            hkReal maxLength = springData->m_atoms.m_spring.m_maxLength;
            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);
            springData->setInBodySpace(pivotA,pivotB,minLength, maxLength);
            break;
        }
    case hkpConstraintData::CONSTRAINT_TYPE_WHEEL:
        {
            hkpWheelConstraintData* wheelData = static_cast<hkpWheelConstraintData*>(data);

            hkTransform aTransform = wheelData->m_atoms.m_suspensionBase.m_transformA;
            hkTransform bTransform = wheelData->m_atoms.m_suspensionBase.m_transformB;
            hkVector4 pivotA = aTransform.getTranslation();
            hkVector4 pivotB = bTransform.getTranslation();
            hkVector4 suspensionAxisB = wheelData->m_atoms.m_suspensionBase.m_transformB.getColumn<0>();
            hkVector4 axleA = wheelData->m_atoms.m_steeringBase.m_rotationA.getColumn<0>();
            hkVector4 axleB = wheelData->m_atoms.m_steeringBase.m_rotationB.getColumn<0>();
            pivotA.setTransformedPos(transformationA,pivotA);
            pivotB.setTransformedPos(transformationB,pivotB);
            hkVector4 steeringAxisB = wheelData->m_atoms.m_steeringBase.m_rotationB.getColumn<1>();
            suspensionAxisB.setRotatedDir(transformationB.getRotation(), suspensionAxisB);
            axleA.setRotatedDir(transformationA.getRotation(), axleA);
            axleB.setRotatedDir(transformationB.getRotation(), axleB);
            steeringAxisB.setRotatedDir(transformationB.getRotation(), steeringAxisB);
            wheelData->setInBodySpace(pivotA, pivotB, axleA, axleB, suspensionAxisB, steeringAxisB);
            break;
        }
    default:
        {
            HK_WARN(0x0606a378, "Transforming this constraint is not supported");
            break;
        }
    }
}



//////////////////////////////////////////////////////////////////////////
// These functions are only necessary due to HVK-6555, could remove them when that is solved.
//////////////////////////////////////////////////////////////////////////

#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>
#include <Physics/Constraint/Motor/SpringDamper/hkpSpringDamperConstraintMotor.h>
#include <Physics/Constraint/Motor/Velocity/hkpVelocityConstraintMotor.h>

bool HK_CALL hkpConstraintDataUtils::isMotorActive( _In_opt_ const hkpConstraintMotor* motor )
{
    if ( motor != HK_NULL )
    {
        switch ( motor->getType() )
        {
            case hkpConstraintMotor::TYPE_POSITION:
            {
                const hkpPositionConstraintMotor* positionMotor = static_cast< const hkpPositionConstraintMotor* >( motor );
                if ( ( positionMotor->m_tau == 0.0f ) || ( positionMotor->m_maxForce == 0.0f ) )
                {
                    return false;
                }
                break;
            }
            case hkpConstraintMotor::TYPE_VELOCITY:
            {
                const hkpVelocityConstraintMotor* velocityMotor = static_cast< const hkpVelocityConstraintMotor* >( motor );
                if ( ( velocityMotor->m_tau == 0.0f ) || ( velocityMotor->m_maxForce == 0.0f ) )
                {
                    return false;
                }
                break;
            }
            case hkpConstraintMotor::TYPE_SPRING_DAMPER:
            {
                const hkpSpringDamperConstraintMotor* springMotor = static_cast< const hkpSpringDamperConstraintMotor* >( motor );
                if ( ( springMotor->m_springConstant == 0.0f ) || ( springMotor->m_maxForce == 0.0f ) )
                {
                    return false;
                }
                break;
            }
            default:
            {
                HK_WARN_ONCE( 0x22440365, "Unhandled motor type" );
                return false;
            }
        }

        // This motor is active
        return true;
    }

    // Null motor is inactive
    return false;
}

bool HK_CALL hkpConstraintDataUtils::hasActiveMotor( _In_opt_ const hkpConstraintData* constraintData )
{
    if ( constraintData )
    {
        switch ( constraintData->getType() )
        {
            case hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE:
            {
                const hkpLimitedHingeConstraintData* lhc = static_cast<const hkpLimitedHingeConstraintData*> ( constraintData );
                return lhc->isMotorEnabled() && isMotorActive( lhc->getMotor() );
            }

            case hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL:
            {
                const hkpRagdollConstraintData* rc = static_cast<const hkpRagdollConstraintData*> ( constraintData );
                if ( rc->areMotorsEnabled() )
                {
                    for ( int i = 0; i < hkpRagdollConstraintData::NUM_MOTOR_INDICES; i++ )
                    {
                        if ( isMotorActive( rc->getMotor( ( hkpRagdollConstraintData::MotorIndex ) i ) ) )
                        {
                            return true;
                        }
                    }
                }
                break;
            }

            case hkpConstraintData::CONSTRAINT_TYPE_6DOF:
            {
                const hkp6DofConstraintData* rc = static_cast<const hkp6DofConstraintData*> ( constraintData );
                if ( rc->areAngularMotorsEnabled() )
                {
                    for ( int i = 0; i < 3; i++ )
                    {
                        if ( isMotorActive( rc->getAngularMotor( ( hkp6DofConstraintData::MotorIndex) i ) ) )
                        {
                            return true;
                        }
                    }
                }
                for ( int i = 0; i < 3; i++ )
                {
                    if ( rc->isLinearMotorEnabled(i) && isMotorActive( rc->getLinearMotor(i) ) )
                    {
                        return true;
                    }
                }
                break;
            }
        }
    }
    return false;
}

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
