// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics/Constraint/Data/hkpConstraintData.h>


/// A stiff spring constraint. It holds the constrained bodies apart at a specified distance,
/// as if they were attached at each end of an invisible rod.
class HK_EXPORT_PHYSICS hkpStiffSpringConstraintData : public hkpConstraintData
{
    public:

        enum
        {
            SOLVER_RESULT_LIN_0 = 0,    // linear constraint
            SOLVER_RESULT_MAX   = 1
        };

        struct Runtime
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkpStiffSpringConstraintData::Runtime );

            hkpSolverResults m_solverResults[SOLVER_RESULT_MAX];
        };

        struct Atoms
        {
            //+version(1)
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkpStiffSpringConstraintData::Atoms );
            HK_DECLARE_REFLECTION();

            Atoms() {}

            // get a pointer to the first atom
            const hkpConstraintAtom* getAtoms() const { return &m_pivots; }

            // get the size of all atoms (we can't use sizeof(*this) because of align16 padding)
            int getSizeOfAllAtoms() const { return hkGetByteOffsetInt(this, &m_spring+1); }

            hkpSetLocalTranslationsConstraintAtom   m_pivots;
            hkpSetupStabilizationAtom               m_setupStabilization;
            hkpStiffSpringConstraintAtom            m_spring;
        };

    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
        HK_DECLARE_REFLECTION();

        /// Constructor.
        hkpStiffSpringConstraintData();

        /// Sets the spring up with world space information.
        /// Will compute a rest length too (so call setlength after this if needed)
        /// \param pivotA           bodyA's pivot point, specified in world space.
        /// \param pivotB           bodyB's pivot point, specified in world space.
        inline void setInWorldSpace(const hkTransform& bodyATransform, const hkTransform& bodyBTransform,
                                const hkVector4& pivotAW, const hkVector4& pivotBW);

        /// Sets the spring up with body space information.
        /// \param pivotA           bodyA's pivot point, specified in bodyA's space.
        /// \param pivotB           bodyB's pivot point, specified in bodyB's space.
        /// \param restLength       The length of the stiff spring when at rest
        inline void setInBodySpace(const hkVector4& pivotA, const hkVector4& pivotB, hkReal restLength);
        inline void setInBodySpace(const hkVector4& pivotA, const hkVector4& pivotB, hkReal minLength, hkReal maxLength);

        /// Gets the length of the stiff spring.
        inline hkReal getSpringMinLength() const;
        inline hkReal getSpringMaxLength() const;

        /// Sets the length of the stiff spring when at rest.
        inline void setSpringLength( hkReal minLength, hkReal maxLength );

        /// Sets the length of the stiff spring when at rest.
        inline void setSpringLength( hkReal length );

        inline const Runtime* getRuntime( _In_opt_ hkpConstraintRuntime* runtime ) const { return reinterpret_cast<Runtime*>(runtime); }

        //
        // hkpConstraintData implementation
        //

        virtual hkBool isValid() const HK_OVERRIDE;

        virtual int getType() const HK_OVERRIDE;

        virtual void getConstraintInfo( ConstraintInfo& infoOut ) const HK_OVERRIDE;

        virtual void getRuntimeInfo( hkBool wantRuntime, hkpConstraintData::RuntimeInfo& infoOut ) const HK_OVERRIDE;

        virtual void getAppliedLinearImpulse(const hkTransform& worldFromBodyA, const hkTransform& worldFromBodyB,
            _In_ const struct hkpConstraintRuntime* runtime, _Out_ hkVector4& impulseOut) const HK_OVERRIDE;

    public:

        Atoms m_atoms;
};

#include <Physics/Constraint/Data/StiffSpring/hkpStiffSpringConstraintData.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
