// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics/Constraint/Data/hkpConstraintData.h>


/// Full hinge constraint with limits and motor.
///
/// By default the motor is disabled. When the motor is enabled friction is automatically disabled.
class HK_EXPORT_PHYSICS hkpLimitedHingeConstraintData : public hkpConstraintData
{
    public:

        enum
        {
            SOLVER_RESULT_MOTOR             = 0,    // the motor
            SOLVER_RESULT_MOTOR_INTERNAL    = 1,
            SOLVER_RESULT_FRICTION          = 2,    // or friction
            SOLVER_RESULT_FRICTION_INTERNAL = 3,
            SOLVER_RESULT_LIMIT             = 4,    // limits defined around m_freeAxisA
            SOLVER_RESULT_ANG_0             = 5,    // angular constraint 0
            SOLVER_RESULT_ANG_1             = 6,    // angular constraint 1

            SOLVER_RESULT_LIN_0             = 7,    // linear constraint
            SOLVER_RESULT_LIN_1             = 8,    // linear constraint
            SOLVER_RESULT_LIN_2             = 9,    // linear constraint
            SOLVER_RESULT_MAX               = 10
        };

        struct Runtime
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkpLimitedHingeConstraintData::Runtime );

            /// Returns current angle position
            inline hkReal getCurrentPos() const;

            HK_ALIGN16( hkpSolverResults m_solverResults[SOLVER_RESULT_MAX] );
            hkUint8 m_initialized;          ///< Whether the previous target angle has been initialized.
            hkReal m_previousTargetAngle;   ///< The previous target angle
        };

        struct Atoms
        {
            HK_DECLARE_CLASS(Atoms, Reflect, New, Version(1), Validate);

            enum Axis
            {
                AXIS_AXLE = 0,
                AXIS_PERP_TO_AXLE_1 = 1,
                AXIS_PERP_TO_AXLE_2 = 2
            };

            Atoms() {}

            void afterReflectNew();

            // get a pointer to the first atom
            const hkpConstraintAtom* getAtoms() const { return &m_transforms; }

            // get the size of all atoms (we can't use sizeof(*this) because of align16 padding)
            int getSizeOfAllAtoms() const { return hkGetByteOffsetInt(this, &m_ballSocket + 1); }

            hkpSetLocalTransformsConstraintAtom m_transforms;
            hkpSetupStabilizationAtom           m_setupStabilization;
            hkpAngMotorConstraintAtom           m_angMotor;
            hkpAngFrictionConstraintAtom        m_angFriction;
            hkpAngLimitConstraintAtom           m_angLimit;
            hkp2dAngConstraintAtom              m_2dAng;
            hkpBallSocketConstraintAtom         m_ballSocket;
        };

    public:

        HK_DECLARE_CLASS(hkpLimitedHingeConstraintData, Reflect, New, Validate);

        /// Constructor.
        hkpLimitedHingeConstraintData();

        /// Destructor.
        ~hkpLimitedHingeConstraintData();

        /// Set the data for a Limited Hinge constraint with information given in body space.
        /// \param pivotA The constraint pivot point, specified in bodyA space.
        /// \param pivotB The constraint pivot point, specified in bodyB space.
        /// \param axisA The hinge axis, specified in bodyA space.
        /// \param axisB The hinge axis, specified in bodyB space.
        /// \param axisAPerp Axis perpendicular to the hinge axis, specified in bodyA space.
        /// \param axisBPerp Axis perpendicular to the hinge axis, specified in bodyB space.
        void setInBodySpace(const hkVector4& pivotA, const hkVector4& pivotB,
                            const hkVector4& axisA, const hkVector4& axisB,
                            const hkVector4& axisAPerp, const hkVector4& axisBPerp);

        /// Set the data for a Limited Hinge constraint with information given in world space.
        /// \param bodyA The first rigid body transform.
        /// \param bodyB The second rigid body transform.
        /// \param pivot The pivot point, specified in world space.
        /// \param axis The hinge axis, specified in world space.
        void setInWorldSpace(   const hkTransform& bodyATransform, const hkTransform& bodyBTransform,
                                const hkVector4& pivot, const hkVector4& axis);

        static inline Runtime* HK_CALL getRuntime( _In_opt_ hkpConstraintRuntime* runtime ) { return reinterpret_cast<Runtime*>(runtime); }

        //
        // Methods to set and get angle limits and friction
        //

        /// Set the friction value. Set this before adding to the system.
        /// Note that this value is an absolute torque value and is therefore dependent on the masses of constrained bodies
        /// and not limited between 0.0f and 1.0f.
        /// If trying to stiffen up hinge constraints, try setting this value sufficiently high so that constraints are completely stiff and then
        /// reduce until the desired behavior has been achieved.
        inline void setMaxFrictionTorque(hkReal tmag);

        /// Get the friction value.
        inline hkReal getMaxFrictionTorque() const;

        /// Set the maximum angular limit.
        inline void setMaxAngularLimit(hkReal rad);

        /// Set the minimum angular limit.
        inline void setMinAngularLimit(hkReal rad);

        /// Get the maximum angular limit.
        inline hkReal getMaxAngularLimit() const;

        /// Get the minimum angular limit.
        inline hkReal getMinAngularLimit() const;

        /// Set the m_angularLimitsTauFactor. This is a factor in the range between 0 and 1
        /// which controls the stiffness of the angular limits. If you slowly increase
        /// this value from 0 to 1 for a newly created constraint,
        /// you can nicely blend in the limits.
        inline void setAngularLimitsTauFactor( hkReal mag );

        /// Get the m_angularLimitsTauFactor;
        inline hkReal getAngularLimitsTauFactor() const;

        /// This is the preferable way to disable angular limits when motor is enabled.
        inline void disableLimits();

        //
        // Motor related methods
        //

        /// Set the motor. Setting this to null will disable any motor computations.
        /// The angle passed to the hkpConstraintMotor::motor() callback will be the relative angle
        /// between the attached and reference body. You need to set the desired target angle of
        /// your motor each step.
        /// Increments reference count of new motor, decrements reference count of replaced motor (if any).
        void setMotor( _In_opt_ hkpConstraintMotor* motor );

        /// Get the motor. Default is HK_NULL.
        inline hkpConstraintMotor* getMotor() const;

        /// Turn the motor on or off.
        void setMotorEnabled( _In_opt_ hkpConstraintRuntime* runtime, hkBool isEnabled );

        /// Is the motor enabled?
        inline hkBool isMotorEnabled() const;

        /// Set the target angle for the motor. Only used by motors which use positions
        inline void setMotorTargetAngle( hkReal angle );

        /// Get the target angle for the motor
        inline hkReal getMotorTargetAngle() const;

        //
        // hkpConstraintData implementation
        //

        /// Get type from this constraint
        virtual int getType() const HK_OVERRIDE;

        /// Check consistency of constraint members
        virtual hkBool isValid() const HK_OVERRIDE;

        virtual void getConstraintInfo( ConstraintInfo& infoOut ) const HK_OVERRIDE;

        /// Set the impulse above which this constraint is reported as breached.
        /// Set it to HK_REAL_MAX to disable breach reporting.
        virtual void setBreachImpulse(hkReal breachImpulse) HK_OVERRIDE;

        /// Get the impulse above which this constraint is reported as breached.
        virtual hkReal getBreachImpulse() const HK_OVERRIDE;

        /// Choose the body to be notified when the constraint's maximum impulse is breached.
        virtual void setBodyToNotify(int bodyIdx) HK_OVERRIDE;

        /// Returns the index of the body that is notified when the constraint's maximum impulse limit is breached.
        virtual hkUint8 getNotifiedBodyIndex() const HK_OVERRIDE;

        /// Set the solving method for this constraint. Use one of the hkpConstraintAtom::SolvingMethod as a value for method.
        virtual void setSolvingMethod(hkpConstraintAtom::SolvingMethod method) HK_OVERRIDE;

        /// Get the inertia stabilization factor, returns HK_FAILURE if the factor is not defined for the given constraint.
        virtual hkResult getInertiaStabilizationFactor( _Out_ hkReal& inertiaStabilizationFactorOut) const HK_OVERRIDE;

        /// Set the inertia stabilization factor, return HK_FAILURE if the factor is not defined for the given constraint.
        virtual hkResult setInertiaStabilizationFactor(const hkReal inertiaStabilizationFactorIn) HK_OVERRIDE;

        virtual void getRuntimeInfo( hkBool wantRuntime, hkpConstraintData::RuntimeInfo& infoOut ) const HK_OVERRIDE;

        /// Returns the linear impulse applied by the solver
        virtual void getAppliedLinearImpulse(const hkTransform& worldFromBodyA, const hkTransform& worldFromBodyB,
            _In_ const struct hkpConstraintRuntime* runtime, _Out_ hkVector4& impulseOut) const HK_OVERRIDE;

    public:

        Atoms m_atoms;
};

#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
