// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics/Constraint/Data/hkpConstraintData.h>


/// This constraint represents interaction between two cogwheels.
///
/// This is a simple one-dimensional constraint only. In order to create a meaningful mechanism, the elements linked
/// must be connected to other external constraints with additional hinge constraints. hkpConstraintUtils has helper
/// functions that will allow you to easily create those.
///
/// Note that the axes of the cog wheels don't have to be fixed at one relative position. The wheel axes are free
/// to move in a (physically sensible) range. And the constraints should properly track the contact point of the wheels.
/// It will also work, when the cogs are physically disjoint at times.
///
/// If you experience stability problems in bigger setups, the external prismatic and hinge constraints should be given
/// higher priority, or the rack-and-pinion constraint added first.
///
class HK_EXPORT_PHYSICS hkpCogWheelConstraintData : public hkpConstraintData
{
    public:

        enum
        {
            SOLVER_RESULT_COG_WHEEL = 0,
            SOLVER_RESULT_MAX       = 1
        };

        struct Runtime
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkpCogWheelConstraintData::Runtime );

            hkpSolverResults m_solverResults[SOLVER_RESULT_MAX];

            // This stores the initial relative angular orientation of the bodies. 0 if uninitialized (triggers
            // initialization of m_revolutionCounter too). Non-zero when initialized (set to a tiny number
            // (< HK_REAL_EPSILON) for zero values).
            HK_ALIGN(hkReal m_initialAngleOffset[2],HK_REAL_SIZE);

            // Previously calculated relative angle.
            HK_ALIGN(hkReal m_prevAngle[2],HK_REAL_SIZE);

            // This counts revolutions of the cog wheel to properly calculate the Rhs. Auto-initialized to 0.
            hkInt32 m_revolutionCounter[2];
        };

        struct Atoms
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkpCogWheelConstraintData::Atoms );
            HK_DECLARE_REFLECTION();

            Atoms() {}

            // get a pointer to the first atom
            const hkpConstraintAtom* getAtoms() const { return &m_transforms; }

            // get the size of all atoms (we can't use sizeof(*this) because of align16 padding)
            int getSizeOfAllAtoms() const { return hkGetByteOffsetInt(this, &m_cogWheels+1); }

            hkpSetLocalTransformsConstraintAtom m_transforms;
            hkpCogWheelConstraintAtom           m_cogWheels;
        };

    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
        HK_DECLARE_REFLECTION();

        /// Constructor
        hkpCogWheelConstraintData();

        /// Sets the cog wheels up with world space information.
        void setInWorldSpace(   const hkTransform& bodyATransform, const hkTransform& bodyBTransform,
                                const hkVector4& rotationPivotA, const hkVector4& rotationAxisA, hkReal radiusA,
                                const hkVector4& rotationPivotB, const hkVector4& rotationAxisB, hkReal radiusB);

        /// Sets the cog wheels up with body space information.
        /// The parameters are analogical to setInWorldSpace().
        void setInBodySpace( const hkVector4& rotationPivotAInA, const hkVector4& rotationAxisAInA, hkReal radiusA,
                             const hkVector4& rotationPivotBInB, const hkVector4& rotationAxisBInB, hkReal radiusB);

        inline Runtime* getRuntime( _In_opt_ hkpConstraintRuntime* runtime ){ return reinterpret_cast<Runtime*>(runtime); }

        //
        // hkpConstraintData implementation
        //

        virtual int getType() const HK_OVERRIDE;

        virtual hkBool isValid() const HK_OVERRIDE;

        virtual void getConstraintInfo( hkpConstraintData::ConstraintInfo& infoOut ) const HK_OVERRIDE;

        virtual void getRuntimeInfo( hkBool wantRuntime, hkpConstraintData::RuntimeInfo& infoOut ) const HK_OVERRIDE;

        /// Returns the linear impulse applied by the solver
        virtual void getAppliedLinearImpulse(const hkTransform& worldFromBodyA, const hkTransform& worldFromBodyB,
            _In_ const struct hkpConstraintRuntime* runtime, _Out_ hkVector4& impulseOut) const HK_OVERRIDE;

    public:

        Atoms m_atoms;
};

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
