// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS PHYSICS_2012
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Physics/Constraint/Data/hkpConstraintData.h>

/// A six degree of freedom (6DOF) constraint.
/// Each degree of freedom can be independently restricted and/or motorized.
///
/// Implementation notes:
///   - Because of the huge amount of possible combinations, this constraint compiles all user settings into a list of
///     atoms. As a result the size of the atoms can change dynamically, which mean that we can not use this constraint
///     with Physics2012.
///   - 6DOF constraint allows you to activate motors to drive the 3 angles or directions. If you are using a motor
///     with a target position to set, set it to 0. If using an hkpCallbackConstraintMotor, the difference between the
///     current angle the desired angle will be passed in to the callback. This will result in your motor driving the
///     difference to zero.
///   - If the motor is enabled, all friction will be disabled, however once you disable the motor, friction will be
///     enabled again.
class HK_EXPORT_PHYSICS hkp6DofConstraintData : public hkpConstraintData
{
    public:

        struct Result
        {
            enum Enum
            {
                LIN_0,  // linear constraint defined by (1,0,0) in object space A
                LIN_1,  // linear constraint defined by (0,1,0) in object space A
                LIN_2,  // linear constraint defined by (0,0,1) in object space A
                LIN_LIMIT, // linear limit

                ANG_MOTOR_0             ,   // the angular motor or friction value
                ANG_MOTOR_0_INTERNAL    ,
                ANG_MOTOR_1             ,   // the angular motor value
                ANG_MOTOR_1_INTERNAL    ,
                ANG_MOTOR_2             ,   // the angular motor value
                ANG_MOTOR_2_INTERNAL    ,

                LIN_MOTOR_0             ,   // the linear motor value or friction
                LIN_MOTOR_0_INTERNAL    ,
                LIN_MOTOR_1             ,   // the linear motor value
                LIN_MOTOR_1_INTERNAL    ,
                LIN_MOTOR_2             ,   // the linear motor value
                LIN_MOTOR_2_INTERNAL    ,

                TWIST,      // angular limits defined around twist axis
                SWING0,     // angular constraint defined by swing0 axis
                SWING1,     // angular constraint defined by swing1 axis

                MAX
            };
        };

        struct Runtime
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DYNAMICS, hkp6DofConstraintData::Runtime );

            /// Function to get the solver results for a specific DOF.
            /// Note since this constraint data builds a dynamic list of atoms, you have to use this function to extract
            /// a specific value.
            HK_INLINE hkpSolverResults& get( Result::Enum type, _In_ hkp6DofConstraintData* data );

            /// const version
            HK_INLINE const hkpSolverResults& get( Result::Enum type, _In_ const hkp6DofConstraintData* data ) const;

            HK_ALIGN16( hkpSolverResults m_solverResults[Result::MAX] );

            /// Tells whether the previous target angles have been initialized.
            /// This is zeroed when created.
            hkUint8 m_ragdollMotorsInitialized[3];
            hkUint8 m_linearMotorsInitialized[3];

            /// Target angles for the individual motors
            HK_ALIGN(hkReal m_previousTargetAngles[3],HK_REAL_SIZE);
            hkReal m_linearMotorPreviousTargetPosition[3];
        };

        enum MotorIndex
        {
            MOTOR_TWIST = 0,    ///< Twist motor
            MOTOR_SWING0 = 1,   ///< Swing0 motor
            MOTOR_SWING1  = 2,  ///< Swing1 motor
            MOTOR_ALL     = 3   ///< Don't destiguish between different motors for different axis, just apply the motor on all axis
        };

        enum Axis
        {
            AXIS_TWIST = 0,
            AXIS_SWING0 = 1,
            AXIS_SWING1 = 2
        };

    public:

        // -------------------------------------------------------------------------------------------------------------
        // Construction
        // -------------------------------------------------------------------------------------------------------------

        HK_DECLARE_CLASS(hkp6DofConstraintData, Reflect, New, Validate);

        /// Constructor.
        hkp6DofConstraintData();

        /// Serialization constructor.
        void afterReflectNew();

        /// Destructor. Will dereference any motors held.
        ~hkp6DofConstraintData();

        /// Setup a 6DOF constraint based on information in world space.
        /// Use the set angle limit methods to add in the extra data.
        /// Use the set motors and set angle limit methods to add in the extra data.
        /// Will set the target frame B to match the current setup frame for B
        /// \param bodyA            The attached body transform
        /// \param bodyB            The reference body transform
        /// \param pivot            The pivot point, specified in world space.
        /// \param twistAxisW       The twist axis, specified in world space.
        /// \param swingAxis0W      The plane axis, specified in world space.
        void setInWorldSpace(
            const hkTransform& bodyATransform, const hkTransform& bodyBTransform,
            const hkVector4& pivot, const hkVector4& twistAxisW,
            const hkVector4& swingAxis0W );

        /// Setup a 6DOF constraint based on information in body space.
        /// Use the set angle limit methods to add in the extra data.
        /// Use the set motors and set angle limit methods to add in the extra data.
        /// Will set the target frame B to match the current setup frame for B
        /// \param pivotA           The pivot point, specified in bodyA space.
        /// \param pivotB           The pivot point, specified in bodyB space.
        /// \param twistAxisA       The twist axis, specified in bodyA space.
        /// \param twistAxisB       The twist axis, specified in bodyB space.
        /// \param swingAxis0A      The plane axis, specified in bodyA space.
        /// \param swingAxis0B      The plane axis, specified in bodyB space.
        void setInBodySpace( const hkVector4& pivotA, const hkVector4& pivotB,
            const hkVector4& twistAxisA,  const hkVector4& twistAxisB,
            const hkVector4& swingAxis0A, const hkVector4& swingAxis0B  );

        /// Get the constraint frame.
        /// \param constraintFrameA Column 0 = twist axis, Column 1 = plane, Column 2 = twist cross plane.
        void getConstraintFrameA( _Out_ hkMatrix3& constraintFrameA ) const;

        /// Get the constraint frame.
        /// \param constraintFrameB Column 0 = twist axis, Column 1 = plane, Column 2 = twist cross plane.
        void getConstraintFrameB( _Out_  hkMatrix3& constraintFrameB ) const;

        // -------------------------------------------------------------------------------------------------------------
        // Limits for humans
        // -------------------------------------------------------------------------------------------------------------

        enum AxisIdentifier
        {
            LINEAR_X, ///< Translation along the twist axis
            LINEAR_Y, ///< Translation along the first swing axis
            LINEAR_Z, ///< Translation along the second swing axis
            ANGULAR_X, ///< Rotation around the twist axis
            ANGULAR_Y, ///< Rotation around the first swing axis
            ANGULAR_Z, ///< Rotation around the second swing axis

            MAX_AXES
        };

        enum AxisMode
        {
            UNLIMITED, ///< No restriction is placed on the axis
            LIMITED, ///< Movement is permitted between the min/max limits 
            FIXED ///< No movement is permitted on the axis
        };

        /// Set axis \a axis to behave as \a mode. See AxisMode for details
        void setAxisMode(AxisIdentifier axis, AxisMode mode);

        /// Return the axis mode for the \a axis.
        AxisMode getAxisMode(AxisIdentifier axis) const; 

        /// Set the minimum limit on axis \a axis to \a limit. Only has an effect for LIMITED mode.
        void setAxisMinLimit(AxisIdentifier axis, hkReal limit);

        /// Return the minimum limit for axis \a axis.
        hkReal getAxisMinLimit(AxisIdentifier axis) const;

        /// Set the maximum limit on axis \a axis to \a limit. Only has an effect for LIMITED mode.
        void setAxisMaxLimit(AxisIdentifier axis, hkReal limit);

        /// Return the maximum limit for axis \a axis.
        hkReal getAxisMaxLimit(AxisIdentifier axis) const;

        /// Set the friction for axis \a axis to \a friction. Ignored if \a axis has a motor.
        void setAxisFriction(AxisIdentifier axis, hkReal friction);

        /// Get the friction for axis \a axis.
        hkReal getAxisFriction(AxisIdentifier axis) const;

        /// Set the motor for a particular \a axis. To disable the motor, specify HK_NULL. Targets
        /// are not currently specified for each axis independently. Use
        /// setLinearMotorTargetPosition() and setAngularTarget()/setTargetRelativeOrientationOfBodies()
        /// to specify the target.
        void setAxisMotor(AxisIdentifier axis, hkpConstraintMotor* motor);

        /// Read-only access to the constraint motor for the axis \a axis.
        const hkpConstraintMotor* getAxisMotor(AxisIdentifier axis) const;

        /// Read-write access to the constraint motor for the axis \a axis.
        hkpConstraintMotor* getAxisMotor(AxisIdentifier axis);
        

        // -------------------------------------------------------------------------------------------------------------
        // Angular limits - these interfaces are deprecated and likely to be removed
        // -------------------------------------------------------------------------------------------------------------

        /// Get the minimum twist limit (in radians).
        inline hkReal getTwistMinAngularLimit() const;

        /// Get the maximum twist limit (in radians).
        inline hkReal getTwistMaxAngularLimit() const;

        /// Set the minimum twist limit (in radians).
        void setTwistMinAngularLimit(hkReal rad);

        /// Set the maximum twist limit (in radians).
        void setTwistMaxAngularLimit(hkReal rad);


        /// Get the angle limit around the swing0 axis (in radians).
        inline hkReal getSwing0Limit() const;

        /// Set the angle limit around the swing0 axis (in radians).
        void setSwing0Limit(hkReal rad);

        void setSwing0Min(hkReal min);
        void setSwing0Max(hkReal max);

        /// Set the angle limit around the swing1 axis (in radians).
        void setSwing1Limit(hkReal rad);

        void setSwing1Min(hkReal min);
        void setSwing1Max(hkReal max);

        /// Get the angle limit around the swing1 axis (in radians).
        inline hkReal getSwing1Limit() const;

        /// Set the m_angularLimitsTauFactor. This is a factor in the range
        /// between 0 and 1 which controls the stiffness of the angular limits.
        /// If you slowly increase this value from 0 to 1 for a newly created
        /// constraint, you can nicely blend in the limits.
        inline void setAngularLimitsTauFactor( hkReal mag );

        /// Get the angular limits tau factor for this constraint.
        inline hkReal getAngularLimitsTauFactor() const;


        /// Get the cone angle limit (in radians).
        inline hkReal getConeAngularLimit() const;

        /// Get the state of cone limit stabilization.
        inline hkBool getConeLimitStabilization() const;

        // -------------------------------------------------------------------------------------------------------------
        // Angular motors
        // -------------------------------------------------------------------------------------------------------------

        /// Returns the motor at a specific index.
        hkpConstraintMotor* getAngularMotor( MotorIndex index ) const;

        /// Set the motor specified by the index.
        /// Increments reference of new motor, decrements counter of
        /// replaced motor (if any, can be null).
        /// Also enables/reenables this motor.
        void setAngularMotor( MotorIndex index, _In_opt_ hkpConstraintMotor* newMotor);

        /// Turn the motors on or off
        /// \param runtime      The constraint instance's runtime data.
        /// \param areEnabled   The desired state of the motors.
        void setAngularMotorsEnabled( bool enable );

        /// Check is the motors are active.
        inline bool areAngularMotorsEnabled() const;

        /// Set target relative orientation between the constraints reference frames attached to
        /// each of the bodies.
        /// See also: setTargetRelativeOrientationOfBodies()
        void setAngularTarget( const hkMatrix3& target_cbRca );

        /// Set target relative orientation between bodies.
        /// This bypasses and ignores actual orientation of the constraint space in relation to each of the bodies.
        /// See also: setAngularTarget()
        void setTargetRelativeOrientationOfBodies( const hkRotation& bRa );

        /// Get the target frame
        void getAngularTarget( hkMatrix3& target_cbRca );


        // -------------------------------------------------------------------------------------------------------------
        // Linear limits - these interfaces are deprecated and likely to be removed
        // -------------------------------------------------------------------------------------------------------------

        /// Get the linear limit, see setLinearLimit() for details.
        HK_INLINE hkReal getLinearLimit() const;

        /// Set the linear limit.
        /// If more than one degree of freedom is enabled, this result in a circle or spherical motion.
        void setLinearLimit( hkReal distance );

        /// If at least one of the linear axes is free, this sets the stiffness of the limit
        void setLinearLimitStrength( hkReal springConstrant, hkReal springDamping );

        /// Set linear limit for a specific axis. Setting distance to HK_REAL_MAX will unlimit this axis.
        void setLinearAxisMinLimit( int axis, hkReal distance );
        void setLinearAxisMaxLimit( int axis, hkReal distance );

        // Set linear axis friction for a specific axis. Setting to 0 will disable friction.
        void setLinearAxisFriction( int axis, hkReal friction );

        /// Get the friction value.
        inline hkReal getMaxFrictionTorque() const;

        // -------------------------------------------------------------------------------------------------------------
        // Linear motors
        // -------------------------------------------------------------------------------------------------------------

        /// Set the friction value. Set this before adding to the system.
        /// Note that this value is an absolute torque value and is therefore
        /// dependent on the masses of constrained bodies and not limited
        /// between 0.0f and 1.0f. If trying to stiffen up 6DOF constraints,
        /// try setting this value sufficiently high so that constraints are
        /// completely stiff and then reduce until the desired behavior has
        /// been achieved.
        void setMaxFrictionTorque(hkReal tmag);

        /// Set the motor. Setting this to null will disable any motor computations.
        /// Increments reference of new motor, decrements counter of replaced motor ( if any ).
        /// Also (re)enables this motor.
        void setLinearMotor( int axis, _In_opt_ hkpConstraintMotor* motor );

        /// Get the motor.
        HK_INLINE hkpConstraintMotor* getLinearMotor(int axis) const;

        /// Turn the motor on or off.
        void setLinearMotorEnabled( int axis, bool isEnabled );

        /// Is the motor enabled?
        HK_INLINE bool isLinearMotorEnabled(int axis) const;

        /// Set the target position for the motor. Only used by motors which use positions.
        void setLinearMotorTargetPosition( hkVector4Parameter position );

        /// Get the target angle for the motor.
        void getLinearMotorTargetPosition( _Out_ hkVector4& positionOut );

        //
        // hkpConstraintData implementation
        //

        /// Get type from this constraint
        virtual int getType() const HK_OVERRIDE;

        /// Check consistency of constraint
        virtual hkBool isValid() const HK_OVERRIDE;

        virtual void getConstraintInfo( hkpConstraintData::ConstraintInfo& infoOut ) const HK_OVERRIDE;

        /// Sets the impulse above which this constraint is reported as breached.
        /// Set it to HK_REAL_MAX to disable breach reporting.
        virtual void setBreachImpulse(hkReal breachImpulse) HK_OVERRIDE;

        /// Gets the impulse above which this constraint is reported as breached.
        virtual hkReal getBreachImpulse() const HK_OVERRIDE;

        /// Choose the body to be notified when the constraint's maximum impulse is breached.
        virtual void setBodyToNotify(int bodyIdx) HK_OVERRIDE;

        /// Returns the index of the body that is notified when the constraint's maximum impulse limit is breached.
        virtual hkUint8 getNotifiedBodyIndex() const HK_OVERRIDE;

        /// Sets the solving method for this constraint. Use one of the hkpConstraintAtom::SolvingMethod as a value for method. Not implemented,
        /// always falling back to METHOD_STABILIZED
        virtual void setSolvingMethod(hkpConstraintAtom::SolvingMethod method) HK_OVERRIDE;

        /// Gets the inertia stabilization factor, returns HK_FAILURE if the factor is not defined for the given constraint.
        virtual hkResult getInertiaStabilizationFactor( _Out_ hkReal& inertiaStabilizationFactorOut ) const HK_OVERRIDE;

        /// Sets the inertia stabilization factor, return HK_FAILURE if the factor is not defined for the given constraint.
        virtual hkResult setInertiaStabilizationFactor(const hkReal inertiaStabilizationFactorIn) HK_OVERRIDE;

        virtual void getRuntimeInfo( hkBool wantRuntime, hkpConstraintData::RuntimeInfo& infoOut ) const HK_OVERRIDE;

        /// compile the blueprints into the compiled version
        virtual UpdateAtomsResult::Enum updateDirtyAtoms() HK_OVERRIDE;

        /// Returns the linear impulse applied by the solver
        virtual void getAppliedLinearImpulse(const hkTransform& worldFromBodyA, const hkTransform& worldFromBodyB,
            _In_ const struct hkpConstraintRuntime* runtime, _Out_ hkVector4& impulseOut) const HK_OVERRIDE;

    public:

        /// helper enums used in combination m_atom2CompiledAtomOffset to map from the blueprint atoms to the compiled atoms
        /// this allows for directly modifying the compiled atom to avoid setting the dirty flag
        struct AtomTypes
        {
            enum Enum
            {
                SET_TRANSFORMS,
                SETUP_STABILIZATION,
                RAGDOLL_MOTORS,
                ANG_FRICTION_X,
                ANG_FRICTION_Y,
                ANG_FRICTION_Z,
                TWIST_LIMIT,
                ELLIPTICAL_LIMIT,
                STIFF_SPRING,
                BALL_SOCKET,

                LINEAR_LIMIT_X,
                LINEAR_LIMIT_Y,
                LINEAR_LIMIT_Z,

                LINEAR_FRICTION_X,
                LINEAR_FRICTION_Y,
                LINEAR_FRICTION_Z,

                LINEAR_MOTOR0,
                LINEAR_MOTOR1,
                LINEAR_MOTOR2,
                MAX,
                UNMAPPED = MAX,
            };
        };

        struct Blueprints
        {
            HK_DECLARE_CLASS(Blueprints, Reflect, New, Validate);
            HK_RECORD_ATTR(hk::Version(1));

            Blueprints() {}
            void afterReflectNew();

            HK_INLINE hkpLinMotorConstraintAtom& linearMotor( int i ) { return *(&m_linearMotor0 + i); }
            HK_INLINE const hkpLinMotorConstraintAtom& linearMotor( int i ) const { return *(&m_linearMotor0 + i); }

            hkEnum<AxisMode, hkUint8> m_axisMode[MAX_AXES];
            hkReal m_ellipticalMinMax[4]; // In the order swing0Min, swing0Max, swing1Min, swing1Max

            HK_ALIGN_REAL(hkpSetLocalTransformsConstraintAtom)  m_transforms;
            HK_ALIGN_REAL(hkpSetupStabilizationAtom)            m_setupStabilization;
            HK_ALIGN_REAL(hkpRagdollMotorConstraintAtom)        m_ragdollMotors;
            HK_ALIGN_REAL(hkpAngFrictionConstraintAtom)         m_angFrictions[3];
            HK_ALIGN_REAL(hkpTwistLimitConstraintAtom)          m_twistLimit;
            HK_ALIGN_REAL(hkpEllipticalLimitConstraintAtom)     m_ellipticalLimit;  ///< might be converted to a cone / plane limit
            HK_ALIGN_REAL(hkpStiffSpringConstraintAtom)         m_stiffSpring;
            HK_ALIGN_REAL(hkpLinLimitConstraintAtom)            m_linearLimits[3];
            HK_ALIGN_REAL(hkpLinFrictionConstraintAtom)         m_linearFriction[3];
            HK_ALIGN_REAL(hkpLinMotorConstraintAtom)            m_linearMotor0; 
            HK_ALIGN_REAL(hkpLinMotorConstraintAtom)            m_linearMotor1;
            HK_ALIGN_REAL(hkpLinMotorConstraintAtom)            m_linearMotor2;
            HK_ALIGN_REAL(hkpBallSocketConstraintAtom)          m_ballSocket;
        };

        static hkpSolverResults s_unusedSolverResults;  // for unmapped solver results.

        static inline Runtime* HK_CALL getRuntime( _In_opt_ hkpConstraintRuntime* runtime )
        {
            return reinterpret_cast<Runtime*>(runtime);
        }

    public:

        /// The "blueprint" atoms, used to build m_compiledAtoms
        HK_ALIGN_REAL( Blueprints m_blueprints );

        hkBool m_isDirty;   ///< if set, the compiled atoms need to be recompiled.
        int m_numRuntimeElements;
        int m_atomToCompiledAtomOffset[AtomTypes::MAX];
        int m_resultToRuntime[Result::MAX];

        /// The compiled atoms
        /// Note: These are not serialized, they are recompiled on deserialized.
        hkArray<hkUchar> m_compiledAtoms;   //+nosave
};

HK_REFLECT_ENUM(HK_EXPORT_PHYSICS, hkp6DofConstraintData::AxisMode);

#include <Physics/Constraint/Data/6Dof/hkp6DofConstraintData.inl>

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
