// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <Physics/Constraint/hkpConstraint.h>
#include <Physics/Constraint/Data/6Dof/hkp6DofConstraintData.h>
#include <Physics/Constraint/Motor/hkpConstraintMotor.h>

#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Math/Vector/Mx/hkMxUnroll.h>

hkpSolverResults hkp6DofConstraintData::s_unusedSolverResults;


//////////////////////////////////////////////////////////////////////////
// hkpEllipticalLimitConstraintAtom
//////////////////////////////////////////////////////////////////////////

namespace
{
    static HK_INLINE hkReal spherePlanarProject( hkReal rad )
    {
        if ( rad >= HK_REAL_PI - 0.0001f )
        {
            return HK_REAL_PI* 1.2f;    // just make huge to avoid stability problems, cone angle should take care of this
        }
        if ( rad <= 0.01 )  // (sqrt(HK_REAL_EPS) to avoid numerical issues in the sphere to planar project algorithm
        {
            return 0.01;
        }
        hkReal x = hkMath::sin(rad);
        hkReal z = hkMath::cos(rad);
        hkReal result = x * hkMath::sqrt( 2/(1+z) );
        return result;
    };
}

void hkpEllipticalLimitConstraintAtom::makeStableByUsingConeAngle( hkReal maxSwingAngleDiff )
{
    m_coneLimitEnabled = false;
    m_elipticalLimitEnabled = false;

    if( m_angle0 == m_angle1 )
    {
        m_coneAngle = m_angle0;
        m_coneAngleCorrected = spherePlanarProject( m_coneAngle );
        m_coneLimitEnabled = true;
        return;
    }

    m_elipticalLimitEnabled = true;

    hkReal angles[2] = {m_angle0, m_angle1};
    int minAngle = 0;
    int maxAngle = 1;
    if ( m_angle0 > m_angle1)
    {
        minAngle = 1;
        maxAngle = 0;
    }

    if ( angles[maxAngle] > angles[minAngle] * maxSwingAngleDiff )
    {
        // set cone if needed
        m_coneAngle = angles[maxAngle];
        m_coneAngleCorrected = spherePlanarProject( m_coneAngle );
        m_coneLimitEnabled = true;

        hkReal ratio = angles[maxAngle] / (angles[minAngle] * maxSwingAngleDiff);
        ratio = hkMath::clamp( ratio, 1.0f, 1.5f );
        // now map to 0.0 and 1.f
        ratio = (ratio - 1.0f) / (1.5f - 1.f);
        hkReal a = 1.1f * angles[maxAngle];
        hkReal b = a * 1.2f + ( (a>0) ? 0.2f : -0.2f );
        hkReal c = a + ratio * (b-a); // interpolate
        angles[maxAngle] = c;
    }

    m_angleCorrected0 = spherePlanarProject(angles[0]);
    m_angleCorrected1 = spherePlanarProject(angles[1]);

    m_angleCorrected0Inv = 1.0f / m_angleCorrected0;
    m_angleCorrected1Inv = 1.0f / m_angleCorrected1;
}

void hkpEllipticalLimitConstraintAtom::getLimitProjectedOntoSphere( hkReal alpha, _Out_ hkVector4 &posOnSphereOut, _Out_ hkBool& isEllipseOut ) const
{
    const hkpEllipticalLimitConstraintAtom* atom = this;
    hkReal x = hkMath::sin(alpha);
    hkReal y = hkMath::cos(alpha);
    hkReal px = x;
    hkReal py = y;
    if ( m_elipticalLimitEnabled )
    {
        px = x * atom->m_angleCorrected1;
        py = y * atom->m_angleCorrected0;
        isEllipseOut = true;
    }
    else
    {
        px *= HK_REAL_PI;   // huge sphere
        py *= HK_REAL_PI;   // huge sphere
    }

    hkReal pxy2 = px*px+py*py;
    if (m_coneLimitEnabled )
    {
        // clip to sphere
        hkReal f = hkMath::sqrt(pxy2);
        if ( f > atom->m_coneAngleCorrected )
        {
            f = atom->m_coneAngleCorrected / f;
            px *= f;
            py *= f;
            pxy2 = px*px+py*py;
            isEllipseOut = false;
        }
    }
    hkReal nx = hkMath::sqrt( 1- pxy2*0.25f )*px;
    hkReal ny = hkMath::sqrt( 1- pxy2*0.25f )*py;
    hkReal nz = 1.0f - pxy2*0.5f;
    posOnSphereOut.set( nz, nx, ny );
}


//////////////////////////////////////////////////////////////////////////
// hkp6DofConstraintData::Blueprints
//////////////////////////////////////////////////////////////////////////

void hkp6DofConstraintData::Blueprints::afterReflectNew()
{
    // Set motor offsets
    m_ragdollMotors.m_initializedOffset             = HK_OFFSET_OF( Runtime, m_ragdollMotorsInitialized );
    m_ragdollMotors.m_previousTargetAnglesOffset    = HK_OFFSET_OF( Runtime, m_previousTargetAngles );
    hkMxUNROLL_3({
        linearMotor(hkMxI).m_initializedOffset            = (hkUint16)HK_OFFSET_OF(Runtime, m_linearMotorsInitialized[hkMxI]);
        linearMotor(hkMxI).m_previousTargetPositionOffset = (hkUint16)HK_OFFSET_OF(Runtime, m_linearMotorPreviousTargetPosition[hkMxI]);
    });
}


//////////////////////////////////////////////////////////////////////////
// hkp6DofConstraintData
//////////////////////////////////////////////////////////////////////////

namespace
{
    // Helper
    template<typename T>
    void updateCompiledAtom( hkp6DofConstraintData* data, T& source, hkp6DofConstraintData::AtomTypes::Enum type )
    {
        if( !data->m_isDirty )
        {
            int offset = data->m_atomToCompiledAtomOffset[type];
            if( offset >= 0 )
            {
                T* dest = (T*)hkAddByteOffset( data->m_compiledAtoms.begin(), offset );
                //HK_ASSERT_NO_MSG( 0xf034fdff, (sizeof(T) & 0xf) == 0);    // for memcpy16
                hkString::memCpy( dest, &source, sizeof(T) );
            }
            else
            {
                data->m_isDirty = true;
            }
        }
    };
}

hkp6DofConstraintData::hkp6DofConstraintData()
{
    m_blueprints.m_transforms.m_transformA.setIdentity();
    m_blueprints.m_transforms.m_transformB.setIdentity();

    for(hkUint8 i = 0; i < 3; i++)
    {
        m_blueprints.m_angFrictions[i].m_firstFrictionAxis = i;
        m_blueprints.m_angFrictions[i].m_numFrictionAxes = 1;
        m_blueprints.m_angFrictions[i].m_maxFrictionTorque = 0.0f;
    }

    // this is the unpowered version of the constraint
    m_blueprints.m_ragdollMotors.m_isEnabled = false;
    m_blueprints.m_ragdollMotors.m_target_bRca.setIdentity();
    m_blueprints.m_ragdollMotors.m_motors[0] = HK_NULL;
    m_blueprints.m_ragdollMotors.m_motors[1] = HK_NULL;
    m_blueprints.m_ragdollMotors.m_motors[2] = HK_NULL;
    m_blueprints.m_ragdollMotors.m_initializedOffset = HK_OFFSET_OF( Runtime, m_ragdollMotorsInitialized );
    m_blueprints.m_ragdollMotors.m_previousTargetAnglesOffset = HK_OFFSET_OF( Runtime, m_previousTargetAngles );

    m_blueprints.m_twistLimit.m_twistAxis = AXIS_TWIST;
    m_blueprints.m_twistLimit.m_refAxis   = AXIS_SWING0;
    m_blueprints.m_twistLimit.m_minAngle = hkReal(-HK_REAL_MAX);
    m_blueprints.m_twistLimit.m_maxAngle = hkReal( HK_REAL_MAX);
    m_blueprints.m_twistLimit.m_angularLimitsTauFactor = hkReal(0.8f);
    m_blueprints.m_twistLimit.m_angularLimitsDampFactor = hkReal(1);

    m_blueprints.m_stiffSpring.m_length    = 0.0f;
    m_blueprints.m_stiffSpring.m_maxLength = HK_REAL_MAX;
    m_blueprints.m_stiffSpring.m_springConstant = 100;
    m_blueprints.m_stiffSpring.m_springDamping  = -1;   // disabled

    m_blueprints.m_ellipticalLimit.m_isEnabled = true;
    m_blueprints.m_ellipticalLimit.m_angle0 = HK_REAL_PI;
    m_blueprints.m_ellipticalLimit.m_angle1 = HK_REAL_PI;
    m_blueprints.m_ellipticalLimit.m_angularLimitsTauFactor = hkReal(0.8f);
    m_blueprints.m_ellipticalLimit.m_angularLimitsDampFactor = hkReal(1);
    m_blueprints.m_ellipticalMinMax[0] = -HK_REAL_PI;
    m_blueprints.m_ellipticalMinMax[1] =  HK_REAL_PI;
    m_blueprints.m_ellipticalMinMax[2] = -HK_REAL_PI;
    m_blueprints.m_ellipticalMinMax[3] =  HK_REAL_PI;

    for(int i = 0; i < MAX_AXES; i++)
    {
        m_blueprints.m_axisMode[i] = UNLIMITED;
    }

    for(hkUint8 i = 0; i < 3; i++)
    {
        m_blueprints.m_linearLimits[i].m_axisIndex = i;
        m_blueprints.m_linearLimits[i].m_min = -HK_REAL_MAX;
        m_blueprints.m_linearLimits[i].m_max = HK_REAL_MAX;
        
        m_blueprints.m_linearFriction[i].m_isEnabled = true; 
        m_blueprints.m_linearFriction[i].m_frictionAxis = i;
        m_blueprints.m_linearFriction[i].m_maxFrictionForce = 0.0f;
    }

    hkString::memSet4( m_atomToCompiledAtomOffset, -1, sizeof(m_atomToCompiledAtomOffset)/ 4 );

    m_numRuntimeElements = 0;
    hkString::memSet4( m_resultToRuntime, -1, sizeof(m_resultToRuntime)/4 );

    m_isDirty = true;

    hkMxUNROLL_3({
        m_blueprints.linearMotor(hkMxI).m_isEnabled = false;
        m_blueprints.linearMotor(hkMxI).m_targetPosition = hkReal(0);
        m_blueprints.linearMotor(hkMxI).m_motor = HK_NULL;
        m_blueprints.linearMotor(hkMxI).m_motorAxis = hkUint8(hkMxI);
        m_blueprints.linearMotor(hkMxI).m_initializedOffset            = (hkUint16)HK_OFFSET_OF(Runtime, m_linearMotorsInitialized[hkMxI]);
        m_blueprints.linearMotor(hkMxI).m_previousTargetPositionOffset = (hkUint16)HK_OFFSET_OF(Runtime, m_linearMotorPreviousTargetPosition[hkMxI]);
    });
}

void hkp6DofConstraintData::afterReflectNew()
{
    // Generate compiled atoms as they are not serialized
    m_isDirty = true;
    updateDirtyAtoms();
}

hkp6DofConstraintData::~hkp6DofConstraintData()
{
    for (int m = 0; m < 3; m++)
    {
        if (m_blueprints.linearMotor(m).m_motor != HK_NULL)
        {
            m_blueprints.linearMotor(m).m_motor->removeReference();
        }
        if( m_blueprints.m_ragdollMotors.m_motors[m] != HK_NULL)
        {
            m_blueprints.m_ragdollMotors.m_motors[m]->removeReference();
        }
    }
}

int hkp6DofConstraintData::getType() const
{
    return hkpConstraintData::CONSTRAINT_TYPE_6DOF;
}

void hkp6DofConstraintData::setAxisMode(AxisIdentifier axis, AxisMode mode)
{
    if(m_blueprints.m_axisMode[axis] != mode)
    {
        m_blueprints.m_axisMode[axis] = mode;
        m_isDirty = true;
    }
}

hkp6DofConstraintData::AxisMode hkp6DofConstraintData::getAxisMode(AxisIdentifier axis) const
{
    return m_blueprints.m_axisMode[axis];
}

void hkp6DofConstraintData::setAxisMinLimit(AxisIdentifier axis, hkReal limit)
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        hkReal& cur = m_blueprints.m_linearLimits[axis - LINEAR_X].m_min;
        if(cur != limit)
        {
            cur = limit;
            updateCompiledAtom( this, m_blueprints.m_linearLimits[axis - LINEAR_X], (AtomTypes::Enum)(AtomTypes::LINEAR_LIMIT_X + axis - LINEAR_X));
        }
    }
    else if(axis == ANGULAR_X)
    {
        hkReal& cur = m_blueprints.m_twistLimit.m_minAngle;
        if(cur != limit)
        {
            cur = limit;
            updateCompiledAtom( this, m_blueprints.m_twistLimit, AtomTypes::TWIST_LIMIT );
        }
    }
    else
    {
        hkReal& cur = m_blueprints.m_ellipticalMinMax[2 * (axis - ANGULAR_Y)];
        if(cur != limit)
        {
            cur = limit;
            m_isDirty = true;
        }
    }
}

hkReal hkp6DofConstraintData::getAxisMinLimit(AxisIdentifier axis) const
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        return m_blueprints.m_linearLimits[axis - LINEAR_X].m_min;
    }
    if(axis == ANGULAR_X)
    {
        return m_blueprints.m_twistLimit.m_minAngle;
    }
    else
    {
        return m_blueprints.m_ellipticalMinMax[2 * (axis - ANGULAR_Y)];
    }
}

void hkp6DofConstraintData::setAxisMaxLimit(AxisIdentifier axis, hkReal limit)
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        hkReal& cur = m_blueprints.m_linearLimits[axis - LINEAR_X].m_max;
        if(cur != limit)
        {
            cur = limit;
            updateCompiledAtom( this, m_blueprints.m_linearLimits[axis - LINEAR_X], (AtomTypes::Enum)(AtomTypes::LINEAR_LIMIT_X + axis - LINEAR_X));
        }
    }
    else if(axis == ANGULAR_X)
    {
        hkReal& cur = m_blueprints.m_twistLimit.m_maxAngle;
        if(cur != limit)
        {
            cur = limit;
            updateCompiledAtom( this, m_blueprints.m_twistLimit, AtomTypes::TWIST_LIMIT );
        }
    }
    else
    {
        hkReal& cur = m_blueprints.m_ellipticalMinMax[2 * (axis - ANGULAR_Y) + 1];
        if(cur != limit)
        {
            cur = limit;
            m_isDirty = true;
        }
    }
}

hkReal hkp6DofConstraintData::getAxisMaxLimit(AxisIdentifier axis) const
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        return m_blueprints.m_linearLimits[axis - LINEAR_X].m_max;
    }
    if(axis == ANGULAR_X)
    {
        return m_blueprints.m_twistLimit.m_maxAngle;
    }
    else
    {
        return m_blueprints.m_ellipticalMinMax[2 * (axis - ANGULAR_Y) + 1];
    }
}

void hkp6DofConstraintData::setAxisFriction(AxisIdentifier axis, hkReal friction)
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        hkReal& cur = m_blueprints.m_linearFriction[axis - LINEAR_X].m_maxFrictionForce;
        if(cur != friction)
        {
            cur = friction;
            updateCompiledAtom( this, m_blueprints.m_linearFriction[axis - LINEAR_X], (AtomTypes::Enum)(AtomTypes::LINEAR_FRICTION_X + axis - LINEAR_X));
        }
    }
    else
    {
        
        hkReal& cur = m_blueprints.m_angFrictions[axis - ANGULAR_X].m_maxFrictionTorque;
        if(friction != cur)
        {
            cur = friction;
            updateCompiledAtom( this, m_blueprints.m_angFrictions[axis - ANGULAR_X], AtomTypes::Enum(AtomTypes::ANG_FRICTION_X + axis - ANGULAR_X) );
        }
    }
}

hkReal hkp6DofConstraintData::getAxisFriction(AxisIdentifier axis) const
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        return m_blueprints.m_linearFriction[axis - LINEAR_X].m_maxFrictionForce;
    }
    else
    {
        return m_blueprints.m_angFrictions[axis - ANGULAR_X].m_maxFrictionTorque;
    }
}

void hkp6DofConstraintData::setAxisMotor(AxisIdentifier axis, hkpConstraintMotor* newMotor)
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        hkpLinMotorConstraintAtom& linMotor = m_blueprints.linearMotor(axis - LINEAR_X);
        if( newMotor == linMotor.m_motor )
        {
            return;
        }

        if( newMotor )
        {
            newMotor->addReference();
        }

        if( linMotor.m_motor )
        {
            linMotor.m_motor->removeReference();
        }
        linMotor.m_motor = newMotor;
        linMotor.m_isEnabled = newMotor != HK_NULL;
        updateCompiledAtom( this, linMotor, AtomTypes::Enum(AtomTypes::LINEAR_MOTOR0 + axis - LINEAR_X) );
    }
    else
    {
        hkpConstraintMotor*& pMotor = m_blueprints.m_ragdollMotors.m_motors[axis - ANGULAR_X];
        if( pMotor == newMotor )
        {
            return;
        }

        if( newMotor )
        {
            newMotor->addReference();
        }

        if( pMotor )
        {
            pMotor->removeReference();
        }

        pMotor = newMotor;
        m_blueprints.m_ragdollMotors.m_isEnabled = true;
        updateCompiledAtom( this, m_blueprints.m_ragdollMotors, AtomTypes::RAGDOLL_MOTORS );
    }
}

const hkpConstraintMotor* hkp6DofConstraintData::getAxisMotor(AxisIdentifier axis) const
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        return m_blueprints.linearMotor(axis - LINEAR_X).m_motor;
    }
    else
    {
        return m_blueprints.m_ragdollMotors.m_motors[axis - ANGULAR_X];
    }
}

hkpConstraintMotor* hkp6DofConstraintData::getAxisMotor(AxisIdentifier axis)
{
    if(axis >= LINEAR_X && axis <= LINEAR_Z)
    {
        return m_blueprints.linearMotor(axis - LINEAR_X).m_motor;
    }
    else
    {
        return m_blueprints.m_ragdollMotors.m_motors[axis - ANGULAR_X];
    }
}

void hkp6DofConstraintData::setTwistMinAngularLimit(hkReal rad)
{
    if ( m_blueprints.m_twistLimit.m_minAngle != rad )
    {
        m_blueprints.m_twistLimit.m_minAngle = rad;
        updateCompiledAtom( this, m_blueprints.m_twistLimit, AtomTypes::TWIST_LIMIT );
    }
}

void hkp6DofConstraintData::setTwistMaxAngularLimit(hkReal rad)
{
    if ( m_blueprints.m_twistLimit.m_maxAngle != rad )
    {
        m_blueprints.m_twistLimit.m_maxAngle = rad;
        updateCompiledAtom( this, m_blueprints.m_twistLimit, AtomTypes::TWIST_LIMIT );
    }
}

void hkp6DofConstraintData::setSwing0Limit(hkReal rad)
{
    hkReal newValue = hkMath::clamp(rad, 0, HK_REAL_PI);
    if ( newValue != m_blueprints.m_ellipticalLimit.m_angle0 )
    {
        m_blueprints.m_ellipticalMinMax[0] = -newValue;
        m_blueprints.m_ellipticalMinMax[1] = newValue;
        m_blueprints.m_ellipticalLimit.m_angle0 = newValue;
        m_blueprints.m_ellipticalLimit.makeStableByUsingConeAngle();
        //updateCompiledAtom( this, m_blueprints.m_ellipticalLimit, AtomTypes::ELLIPTICAL_LIMIT );
        m_isDirty = true;
    }
}

void hkp6DofConstraintData::setSwing0Min(hkReal min)
{
    if(min != m_blueprints.m_ellipticalMinMax[0])
    {
        m_blueprints.m_ellipticalMinMax[0] = min;
        m_isDirty = true;
    }
}

void hkp6DofConstraintData::setSwing0Max(hkReal max)
{
    if(max != m_blueprints.m_ellipticalMinMax[1])
    {
        m_blueprints.m_ellipticalMinMax[1] = max;
        m_isDirty = true;
    }
}

void hkp6DofConstraintData::setSwing1Limit(hkReal rad)
{
    hkReal newValue = hkMath::clamp(rad, 0, HK_REAL_PI);
    if ( m_blueprints.m_ellipticalLimit.m_angle1 != newValue )
    {
        m_blueprints.m_ellipticalMinMax[2] = -newValue;
        m_blueprints.m_ellipticalMinMax[3] = newValue;

        m_blueprints.m_ellipticalLimit.m_angle1 = newValue;
        m_blueprints.m_ellipticalLimit.makeStableByUsingConeAngle();
        //updateCompiledAtom( this, m_blueprints.m_ellipticalLimit, AtomTypes::ELLIPTICAL_LIMIT );
        m_isDirty = true;
    }
}

void hkp6DofConstraintData::setSwing1Min(hkReal min)
{
    if(min != m_blueprints.m_ellipticalMinMax[2])
    {
        m_blueprints.m_ellipticalMinMax[2] = min;
        m_isDirty = true;
    }
}

void hkp6DofConstraintData::setSwing1Max(hkReal max)
{
    if(max != m_blueprints.m_ellipticalMinMax[3])
    {
        m_blueprints.m_ellipticalMinMax[3] = max;
        m_isDirty = true;
    }
}


void hkp6DofConstraintData::setInWorldSpace(
    const hkTransform& bodyATransform, const hkTransform& bodyBTransform,
    const hkVector4& pivot, const hkVector4& twistAxisW,
    const hkVector4& swingAxis0W )
{
    HK_ASSERT(0x77ad7e93, hkMath::equal( twistAxisW.dot<3>(swingAxis0W).getReal(), 0.0f), "twistAxisW && swingAxis0W should be perpendicular");

    // Set relative orientation
    {
        hkVector4 constraintBaseInW[3];

        constraintBaseInW[AXIS_TWIST] = twistAxisW; constraintBaseInW[AXIS_TWIST].normalize<3>();
        constraintBaseInW[AXIS_SWING0] = swingAxis0W; constraintBaseInW[AXIS_SWING0].normalize<3>();
        constraintBaseInW[AXIS_SWING1].setCross( constraintBaseInW[AXIS_TWIST], constraintBaseInW[AXIS_SWING0] );

        hkVector4Util::rotateInversePoints( bodyATransform.getRotation(), constraintBaseInW, 3, &m_blueprints.m_transforms.m_transformA.getRotation().getColumn(0) );
        hkVector4Util::rotateInversePoints( bodyBTransform.getRotation(), constraintBaseInW, 3, &m_blueprints.m_transforms.m_transformB.getRotation().getColumn(0) );
    }

    // Set pivot points
    m_blueprints.m_transforms.m_transformA.getTranslation().setTransformedInversePos( bodyATransform, pivot );
    m_blueprints.m_transforms.m_transformB.getTranslation().setTransformedInversePos( bodyBTransform, pivot );

    m_blueprints.m_ragdollMotors.m_target_bRca = m_blueprints.m_transforms.m_transformB.getRotation();

    HK_ASSERT(0xadea65ee, isValid(), "Members of 6dof constraint inconsistent.");
    m_isDirty  = true;
}


void hkp6DofConstraintData::setInBodySpace(
    const hkVector4& pivotA,const hkVector4& pivotB,
    const hkVector4& twistAxisA, const hkVector4& twistAxisB,
    const hkVector4& swingAxis0A,const hkVector4& swingAxis0B )
{
    hkVector4* baseA = &m_blueprints.m_transforms.m_transformA.getColumn(0);
    baseA[AXIS_TWIST] = twistAxisA; baseA[AXIS_TWIST].normalize<3>();
    baseA[AXIS_SWING0] = swingAxis0A; baseA[AXIS_SWING0].normalize<3>();
    baseA[AXIS_SWING1].setCross( baseA[AXIS_TWIST], baseA[AXIS_SWING0] );
    m_blueprints.m_transforms.m_transformA.getTranslation() = pivotA;

    hkVector4* baseB = &m_blueprints.m_transforms.m_transformB.getColumn(0);
    baseB[AXIS_TWIST] = twistAxisB; baseB[AXIS_TWIST].normalize<3>();
    baseB[AXIS_SWING0] = swingAxis0B; baseB[AXIS_SWING0].normalize<3>();
    baseB[AXIS_SWING1].setCross( baseB[AXIS_TWIST], baseB[AXIS_SWING0] );
    m_blueprints.m_transforms.m_transformB.getTranslation() = pivotB;

    m_blueprints.m_ragdollMotors.m_target_bRca = m_blueprints.m_transforms.m_transformB.getRotation();

    HK_ASSERT(0xadea65ef, isValid(), "Members of 6dof constraint inconsistent.");
    m_isDirty  = true;
}

void hkp6DofConstraintData::setBreachImpulse(hkReal breachImpulse)
{
    HK_ASSERT_NOT_IMPLEMENTED( 0xf04545fe);
    m_blueprints.m_ballSocket.setBreachImpulse(breachImpulse);
}

hkReal hkp6DofConstraintData::getBreachImpulse() const
{
    return m_blueprints.m_ballSocket.getBreachImpulse();
}

void hkp6DofConstraintData::setBodyToNotify(int bodyIdx)
{
    HK_ASSERT_NOT_IMPLEMENTED( 0xf04545fe );
    HK_ASSERT(0xad808071, bodyIdx >= 0 && bodyIdx <= 1, "Notify body index must be 0 or 1.");
    m_blueprints.m_ballSocket.m_bodiesToNotify = hkUint8(0x01 << bodyIdx);
}

hkUint8 hkp6DofConstraintData::getNotifiedBodyIndex() const
{
    //return m_blueprints.m_ballSocket.m_bodiesToNotify;
    HK_ASSERT(0xad809032, m_blueprints.m_ballSocket.m_bodiesToNotify & 0x3, "Body to be notified not set.");
    return m_blueprints.m_ballSocket.m_bodiesToNotify >> 1;
}

void hkp6DofConstraintData::getConstraintInfo( hkpConstraintData::ConstraintInfo& infoOut ) const
{
    HK_ASSERT( 0xf04565f1, !m_isDirty, "If you make changes to the hkp6DofConstraintData, you must call updateDirtyAtoms()");
    getConstraintInfoUtil( (const hkpConstraintAtom*)m_compiledAtoms.begin(), m_compiledAtoms.getSize(), infoOut );
    infoOut.m_sizeOfSchemas += 0x100;    // because of some dynamic atoms, in P2014 this does not matter
    infoOut.m_numSolverElemTemps += 4;
}

void hkp6DofConstraintData::getRuntimeInfo( hkBool wantRuntime, hkpConstraintData::RuntimeInfo& infoOut ) const
{
    // we need runtime data to be able to support lastAngle and friction
    infoOut.m_numSolverResults = Result::MAX;
    infoOut.m_sizeOfExternalRuntime = sizeof( Runtime );
}


void hkp6DofConstraintData::setMaxFrictionTorque(hkReal tmag)
{
    setAxisFriction(ANGULAR_X, tmag);
    setAxisFriction(ANGULAR_Y, tmag);
    setAxisFriction(ANGULAR_Z, tmag);
}

void hkp6DofConstraintData::setLinearLimitStrength( hkReal springConstrant, hkReal springDamping )
{
    m_blueprints.m_stiffSpring.m_springConstant = springConstrant;
    m_blueprints.m_stiffSpring.m_springDamping = springDamping;
    updateCompiledAtom( this, m_blueprints.m_stiffSpring, AtomTypes::STIFF_SPRING );
}

void hkp6DofConstraintData::setLinearLimit( hkReal distance )
{
    hkReal newValue = hkMath::max2<hkReal>( 0.0f, distance);
    if ( m_blueprints.m_stiffSpring.m_maxLength != newValue )
    {
        m_blueprints.m_stiffSpring.m_maxLength = newValue;
        updateCompiledAtom( this, m_blueprints.m_stiffSpring, AtomTypes::STIFF_SPRING );
    }
}

void hkp6DofConstraintData::setLinearAxisMinLimit( int axis, hkReal distance )
{
    HK_ASSERT_NO_MSG(0x301759E6, axis >= 0 && axis <= 2);
    m_blueprints.m_linearLimits[axis].m_min = distance;
    m_isDirty = true;
}

void hkp6DofConstraintData::setLinearAxisMaxLimit( int axis, hkReal distance )
{
    HK_ASSERT_NO_MSG(0x301759E7, axis >= 0 && axis <= 2);
    m_blueprints.m_linearLimits[axis].m_max = distance;
    m_isDirty = true;
}

void hkp6DofConstraintData::setLinearAxisFriction(int axis, hkReal friction )
{
    HK_ASSERT_NO_MSG(0x301759E8, axis >= 0 && axis <= 2);
    m_blueprints.m_linearFriction[axis].m_maxFrictionForce = friction;
    m_isDirty = true;
}

hkBool hkp6DofConstraintData::isValid() const
{
    // In stable mode, we need the setupStabilization atom enabled!
    if ( (m_blueprints.m_ballSocket.m_solvingMethod == hkpConstraintAtom::METHOD_STABILIZED) &&
        !m_blueprints.m_setupStabilization.m_enabled )
    {
        return false;
    }

    hkBool valid = true;
    valid = valid && m_blueprints.m_transforms.m_transformA.getRotation().isOrthonormal();
    valid = valid && m_blueprints.m_transforms.m_transformB.getRotation().isOrthonormal();
    valid = valid && ( m_blueprints.m_ellipticalLimit.m_angle0 >= 0.0f );
    valid = valid && ( m_blueprints.m_ellipticalLimit.m_angle0 <= HK_REAL_PI );
    valid = valid && ( m_blueprints.m_ellipticalLimit.m_angle1 >= 0.0f );
    valid = valid && ( m_blueprints.m_ellipticalLimit.m_angle1 <= HK_REAL_PI );
    valid = valid && ( m_blueprints.m_twistLimit.m_minAngle <= m_blueprints.m_twistLimit.m_maxAngle );

    // Check ranges on other angular limits ??
    // No wacky behavior seems to result

    return valid;
}

void hkp6DofConstraintData::getConstraintFrameA( _Out_ hkMatrix3& constraintFrameA ) const
{
    constraintFrameA = m_blueprints.m_transforms.m_transformA.getRotation();
}

void hkp6DofConstraintData::getConstraintFrameB( _Out_ hkMatrix3& constraintFrameB ) const
{
    constraintFrameB = m_blueprints.m_transforms.m_transformB.getRotation();
}

void hkp6DofConstraintData::setLinearMotor( int axis, _In_opt_ hkpConstraintMotor* newMotor )
{
    hkpLinMotorConstraintAtom& linMotor = m_blueprints.linearMotor(axis);
    if( newMotor == linMotor.m_motor )
    {
        return;
    }

    if( newMotor )
    {
        newMotor->addReference();
    }

    if( linMotor.m_motor )
    {
        linMotor.m_motor->removeReference();
    }
    HK_ASSERT_NO_MSG( 0xf0fc34cd, axis>=0 && axis < 3);
    linMotor.m_motor = newMotor;
    linMotor.m_isEnabled = true;
    updateCompiledAtom( this, linMotor, AtomTypes::Enum(AtomTypes::LINEAR_MOTOR0+axis) );
}

void hkp6DofConstraintData::setLinearMotorEnabled( int axis, bool isEnabled )
{
    HK_ASSERT_NO_MSG( 0xf0fc34cd, axis>=0 && axis < 3);
    hkpLinMotorConstraintAtom& linMotor = m_blueprints.linearMotor(axis);
    if ( linMotor.m_isEnabled != isEnabled )
    {
        linMotor.m_isEnabled = isEnabled;
        updateCompiledAtom( this, linMotor, AtomTypes::Enum(AtomTypes::LINEAR_MOTOR0+axis) );
    }
}

void hkp6DofConstraintData::setLinearMotorTargetPosition( hkVector4Parameter position )
{
    for (int i =0; i < 3; i++)
    {
        hkpLinMotorConstraintAtom& linMotor = m_blueprints.linearMotor(i);
        if ( linMotor.m_targetPosition != position(i) )
        {
            linMotor.m_targetPosition = position(i);
            if ( linMotor.m_motor )
            {
                updateCompiledAtom( this, linMotor, AtomTypes::Enum(AtomTypes::LINEAR_MOTOR0+i) );
            }
        }
    }
}

void hkp6DofConstraintData::getLinearMotorTargetPosition( _Out_ hkVector4& positionOut )
{
    positionOut(0) = m_blueprints.linearMotor(0).m_targetPosition;
    positionOut(1) = m_blueprints.linearMotor(1).m_targetPosition;
    positionOut(2) = m_blueprints.linearMotor(2).m_targetPosition;
    positionOut(3) = 0;
}


//////////////////////////////////////////////////////////////////////////
//  Motor related methods
//////////////////////////////////////////////////////////////////////////

void hkp6DofConstraintData::setAngularTarget( const hkMatrix3& target_cbRca )
{
    // we assign :
    //  m_target_bRca = (bRcb * cbRca) * (caRa * aRca);
    // where :
    //  bRcb -- is rotation from bodyB's constraint space to bodyB's space // it's stored in m_blueprints.m_transforms.m_transformB.getRotation()
    //  aRca -- is rotation from bodyA's constraint space to bodyA's space // it's stored in m_blueprints.m_transforms.m_transformA.getRotation()
    //  caRa == aRca^-1
    //  cbRca -- is the target relative orientation of constraint reference frames attached to bodies A & B
    //
    m_blueprints.m_ragdollMotors.m_target_bRca.setMul( m_blueprints.m_transforms.m_transformB.getRotation(), target_cbRca );
    updateCompiledAtom( this, m_blueprints.m_ragdollMotors, AtomTypes::RAGDOLL_MOTORS );
}

void hkp6DofConstraintData::setTargetRelativeOrientationOfBodies( const hkRotation& target_bRa )
{
    // we assign :
    //  m_target_bRca = (bRcb * cbRca * caRa) * aRca;
    // for explanation of the indiviual rotation variables see info in setAngularTarget().

    HK_ASSERT(0xad67dd88, hkp6DofConstraintData::AXIS_TWIST == 0, "Assuming twist axis has index 0 in base");
    m_blueprints.m_ragdollMotors.m_target_bRca.setMul( target_bRa, m_blueprints.m_transforms.m_transformA.getRotation() );
    updateCompiledAtom( this, m_blueprints.m_ragdollMotors, AtomTypes::RAGDOLL_MOTORS );
}

void hkp6DofConstraintData::getAngularTarget( hkMatrix3& target_cbRca )
{
    target_cbRca.setMulInverseMul(m_blueprints.m_transforms.m_transformB.getRotation(), m_blueprints.m_ragdollMotors.m_target_bRca);
}

hkpConstraintMotor* hkp6DofConstraintData::getAngularMotor( MotorIndex index ) const
{
    if ( index == MOTOR_ALL)
    {
        HK_ASSERT( 0xf0454565, m_blueprints.m_ragdollMotors.m_motors[0] == m_blueprints.m_ragdollMotors.m_motors[1] && m_blueprints.m_ragdollMotors.m_motors[0] == m_blueprints.m_ragdollMotors.m_motors[2], "You are trying to get a motor MotorIndex::MOTOR_ALL, but the motors are different");
        return m_blueprints.m_ragdollMotors.m_motors[0];
    }
    hkpConstraintMotor* motor = m_blueprints.m_ragdollMotors.m_motors[index];
    return motor;
}

void hkp6DofConstraintData::setAngularMotor( MotorIndex index, _In_opt_ hkpConstraintMotor* newMotor)
{
    if ( index == MOTOR_ALL )
    {
        setAngularMotor(MOTOR_SWING0, newMotor);
        setAngularMotor(MOTOR_SWING1, newMotor);
        setAngularMotor(MOTOR_TWIST, newMotor);
        return;
    }
    hkpConstraintMotor** pMotor = &m_blueprints.m_ragdollMotors.m_motors[index];
    if( (*pMotor)  == newMotor )
    {
        return;
    }

    if( newMotor )
    {
        newMotor->addReference();
    }

    if( (*pMotor) )
    {
        (*pMotor)->removeReference();
    }

    (*pMotor) = newMotor;
    m_blueprints.m_ragdollMotors.m_isEnabled = true;
    updateCompiledAtom( this, m_blueprints.m_ragdollMotors, AtomTypes::RAGDOLL_MOTORS );
}

void hkp6DofConstraintData::setAngularMotorsEnabled( bool enable )
{
    
    setAxisMotor(hkp6DofConstraintData::ANGULAR_X, HK_NULL);
    setAxisMotor(hkp6DofConstraintData::ANGULAR_Y, HK_NULL);
    setAxisMotor(hkp6DofConstraintData::ANGULAR_Z, HK_NULL);

    m_isDirty = true;
}

void hkp6DofConstraintData::setSolvingMethod(hkpConstraintAtom::SolvingMethod method)
{
//  switch ( method )
//  {
//  case hkpConstraintAtom::METHOD_STABILIZED:
//      {
//          m_blueprints.m_setupStabilization.m_enabled = true;
//          m_blueprints.m_ballSocket.m_solvingMethod   = hkpConstraintAtom::METHOD_STABILIZED;
//      }
//      break;
//
//  case hkpConstraintAtom::METHOD_OLD:
//      {
//          m_blueprints.m_setupStabilization.m_enabled = false;
//          m_blueprints.m_ballSocket.m_solvingMethod   = hkpConstraintAtom::METHOD_OLD;
//      }
//      break;
//
//  default:
//      {
//          HK_ASSERT(0x3ce72932, false, "Unknown solving method! Please use one of the values in hkpConstraintAtom::SolvingMethod!");
//      }
//      break;
//  }
//  m_isDirty = true;
}

hkResult hkp6DofConstraintData::getInertiaStabilizationFactor( _Out_ hkReal& inertiaStabilizationFactorOut) const
{
    inertiaStabilizationFactorOut = m_blueprints.m_ballSocket.getInertiaStabilizationFactor();
    return HK_SUCCESS;
}

hkResult hkp6DofConstraintData::setInertiaStabilizationFactor(const hkReal inertiaStabilizationFactorIn)
{
    m_blueprints.m_ballSocket.setInertiaStabilizationFactor(inertiaStabilizationFactorIn);
    updateCompiledAtom( this, m_blueprints.m_ballSocket, AtomTypes::BALL_SOCKET );
    return HK_SUCCESS;
}


//////////////////////////////////////////////////////////////////////////
//  Internal stuff
//////////////////////////////////////////////////////////////////////////

namespace
{
    // Helper used by updateDirtyAtoms()
    template<typename T>
    void appendAtom( hkp6DofConstraintData* data, T& source, hkp6DofConstraintData::AtomTypes::Enum type, hkp6DofConstraintData::Result::Enum rt )
    {
        int offset = data->m_compiledAtoms.getSize();

        const int size = sizeof(T);
        hkUchar* dest = data->m_compiledAtoms.expandBy(size);
        if ( type != hkp6DofConstraintData::AtomTypes::UNMAPPED )
        {
            HK_ASSERT_NO_MSG(0xf04fde35, data->m_atomToCompiledAtomOffset[type]  < 0 );
            data->m_atomToCompiledAtomOffset[type] = offset;
        }
        HK_ASSERT_NO_MSG(0xf04fde34, (size & 0xf)  == 0);
        hkString::memCpy16(dest, &source, size/16 );

        if ( rt != hkp6DofConstraintData::Result::MAX )
        {
            int numRuntimes = source.numSolverResults();
            if ( numRuntimes )
            {
                for (int i =0; i < numRuntimes; i++ )
                {
                    int k = i + rt;
                    HK_ASSERT_NO_MSG( 0xf0342343, data->m_resultToRuntime[k] == -1 );
                    data->m_resultToRuntime[k] = data->m_numRuntimeElements++;
                }
            }
        }
    };

    // Helper used by updateDirtyAtoms()
    void appendRuntimeManual( hkp6DofConstraintData* data, hkp6DofConstraintData::Result::Enum rt )
    {
        if ( rt != hkp6DofConstraintData::Result::MAX )
        {
            int k = rt;
            HK_ASSERT_NO_MSG( 0xf0342343, data->m_resultToRuntime[k] == -1 );
            data->m_resultToRuntime[k] = data->m_numRuntimeElements++;
        }
    }
}

hkpConstraintData::UpdateAtomsResult::Enum hkp6DofConstraintData::updateDirtyAtoms()
{
    if( !m_isDirty )
    {
        return hkpConstraintData::UpdateAtomsResult::RUNTIME_UNCHANGED;
    }

    m_isDirty = false;
    hkString::memSet4( m_atomToCompiledAtomOffset, -1, sizeof(m_atomToCompiledAtomOffset)/ sizeof(m_atomToCompiledAtomOffset[0]) );
    m_numRuntimeElements = 0;
    hkString::memSet4( m_resultToRuntime, -1, sizeof(m_resultToRuntime)/ sizeof(m_resultToRuntime[0]) );

    m_compiledAtoms.setSize(0);
    const Blueprints* bp = &m_blueprints;
    appendAtom( this, bp->m_transforms,       AtomTypes::SET_TRANSFORMS, Result::MAX );
    appendAtom( this, bp->m_setupStabilization, AtomTypes::SETUP_STABILIZATION, Result::MAX );

    // -------------------------------------------------------------------------------------------------------------
    // Angular part
    // -------------------------------------------------------------------------------------------------------------
    
    int fixSwing = (bp->m_axisMode[ANGULAR_Y] == FIXED) + (bp->m_axisMode[ANGULAR_Z] == FIXED);
    int fixTwist = (bp->m_axisMode[ANGULAR_X] == FIXED);
    int fixAngDof = fixTwist + fixSwing;
    int limitTwist = bp->m_axisMode[ANGULAR_X] == LIMITED;
    // motors
    if ( fixAngDof < 3 )
    {
        if ( bp->m_ragdollMotors.m_isEnabled )
        {
            appendAtom( this, bp->m_ragdollMotors, AtomTypes::RAGDOLL_MOTORS, Result::ANG_MOTOR_0 );
        }
        else
        {
            for(int i = 0; i < 3; i++)
            {
                if( bp->m_angFrictions[i].m_isEnabled && bp->m_axisMode[ANGULAR_X + i] != FIXED )
                {
                    appendAtom( this, bp->m_angFrictions[i], AtomTypes::Enum(AtomTypes::ANG_FRICTION_X + i), Result::Enum(Result::ANG_MOTOR_0 + i * 2) );
                }
            }
        }
    }


    if ( fixAngDof == 3 )
    {
        HK_ALIGN16(hkpAngConstraintAtom) angAtom;
        angAtom.m_constrainedAxes[0] = 0;
        angAtom.m_constrainedAxes[1] = 1;
        angAtom.m_constrainedAxes[2] = 2;
        angAtom.m_numConstrainedAxes = 3;
        appendAtom( this, angAtom, AtomTypes::UNMAPPED, Result::TWIST );
    }
    else if ( fixSwing == 2)
    {
        // hinge
        HK_ALIGN16(hkpAngConstraintAtom) angAtom;
        angAtom.m_constrainedAxes[0] = 1;
        angAtom.m_constrainedAxes[1] = 2;
        angAtom.m_constrainedAxes[2] = 0;
        angAtom.m_numConstrainedAxes = 2;
        appendAtom( this, angAtom, AtomTypes::UNMAPPED, Result::SWING0 );

        // twist limit
        if ( limitTwist )
        {
            appendAtom( this, bp->m_twistLimit, AtomTypes::TWIST_LIMIT, Result::TWIST );
        }
    }
    else
    {
        // check if one swing axis is 0, if true, create a hinge like constraint

        if ( fixSwing == 1 )
        {
            int freeConstraintAxis = (bp->m_axisMode[ANGULAR_Y] == FIXED) ? 2 : 1;

            HK_ALIGN16(hkpAngConstraintAtom) angAtom;
            if ( fixTwist )
            {
                // we need to constraint axis (0,1) or (0,2)
                if ( freeConstraintAxis == 1 )
                {
                    angAtom.m_constrainedAxes[0] = 2;
                    angAtom.m_constrainedAxes[1] = 0;
                    angAtom.m_constrainedAxes[2] = 1;
                }
                else
                {
                    angAtom.m_constrainedAxes[0] = 0;
                    angAtom.m_constrainedAxes[1] = 1;
                    angAtom.m_constrainedAxes[2] = 2;
                }
                angAtom.m_numConstrainedAxes = 2;
            }
            else
            {
                if ( limitTwist )
                {
                    appendAtom( this, bp->m_twistLimit, AtomTypes::TWIST_LIMIT, Result::TWIST );
                }
                angAtom.m_constrainedAxes[0] = 2; 
                angAtom.m_constrainedAxes[1] = 0;
                angAtom.m_constrainedAxes[2] = hkUint8(freeConstraintAxis);
                angAtom.m_numConstrainedAxes = 1;
            }

            appendAtom( this, angAtom, AtomTypes::UNMAPPED, Result::MAX );
            if ( fixTwist )
            {
                // os.todo.aa Note the next lines could be in the wrong order, needs to be tested (also dependent on constraintAxis)
                appendRuntimeManual( this, Result::TWIST );
                appendRuntimeManual( this, freeConstraintAxis==1 ? Result::SWING0: Result::SWING1 );
            }
            else
            {
                appendRuntimeManual( this, Result::Enum(Result::TWIST+freeConstraintAxis) );
            }

            if ( bp->m_axisMode[ANGULAR_X + freeConstraintAxis] == LIMITED )
            {
                HK_ALIGN16(hkpAngLimitConstraintAtom) angLimitAtom;
                angLimitAtom.m_isEnabled = true;
                angLimitAtom.m_limitAxis = hkUint8(freeConstraintAxis);
                angLimitAtom.m_cosineAxis = 0;
                angLimitAtom.m_minAngle = bp->m_ellipticalMinMax[(freeConstraintAxis - 1) * 2];
                angLimitAtom.m_maxAngle = bp->m_ellipticalMinMax[(freeConstraintAxis - 1) * 2 + 1];
                angLimitAtom.m_angularLimitsDampFactor = 1.0f;
                angLimitAtom.m_angularLimitsTauFactor  = 0.8f;
                appendAtom( this, angLimitAtom, AtomTypes::UNMAPPED, freeConstraintAxis==1 ? Result::SWING1: Result::SWING0 );
            }
        }
        else
        {
            // twist limit
             if( fixTwist )
            {
                HK_ALIGN16(hkpAngConstraintAtom) angAtom;
                angAtom.m_constrainedAxes[0] = 0;
                angAtom.m_constrainedAxes[1] = 1;
                angAtom.m_constrainedAxes[2] = 2;
                angAtom.m_numConstrainedAxes = 1;
                appendAtom( this, angAtom, AtomTypes::UNMAPPED, Result::TWIST );
            }
            else if ( limitTwist )
            {
                appendAtom( this, bp->m_twistLimit, AtomTypes::TWIST_LIMIT, Result::TWIST );
            }

            // full elliptical limit
            {
                hkReal reorientAngle0 = bp->m_ellipticalMinMax[0] * 0.5f + bp->m_ellipticalMinMax[1] * 0.5f;
                hkReal reorientAngle1 = bp->m_ellipticalMinMax[2] * 0.5f + bp->m_ellipticalMinMax[3] * 0.5f;

                if(reorientAngle0 != 0.0f || reorientAngle1 != 0.0f)
                {
                    hkQuaternion adjust0;
                    adjust0.setAxisAngle( bp->m_transforms.m_transformB.getColumn<1>(), reorientAngle0 );
                    hkRotation a0; a0.set(adjust0);

                    hkQuaternion adjust1;
                    adjust1.setAxisAngle( bp->m_transforms.m_transformB.getColumn<2>(), reorientAngle1 );
                    hkRotation a1; a1.set(adjust1);

                    a1.mul(a0);
                    hkpSetLocalTransformsConstraintAtom extraTransforms;
                    extraTransforms.m_transformA = bp->m_transforms.m_transformA;
                    extraTransforms.m_transformB.setTranslation(bp->m_transforms.m_transformB.getTranslation());
                    extraTransforms.m_transformB.getRotation().setMul(a1, bp->m_transforms.m_transformB.getRotation());
                    appendAtom( this, extraTransforms, AtomTypes::UNMAPPED, Result::MAX );
                }

                m_blueprints.m_ellipticalLimit.m_angle0 = bp->m_ellipticalMinMax[1] - reorientAngle0;
                m_blueprints.m_ellipticalLimit.m_angle1 = bp->m_ellipticalMinMax[3] - reorientAngle1;
                m_blueprints.m_ellipticalLimit.makeStableByUsingConeAngle();
                appendAtom( this, bp->m_ellipticalLimit, AtomTypes::ELLIPTICAL_LIMIT, Result::SWING0 );

                if(reorientAngle0 != 0.0f || reorientAngle1 != 0.0f)
                {
                    appendAtom( this, bp->m_transforms, AtomTypes::UNMAPPED, Result::MAX );
                }
            }
        }
    }

    // -------------------------------------------------------------------------------------------------------------
    // Linear part
    // -------------------------------------------------------------------------------------------------------------

    int fixLinDof = (bp->m_axisMode[LINEAR_X] == FIXED) + (bp->m_axisMode[LINEAR_Y] == FIXED) + (bp->m_axisMode[LINEAR_Z] == FIXED);
    if ( (bp->m_stiffSpring.m_maxLength == 0.0f) && (bp->m_stiffSpring.m_springDamping < 0.0f) )
    {
        fixLinDof = 3;  // if our limit is 0 and no spring created, create a ball socket constraint
    }

    if ( fixLinDof == 3)
    {
        appendAtom( this, bp->m_ballSocket, AtomTypes::BALL_SOCKET, Result::LIN_0 );
    }
    else
    {
        for (int i=0; i < 3; i++ )
        {
            if (bp->m_axisMode[i] != FIXED )
            {
                if ( bp->linearMotor(i).m_isEnabled )
                {
                    appendAtom( this, bp->linearMotor(i), (AtomTypes::Enum)(AtomTypes::LINEAR_MOTOR0+i), Result::Enum(Result::LIN_MOTOR_0+2*i) );
                }
            }
        }

        {
            
            appendAtom( this, bp->m_stiffSpring, AtomTypes::STIFF_SPRING, Result::LIN_LIMIT );

            HK_ALIGN16(hkpLinConstraintAtom linAtom);
            for (int i=0;i<3;i++)
            {

                if (bp->m_axisMode[i] == FIXED)
                {
                    linAtom.m_axisIndex = hkUint8(i);
                    appendAtom( this, linAtom,  AtomTypes::UNMAPPED, Result::Enum(Result::LIN_0+i) );
                }
                else if(bp->m_axisMode[i] == LIMITED)
                {
                    appendAtom( this, m_blueprints.m_linearLimits[i], AtomTypes::Enum(AtomTypes::LINEAR_LIMIT_X+i), Result::Enum(Result::LIN_0+i) );
                }

                if(bp->m_linearFriction[i].m_maxFrictionForce != 0.0f && !bp->linearMotor(i).m_isEnabled)
                {
                    appendAtom( this, bp->m_linearFriction[i], AtomTypes::Enum(AtomTypes::LINEAR_FRICTION_X+i), Result::Enum(Result::LIN_MOTOR_0 + i * 2) );
                }
            }
        }
    }

    return hkpConstraintData::UpdateAtomsResult::RUNTIME_CHANGED;
}

//
//  Returns the linear impulse applied by the solver

void hkp6DofConstraintData::getAppliedLinearImpulse(const hkTransform& worldFromBodyA, const hkTransform& worldFromBodyB, _In_ const hkpConstraintRuntime* runtimeIn, _Out_ hkVector4& impulseOut) const
{
    const Runtime* rt = reinterpret_cast<const Runtime*>(runtimeIn);

    impulseOut.set(
        rt->get( hkp6DofConstraintData::Result::LIN_0, this ).getImpulseApplied(),
        rt->get( hkp6DofConstraintData::Result::LIN_1, this ).getImpulseApplied(),
        rt->get( hkp6DofConstraintData::Result::LIN_2, this ).getImpulseApplied()
        );
}

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
