// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Geometry/Internal/DataStructures/StaticTree/hkcdStaticTreeCodecs.h>

namespace hkcdStaticTree
{

    /// Dynamic storage
    template <typename CODEC>
    struct HK_EXPORT_COMMON DynamicStorage
    {
        HK_DECLARE_CLASS(DynamicStorage, Reflect, NewPlacement);
        HK_INLINE DynamicStorage() {}
        HK_INLINE DynamicStorage(const DynamicStorage& other)
        {
            m_nodes.append(other.m_nodes);
        }
        HK_INLINE ~DynamicStorage() {}

        typedef CODEC   Node;

        /// Clear the storage.
        HK_INLINE void      clear() { m_nodes.clear(); }

        /// Return the number of allocated nodes.
        HK_INLINE int           getNumNodes() const { return m_nodes.getSize(); }

        /// Reserve nodes.
        HK_INLINE void      reserveNodes(int numNodes) { m_nodes.setSize(numNodes); }

        /// Reserve leaves.
        HK_INLINE void      reserveLeaves(int numLeaves) { reserveNodes(numLeaves*2-1); }

        /// Get a packed node pointer.
        HK_INLINE const Node*   getNode(int index, bool useCache = false) const
        {
            {
                return &m_nodes[index];
            }
        }

        /// Get a packed node pointer.
        HK_INLINE Node*     getNode(int index) { return &m_nodes[index]; }

        /// Return the memory foot-print of the tree in bytes.
        HK_INLINE int           getMemoryFootPrint(int treeSize) const  { return treeSize + sizeof(Node) * m_nodes.getSize(); }

        /// Creates a shallow copy of the tree.
        inline void                 shallowCopy(const DynamicStorage<CODEC>& src);

        hkArray<Node>   m_nodes;    ///< Nodes.
    };

    //
    // Storage specializations
    //

        /// 4 bytes per node, up to 2^7 leaves and 7 bits data range.
    typedef DynamicStorage<Codec3Axis4> DynamicStorage4;

        /// 5 bytes per node, up to 2^15 leaves and 15 bits data range.
    typedef DynamicStorage<Codec3Axis5> DynamicStorage5;

        /// 6 bytes per node, up to 2^23 leaves and 23 bits data range.
    typedef DynamicStorage<Codec3Axis6> DynamicStorage6;

        /// 32 bytes per node, full precision, up to 2^23 leaves and 23 bits data range.
    typedef DynamicStorage<CodecRaw> DynamicStorage32;

    #ifdef HK_DYNAMIC_DLL  // then as we are exporting the templates, we want to export the specializations
        HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON DynamicStorage< Codec3Axis4 >;
        HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON DynamicStorage< Codec3Axis5 >;
        HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON DynamicStorage< Codec3Axis6 >;
        HK_EXPORT_COMMON_TEMPLATE_SPECIALIZATION template struct HK_EXPORT_COMMON DynamicStorage< CodecRaw >;
    #endif

}

HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DynamicStorage4, hkcdStaticTree_DynamicStorage4);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DynamicStorage5, hkcdStaticTree_DynamicStorage5);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DynamicStorage6, hkcdStaticTree_DynamicStorage6);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdStaticTree::DynamicStorage32, hkcdStaticTree_DynamicStorage32);

#include <Geometry/Internal/_Auto/TemplateTypes/hkcdStaticTreeStorages_Types.inl>

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
