// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

/// Simple ray cast query implementation.
struct RayCastQuery : public BaseUnaryQuery
{
    HK_INLINE RayCastQuery(_In_ const hkcdStaticMeshTree* mesh, hkBool ts, const hkSimdReal& fraction) : BaseUnaryQuery(mesh), m_fraction(fraction), m_hitKey((hkUint32)INVALID_PRIMITIVE_KEY), m_flags(ts ? hkcdRayQueryFlags::NO_FLAGS : hkcdRayQueryFlags::DISABLE_BACK_FACING_TRIANGLE_HITS), m_layer(0) {}

    HK_INLINE   hkSimdReal  getFraction() const { return m_fraction; }

    HK_INLINE bool          isDone() const { return false; }

    HK_INLINE   hkBool32    enterSubTree(const hkcdStaticMeshTree& mesh, _Pre_valid_ _Outref_ const Section*& section, const NodeContext& context, hkcdRay&) { return BaseUnaryQuery::enterSubTree(mesh, section, context); }

    HK_INLINE   void        leaveSubTree(const Section& section, const hkcdStaticMeshTree& mesh, const hkcdRay&) const {}

    HK_INLINE bool      processLeaf(int primitiveIndex, const hkcdRay& ray)
    {
        if(BaseUnaryQuery::m_decoder.isPrimitiveTriangleOrQuad(primitiveIndex))
        {
            hkVector4   vertices[4]; BaseUnaryQuery::m_decoder.decodePrimitive(primitiveIndex, vertices);
            hkSimdReal  fractions[2] = { m_fraction, m_fraction };
            hkVector4   normals[2];

            hkBool32    t0 = hkcdRayCastTriangle::safeUsingDynamicTolerance(ray, vertices[0], vertices[1], vertices[2], m_flags, fractions[0], normals[0] ).isHit();
            hkBool32    t1 = hkcdRayCastTriangle::safeUsingDynamicTolerance(ray, vertices[0], vertices[2], vertices[3], m_flags, fractions[1], normals[1] ).isHit();
            if(t0 | t1)
            {
                const int triangleIndex = fractions[0] < fractions[1] ? 0 : 1;
                m_hitKey = BaseUnaryQuery::m_decoder.getPrimitiveKey(primitiveIndex, triangleIndex);
                m_normal = normals[triangleIndex];
                m_layer = BaseUnaryQuery::m_decoder.m_section->m_layerData;
                m_fraction = fractions[triangleIndex];
                return true;
            }
        }
        return false;
    }

    HK_INLINE void      processLeaf(const Section::NodeContext& leaf, const hkcdRay& ray)
    {
        processLeaf(leaf->getData(), ray);
    }

    HK_INLINE bool      processSimdTreeLeaf(hkUint32 key, const hkcdRay& ray)
    {
        ExpandedPrimitiveKey expKey(key);
        if(BaseUnaryQuery::m_decoder.m_sectionIndex != expKey.m_section)
        {
            BaseUnaryQuery::m_decoder.setSection(expKey.m_section);
            BaseUnaryQuery::m_decoder.waitForCompletion();
        }
        return processLeaf(expKey.m_primitive, ray);
    }

    hkVector4               m_normal;
    hkSimdReal              m_fraction;
    hkUint32                m_hitKey;
    hkFlags<hkcdRayQueryFlags::Enum,hkUint32>   m_flags;
    int                     m_layer;
};

/// Unsigned distance from point query.
struct UnsignedDistanceQuery : public BaseUnaryQuery
{
    HK_INLINE UnsignedDistanceQuery(_In_opt_ const hkcdStaticMeshTree* mesh, const hkVector4& prj)
        : BaseUnaryQuery(mesh)
        , m_projection(prj)
        , m_key(HKCD_INVALID_KEY) {}

    inline void             setInitialKey(const hkVector4& x, hkUint32 key)
    {
        hkVector4           v[3]; BaseUnaryQuery::m_decoder.m_mesh->getTriangleVertices(key,v);
        hkVector4           p;
        const hkSimdReal    distanceToQuery = hkcdPointTriangleProject(x, v[0], v[1], v[2], &p);
        if(distanceToQuery.isLess(m_projection.getW()))
        {
            m_key           =   key;
            m_projection.setXYZ_W(p, distanceToQuery);
        }
    }

    HK_INLINE bool          isDone() const { return false; }

    HK_INLINE void  processLeaf(int localIndex, const hkVector4& x, hkSimdReal& distanceSquared)
    {
        if(BaseUnaryQuery::m_decoder.isPrimitiveTriangleOrQuad(localIndex))
        {
            hkVector4   vertices[4];
            const int   numTriangles = BaseUnaryQuery::m_decoder.decodePrimitive(localIndex, vertices);
            for(int i=0; i<numTriangles; ++i)
            {
                hkVector4 projection;
                const hkSimdReal    distanceToQuery = hkcdPointTriangleProject(x, vertices[0], vertices[1+i], vertices[2+i], &projection);
                if(distanceToQuery.isLess(distanceSquared))
                {
                    distanceSquared =   distanceToQuery;
                    m_key           =   BaseUnaryQuery::m_decoder.getPrimitiveKey(localIndex, i);
                    m_projection.setXYZ_W( projection, distanceToQuery);
                }
            }
        }
    }

    HK_INLINE void  processLeaf(const Section::NodeContext& leaf, const hkVector4& x, hkSimdReal& distanceSquared)
    {
        processLeaf(leaf->getData(), x, distanceSquared);
    }

    HK_INLINE void  processSimdTreeLeaf(hkUint32 key, const hkVector4& x, hkSimdReal& distanceSquared)
    {
        const hkcdStaticMeshTreeBase::ExpandedPrimitiveKey expKey(key);
        BaseUnaryQuery::m_decoder.setSection(expKey.m_section);
        BaseUnaryQuery::m_decoder.waitForCompletion();

        processLeaf(expKey.m_primitive, x, distanceSquared);
    }

    hkVector4   m_projection;
    hkUint32    m_key;
};

/// Unsigned distance from point query.
struct UnsignedDistanceQueryWithNmp : public UnsignedDistanceQuery
{
    HK_INLINE UnsignedDistanceQueryWithNmp(_In_opt_ const hkcdStaticMeshTree* mesh, const hkVector4& prj) : UnsignedDistanceQuery(mesh, prj)
    {
        m_nextProjection = UnsignedDistanceQuery::m_projection;
        m_nextKey = UnsignedDistanceQuery::m_key;
    }

    HK_INLINE bool      isDone() const { return false; }

    HK_INLINE void  setInitialKey(const hkVector4& x, hkUint32 key)
    {
        UnsignedDistanceQuery::setInitialKey(x, key);
        m_nextProjection = UnsignedDistanceQuery::m_projection;
        m_nextKey = UnsignedDistanceQuery::m_key;
    }

    HK_INLINE void  processLeaf(int localIndex, const hkVector4& x, hkSimdReal& distanceSquared)
    {
        if ( BaseUnaryQuery::m_decoder.isPrimitiveTriangleOrQuad( localIndex ) )
        {
            hkVector4   vertices[ 4 ];
            const int   numTriangles = BaseUnaryQuery::m_decoder.decodePrimitive( localIndex, vertices );
            for ( int triangleIndex = 0; triangleIndex < numTriangles; ++triangleIndex )
            {
                hkVector4   projection;
                hkSimdReal  distanceToQuery = hkcdPointTriangleProject( x, vertices[ 0 ], vertices[ triangleIndex + 1 ], vertices[ triangleIndex + 2 ], &projection );
                if ( distanceToQuery.isLess( distanceSquared ) )
                {
                    const hkUint32 key = BaseUnaryQuery::m_decoder.getPrimitiveKey( localIndex, triangleIndex );
                    const hkSimdReal nearest = UnsignedDistanceQuery::m_projection.getW();
                    if ( distanceToQuery.isLess( nearest ) )
                    {
                        m_nextKey = UnsignedDistanceQuery::m_key;
                        m_nextProjection = UnsignedDistanceQuery::m_projection;

                        UnsignedDistanceQuery::m_key = key;
                        UnsignedDistanceQuery::m_projection.setXYZ_W( projection, distanceToQuery );

                        distanceSquared = nearest;
                    }
                    else
                    {
                        m_nextKey = key;
                        m_nextProjection.setXYZ_W( projection, distanceToQuery );

                        distanceSquared = distanceToQuery;
                    }
                }
            }
        }
    }

    HK_INLINE void  processLeaf(const Section::NodeContext& leaf, const hkVector4& x, hkSimdReal& distanceSquared)
    {
        processLeaf(leaf->getData(), x, distanceSquared);
    }

    HK_INLINE void  processSimdTreeLeaf(hkUint32 key, const hkVector4& x, hkSimdReal& distanceSquared)
    {
        const hkcdStaticMeshTreeBase::ExpandedPrimitiveKey expKey(key);
        BaseUnaryQuery::m_decoder.setSection(expKey.m_section);
        BaseUnaryQuery::m_decoder.waitForCompletion();

        processLeaf(expKey.m_primitive, x, distanceSquared);
    }

    hkVector4   m_nextProjection;
    hkUint32    m_nextKey;
};

/// Collect in range query.
struct CollectInRangeQuery : public BaseUnaryQuery
{
    HK_INLINE   CollectInRangeQuery(_In_opt_ const hkcdStaticMeshTree* mesh, const hkVector4& pointAndRange, _In_count_(maxKeys) hkUint32* keys, int maxKeys)
        : BaseUnaryQuery(mesh)
    {
        m_point         =   pointAndRange;
        m_rangeSquared  =   m_point.getComponent<3>() * m_point.getComponent<3>();
        m_keys          =   keys;
        m_maxKeys       =   maxKeys;
        m_count         =   0;
        m_overflow      =   false;
    }

    template <typename TREE, typename CONTEXT>
    HK_INLINE hkBool32  processNode(const CONTEXT& node) const
    {
        if(m_overflow)  return 0;

        hkVector4   x; x.setClamped(m_point, node.m_aabb.m_min, node.m_aabb.m_max);

        return x.distanceToSquared(m_point).isLessEqual(m_rangeSquared);
    }

    HK_INLINE void      processLeaf(const Section::NodeContext& leaf)
    {
        if(!m_overflow)
        {
            const int   localIndex = leaf->getData();
            if(BaseUnaryQuery::m_decoder.isPrimitiveTriangleOrQuad(localIndex))
            {
                hkVector4   vertices[4];
                const int   numTriangles = BaseUnaryQuery::m_decoder.decodePrimitive(localIndex, vertices);
                for(int i=0; i<numTriangles; ++i)
                {
                    hkVector4   projection;
                    hkSimdReal  distanceToQuery = hkcdPointTriangleProject( m_point, vertices[ 0 ], vertices[ i + 1 ], vertices[ i + 2 ], &projection );
                    if(distanceToQuery.isLessEqual(m_rangeSquared))
                    {
                        if(m_count < m_maxKeys)
                        {
                            m_keys[m_count++] = BaseUnaryQuery::m_decoder.getPrimitiveKey(localIndex, i);
                        }
                        else
                        {
                            m_overflow = true;
                            break;
                        }
                    }
                }
            }
        }
    }

    hkVector4   m_point;
    hkSimdReal  m_rangeSquared;
    hkUint32*   m_keys;
    int         m_maxKeys;
    int         m_count;
    bool        m_overflow;
};

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
