// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

//
// Perform occlusion query.
//
struct OcclusionQuery : public BaseUnaryQuery
{
    enum States
    {
        ST_UNPROCESSED  = 0,
        ST_PENDING      = 1,
        ST_PROCESSED    = 2,
    };

    struct Triangle
    {
        hkUint32    m_key;
        hkUint32    m_state;
        bool        m_hit;
        hkVector4   m_vertices[3];
    };

    enum IndexType
    {
        IT_UNMAPPED =   -1,
        IT_EXCLUDED =   -2
    };

    //
    OcclusionQuery()
    {
        m_triangles.reserve(1024);
    }

    //
    HK_INLINE bool isLinePenetratingPlanesSet(const hkVector4& a, const hkVector4& b) const
    {
        hkVector4   endPoints[]={a,b};
        for(int i=0; i<m_planes.getSize(); ++i)
        {
            hkSimdReal  d0 = m_planes[i].dot4xyz1(endPoints[0]);
            hkSimdReal  d1 = m_planes[i].dot4xyz1(endPoints[1]);
            if((d0 * d1).isLessZero())
            {
                endPoints[d0 > d1? 0 : 1].setInterpolate(endPoints[0], endPoints[1], d0 / (d0 - d1));
            } else if(!d0.isLessZero()) return false;
        }
        return true;
    }

    //
    static HK_INLINE void   buildTriangleAabb(const hkVector4& a, const hkVector4& b, const hkVector4& c, hkAabb& aabb)
    {
        aabb.m_min.setMin(a, b);
        aabb.m_max.setMax(a, b);
        aabb.m_min.setMin(aabb.m_min, c);
        aabb.m_max.setMax(aabb.m_max, c);
    }

    //
    HK_INLINE   hkBool32    enterSubTree(const hkcdStaticMeshTree& mesh, const Section*& section, const NodeContext&, hkcdRay&)
    {
        const int sectionIndex = (int)(section - mesh.m_sections.begin());
        if(sectionIndex != m_exclude)
        {
            BaseUnaryQuery::m_decoder.setSection(sectionIndex);
            return 1;
        }
        return 0;
    }

    //
    HK_INLINE   void        leaveSubTree(const Section& section, const hkcdStaticMeshTree& mesh, hkcdRay&)
    {}

    //
    HK_INLINE void      processLeaf(const Section::NodeContext& leaf, hkcdRay& ray)
    {
        const int       primitive = leaf->getData();
        if(BaseUnaryQuery::m_decoder.isPrimitiveTriangleOrQuad(primitive))
        {
            hkVector4       vertices[4], dummyNormal;
            const int       numTriangles = BaseUnaryQuery::m_decoder.decodePrimitive(primitive, vertices);
            hkSimdReal      fraction0 = ray.getFraction();
            hkSimdReal      fraction1 = ray.getFraction();
            const hkBool32  hits[] = {  hkcdRayCastTriangle::fastUsingZeroTolerance(ray, vertices[0], vertices[1], vertices[2], hkcdRayQueryFlags::NO_FLAGS, fraction0, dummyNormal ).isHit(),
                                        hkcdRayCastTriangle::fastUsingZeroTolerance(ray, vertices[0], vertices[2], vertices[3], hkcdRayQueryFlags::NO_FLAGS, fraction1, dummyNormal ).isHit() };
            hkAabb          aabbs[2];
            buildTriangleAabb(vertices[0], vertices[1], vertices[2], aabbs[0]);
            buildTriangleAabb(vertices[0], vertices[2], vertices[3], aabbs[1]);
            for(int i=0; i<numTriangles; ++i)
            {
                if(hits[i] && !(aabbs[i].overlaps(m_cells[0]) | aabbs[i].overlaps(m_cells[1])))
                {
                    Triangle&   t   =   m_triangles.expandOne();
                    t.m_hit         =   1;
                    t.m_key         =   BaseUnaryQuery::m_decoder.getPrimitiveKey(primitive, i);
                    t.m_state       =   ST_UNPROCESSED;
                    t.m_vertices[0] =   vertices[0];
                    t.m_vertices[1] =   vertices[i+1];
                    t.m_vertices[2] =   vertices[i+2];

                    m_keyToTriangle[t.m_key]    =   m_triangles.getSize() - 1;
                }
            }
        }
    }

    //
    bool    isOccluded(_In_ const hkcdStaticMeshTree* mesh, _In_reads_(_Inexpressible_()) const Links* connectivity, const hkAabb& cellA, const hkAabb& cellB, int excludeSection)
    {
        if(!cellA.overlaps(cellB))
        {
            // Initialize.
            {
                m_cells[0]  =   cellA;
                m_cells[1]  =   cellB;
                m_exclude   =   excludeSection;

                // Cleanup planes.
                m_planes.clear();

                // Adjust key map if required.
                const int keyMapCapacity = mesh->m_maxKeyValue + 1;
                if(m_keyToTriangle.getSize() < keyMapCapacity)
                {
                    m_keyToTriangle.clear();
                    m_keyToTriangle.setSize(keyMapCapacity, IT_UNMAPPED);
                }
                else
                {
                    for(int i=0; i<m_triangles.getSize(); ++i) m_keyToTriangle[m_triangles[i].m_key] = IT_UNMAPPED;
                }
                m_triangles.clear();

                // Initialize decoder.
                BaseUnaryQuery::m_decoder.initialize(mesh, true);
            }

            // Build the convex planes of cellA || cellB.
            {
                // Some consts.
                hkSimdReal  zero = hkSimdReal::getConstant<HK_QUADREAL_0>();
                hkSimdReal  pos = hkSimdReal::getConstant<HK_QUADREAL_1>();
                hkSimdReal  neg = hkSimdReal::getConstant<HK_QUADREAL_MINUS1>();

                // Build cellA || cellB.
                hkAabb      compound;
                compound.m_min.setMin(cellA.m_min, cellB.m_min);
                compound.m_max.setMax(cellA.m_max, cellB.m_max);

                // Flatten cellA and cellB components.
                const hkSimdReal    flattenA[] = {  cellA.m_min.getComponent<0>(), cellA.m_min.getComponent<1>(), cellA.m_min.getComponent<2>(),
                                                    cellA.m_max.getComponent<0>(), cellA.m_max.getComponent<1>(), cellA.m_max.getComponent<2>() };
                const hkSimdReal    flattenB[] = {  cellB.m_min.getComponent<0>(), cellB.m_min.getComponent<1>(), cellB.m_min.getComponent<2>(),
                                                    cellB.m_max.getComponent<0>(), cellB.m_max.getComponent<1>(), cellB.m_max.getComponent<2>() };

                // Compute shaft planes lookup index.
                int typeField;
                typeField =     cellB.m_min.equal(compound.m_min).getMask<hkVector4ComparisonMask::MASK_XYZ>();
                typeField |=    cellB.m_max.equal(compound.m_max).getMask<hkVector4ComparisonMask::MASK_XYZ>() << 3;

                // Append shaft planes.
                const int       first = typeField > 31 ? 1 : 0;
                const hkUint8*  indices = s_aabbShaftsTable[hkMath::min2(typeField, 63 - typeField)];
                const int       numPlanes = *indices++;
                for(int i=0; i<numPlanes; ++i, indices += 2)
                {
                    const int   vertices[] = { indices[first], indices[1-first] };
                    const int   edge[] = { vertices[0] % 3, vertices[1] % 3 };
                    hkVector4   plane; plane.setZero();
                    plane.setComponent(edge[0], flattenA[vertices[1]] - flattenB[vertices[1]]);
                    plane.setComponent(edge[1], flattenB[vertices[0]] - flattenA[vertices[0]]);
                    plane.setComponent<3>( - (flattenA[vertices[0]] * plane.getComponent(edge[0]) + flattenA[vertices[1]] * plane.getComponent(edge[1])));
                    m_planes.pushBack(plane);
                }

                // Append outer planes.
                m_planes.expandOne().set(pos, zero, zero, -compound.m_max.getComponent<0>());
                m_planes.expandOne().set(neg, zero, zero,  compound.m_min.getComponent<0>());
                m_planes.expandOne().set(zero, pos, zero, -compound.m_max.getComponent<1>());
                m_planes.expandOne().set(zero, neg, zero,  compound.m_min.getComponent<1>());
                m_planes.expandOne().set(zero, zero, pos, -compound.m_max.getComponent<2>());
                m_planes.expandOne().set(zero, zero, neg,  compound.m_min.getComponent<2>());
            }

            // Extract pontential occluders roots.
            {
                hkVector4   from; cellA.getCenter(from);
                hkVector4   to; cellB.getCenter(to);
                hkcdDynamicStackTreeQueries::rayCast(*mesh, from, to, *this);
            }

            // Traverse connectivity and compute occlusion.
            {
                for(int triangleIndex=0, numRoots = m_triangles.getSize(); triangleIndex < numRoots; ++triangleIndex)
                {
                    // Traverse the connectivity.
                    if(m_triangles[triangleIndex].m_state == ST_UNPROCESSED)
                    {
                        bool    hasExternalBoundary = false;
                        bool    hasInternalBoundary = false;
                        bool    hasOddIntersections = false;
                        m_stack.pushBack(triangleIndex);
                        m_triangles[triangleIndex].m_state++;
                        do
                        {
                            // Pop triangle from the stack.
                            Triangle&   tr = m_triangles[m_stack.back()]; m_stack.popBack();

                            // Update triangle state.
                            HK_ASSERT(0x7B4B6664, tr.m_state == ST_PENDING, "Internal state error");
                            tr.m_state++;

                            // Update intersections state.
                            if(tr.m_hit) hasOddIntersections = !hasOddIntersections;

                            // Stack neighbor triangles.
                            const Links&    tl = connectivity[tr.m_key];
                            for(int ei=2,ej=0; ej<3; ei=ej++)
                            {
                                const hkcdStaticMeshTreeBase::Edge  keyEdge = tl.m_links[ei];
                                const bool  isInnerEdge = isLinePenetratingPlanesSet(tr.m_vertices[ei], tr.m_vertices[ej]);
                                const bool  isBoundEdge = keyEdge.isValid();
                                if(isBoundEdge)
                                {
                                    if(isInnerEdge)
                                    {
                                        int& linkIndex = m_keyToTriangle[keyEdge.key()];

                                        // Create triangle if not exist.
                                        if(linkIndex < 0)
                                        {
                                            // If already excluded, just leave scope.
                                            if(linkIndex == IT_EXCLUDED) continue;

                                            // Mark as an internal boundary if linked to an excluded section.
                                            if(keyEdge.key() >> 8 == (hkUint32) m_exclude)
                                            {
                                                linkIndex = IT_EXCLUDED;
                                                hasInternalBoundary = true;
                                                hkUint32 key = keyEdge.key();
                                                m_excludedKeys.pushBack(key);
                                                continue;
                                            }

                                            // Fetch vertices.
                                            hkVector4   vertices[3];
                                            const int   eli = (keyEdge.index() + 1) % 3;
                                            const int   elj = (keyEdge.index() + 2) % 3;
                                            vertices[keyEdge.index()]   =   tr.m_vertices[ej];
                                            vertices[eli]               =   tr.m_vertices[ei];
                                            BaseUnaryQuery::m_decoder.setSection(keyEdge.key()>>8);
                                            BaseUnaryQuery::m_decoder.decodeVertex(keyEdge.key(), elj, vertices[elj]);

                                            // Mark as an internal boundary if overlaps cells.
                                            hkAabb  aabb; buildTriangleAabb(vertices[0], vertices[1], vertices[2], aabb);
                                            if(aabb.overlaps(cellA) | aabb.overlaps(cellB))
                                            {
                                                linkIndex = IT_EXCLUDED;
                                                hasInternalBoundary = true;
                                                hkUint32 key = keyEdge.key();
                                                m_excludedKeys.pushBack(key);
                                                continue;
                                            }

                                            // Create new triangle.
                                            Triangle& t = m_triangles.expandOne();
                                            t.m_hit         =   0;
                                            t.m_key         =   keyEdge.key();
                                            t.m_state       =   ST_UNPROCESSED;
                                            t.m_vertices[0] =   vertices[0];
                                            t.m_vertices[1] =   vertices[1];
                                            t.m_vertices[2] =   vertices[2];
                                            linkIndex = m_triangles.getSize() - 1;
                                        }

                                        // Stack it up.
                                        Triangle& nt = m_triangles[linkIndex];
                                        if(nt.m_state == ST_UNPROCESSED)
                                        {
                                            nt.m_state++;
                                            m_stack.pushBack(linkIndex);
                                        }
                                    }
                                    else
                                    {
                                        // External naked edge, set hasExternalBoundary to true.
                                        hasExternalBoundary = true;
                                    }
                                }
                                else
                                {
                                    // Internal naked edge, set hasInternalBoundary to true.
                                    if(isInnerEdge)
                                        hasInternalBoundary = true;
                                    else
                                        hasExternalBoundary = true;
                                }
                            }
                        } while(m_stack.getSize());

                        // Cleanup lookups and triangles.
                        for(int i=numRoots; i<m_triangles.getSize(); ++i)   m_keyToTriangle[m_triangles[i].m_key] = IT_UNMAPPED;
                        for(int i=0; i<m_excludedKeys.getSize(); ++i)       m_keyToTriangle[m_excludedKeys[i]] = IT_UNMAPPED;
                        m_triangles.popBack(m_triangles.getSize() - numRoots);
                        m_excludedKeys.clear();

                        // Check visibility conditions.
                        if(hasExternalBoundary && hasOddIntersections && !hasInternalBoundary)
                        {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    hkInplaceArray<int, 256>        m_stack;
    hkInplaceArray<hkUint32, 256>   m_excludedKeys;
    hkInplaceArray<hkVector4, 14>   m_planes;
    hkArray<int>                    m_keyToTriangle;
    hkArray<Triangle>               m_triangles;
    hkAabb                          m_cells[2];
    int                             m_exclude;
};

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
