// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

/// hkGeometry provider.
struct  DefaultGeometryProvider
{
    HK_INLINE DefaultGeometryProvider(const hkGeometry& geometry, _In_opt_count_(_Inexpressible_()) const hkUint8* layers = HK_NULL) : m_geometry(geometry), m_layers(layers) {}
    HK_INLINE   void    overrideDomain(hkAabb& domain) const {}
    HK_INLINE   int     getNumVertices() const { return m_geometry.m_vertices.getSize(); }
    HK_INLINE   int     getNumTriangles() const { return m_geometry.m_triangles.getSize(); }
    HK_INLINE void  getVertex(int vi, hkVector4& vertexOut) const { vertexOut = m_geometry.m_vertices[vi]; }
    HK_INLINE void  getIndices(int ti, _Out_writes_all_(3) int* indices) const { const hkGeometry::Triangle& t = m_geometry.m_triangles[ti]; indices[0] = t.m_a; indices[1] = t.m_b; indices[2] = t.m_c; }
    template <typename T>
    HK_INLINE void  getTriangleData(int ti, T& data) const { data = (T) m_geometry.m_triangles[ti].m_material; }

    HK_INLINE hkUint8   getTriangleLayerData(int ti) const { return m_layers ? m_layers[ti] : 0; }

    HK_INLINE int       getNumCustomPrimitives() const { return 0; }
    HK_INLINE void  getCustomPrimitiveInfos(int cpi, BuildCustomInfos& infos) const { HK_ERROR(0x74786ED6, "Not supported"); }
    HK_INLINE void  getCustomPrimitiveVertex(int cpi, int vi, hkVector4& vertexOut) const { vertexOut.setZero(); HK_ERROR(0x74786ED6, "Not supported"); }
    template <typename T>
    HK_INLINE   void    getCustomPrimitiveData(int cpi, T& data) const { HK_ERROR(0x74786ED6, "Not supported"); }

    const hkGeometry&   m_geometry;
    const hkUint8*      m_layers;
};

/// Build geometry provider.
/// Used as a wrapper around the user provided one during build.
template <typename PROVIDER>
struct BuildGeometryProvider : public hkGeometryUtils::IVertices
{
    HK_INLINE   BuildGeometryProvider(const PROVIDER& provider) : m_provider(provider)
    {
        m_vMap.setSize(provider.getNumVertices());
        for(int i=0; i<m_vMap.getSize(); ++i) m_vMap[i] = i;

        m_validTriangles.resize(0, provider.getNumTriangles());
        m_validTriangles.assignAll(1);
    }

    HK_INLINE   void    overrideDomain(hkAabb& domain) const { m_provider.overrideDomain(domain); }
    HK_INLINE   int     getNumVertices() const { return m_provider.getNumVertices(); }
    HK_INLINE   int     getNumTriangles() const { return m_provider.getNumTriangles(); }
    HK_INLINE void  getVertex(int vi, hkVector4& vertexOut) const { m_provider.getVertex(vi, vertexOut); }
    HK_INLINE void  getIndices(int ti, _Out_writes_all_(3) int* indices) const { m_provider.getIndices(ti,indices); indices[0] = m_vMap[indices[0]]; indices[1] = m_vMap[indices[1]]; indices[2] = m_vMap[indices[2]]; }
    template <typename T>
    HK_INLINE void  getTriangleData(int ti, T& data) const { m_provider.getTriangleData(ti,data); }

    HK_INLINE hkUint8   getTriangleLayerData(int ti) const { return m_provider.getTriangleLayerData(ti); }

    HK_INLINE int       getNumCustomPrimitives() const { return m_provider.getNumCustomPrimitives(); }
    HK_INLINE void  getCustomPrimitiveInfos(int cpi, BuildCustomInfos& infos) const { m_provider.getCustomPrimitiveInfos(cpi,infos); }
    HK_INLINE void  getCustomPrimitiveVertex(int cpi, int vi, hkVector4& vertexOut) const { m_provider.getCustomPrimitiveVertex(cpi,vi,vertexOut); }
    template <typename T>
    HK_INLINE   void    getCustomPrimitiveData(int cpi, T& data) const { m_provider.getCustomPrimitiveData(cpi,data); }

    HK_INLINE void  getVertices(int ti, _Inout_updates_all_(3) hkVector4* verticesOut) const { int i[3]; getIndices(ti,i); getVertex(i[0],verticesOut[0]); getVertex(i[1],verticesOut[1]); getVertex(i[2],verticesOut[2]); }
    HK_INLINE int       getIndex(int ti, int vi) const                      { int i[3]; getIndices(ti,i); return i[vi]; }
    HK_INLINE int       isValid(int ti) const                               { return m_validTriangles.get(ti); }
    HK_INLINE void  setValid(int ti)                                    { m_validTriangles.set(ti); }
    HK_INLINE void  clrValid(int ti)                                    { m_validTriangles.clear(ti); }

    const PROVIDER& m_provider;         ///< User provider.
    hkArray<int>    m_vMap;             ///< Vertices index map (original to welded).
    hkBitField  m_validTriangles;   ///< Valid triangles bitfield.
};

/// Master tree used to build sub-trees.
struct DynamicTree : public hkcdDynamicTree::DefaultTreePtr
{};

/// TriangleRef.
struct  TriangleRef
{
    HK_INLINE TriangleRef() {}
    HK_INLINE TriangleRef(int a, int b, int c, int i = -1)
    {
        int mi = a<b? (a<c? 0 : 2) : (b<c? 1 : 2);
        m_indices[0]    =   a;
        m_indices[1]    =   b;
        m_indices[2]    =   c;
        m_index         =   i;
        while(mi--)
        {
            const int t=m_indices[0];
            m_indices[0] = m_indices[1];
            m_indices[1] = m_indices[2];
            m_indices[2] = t;
        }
        m_hash = (m_indices[0] * 56320463) ^ (m_indices[1] * 74312107) ^ (m_indices[2] * 92552963);
    }

    HK_INLINE int   getHash() const { return m_hash; }

    HK_INLINE bool  operator==(const TriangleRef& other) const  {   return  other.m_indices[0] == m_indices[0] &&
                                                                                    other.m_indices[1] == m_indices[1] &&
                                                                                    other.m_indices[2] == m_indices[2]; }

    int             m_indices[3];   ///< Original triangle vertex indices.
    int             m_index;        ///< Original triangle index.
    int             m_hash;         ///< Hash code.
};

/// EdgeBase
struct  EdgeBase
{
    HK_INLINE   EdgeBase() {}
    HK_INLINE   EdgeBase(int _t,int _s) : m_t((hkUint32)_t), m_s((hkUint32)_s) {}
    HK_INLINE int   t() const { return (int) m_t; }
    HK_INLINE int   s() const { return (int) m_s; }
    HK_INLINE int   s(int i) const { return (((int) m_s) + i) % 3; }

    hkUint32    m_t:30; ///< Triangle.
    hkUint32    m_s:2;  ///< Side.
};

/// HaldEdge, used to build quads.
struct  HalfEdge
{
    HK_INLINE   HalfEdge() {}
    HK_INLINE   HalfEdge(int a, int b, int t, int s) : m_va(hkMath::min2(a,b)), m_vb(hkMath::max2(a,b)), m_e(t,s) {}

    HK_INLINE bool operator<(const HalfEdge& other) const { return compare(other) < 0; }
    HK_INLINE bool operator==(const HalfEdge& other) const { return compare(other) == 0; }

    HK_INLINE int compare(const HalfEdge& other) const
    {
        if(m_va < other.m_va) return -1;
        if(m_va > other.m_va) return +1;
        if(m_vb < other.m_vb) return -1;
        if(m_vb > other.m_vb) return +1;
        return 0;
    }

    int         m_va;
    int         m_vb;
    EdgeBase    m_e;
};

/// Quad.
struct  Quad
{
    HK_INLINE   Quad() {}

    HK_INLINE hkBool32  operator<(const Quad& other) const { return m_value < other.m_value; }

    template <typename GEOMETRY_PROVIDER>
    HK_INLINE int           index(const GEOMETRY_PROVIDER& provider, int i) const
    {
        switch(i)
        {
        case    0:  return provider.getIndex(m_edges[0].t(), m_edges[0].s(1));
        case    1:  return provider.getIndex(m_edges[0].t(), m_edges[0].s(2));
        case    2:  return provider.getIndex(m_edges[1].t(), m_edges[1].s(1));
        case    3:  return provider.getIndex(m_edges[1].t(), m_edges[1].s(2));
        }
        return -1;
    }

    EdgeBase    m_edges[2];
    hkBool      m_isFlatConvex; ///< set if the quad is flat and convex and this information wants to be encoded
    hkReal      m_value;
};

/// PrimitiveMap.
struct  PrimitiveMap
{
    int     m_sources[2];
};

/// Data sorter.
struct  DataSorter
{
    DataSorter() {}
    DataSorter(_In_ const TriangleDataType* data) : m_data(data) {}

    HK_INLINE bool operator()(int x, int y) const { return m_data[x] < m_data[y]; }

    const TriangleDataType* m_data;
};

/// Morton code sorter.
struct  MortonSorter
{
    HK_INLINE bool  operator()(int x, int y) const { return m_codes[x] < m_codes[y]; }

    const int*  m_codes;
};

/// SubTree.
struct  SubTree
{
    /// VertexCounter
    template <typename GEOMETRY_PROVIDER>
    struct VertexCounter
    {
        HK_INLINE   VertexCounter(const GEOMETRY_PROVIDER& provider, const hkArray<Quad>& quads) : m_provider(provider), m_quads(quads) {}

        template <typename TREE, typename INDEX>
        HK_INLINE void append(_In_ const TREE* tree, INDEX leaf);

        const GEOMETRY_PROVIDER&        m_provider;
        const hkArray<Quad>&            m_quads;
        hkFixedInplaceArray<int,256>    m_vertices;
    };

    HK_INLINE SubTree() : m_node(0), m_layer(-1), m_numLeaves(0), m_numVertices(-1) {}
    HK_INLINE SubTree(const DynamicTree& tree, typename DynamicTree::Index node, int layer) : m_node(node), m_layer(layer) { m_numLeaves = tree.countLeaves(node); m_numVertices = -1; }

    template <typename GEOMETRY_PROVIDER>
    inline int  countVertices(const DynamicTree& tree, const GEOMETRY_PROVIDER& provider, const hkArray<Quad>& quads) const;

    typename DynamicTree::Index m_node;
    int                         m_layer;
    int                         m_numLeaves;
    mutable int                 m_numVertices;
};

/// BranchLeavesRemapCollector
struct  BranchLeavesRemapCollector
{
    HK_INLINE BranchLeavesRemapCollector() : m_index(0) {}

    template <typename TREE, typename INDEX>
    HK_INLINE void append(_In_ const TREE* tree, INDEX leaf) { const_cast<TREE*>(tree)->getNode(leaf)->setData((INDEX)(m_index++)); }

    int m_index;
};

/// Edge, used to build connectivity.
struct  BuildEdge
{
    HK_DECLARE_CLASS(Edge, NewOpaque);

    /// Actually check for match, not raw equality.
    HK_INLINE bool operator==(const BuildEdge& other) const
    {
        return other.m_vertices[0] == m_vertices[1] && other.m_vertices[1] == m_vertices[0];
    }

    hkUint32    m_vertices[2];  ///< Vertice ids.
    hkUint32    m_key:30;       ///< Primitive key.
    hkUint32    m_idx:2;        ///< To which side this edge belong to.
};

/// RootToSectionData
struct  RootToSectionData
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE, RootToSectionData);

    RootToSectionData() { }
    HK_INLINE   int numPrimitives() const { return m_data.getSize(); }

    hkAabb                                                  m_aabb;
    hkFixedInplaceArray<hkUlong,128>                        m_data;
    hkFixedInplaceArray<typename DynamicTree::Index,128>    m_leaves;
};

/// LeavesDataCollector
struct  LeavesDataCollector
{
    HK_INLINE LeavesDataCollector(RootToSectionData& rts) : m_rts(rts) {}

    template <typename TREE, typename INDEX>
    HK_INLINE void append(_In_ const TREE* tree, INDEX leaf) { m_rts.m_leaves.pushBack(leaf); m_rts.m_data.pushBack((hkUlong) tree->getNode(leaf)->getData()); }

    RootToSectionData&  m_rts;
};

/// Vertex mapping.
struct  VertexMap
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_COLLIDE, VertexMap);

    HK_INLINE   VertexMap() : m_section(-1), m_index(-1) {}

    int m_section;  ///< Section to which this vertex belong to or: -1 if unassigned, -2 if shared.
    int m_index;    ///< Local or global index, or -1 if unassigned.
};

/// Utilities.
struct  Utilities
{
    static HK_INLINE hkSimdReal volume(const hkAabb& aabb)
    {
        hkVector4   e; e.setSub(aabb.m_max, aabb.m_min);
        return e.getComponent<0>() * e.getComponent<1>() * e.getComponent<2>();
    }

    static HK_INLINE hkSimdReal surface(const hkAabb& aabb)
    {
        hkVector4   e; e.setSub(aabb.m_max, aabb.m_min);
        hkVector4 eYZX; eYZX.setPermutation<hkVectorPermutation::YZXW>(e);
        return e.dot<3>(eYZX);
    }

    static HK_INLINE hkSimdReal intersectionSurface(const hkAabb& a, const hkAabb& b)
    {
        hkAabb  isec;
        isec.m_min.setMax(a.m_min, b.m_min);
        isec.m_max.setMin(a.m_max, b.m_max);
        return surface(isec);
    }

    //
    // Build steps.
    //

    //
    struct QuantizersInfos
    {
        enum { COUNT = CM_AUTO };

        HK_INLINE QuantizersInfos() : m_numTags(0) { for(int i=0; i<COUNT; ++i) m_size[i] = m_count[i] = 0; }
        int m_size[COUNT];
        int m_count[COUNT];
        int m_numTags;
    };

    // 1. Process geometry.
    // 2. Build master tree.
    template <typename GEOMETRY_PROVIDER>
    static HK_NEVER_INLINE  bool    buildStep12(    const BuildConfig& config,
                                                    GEOMETRY_PROVIDER& geometry,
                                                    BuildMapping& mapping,
                                                    hkArray<Quad>& quads,
                                                    hkArray<SubTree>& roots,
                                                    DynamicTree& masterTree);

    // 3. Build the static mesh tree.
    template <typename GEOMETRY_PROVIDER>
    static HK_NEVER_INLINE  bool    buildStep3(     const BuildConfig& config,
                                                    QuantizersInfos& qInfos,
                                                    GEOMETRY_PROVIDER& geometry,
                                                    BuildMapping& mapping,
                                                    hkArray<Quad>& quads,
                                                    hkArray<SubTree>& roots,
                                                    DynamicTree& masterTree,
                                                    hkcdStaticMeshTree& mesh);
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
