// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

#include <Geometry/Internal/DataStructures/DynamicTree/hkcdDynamicTreeCodecs.h>
#include <Geometry/Internal/DataStructures/DynamicTree/hkcdDynamicTreeMetrics.h>

namespace hkcdDynamicTree
{
    ///
    /// Inplace storage.
    ///
    template <int MAX_LEAVES, typename METRIC, typename CODEC>
    struct  InplaceStorage : public METRIC
    {
        HK_DECLARE_CLASS(InplaceStorage, New, Reflect, Pod);

        typedef METRIC                  Metric;     ///< Metric.
        typedef CODEC                   Node;       ///< Node.
        typedef typename CODEC::Index   Index;      ///< Index.
        typedef typename CODEC::PackAabb    PackAabb;   ///< Storage AABB.
        typedef typename CODEC::RawAabb     RawAabb;    ///< Processing AABB.

        enum { CAPACITY = MAX_LEAVES*2, LEAF_CAPACITY = MAX_LEAVES };

        HK_INLINE               InplaceStorage()                    { releaseAll(); }
        HK_INLINE _Ret_maybenull_ const Node* getNodesBuffer() const { return m_nodes; }
        HK_INLINE _Ret_maybenull_ Node* getNodesBuffer() { return m_nodes; }
        HK_INLINE int           getNodeCapacity() const             { return (int) CAPACITY; }
        HK_INLINE void      exchange(InplaceStorage& other)     { hkMath::swap(m_firstFree,other.m_firstFree); for(int i=0;i<CAPACITY;++i) hkMath::swap(m_nodes[i],other.m_nodes[i]); }
        HK_INLINE void      reserveLeaves(int numAdditionalLeaves) const    {}
        HK_NEVER_INLINE void        reserveNodes(int numAdditionalNodes) const      {}
        HK_INLINE Index     getFirstFreeNode() const            { return m_firstFree; }
        HK_INLINE Index     getNextFreeNode(Index node) const   { return (const Index&)m_nodes[(int)node]; }
        HK_NEVER_INLINE void        releaseAll()
        {
            hkString::memSet(m_nodes, 0, sizeof(Node) * CAPACITY);
            for (int i = 1; i < (CAPACITY-1); ++i) ((Index&)m_nodes[i]) = (Index)(i+1);
            m_firstFree                     = 1;
            ((Index&)m_nodes[CAPACITY-1])   = 0;
        }
        HK_INLINE Index     allocateNode()
        {
            HK_COMPILE_TIME_ASSERT(sizeof(Node) >= sizeof(Index));
            HK_ASSERT(0x6CFD82A7, m_firstFree, "Out of memory");
            Index newNode   = m_firstFree;
            m_firstFree     = (Index&)m_nodes[(int)m_firstFree];
            return newNode;
        }
        HK_INLINE Index     tryAllocateNode(hkResult & res)
        {
            HK_COMPILE_TIME_ASSERT(sizeof(Node) >= sizeof(Index));
            if (HK_VERY_UNLIKELY(!m_firstFree))
            {
                res = HK_FAILURE;
                return 0;
            }
            else
            {
                res = HK_SUCCESS;
            }
            HK_ASSERT(0x6CFD82A7, m_firstFree, "Out of memory");
            Index newNode   = m_firstFree;
            m_firstFree     = (Index&)m_nodes[(int)m_firstFree];
            return newNode;
        }
        HK_INLINE void      releaseNode(Index node)
        {
            HK_ASSERT(0xC53DB6F6, node, "Null node");
            ((Index&)m_nodes[(int)node])    = m_firstFree;
            m_firstFree                     = node;
        }

        Node    m_nodes[InplaceStorage::CAPACITY];  ///< Nodes.
        Index   m_firstFree;        ///< First free node.
    };

    ///
    /// Extern storage.
    ///
    template <typename METRIC, typename CODEC>
    struct  ExternStorage : public METRIC
    {
        HK_DECLARE_CLASS(ExternStorage, New, Reflect, Pod);

        typedef METRIC                  Metric;     ///< Metric.
        typedef CODEC                   Node;       ///< Node.
        typedef typename CODEC::Index   Index;      ///< Index.
        typedef typename CODEC::PackAabb    PackAabb;   ///< Storage AABB.
        typedef typename CODEC::RawAabb     RawAabb;    ///< Processing AABB.

        HK_INLINE ExternStorage()   { m_nodes=HK_NULL;m_firstFree=0;m_capacity=0; }
        HK_INLINE ExternStorage(_In_bytecount_(size) void* buffer, int size){ setExternBuffer(buffer,size); }
        HK_INLINE _Ret_maybenull_ const Node* getNodesBuffer() const { return m_nodes; }
        HK_INLINE _Ret_maybenull_ Node* getNodesBuffer() { return m_nodes; }
        HK_INLINE int           getNodeCapacity() const             { return (int) m_capacity; }
        HK_INLINE void      exchange(ExternStorage& other)      { hkMath::swap(m_nodes,other.m_nodes); hkMath::swap(m_firstFree,other.m_firstFree); hkMath::swap(m_capacity,other.m_capacity); }
        HK_INLINE Index     getFirstFreeNode() const            { return m_firstFree; }
        HK_INLINE Index     getNextFreeNode(Index node) const   { return (const Index&)m_nodes[(int)node]; }
        HK_NEVER_INLINE void        releaseAll()
        {
            hkString::memSet(m_nodes, 0, sizeof(Node) * m_capacity);
            for (int i = 1; i < (m_capacity-1); ++i) ((Index&)m_nodes[i]) = (Index)(i+1);
            m_firstFree                             = 1;
            ((Index&)m_nodes[(int)(m_capacity-1)])  = 0;
        }
        HK_INLINE Index     allocateNode()
        {
            HK_COMPILE_TIME_ASSERT(sizeof(Node) >= sizeof(Index));
            HK_ASSERT(0x6CFD82A7, m_firstFree, "Out of memory");
            Index newNode   = m_firstFree;
            m_firstFree     = (Index&)m_nodes[(int)m_firstFree];
            return newNode;
        }
        HK_INLINE Index     tryAllocateNode(hkResult & res)
        {
            HK_COMPILE_TIME_ASSERT(sizeof(Node) >= sizeof(Index));
            if (HK_VERY_UNLIKELY(!m_firstFree))
            {
                res = HK_FAILURE;
                return;
            }
            else
            {
                res = HK_SUCCESS;
            }
            Index newNode = m_firstFree;
            m_firstFree = (Index&)m_nodes[(int)m_firstFree];
            return newNode;
        }
        HK_INLINE void      releaseNode(Index node)
        {
            HK_ASSERT(0xC53DB6F6, node, "Null node");
            ((Index&)m_nodes[(int)node])    = m_firstFree;
            m_firstFree             = node;
        }
        HK_INLINE void setExternBuffer(_In_bytecount_(size) void* buffer, int size)
        {
            m_nodes     = (Node*)buffer;
            m_capacity  = hkUint32(size/sizeof(Node));
            releaseAll();
        }

        Node*                   m_nodes;            ///< Nodes.
        Index                   m_firstFree;        ///< First free node.
        hkUint32                m_capacity;         ///< Capacity.
    };

    ///
    /// Dynamic storage.
    ///
    template <int GROWTH, typename METRIC, typename CODEC>
    struct  DynamicStorage : public METRIC
    {
        HK_DECLARE_CLASS(DynamicStorage, New, Reflect);

        typedef METRIC                  Metric;     ///< Metric.
        typedef CODEC                   Node;       ///< Node.
        typedef typename CODEC::Index   Index;      ///< Index.
        typedef typename CODEC::PackAabb    PackAabb;   ///< Storage AABB.
        typedef typename CODEC::RawAabb     RawAabb;    ///< Processing AABB.

        HK_INLINE               DynamicStorage()                        { releaseAll(); }
        HK_INLINE _Ret_maybenull_ const Node* getNodesBuffer() const { return m_nodes.begin(); }
        HK_INLINE _Ret_maybenull_ Node* getNodesBuffer() { return m_nodes.begin(); }
        HK_INLINE int           getNodeCapacity() const                 { return (int) m_nodes.getSize(); }
        HK_INLINE void      exchange(DynamicStorage& other)         { m_nodes.swap(other.m_nodes); hkMath::swap(m_firstFree,other.m_firstFree); }
        HK_INLINE void      releaseAll()                            { m_nodes.clearAndDeallocate(); m_firstFree=0; }
        HK_INLINE Index     getFirstFreeNode() const                { return m_firstFree; }
        HK_INLINE Index     getNextFreeNode(Index node) const       { return (const Index&)m_nodes[(int)node]; }
        HK_INLINE hkResult  reserveLeaves(int numAdditionalLeaves)  { return reserveNodes(numAdditionalLeaves * 2); }
        HK_NEVER_INLINE hkResult    reserveNodes(int numAdditionalNodes)
        {
            if(numAdditionalNodes)
            {
                hkResult resizeResult;
                const int   oldCapacity = hkMath::max2(1, m_nodes.getSize());
                if(GROWTH)
                {
                    resizeResult = m_nodes.reserveExactly(m_nodes.getSize() + numAdditionalNodes);
                }
                else
                {
                    resizeResult = m_nodes.reserve(m_nodes.getSize() + numAdditionalNodes + 1);
                }

                if(resizeResult.isFailure())
                {
                    return HK_FAILURE;
                }

                const int prevSize = m_nodes.getSize();

                m_nodes.expandBy(m_nodes.getCapacity() - m_nodes.getSize());

                hkString::memSet(m_nodes.begin() + prevSize, 0, sizeof(Node) * (m_nodes.getSize() - prevSize));

                HK_ASSERT(0x8DABC363, m_nodes.getSize() == (int)(Index)m_nodes.getSize(), "Node index overflow");

                for (int i = oldCapacity, last = m_nodes.getSize()-1; i <= last; ++i)
                {
                    ((Index&)m_nodes[i]) = i < last ? (Index)(i+1) : m_firstFree;
                }

                m_firstFree = (Index)oldCapacity;
            }

            return HK_SUCCESS;
        }
        HK_INLINE   Index       allocateNode()
        {
            HK_COMPILE_TIME_ASSERT(sizeof(Node) >= sizeof(Index));
            if ( !m_firstFree ) reserveNodes(GROWTH?GROWTH:1);

            Index newNode   = m_firstFree;
            m_firstFree     = (Index&)m_nodes[(int)m_firstFree];

            return newNode;
        }
            // Allocate as long as there's memory. Success/failure is output in "res".
        HK_INLINE Index tryAllocateNode(hkResult& res)
        {
            HK_COMPILE_TIME_ASSERT(sizeof(Node) >= sizeof(Index));
            if ( !m_firstFree )
            {
                res = this->reserveNodes(GROWTH?GROWTH:1);
                if (res.isFailure())
                    return 0;
            }
            else
            {
                res = HK_SUCCESS;
            }

            Index newNode   = m_firstFree;
            m_firstFree     = (Index&)m_nodes[(int)m_firstFree];

            return newNode;
        }

        HK_INLINE void      releaseNode(Index node)
        {
            HK_ASSERT(0xC53DB6F6, node, "Null node");

            ((Index&)m_nodes[(int)node])    =   m_firstFree;
            m_firstFree                     =   node;
        }

        hkArray<Node>   m_nodes;            ///< Nodes.
        Index           m_firstFree;        ///< First free node.
    };

    //
    // Storage specializations
    //

        /// Partial specialization.
    template <typename CODEC>
    struct DefaultDynamicStorage : public DynamicStorage<0, AnisotropicMetric, CODEC>
    {
        HK_DECLARE_CLASS(DefaultDynamicStorage, New, Reflect);

        HK_INLINE   DefaultDynamicStorage() : DynamicStorage<0, AnisotropicMetric, CODEC>() { }
    };

        /// Pointer size storage.
    typedef DefaultDynamicStorage<CodecRawUlong> DynamicStoragePtr;

        /// 32 bits storage data.
    typedef DefaultDynamicStorage<CodecRawUint> DynamicStorage32;

        /// 16 bits storage data.
    typedef DefaultDynamicStorage<Codec32> DynamicStorage16;

    /// Integer 16 bits storage data.
    typedef DefaultDynamicStorage<CodecInt16> DynamicStorageInt16;
}

HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DynamicStoragePtr, hkcdDynamicTree_DynamicStoragePtr);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DynamicStorage32, hkcdDynamicTree_DynamicStorage32);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DynamicStorage16, hkcdDynamicTree_DynamicStorage16);
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkcdDynamicTree::DynamicStorageInt16, hkcdDynamicTree_DynamicStorageInt16);

#include <Geometry/Internal/_Auto/TemplateTypes/hkcdDynamicTreeStorages_Types.inl>

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
