// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

#include <Geometry/Internal/DataStructures/DynamicTree/hkcdDynamicTreeCodecs.h>

namespace hkcdDynamicTree
{
    ///
    /// Centroid metric
    ///
    struct CentroidMetric
    {
        HK_DECLARE_CLASS(CentroidMetric, New, Reflect);

        typedef hkVector4               MetricData; ///< Metric data.

        CentroidMetric() {}

        template <typename AABB>
        HK_INLINE void metricInitialize(const AABB& aabb, MetricData& metricOut) const
        {
            aabb.getCenter2(metricOut);
        }

        template <typename Node>
        HK_INLINE   int  metricSelect(const MetricData& metricData, _In_ const Node* x, _In_ const Node* y) const
        {
            typename Node::RawAabb  b[2]; x->getAabb(b[0]); y->getAabb(b[1]);
            hkVector4               cx,cy; b[0].getCenter2(cx); b[1].getCenter2(cy);

            cx.sub(metricData); cy.sub(metricData);

            return cx.lengthSquared<3>() > cy.lengthSquared<3>() ? 1 : 0;
        }
    };

    ///
    /// Balance metric, better than CentroidMetric but slightly slower to evaluate.
    ///
    struct BalanceMetric
    {
        HK_DECLARE_CLASS(BalanceMetric, New, Reflect, BypassCtor);

        typedef hkVector4               MetricData; ///< Metric data.

        BalanceMetric() {}

        template <typename AABB>
        HK_INLINE void metricInitialize(const AABB& aabb, MetricData& metricOut) const
        {
            aabb.getCenter2(metricOut);
        }

        template <typename Node>
        HK_INLINE   int  metricSelect(const MetricData& metricData, _In_ const Node* x, _In_ const Node* y) const
        {
            typename Node::RawAabb  b[2]; x->getAabb(b[0]); y->getAabb(b[1]);
            hkVector4               cx,cy; b[0].getCenter2(cx); b[1].getCenter2(cy);
            hkVector4   ex,ey; ex.setSub(b[0].m_max, b[0].m_min); ey.setSub(b[1].m_max, b[1].m_min);

            cx.sub(metricData); cy.sub(metricData);

            return cx.lengthSquared<3>() * ex.horizontalAdd<3>() > cy.lengthSquared<3>() * ey.horizontalAdd<3>() ? 1 : 0;
        }
    };

    ///
    /// Anisotropic metric, the best metric but requires more registers to be efficient (xbox360, PlayStation(R)3).
    ///
    struct HK_EXPORT_COMMON AnisotropicMetric
    {
        HK_DECLARE_CLASS(AnisotropicMetric, New, Reflect);

        /// Metric data.
        struct MetricData
        {
            hkVector4 m_center;
            hkVector4 m_extent;
        };

        AnisotropicMetric() {}

        template <typename AABB>
        HK_INLINE void metricInitialize(const AABB& aabb, MetricData& metricOut) const
        {
            aabb.getCenter2(metricOut.m_center);
            aabb.getExtents(metricOut.m_extent);
        }

        template <typename Node>
        HK_INLINE   int  metricSelect(const MetricData& metricData, _In_ const Node* HK_RESTRICT x, _In_ const Node* HK_RESTRICT y) const
        {
            typename Node::RawAabb  b[2]; x->getAabb(b[0]); y->getAabb(b[1]);
            hkVector4               cx,cy; b[0].getCenter2(cx); b[1].getCenter2(cy);
            hkVector4               ex,ey; b[0].getExtents(ex); b[1].getExtents(ey);

            cx.sub(metricData.m_center);    cy.sub(metricData.m_center);
            ex.add(metricData.m_extent);    ey.add(metricData.m_extent);

            return ((cx.lengthSquared<3>() * ex.horizontalAdd<3>()) > (cy.lengthSquared<3>() * ey.horizontalAdd<3>())) ? 1 : 0;
        }
    };
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
